/***************************************************************************

Universal board numbers (found on the schematics)

Cosmic Guerilla - 7907A
Cosmic Alien    - 7910
Magical Spot	- 8013
Magical Spot II - 8013
Devil Zone      - 8022

***************************************************************************/


#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/z80/z80.h"


PALETTE_INIT( panic );
PALETTE_INIT( cosmica );
PALETTE_INIT( cosmicg );
PALETTE_INIT( magspot2 );
PALETTE_INIT( nomnlnd );
VIDEO_UPDATE( panic );
VIDEO_UPDATE( magspot2 );
VIDEO_UPDATE( devzone );
VIDEO_UPDATE( cosmica );
VIDEO_UPDATE( cosmicg );
VIDEO_UPDATE( nomnlnd );
WRITE_HANDLER( cosmic_color_register_w );
WRITE_HANDLER( cosmic_background_enable_w );


static unsigned int pixel_clock = 0;



/* Schematics show 12 triggers for discrete sound circuits */

static WRITE_HANDLER( panic_sound_output_w )
{
    static int sound_enabled=1;

    /* Sound Enable / Disable */

    if (offset == 11)
    {
    	int count;
    	if (data == 0)
        	for(count=0; count<9; count++) sample_stop(count);

    	sound_enabled = data;
    }

    if (sound_enabled)
    {
        switch (offset)
        {
		case 0:  if (data) sample_start(0, 0, 0); break;  	/* Walk */
        case 1:  if (data) sample_start(0, 5, 0); break;  	/* Enemy Die 1 */
        case 2:  if (data)								  	/* Drop 1 */
				 {
					 if (!sample_playing(1))
					 {
						 sample_stop(2);
						 sample_start(1, 3, 0);
					 }
				 }
				 else
				 	sample_stop(1);
				 	break;

		case 3:	 if (data && !sample_playing(6))			/* Oxygen */
					sample_start(6, 9, 1);
                 break;

        case 4:  break;										/* Drop 2 */
        case 5:  if (data) sample_start(0, 5, 0); break;	/* Enemy Die 2 (use same sample as 1) */
        case 6:  if (data && !sample_playing(1) && !sample_playing(3))   /* Hang */
                 	sample_start(2, 2, 0);
                    break;

		case 7:  if (data) 									/* Escape */
				 {
					 sample_stop(2);
					 sample_start(3, 4, 0);
				 }
				 else
				 	 sample_stop(3);
                     break;

    	case 8:  if (data) sample_start(0, 1, 0); break;	/* Stairs */
    	case 9:  if (data)									/* Extend */
				 	sample_start(4, 8, 0);
				 else
					sample_stop(4);
	  			 break;

        case 10: DAC_data_w(0, data); break;				/* Bonus */
		case 15: if (data) sample_start(0, 6, 0); break;	/* Player Die */
		case 16: if (data) sample_start(5, 7, 0); break;	/* Enemy Laugh */
        case 17: if (data) sample_start(0, 10, 0); break;	/* Coin - Not triggered by software */
        }
    }

    #ifdef MAME_DEBUG
 	logerror("Sound output %x=%x\n",offset,data);
	#endif
}

WRITE_HANDLER( panic_sound_output2_w )
{
	panic_sound_output_w(offset+15, data);
}

WRITE_HANDLER( cosmicg_output_w )
{
	static int march_select;
    static int gun_die_select;
    static int sound_enabled;

    /* Sound Enable / Disable */

    if (offset == 12)
    {
	    int count;

    	sound_enabled = data;
    	if (data == 0)
        	for(count=0;count<9;count++) sample_stop(count);
    }

    if (sound_enabled)
    {
        switch (offset)
        {
		/* The schematics show a direct link to the sound amp  */
		/* as other cosmic series games, but it never seems to */
		/* be used for anything. It is implemented for sake of */
		/* completness. Maybe it plays a tune if you win ?     */
		case 1:  DAC_data_w(0, -data); break;
		case 2:  if (data) sample_start (0, march_select, 0); break;	/* March Sound */
		case 3:  march_select = (march_select & 0xfe) | data; break;
        case 4:  march_select = (march_select & 0xfd) | (data << 1); break;
        case 5:  march_select = (march_select & 0xfb) | (data << 2); break;

        case 6:  if (data)  							/* Killer Attack (crawly thing at bottom of screen) */
					sample_start(1, 8, 1);
				 else
					sample_stop(1);
				 break;

		case 7:  if (data)								/* Bonus Chance & Got Bonus */
				 {
					 sample_stop(4);
					 sample_start(4, 10, 0);
				 }
				 break;

		case 8:  if (data)
				 {
					 if (!sample_playing(4)) sample_start(4, 9, 1);
				 }
				 else
				 	sample_stop(4);
				 break;

		case 9:  if (data) sample_start(3, 11, 0); break;	/* Got Ship */
//		case 11: watchdog_reset_w(0, 0); break;				/* Watchdog */
		case 13: if (data) sample_start(8, 13-gun_die_select, 0); break;  /* Got Monster / Gunshot */
		case 14: gun_die_select = data; break;
		case 15: if (data) sample_start(5, 14, 0); break;	/* Coin Extend (extra base) */
        }
    }

	#ifdef MAME_DEBUG
 	if (offset != 11) logerror("Output %x=%x\n",offset,data);
    #endif
}

static INTERRUPT_GEN( panic_interrupt )
{
	if (cpu_getiloops() != 0)
	{
    	/* Coin insert - Trigger Sample */

        /* mostly not noticed since sound is */
		/* only enabled if game in progress! */

    	if ((input_port_3_r(0) & 0xc0) != 0xc0)
        	panic_sound_output_w(17,1);

		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xcf);	/* RST 08h */
    }
    else
    {
        cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xd7);	/* RST 10h */
    }
}

static INTERRUPT_GEN( cosmica_interrupt )
{
    pixel_clock = (pixel_clock + 2) & 0x3f;

    if (pixel_clock == 0)
    {
		if (readinputport(3) & 1)	/* Left Coin */
			cpu_set_irq_line(0, IRQ_LINE_NMI, PULSE_LINE);
    }
}

static INTERRUPT_GEN( cosmicg_interrupt )
{
    /* Insert Coin */

	/* R Nabet : fixed to make this piece of code sensible.
	I assumed that the interrupt request lasted for as long as the coin was "sensed".
	It makes sense and works fine, but I cannot be 100% sure this is correct,
	as I have no Cosmic Guerilla console :-) . */

	if ((readinputport(2) & 1)) /* Coin */
	{
		/* on tms9980, a 6 on the interrupt bus means level 4 interrupt */
		cpu_set_irq_line_and_vector(0, 0, ASSERT_LINE, 6);
	}
	else
	{
		cpu_set_irq_line(0, 0, CLEAR_LINE);
	}
}

static INTERRUPT_GEN( magspot2_interrupt )
{
	/* Coin 1 causes an IRQ, Coin 2 an NMI */
	if (input_port_4_r(0) & 0x01)
	{
  		cpu_set_irq_line(0, 0, HOLD_LINE);
	}
	else if (input_port_4_r(0) & 0x02)
	{
		cpu_set_irq_line(0, IRQ_LINE_NMI, PULSE_LINE);
	}
}

static INTERRUPT_GEN( nomnlnd_interrupt )
{
	/* Coin causes an NMI */
	if (input_port_4_r(0) & 0x01)
	{
		cpu_set_irq_line(0, IRQ_LINE_NMI, PULSE_LINE);
	}
}



static READ_HANDLER( cosmica_pixel_clock_r )
{
	return pixel_clock;
}

static READ_HANDLER( cosmicg_port_0_r )
{
	/* The top four address lines from the CRTC are bits 0-3 */

	return (input_port_0_r(0) & 0xf0) | ((cpu_getscanline() & 0xf0) >> 4);
}

static READ_HANDLER( magspot2_coinage_dip_r )
{
	return (input_port_5_r(0) & (1 << (7 - offset))) ? 0 : 1;
}


/* Has 8 way joystick, remap combinations to missing directions */

static READ_HANDLER( nomnlnd_port_0_1_r )
{
	int control;
    int fire = input_port_3_r(0);

	if (offset)
		control = input_port_1_r(0);
    else
		control = input_port_0_r(0);

    /* If firing - stop tank */

    if ((fire & 0xc0) == 0) return 0xff;

    /* set bit according to 8 way direction */

    if ((control & 0x82) == 0 ) return 0xfe;    /* Up & Left */
    if ((control & 0x0a) == 0 ) return 0xfb;    /* Down & Left */
    if ((control & 0x28) == 0 ) return 0xef;    /* Down & Right */
    if ((control & 0xa0) == 0 ) return 0xbf;    /* Up & Right */

    return control;
}



static WRITE_HANDLER( flip_screen_w )
{
	flip_screen_set(data&0x80);
}


static MEMORY_READ_START( panic_readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x4000, 0x5fff, MRA_RAM },
	{ 0x6800, 0x6800, input_port_0_r }, /* IN1 */
	{ 0x6801, 0x6801, input_port_1_r }, /* IN2 */
	{ 0x6802, 0x6802, input_port_2_r }, /* DSW */
	{ 0x6803, 0x6803, input_port_3_r }, /* IN0 */
MEMORY_END

static MEMORY_WRITE_START( panic_writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x5fff, MWA_RAM, &videoram, &videoram_size },
	{ 0x6000, 0x601f, MWA_RAM, &spriteram, &spriteram_size },
	{ 0x7000, 0x700b, panic_sound_output_w },
	{ 0x700c, 0x700e, cosmic_color_register_w },
	{ 0x700f, 0x700f, flip_screen_w },
	{ 0x7800, 0x7801, panic_sound_output2_w },
MEMORY_END


static MEMORY_READ_START( cosmica_readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
	{ 0x4000, 0x5fff, MRA_RAM },
	{ 0x6800, 0x6800, input_port_0_r }, /* IN1 */
	{ 0x6801, 0x6801, input_port_1_r }, /* IN2 */
	{ 0x6802, 0x6802, input_port_2_r }, /* DSW */
	{ 0x6803, 0x6803, cosmica_pixel_clock_r },
MEMORY_END

static MEMORY_WRITE_START( cosmica_writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x5fff, MWA_RAM, &videoram, &videoram_size },
	{ 0x6000, 0x601f, MWA_RAM ,&spriteram, &spriteram_size },
	{ 0x700c, 0x700d, cosmic_color_register_w },
	{ 0x700f, 0x700f, flip_screen_w },
MEMORY_END


static MEMORY_READ_START( cosmicg_readmem )
	{ 0x0000, 0x1fff, MRA_ROM },
	{ 0x2000, 0x3fff, MRA_RAM },
MEMORY_END

static MEMORY_WRITE_START( cosmicg_writemem )
	{ 0x0000, 0x1fff, MWA_ROM },
	{ 0x2000, 0x3fff, MWA_RAM, &videoram, &videoram_size },
MEMORY_END

static PORT_READ_START( cosmicg_readport )
	{ 0x00, 0x00, cosmicg_port_0_r },
	{ 0x01, 0x01, input_port_1_r },
PORT_END

static PORT_WRITE_START( cosmicg_writeport )
	{ 0x00, 0x15, cosmicg_output_w },
	{ 0x16, 0x17, cosmic_color_register_w },
PORT_END


static MEMORY_READ_START( magspot2_readmem )
	{ 0x0000, 0x2fff, MRA_ROM },
	{ 0x3800, 0x3807, magspot2_coinage_dip_r },
	{ 0x5000, 0x5000, input_port_0_r },
	{ 0x5001, 0x5001, input_port_1_r },
	{ 0x5002, 0x5002, input_port_2_r },
	{ 0x5003, 0x5003, input_port_3_r },
	{ 0x6000, 0x7fff, MRA_RAM },
MEMORY_END

static MEMORY_WRITE_START( magspot2_writemem )
	{ 0x0000, 0x2fff, MWA_ROM },
	{ 0x4000, 0x401f, MWA_RAM, &spriteram, &spriteram_size},
	{ 0x4800, 0x4800, DAC_0_data_w },
	{ 0x480c, 0x480d, cosmic_color_register_w },
	{ 0x480f, 0x480f, flip_screen_w },
	{ 0x6000, 0x7fff, MWA_RAM, &videoram, &videoram_size},
MEMORY_END


INPUT_PORTS_START( panic )
	PORT_START      /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 )

	PORT_START      /* IN2 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )

	PORT_START      /* DSW */
	PORT_DIPNAME( 0x07, 0x00, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_5C ) )
/* 0x06 and 0x07 disabled */
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x00, "3000" )
	PORT_DIPSETTING(    0x10, "5000" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x20, "4" )
	PORT_DIPNAME( 0xc0, 0x40, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x80, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( 1C_3C ) )

	PORT_START      /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN2 )

INPUT_PORTS_END

INPUT_PORTS_START( cosmica )
	PORT_START      /* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* IN2 */
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_2C ) )
  //PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x30, "5000" )
	PORT_DIPSETTING(    0x20, "10000" )
	PORT_DIPSETTING(    0x10, "15000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	/* The coin slots are not memory mapped.  Coin causes a NMI, */
	/* This fake input port is used by the interrupt */
	/* handler to be notified of coin insertions. We use IMPULSE to */
	/* trigger exactly one interrupt, without having to check when the */
	/* user releases the key. */

	PORT_START	/* FAKE */
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_HIGH, IPT_COIN1, 1 )

INPUT_PORTS_END

/* These are used for the CR handling - This can be used to */
/* from 1 to 16 bits from any bit offset between 0 and 4096 */

/* Offsets are in BYTES, so bits 0-7 are at offset 0 etc.   */

INPUT_PORTS_START( cosmicg )
	PORT_START /* 4-7 */
	PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_SPECIAL )	/* pixel clock */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )

	PORT_START /* 8-15 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_COCKTAIL)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_COCKTAIL)
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x10, "1000" )
	PORT_DIPSETTING(    0x20, "1500" )
	PORT_DIPSETTING(    0x30, "2000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x80, "5" )

	PORT_START      /* Hard wired settings */

	/* The coin slots are not memory mapped. Coin causes INT 4  */
	/* This fake input port is used by the interrupt handler 	*/
	/* to be notified of coin insertions. */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_COIN1 )

	/* This dip switch is not read by the program at any time   */
	/* but is wired to enable or disable the flip screen output */

	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Cocktail ) )

	/* This odd setting is marked as shown on the schematic,    */
	/* and again, is not read by the program, but wired into    */
	/* the watchdog circuit. The book says to leave it off      */

	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unused ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )

INPUT_PORTS_END


INPUT_PORTS_START( magspot2 )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_DIPNAME( 0xc0, 0x40, "Bonus Game" )
	PORT_DIPSETTING(    0x40, "5000" )
	PORT_DIPSETTING(    0x80, "10000" )
	PORT_DIPSETTING(    0xc0, "15000" )
	PORT_DIPSETTING(    0x00, "None" )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN2 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x01, "2000" )
	PORT_DIPSETTING(    0x02, "3000" )
	PORT_DIPSETTING(    0x03, "5000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x10, "4" )
	PORT_DIPSETTING(    0x18, "5" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Cocktail ) )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_VBLANK )
	PORT_BIT( 0x1e, IP_ACTIVE_LOW, IPT_UNUSED )		/* always HI */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SPECIAL )	/* reads what was written to 4808.  Probably not used?? */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	/* Fake port to handle coins */
	PORT_START	/* IN4 */
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_HIGH, IPT_COIN1, 1 )
	PORT_BIT_IMPULSE( 0x02, IP_ACTIVE_HIGH, IPT_COIN2, 1 )

	/* Fake port to handle coinage dip switches. Each bit goes to 3800-3807 */
	PORT_START	/* IN5 */
	PORT_DIPNAME( 0x0f, 0x00, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x0d, DEF_STR( 4C_2C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0x0e, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0x0f, DEF_STR( 4C_4C ) )
	PORT_DIPSETTING(    0x0a, DEF_STR( 3C_3C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x0b, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_5C ) )
	PORT_DIPNAME( 0xf0, 0x00, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x80, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0xd0, DEF_STR( 4C_2C ) )
	PORT_DIPSETTING(    0x50, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x90, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0xe0, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0xf0, DEF_STR( 4C_4C ) )
	PORT_DIPSETTING(    0xa0, DEF_STR( 3C_3C ) )
	PORT_DIPSETTING(    0x60, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0xb0, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x70, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_5C ) )
INPUT_PORTS_END

INPUT_PORTS_START( devzone )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN2 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x01, "4000" )
	PORT_DIPSETTING(    0x02, "6000" )
	PORT_DIPSETTING(    0x03, "8000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x0c, "Use Coin A & B" )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPSETTING(    0x10, "3" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Cocktail ) )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_VBLANK )
	PORT_BIT( 0x3e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	/* Fake port to handle coins */
	PORT_START	/* IN4 */
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_HIGH, IPT_COIN1, 1 )
	PORT_BIT_IMPULSE( 0x02, IP_ACTIVE_HIGH, IPT_COIN2, 1 )

	PORT_START	/* IN5 */
	PORT_DIPNAME( 0x0f, 0x00, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x0d, DEF_STR( 4C_2C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0x0e, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0x0f, DEF_STR( 4C_4C ) )
	PORT_DIPSETTING(    0x0a, DEF_STR( 3C_3C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x0b, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_5C ) )
	PORT_DIPNAME( 0xf0, 0x00, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x80, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0xd0, DEF_STR( 4C_2C ) )
	PORT_DIPSETTING(    0x50, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x90, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0xe0, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0xf0, DEF_STR( 4C_4C ) )
	PORT_DIPSETTING(    0xa0, DEF_STR( 3C_3C ) )
	PORT_DIPSETTING(    0x60, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0xb0, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x70, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_5C ) )
INPUT_PORTS_END


INPUT_PORTS_START( devzone2 )
	PORT_START	/* IN0 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN1 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0x1c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_COCKTAIL )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN2 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x01, "2000" )
	PORT_DIPSETTING(    0x02, "3000" )
	PORT_DIPSETTING(    0x03, "5000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x0c, "Use Coin A & B" )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPSETTING(    0x10, "3" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Cocktail ) )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_VBLANK )
	PORT_BIT( 0x3e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	/* Fake port to handle coins */
	PORT_START	/* IN4 */
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_HIGH, IPT_COIN1, 1 )
	PORT_BIT_IMPULSE( 0x02, IP_ACTIVE_HIGH, IPT_COIN2, 1 )

	PORT_START	/* IN5 */
	PORT_DIPNAME( 0x0f, 0x00, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x0d, DEF_STR( 4C_2C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0x0e, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0x0f, DEF_STR( 4C_4C ) )
	PORT_DIPSETTING(    0x0a, DEF_STR( 3C_3C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x0b, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_5C ) )
	PORT_DIPNAME( 0xf0, 0x00, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0xc0, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x80, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0xd0, DEF_STR( 4C_2C ) )
	PORT_DIPSETTING(    0x50, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x90, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0xe0, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0xf0, DEF_STR( 4C_4C ) )
	PORT_DIPSETTING(    0xa0, DEF_STR( 3C_3C ) )
	PORT_DIPSETTING(    0x60, DEF_STR( 2C_2C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0xb0, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x70, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x40, DEF_STR( 1C_5C ) )
INPUT_PORTS_END


INPUT_PORTS_START( nomnlnd )
	PORT_START	/* Controls - Remapped for game */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x55, IP_ACTIVE_LOW, IPT_SPECIAL )	/* diagonals */

	PORT_START	/* IN1 */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x55, IP_ACTIVE_LOW, IPT_SPECIAL )	/* diagonals */

	PORT_START	/* IN2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x01, "2000" )
	PORT_DIPSETTING(    0x02, "3000" )
	PORT_DIPSETTING(    0x03, "5000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
//	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x10, "5" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Cocktail ) )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_VBLANK )
	PORT_BIT( 0x1e, IP_ACTIVE_LOW, IPT_UNUSED )		/* always HI */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SPECIAL )	/* reads what was written to 4808.  Probably not used?? */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	/* Fake port to handle coin */
	PORT_START	/* IN4 */
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_HIGH, IPT_COIN1, 1 )
INPUT_PORTS_END


INPUT_PORTS_START( nomnlndg )
	PORT_START	/* Controls - Remapped for game */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x55, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN1 */
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x55, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* IN2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x01, "3000" )
	PORT_DIPSETTING(    0x02, "5000" )
	PORT_DIPSETTING(    0x03, "8000" )
	PORT_DIPSETTING(    0x00, "None" )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
//	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPSETTING(    0x10, "3" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Cocktail ) )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	/* IN3 */
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_VBLANK )
	PORT_BIT( 0x1e, IP_ACTIVE_LOW, IPT_UNUSED )		/* always HI */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SPECIAL )	/* reads what was written to 4808.  Probably not used?? */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	/* Fake port to handle coin */
	PORT_START	/* IN4 */
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_HIGH, IPT_COIN1, 1 )
INPUT_PORTS_END


static struct GfxLayout cosmic_spritelayout16 =
{
	16,16,
	RGN_FRAC(1,2),
	2,
	{ RGN_FRAC(0,2), RGN_FRAC(1,2) },
	{  0*8+0,  0*8+1,  0*8+2,  0*8+3,  0*8+4,  0*8+5,  0*8+6,  0*8+7,
	  16*8+0, 16*8+1, 16*8+2, 16*8+3, 16*8+4, 16*8+5, 16*8+6, 16*8+7},
	{ 0*8, 1*8,  2*8,  3*8,  4*8,  5*8,  6*8,  7*8,
	  8*8, 9*8, 10*8, 11*8, 12*8, 13*8, 14*8, 15*8 },
	32*8
};

static struct GfxLayout cosmic_spritelayout32 =
{
	32,32,
	RGN_FRAC(1,2),
	2,
	{ RGN_FRAC(0,2), RGN_FRAC(1,2) },
	{ 0*32*8+0, 0*32*8+1, 0*32*8+2, 0*32*8+3, 0*32*8+4, 0*32*8+5, 0*32*8+6, 0*32*8+7,
	  1*32*8+0, 1*32*8+1, 1*32*8+2, 1*32*8+3, 1*32*8+4, 1*32*8+5, 1*32*8+6, 1*32*8+7,
	  2*32*8+0, 2*32*8+1, 2*32*8+2, 2*32*8+3, 2*32*8+4, 2*32*8+5, 2*32*8+6, 2*32*8+7,
  	  3*32*8+0, 3*32*8+1, 3*32*8+2, 3*32*8+3, 3*32*8+4, 3*32*8+5, 3*32*8+6, 3*32*8+7 },
	{  0*8,  1*8,  2*8,  3*8,  4*8,  5*8,  6*8,  7*8,
	   8*8,  9*8, 10*8, 11*8, 12*8, 13*8, 14*8, 15*8,
	  16*8, 17*8, 18*8, 19*8, 20*8, 21*8, 22*8, 23*8,
	  24*8, 25*8, 26*8, 27*8, 28*8, 29*8, 30*8, 31*8 },
	128*8
};


static struct GfxDecodeInfo panic_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &cosmic_spritelayout16,  0, 8 },
	{ REGION_GFX1, 0, &cosmic_spritelayout32,  0, 8 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo cosmica_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &cosmic_spritelayout16,  0, 16 },
	{ REGION_GFX1, 0, &cosmic_spritelayout32,  0, 16 },
	{ -1 } /* end of array */
};


static struct DACinterface dac_interface =
{
	1,
	{ 100 }
};

static const char *panic_sample_names[] =
{
	"*panic",
	"walk.wav",
    "upordown.wav",
    "trapped.wav",
    "falling.wav",
    "escaping.wav",
	"ekilled.wav",
    "death.wav",
    "elaugh.wav",
    "extral.wav",
    "oxygen.wav",
    "coin.wav",
	0       /* end of array */
};

static struct Samplesinterface panic_samples_interface =
{
	9,	/* 9 channels */
	25,	/* volume */
	panic_sample_names
};

static const char *cosmicg_sample_names[] =
{
	"*cosmicg",
	"cg_m0.wav",	/* 8 Different pitches of March Sound */
	"cg_m1.wav",
	"cg_m2.wav",
	"cg_m3.wav",
	"cg_m4.wav",
	"cg_m5.wav",
	"cg_m6.wav",
	"cg_m7.wav",
	"cg_att.wav",	/* Killer Attack */
	"cg_chnc.wav",	/* Bonus Chance  */
	"cg_gotb.wav",	/* Got Bonus - have not got correct sound for */
	"cg_dest.wav",	/* Gun Destroy */
	"cg_gun.wav",	/* Gun Shot */
	"cg_gotm.wav",	/* Got Monster */
	"cg_ext.wav",	/* Coin Extend */
	0       /* end of array */
};

static struct Samplesinterface cosmicg_samples_interface =
{
	9,	/* 9 channels */
	25,	/* volume */
	cosmicg_sample_names
};


static MACHINE_DRIVER_START( cosmic )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", Z80,10816000/6)	/* 1.802 MHz*/

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(32*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 32*8-1, 4*8, 28*8-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( panic )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(cosmic)
	MDRV_CPU_MODIFY("main")

	MDRV_CPU_MEMORY(panic_readmem,panic_writemem)
	MDRV_CPU_VBLANK_INT(panic_interrupt,2)

	/* video hardware */
	MDRV_GFXDECODE(panic_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(16)
	MDRV_COLORTABLE_LENGTH(8*4)

	MDRV_PALETTE_INIT(panic)
	MDRV_VIDEO_UPDATE(panic)

	/* sound hardware */
	MDRV_SOUND_ADD(SAMPLES, panic_samples_interface)
	MDRV_SOUND_ADD(DAC, dac_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( cosmica )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(cosmic)
	MDRV_CPU_MODIFY("main")

	MDRV_CPU_MEMORY(cosmica_readmem,cosmica_writemem)
	MDRV_CPU_VBLANK_INT(cosmica_interrupt,32)

	/* video hardware */
	MDRV_GFXDECODE(cosmica_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(8)
	MDRV_COLORTABLE_LENGTH(16*4)

	MDRV_PALETTE_INIT(cosmica)
	MDRV_VIDEO_UPDATE(cosmica)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( cosmicg )

	/* basic machine hardware */
	MDRV_CPU_ADD(TMS9980, 1228500)
			/* 9.828 MHz Crystal */
			/* R Nabet : huh ? This would imply the crystal frequency is somehow divided by 2 before being
			fed to the tms9904 or tms9980.  Also, I have never heard of a tms9900/9980 operating under
			1.5MHz.  So, if someone can check this... */
	MDRV_CPU_MEMORY(cosmicg_readmem,cosmicg_writemem)
	MDRV_CPU_PORTS(cosmicg_readport,cosmicg_writeport)
	MDRV_CPU_VBLANK_INT(cosmicg_interrupt,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(0)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(32*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 32*8-1, 4*8, 28*8-1)
    MDRV_PALETTE_LENGTH(16)

	MDRV_PALETTE_INIT(cosmicg)
	MDRV_VIDEO_UPDATE(cosmicg)

	/* sound hardware */
	MDRV_SOUND_ADD(SAMPLES, cosmicg_samples_interface)
	MDRV_SOUND_ADD(DAC, dac_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( magspot2 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(cosmic)
	MDRV_CPU_MODIFY("main")

	MDRV_CPU_MEMORY(magspot2_readmem,magspot2_writemem)
	MDRV_CPU_VBLANK_INT(magspot2_interrupt,1)

	/* video hardware */
	MDRV_GFXDECODE(panic_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(16)
	MDRV_COLORTABLE_LENGTH(8*4)

	MDRV_PALETTE_INIT(magspot2)
	MDRV_VIDEO_UPDATE(magspot2)

	/* sound hardware */
	MDRV_SOUND_ADD(DAC, dac_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( devzone )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(magspot2)

	/* video hardware */
	MDRV_VIDEO_UPDATE(devzone)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( nomnlnd )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(cosmic)
	MDRV_CPU_MODIFY("main")

	MDRV_CPU_MEMORY(magspot2_readmem,magspot2_writemem)
	MDRV_CPU_VBLANK_INT(nomnlnd_interrupt,1)

	/* video hardware */
	MDRV_GFXDECODE(panic_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(8)
	MDRV_COLORTABLE_LENGTH(8*4)

	MDRV_PALETTE_INIT(nomnlnd)
	MDRV_VIDEO_UPDATE(nomnlnd)

	/* sound hardware */
	MDRV_SOUND_ADD(DAC, dac_interface)
MACHINE_DRIVER_END


ROM_START( panic )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "spe1",         0x0000, 0x0800, 0x70ac0888 )
	ROM_LOAD( "spe2",         0x0800, 0x0800, 0x2b910c48 )
	ROM_LOAD( "spe3",         0x1000, 0x0800, 0x03810148 )
	ROM_LOAD( "spe4",         0x1800, 0x0800, 0x119bbbfd )
	ROM_LOAD( "spcpanic.5",   0x2000, 0x0800, 0x5b80f277 )
	ROM_LOAD( "spcpanic.6",   0x2800, 0x0800, 0xb73babf0 )
	ROM_LOAD( "spe7",         0x3000, 0x0800, 0x2894106e )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "spcpanic.11",  0x0000, 0x0800, 0xacea9df4 )
	ROM_LOAD( "spcpanic.12",  0x0800, 0x0800, 0xe83423d0 )
	ROM_LOAD( "spcpanic.10",  0x1000, 0x0800, 0xc9631c2d )
	ROM_LOAD( "spcpanic.9",   0x1800, 0x0800, 0xeec78b4c )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.sp",    0x0000, 0x0020, 0x35d43d2f )

	ROM_REGION( 0x0800, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "spcpanic.8",   0x0000, 0x0800, 0x7da0b321 )
ROM_END

ROM_START( panic2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "spcpanic.1",   0x0000, 0x0800, 0x405ae6f9 )
	ROM_LOAD( "spcpanic.2",   0x0800, 0x0800, 0xb6a286c5 )
	ROM_LOAD( "spcpanic.3",   0x1000, 0x0800, 0x85ae8b2e )
	ROM_LOAD( "spcpanic.4",   0x1800, 0x0800, 0xb6d4f52f )
	ROM_LOAD( "spcpanic.5",   0x2000, 0x0800, 0x5b80f277 )
	ROM_LOAD( "spcpanic.6",   0x2800, 0x0800, 0xb73babf0 )
	ROM_LOAD( "spcpanic.7",   0x3000, 0x0800, 0xfc27f4e5 )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "spcpanic.11",  0x0000, 0x0800, 0xacea9df4 )
	ROM_LOAD( "spcpanic.12",  0x0800, 0x0800, 0xe83423d0 )
	ROM_LOAD( "spcpanic.10",  0x1000, 0x0800, 0xc9631c2d )
	ROM_LOAD( "spcpanic.9",   0x1800, 0x0800, 0xeec78b4c )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.sp",    0x0000, 0x0020, 0x35d43d2f )

	ROM_REGION( 0x0800, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "spcpanic.8",   0x0000, 0x0800, 0x7da0b321 )
ROM_END

ROM_START( panic3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "panica.1",     0x0000, 0x0800, 0x289720ce )
	ROM_LOAD( "spcpanic.2",   0x0800, 0x0800, 0xb6a286c5 )
	ROM_LOAD( "spcpanic.3",   0x1000, 0x0800, 0x85ae8b2e )
	ROM_LOAD( "spcpanic.4",   0x1800, 0x0800, 0xb6d4f52f )
	ROM_LOAD( "spcpanic.5",   0x2000, 0x0800, 0x5b80f277 )
	ROM_LOAD( "spcpanic.6",   0x2800, 0x0800, 0xb73babf0 )
	ROM_LOAD( "panica.7",     0x3000, 0x0800, 0x3641cb7f )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "spcpanic.11",  0x0000, 0x0800, 0xacea9df4 )
	ROM_LOAD( "spcpanic.12",  0x0800, 0x0800, 0xe83423d0 )
	ROM_LOAD( "spcpanic.10",  0x1000, 0x0800, 0xc9631c2d )
	ROM_LOAD( "spcpanic.9",   0x1800, 0x0800, 0xeec78b4c )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.sp",    0x0000, 0x0020, 0x35d43d2f )

	ROM_REGION( 0x0800, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "spcpanic.8",   0x0000, 0x0800, 0x7da0b321 )
ROM_END

ROM_START( panich )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "sph1",         0x0000, 0x0800, 0xf6e9c6ef )
	ROM_LOAD( "sph2",         0x0800, 0x0800, 0x58dbc49b )
	ROM_LOAD( "sph3",         0x1000, 0x0800, 0xc4f275ad )
	ROM_LOAD( "sph4",         0x1800, 0x0800, 0x6e7785de )
	ROM_LOAD( "sph5",         0x2000, 0x0800, 0x1916c9b8 )
	ROM_LOAD( "sph6",         0x2800, 0x0800, 0x54b92314 )
	ROM_LOAD( "sph7",         0x3000, 0x0800, 0x8600b881 )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "spcpanic.11",  0x0000, 0x0800, 0xacea9df4 )
	ROM_LOAD( "spcpanic.12",  0x0800, 0x0800, 0xe83423d0 )
	ROM_LOAD( "spcpanic.10",  0x1000, 0x0800, 0xc9631c2d )
	ROM_LOAD( "spcpanic.9",   0x1800, 0x0800, 0xeec78b4c )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.sp",    0x0000, 0x0020, 0x35d43d2f )

	ROM_REGION( 0x0800, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "spcpanic.8",   0x0000, 0x0800, 0x7da0b321 )
ROM_END

ROM_START( panicger )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "spacepan.001", 0x0000, 0x0800, 0xa6d9515a )
	ROM_LOAD( "spacepan.002", 0x0800, 0x0800, 0xcfc22663 )
	ROM_LOAD( "spacepan.003", 0x1000, 0x0800, 0xe1f36893 )
	ROM_LOAD( "spacepan.004", 0x1800, 0x0800, 0x01be297c )
	ROM_LOAD( "spacepan.005", 0x2000, 0x0800, 0xe0d54805 )
	ROM_LOAD( "spacepan.006", 0x2800, 0x0800, 0xaae1458e )
	ROM_LOAD( "spacepan.007", 0x3000, 0x0800, 0x14e46e70 )

	ROM_REGION( 0x2000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "spcpanic.11",  0x0000, 0x0800, 0xacea9df4 )
	ROM_LOAD( "spcpanic.12",  0x0800, 0x0800, 0xe83423d0 )
	ROM_LOAD( "spcpanic.10",  0x1000, 0x0800, 0xc9631c2d )
	ROM_LOAD( "spcpanic.9",   0x1800, 0x0800, 0xeec78b4c )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "82s123.sp",    0x0000, 0x0020, 0x35d43d2f )

	ROM_REGION( 0x0800, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "spcpanic.8",   0x0000, 0x0800, 0x7da0b321 )
ROM_END

ROM_START( cosmica )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "ca.e3",        0x0000, 0x0800, 0x535ee0c5 )
	ROM_LOAD( "ca.e4",        0x0800, 0x0800, 0xed3cf8f7 )
	ROM_LOAD( "ca.e5",        0x1000, 0x0800, 0x6a111e5e )
	ROM_LOAD( "ca.e6",        0x1800, 0x0800, 0xc9b5ca2a )
	ROM_LOAD( "ca.e7",        0x2000, 0x0800, 0x43666d68 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "ca.n2",        0x0000, 0x0800, 0xaa6c6079 )
	ROM_LOAD( "ca.n1",        0x0800, 0x0800, 0x431e866c )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "ca.d9",        0x0000, 0x0020, 0xdfb60f19 )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "ca.e2",        0x0000, 0x0400, 0xea4ee931 )

	ROM_REGION( 0x0400, REGION_USER2, 0 ) /* starfield generator */
	ROM_LOAD( "ca.sub",       0x0000, 0x0400, 0xacbd4e98 )
ROM_END

ROM_START( cosmica2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "ca.e3",        0x0000, 0x0800, 0x535ee0c5 )
	ROM_LOAD( "c3.bin",       0x0800, 0x0400, 0x699c849e )
	ROM_LOAD( "d4.bin",       0x0c00, 0x0400, 0x168e38da )
	ROM_LOAD( "ca.e5",        0x1000, 0x0800, 0x6a111e5e )
	ROM_LOAD( "ca.e6",        0x1800, 0x0800, 0xc9b5ca2a )
	ROM_LOAD( "i9.bin",       0x2000, 0x0400, 0x3bb57720 )
	ROM_LOAD( "j0.bin",       0x2400, 0x0400, 0x4ff70f45 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "ca.n2",        0x0000, 0x0800, 0xaa6c6079 )
	ROM_LOAD( "ca.n1",        0x0800, 0x0800, 0x431e866c )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "ca.d9",        0x0000, 0x0020, 0xdfb60f19 )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "ca.e2",        0x0000, 0x0400, 0xea4ee931 )

	ROM_REGION( 0x0400, REGION_USER2, 0 ) /* starfield generator */
	ROM_LOAD( "ca.sub",       0x0000, 0x0400, 0xacbd4e98 )
ROM_END

ROM_START( cosmicg )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )  /* 8k for code */
	ROM_LOAD( "cosmicg1.bin", 0x0000, 0x0400, 0xe1b9f894 )
	ROM_LOAD( "cosmicg2.bin", 0x0400, 0x0400, 0x35c75346 )
	ROM_LOAD( "cosmicg3.bin", 0x0800, 0x0400, 0x82a49b48 )
	ROM_LOAD( "cosmicg4.bin", 0x0c00, 0x0400, 0x1c1c934c )
	ROM_LOAD( "cosmicg5.bin", 0x1000, 0x0400, 0xb1c00fbf )
	ROM_LOAD( "cosmicg6.bin", 0x1400, 0x0400, 0xf03454ce )
	ROM_LOAD( "cosmicg7.bin", 0x1800, 0x0400, 0xf33ebae7 )
	ROM_LOAD( "cosmicg8.bin", 0x1c00, 0x0400, 0x472e4990 )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "cosmicg9.bin", 0x0000, 0x0400, 0x689c2c96 )
ROM_END

/* rom 9 not dumped according to readme? */
ROM_START( magspot )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "ms1.bin",	  0x0000, 0x0800, 0x59e9019d )
	ROM_LOAD( "ms2.bin",	  0x0800, 0x0800, 0x98b913b1 )
	ROM_LOAD( "ms3.bin",	  0x1000, 0x0800, 0xea58c124 )
	ROM_LOAD( "ms5.bin",	  0x1800, 0x0800, 0x4302a658 )
	ROM_LOAD( "ms4.bin",	  0x2000, 0x0800, 0x088582ab )
	ROM_LOAD( "ms6.bin",	  0x2800, 0x0800, 0xe6bf492c )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "ms8.bin",	  0x0000, 0x0800, 0x9e1d63a2 )
	ROM_LOAD( "ms7.bin",	  0x0800, 0x0800, 0x1ab338d3 )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "ms.d9",        0x0000, 0x0020, 0x36e2aa2a )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "ms.e2",        0x0000, 0x0400, 0x89f23ebd )
ROM_END

ROM_START( magspot2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "ms.e3",        0x0000, 0x0800, 0xc0085ade )
	ROM_LOAD( "ms.e4",        0x0800, 0x0800, 0xd534a68b )
	ROM_LOAD( "ms.e5",        0x1000, 0x0800, 0x25513b2a )
	ROM_LOAD( "ms.e7",        0x1800, 0x0800, 0x8836bbc4 )
	ROM_LOAD( "ms.e6",        0x2000, 0x0800, 0x6a08ab94 )
	ROM_LOAD( "ms.e8",        0x2800, 0x0800, 0x77c6d109 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "ms.n2",        0x0000, 0x0800, 0x9e1d63a2 )
	ROM_LOAD( "ms.n1",        0x0800, 0x0800, 0x1ab338d3 )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "ms.d9",        0x0000, 0x0020, 0x36e2aa2a )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "ms.e2",        0x0000, 0x0400, 0x89f23ebd )
ROM_END

ROM_START( devzone )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "dv1.e3",       0x0000, 0x0800, 0xc70faf00 )
	ROM_LOAD( "dv2.e4",       0x0800, 0x0800, 0xeacfed61 )
	ROM_LOAD( "dv3.e5",       0x1000, 0x0800, 0x7973317e )
	ROM_LOAD( "dv5.e7",       0x1800, 0x0800, 0xb71a3989 )
	ROM_LOAD( "dv4.e6",       0x2000, 0x0800, 0xa58c5b8c )
	ROM_LOAD( "dv6.e8",       0x2800, 0x0800, 0x3930fb67 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "dv8.n2",       0x0000, 0x0800, 0xda1cbec1 )
	ROM_LOAD( "dv7.n1",       0x0800, 0x0800, 0xe7562fcf )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "ms.d9",        0x0000, 0x0020, 0x36e2aa2a )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "dz9.e2",       0x0000, 0x0400, 0x693855b6 )

	ROM_REGION( 0x0800, REGION_USER2, 0 ) /* grid horizontal line positions */
	ROM_LOAD( "ic12.sub",     0x0000, 0x0800, 0xf61c1c45 )

	ROM_REGION( 0x0020, REGION_USER3, 0 ) /* grid vertical line positions */
	ROM_LOAD( "ic1.sub",      0x0000, 0x0020, 0xdf974878 )
ROM_END

ROM_START( devzone2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "p10_1.e3",     0x0000, 0x0800, 0x38bd45a4  )
	ROM_LOAD( "my4_2.e4",     0x0800, 0x0800, 0xe1637800  )
	ROM_LOAD( "ms6_3.e5",     0x1000, 0x0800, 0xc1952e2f  )
	ROM_LOAD( "mx6_5.e7",     0x1800, 0x0800, 0xc5394215  )
	ROM_LOAD( "my1_4.e6",     0x2000, 0x0800, 0x5d965d93  )
	ROM_LOAD( "mz7_6.e8",     0x2800, 0x0800, 0x8504e8c9  )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "my8_8.n2",     0x0000, 0x0800, 0x18abce02 )
	ROM_LOAD( "mx3_7.n1",     0x0800, 0x0800, 0xc089c9e3 )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "ms.d9",        0x0000, 0x0020, 0x36e2aa2a )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "dz9.e2",       0x0000, 0x0400, 0x693855b6 )

	ROM_REGION( 0x0800, REGION_USER2, 0 ) /* grid horizontal line positions */
	ROM_LOAD( "ic12.sub",     0x0000, 0x0800, 0xf61c1c45 )

	ROM_REGION( 0x0020, REGION_USER3, 0 ) /* grid vertical line positions */
	ROM_LOAD( "ic1.sub",      0x0000, 0x0020, 0xdf974878 )
ROM_END

ROM_START( nomnlnd )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "1.bin",        0x0000, 0x0800, 0xba117ba6 )
	ROM_LOAD( "2.bin",        0x0800, 0x0800, 0xe5ed654f )
	ROM_LOAD( "3.bin",        0x1000, 0x0800, 0x7fc42724 )
	ROM_LOAD( "5.bin",        0x1800, 0x0800, 0x9cc2f1d9 )
	ROM_LOAD( "4.bin",        0x2000, 0x0800, 0x0e8cd46a )
	ROM_LOAD( "6.bin",        0x2800, 0x0800, 0xba472ba5 )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "nml8.n2",      0x0000, 0x0800, 0x739009b4 )
	ROM_LOAD( "nml7.n1",      0x0800, 0x0800, 0xd08ed22f )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "nml.d9",       0x0000, 0x0020, 0x65e911f9 )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "nl9.e2",       0x0000, 0x0400, 0x9e05f14e )

	ROM_REGION( 0x0800, REGION_USER2, 0 ) /* tree + river */
	ROM_LOAD( "nl10.ic4",     0x0000, 0x0400, 0x5b13f64e )
	ROM_LOAD( "nl11.ic7",     0x0400, 0x0400, 0xe717b241 )
ROM_END

ROM_START( nomnlndg )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )	/* 64k for code */
	ROM_LOAD( "nml1.e3",      0x0000, 0x0800, 0xe212ed91 )
	ROM_LOAD( "nml2.e4",      0x0800, 0x0800, 0xf66ef3d8 )
	ROM_LOAD( "nml3.e5",      0x1000, 0x0800, 0xd422fc8a )
	ROM_LOAD( "nml5.e7",      0x1800, 0x0800, 0xd58952ac )
	ROM_LOAD( "nml4.e6",      0x2000, 0x0800, 0x994c9afb )
	ROM_LOAD( "nml6.e8",      0x2800, 0x0800, 0x01ed2d8c )

	ROM_REGION( 0x1000, REGION_GFX1, ROMREGION_DISPOSE )	/* sprites */
	ROM_LOAD( "nml8.n2",      0x0000, 0x0800, 0x739009b4 )
	ROM_LOAD( "nml7.n1",      0x0800, 0x0800, 0xd08ed22f )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "nml.d9",       0x0000, 0x0020, 0x65e911f9 )

	ROM_REGION( 0x0400, REGION_USER1, 0 ) /* color map */
	ROM_LOAD( "nl9.e2",       0x0000, 0x0400, 0x9e05f14e )

	ROM_REGION( 0x0800, REGION_USER2, 0 ) /* tree + river */
	ROM_LOAD( "nl10.ic4",     0x0000, 0x0400, 0x5b13f64e )
	ROM_LOAD( "nl11.ic7",     0x0400, 0x0400, 0xe717b241 )
ROM_END


static DRIVER_INIT( cosmicg )
{
	/* Program ROMs have data pins connected different from normal */

	offs_t offs;

    for (offs =0; offs < memory_region_length(REGION_CPU1); offs++)
	{
        data8_t scrambled = memory_region(REGION_CPU1)[offs];

        data8_t normal = (scrambled >> 3 & 0x11)
                       | (scrambled >> 1 & 0x22)
                       | (scrambled << 1 & 0x44)
                       | (scrambled << 3 & 0x88);

        memory_region(REGION_CPU1)[offs] = normal;
    }


    /* Patch to avoid crash - Seems like duff romcheck routine */
    /* I would expect it to be bitrot, but have two romsets    */
    /* from different sources with the same problem!           */
    memory_region(REGION_CPU1)[0x1e9e] = 0x04;
    memory_region(REGION_CPU1)[0x1e9f] = 0xc0;
}


static DRIVER_INIT( devzone )
{
	install_mem_write_handler(0, 0x4807, 0x4807, cosmic_background_enable_w);
}


static DRIVER_INIT( nomnlnd )
{
	install_mem_read_handler(0, 0x5000, 0x5001, nomnlnd_port_0_1_r);

	install_mem_write_handler(0, 0x4807, 0x4807, cosmic_background_enable_w);
}


GAMEX(1979, cosmicg,  0,       cosmicg,  cosmicg,  cosmicg, ROT270, "Universal", "Cosmic Guerilla", GAME_NO_COCKTAIL )
GAMEX(1979, cosmica,  0,       cosmica,  cosmica,  0,       ROT270, "Universal", "Cosmic Alien", GAME_NO_SOUND )
GAMEX(1979, cosmica2, cosmica, cosmica,  cosmica,  0,       ROT270, "Universal", "Cosmic Alien (older)", GAME_NO_SOUND )
GAME( 1980, panic,    0,       panic,    panic,    0,       ROT270, "Universal", "Space Panic (version E)" )
GAME( 1980, panic2,   panic,   panic,    panic,    0,       ROT270, "Universal", "Space Panic (set 2)" )
GAME( 1980, panic3,   panic,   panic,    panic,    0,       ROT270, "Universal", "Space Panic (set 3)" )
GAME( 1980, panich,   panic,   panic,    panic,    0,       ROT270, "Universal", "Space Panic (harder)" )
GAME( 1980, panicger, panic,   panic,    panic,    0,       ROT270, "Universal (ADP Automaten license)", "Space Panic (German)" )
GAMEX(1980, magspot,  0,	   magspot2, magspot2, 0,       ROT270, "Universal", "Magical Spot", GAME_IMPERFECT_SOUND )
GAMEX(1980, magspot2, 0,       magspot2, magspot2, 0,       ROT270, "Universal", "Magical Spot II", GAME_IMPERFECT_SOUND )
GAMEX(1980, devzone,  0,       devzone,  devzone,  devzone, ROT270, "Universal", "Devil Zone", GAME_IMPERFECT_SOUND )
GAMEX(1980, devzone2, devzone, devzone,  devzone2, devzone, ROT270, "Universal", "Devil Zone (easier)", GAME_IMPERFECT_SOUND )
GAMEX(1980, nomnlnd,  0,       nomnlnd,  nomnlnd,  nomnlnd, ROT270, "Universal", "No Man's Land", GAME_IMPERFECT_SOUND )
GAMEX(1980, nomnlndg, nomnlnd, nomnlnd,  nomnlndg, nomnlnd, ROT270, "Universal (Gottlieb license)", "No Man's Land (Gottlieb)", GAME_IMPERFECT_SOUND )
