/*
    Boogie Wings (aka The Great Ragtime Show)
    Data East, 1992

    PCB No: DE-0379-1

    CPU: DE102
    Sound: HuC6280A, YM2151, YM3012, OKI M6295 (x2)
    OSC: 32.220MHz, 28.000MHz
    DIPs: 8 position (x2)
    PALs: (not dumped) VF-01 (PAL16L8)
                    VF-02 (22CV10P)
                    VF-03 (PAL16L8)
                    VF-04 (PAL 16L8)
                    VF-05 (PAL 16L8)

    PROM: MB7122 (compatible to 82S137, located near MBD-09)

    RAM: 62256 (x2), 6264 (x5)

    Data East Chips:   52 (x2)
                    141 (x2)
                    102
                    104
                    113
                    99
                    71 (x2)
                    200

    ROMs:
    kn00-2.2b   27c2001   \
    kn01-2.4b   27c2001    |  Main program
    kn02-2.2e   27c2001    |
    kn03-2.4e   27c2001   /
    kn04.8e     27c512    \
    kn05.9e     27c512    /   near 141's and MBD-00, MBD-01 and MBD-02
    kn06.18p    27c512        Sound Program
    mbd-00.8b   16M
    mbd-01.9b   16M
    mbd-02.10e  4M
    mbd-03.13b  16M
    mbd-04.14b  16M
    mbd-05.16b  16M
    mbd-06.17b  16M
    mbd-07.18b  16M
    mbd-08.19b  16M
    mbd-09.16p  4M         Oki Samples
    mbd-10.17p  4M         Oki Samples


    Driver by Bryan McPhail and David Haywood.

    Todo:
        Sprite priorities aren't verified to be 100% accurate.
        There may be some kind of fullscreen palette effect (controlled by bit 3 in priority
        word - used at end of each level, and on final boss).
        A shadow effect (used in level 1) is not implemented.
*/

#include "driver.h"
#include "cpu/m68000/m68000.h"
#include "cpu/h6280/h6280.h"
#include "decocrpt.h"
#include "vidhrdw/generic.h"
#include "deco16ic.h"
#include "decoprot.h"

extern void deco102_decrypt(int region, int address_xor, int data_select_xor, int opcode_select_xor);
VIDEO_START(boogwing);
VIDEO_UPDATE(boogwing);

/**********************************************************************************/

static MEMORY_READ16_START( boogwing_readmem )
	{ 0x000000, 0x0fffff, MRA16_ROM },
	{ 0x200000, 0x20ffff, MRA16_RAM },

	{ 0x242000, 0x2427ff, MRA16_RAM },
	{ 0x246000, 0x2467ff, MRA16_RAM },

	{ 0x24e6c0, 0x24e6c1, input_port_1_word_r },
	{ 0x24e138, 0x24e139, input_port_0_word_r },
	{ 0x24e344, 0x24e345, input_port_2_word_r },
	{ 0x24e000, 0x24e7ff, deco16_104_prot_r }, /* Protection device */

	{ 0x264000, 0x265fff, MRA16_RAM },
	{ 0x266000, 0x267fff, MRA16_RAM },
	{ 0x268000, 0x268fff, MRA16_RAM },
	{ 0x26a000, 0x26afff, MRA16_RAM },

	{ 0x274000, 0x275fff, MRA16_RAM },
	{ 0x276000, 0x277fff, MRA16_RAM },
	{ 0x278000, 0x278fff, MRA16_RAM },
	{ 0x27a000, 0x27afff, MRA16_RAM },

	{ 0x284000, 0x285fff, MRA16_RAM },

	{ 0x3c0000, 0x3c004f, MRA16_RAM },
MEMORY_END

static MEMORY_WRITE16_START( boogwing_writemem )
	{ 0x000000, 0x0fffff, MWA16_ROM },
	{ 0x200000, 0x20ffff, MWA16_RAM },

	{ 0x220000, 0x220001, deco16_priority_w }, /* Priority */
	{ 0x220002, 0x22000f, MWA16_NOP }, /* ? */

	{ 0x240000, 0x240001, buffer_spriteram16_w }, /* Triggers DMA for spriteram */
	{ 0x242000, 0x2427ff, MWA16_RAM, &spriteram16, &spriteram_size },
	{ 0x244000, 0x244001, buffer_spriteram16_2_w }, /* Triggers DMA for spriteram */
	{ 0x246000, 0x2467ff, MWA16_RAM, &spriteram16_2, &spriteram_2_size },

	{ 0x24e000, 0x24e7ff, deco16_104_prot_w, &deco16_prot_ram }, /* Protection writes */

	{ 0x260000, 0x26000f, MWA16_RAM, &deco16_pf12_control },
	{ 0x264000, 0x265fff, deco16_pf1_data_w, &deco16_pf1_data },
	{ 0x266000, 0x267fff, deco16_pf2_data_w, &deco16_pf2_data },
	{ 0x268000, 0x268fff, MWA16_RAM, &deco16_pf1_rowscroll },
	{ 0x26a000, 0x26afff, MWA16_RAM, &deco16_pf2_rowscroll },

	{ 0x270000, 0x27000f, MWA16_RAM, &deco16_pf34_control },
	{ 0x274000, 0x275fff, deco16_pf3_data_w, &deco16_pf3_data },
	{ 0x276000, 0x277fff, deco16_pf4_data_w, &deco16_pf4_data },
	{ 0x278000, 0x278fff, MWA16_RAM, &deco16_pf3_rowscroll },
	{ 0x27a000, 0x27afff, MWA16_RAM, &deco16_pf4_rowscroll },

	{ 0x280000, 0x28000f, MWA16_NOP }, /* ? */
	{ 0x282000, 0x282001, MWA16_NOP }, /* Palette setup? */
	{ 0x282008, 0x282009, deco16_palette_dma_w },
	{ 0x284000, 0x285fff, deco16_nonbuffered_palette_w, &paletteram16 },

	{ 0x3c0000, 0x3c004f, MWA16_RAM }, /* Main ram */
MEMORY_END



static MEMORY_READ_START( sound_readmem )
	{ 0x000000, 0x00ffff, MRA_ROM },
	{ 0x100000, 0x100001, MRA_NOP },
	{ 0x110000, 0x110001, YM2151_status_port_0_r },
	{ 0x120000, 0x120001, OKIM6295_status_0_r },
	{ 0x130000, 0x130001, OKIM6295_status_1_r },
	{ 0x140000, 0x140001, soundlatch_r },
	{ 0x1f0000, 0x1f1fff, MRA_BANK8 },
MEMORY_END

static MEMORY_WRITE_START( sound_writemem )
	{ 0x000000, 0x00ffff, MWA_ROM },
	{ 0x100000, 0x100001, MWA_NOP },
	{ 0x110000, 0x110001, YM2151_word_0_w },
	{ 0x120000, 0x120001, OKIM6295_data_0_w },
	{ 0x130000, 0x130001, OKIM6295_data_1_w },
	{ 0x1f0000, 0x1f1fff, MWA_BANK8 },
	{ 0x1fec00, 0x1fec01, H6280_timer_w },
	{ 0x1ff402, 0x1ff403, H6280_irq_status_w },
MEMORY_END

/**********************************************************************************/

INPUT_PORTS_START( boogwing )
	PORT_START	/* 16bit */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, IPT_VBLANK )

	PORT_START	/* 16bit */
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0028, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unknown ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0300, 0x0300, DEF_STR( Lives ) )
	PORT_DIPSETTING(      0x0100, "1" )
	PORT_DIPSETTING(      0x0300, "2" )
	PORT_DIPSETTING(      0x0200, "3" )
	PORT_DIPSETTING(      0x0000, "4" )
	PORT_DIPNAME( 0x0c00, 0x0c00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0800, "Easy" )
	PORT_DIPSETTING(      0x0c00, "Normal" )
	PORT_DIPSETTING(      0x0400, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x1000, 0x1000, "Coin Slots" )
	PORT_DIPSETTING(      0x1000, "Common" )
	PORT_DIPSETTING(      0x0000, "Individual" )
	PORT_DIPNAME( 0x2000, 0x2000, DEF_STR( Unknown ) )
	PORT_DIPSETTING(      0x2000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x4000, 0x4000, DEF_STR( Unknown ) )
	PORT_DIPSETTING(      0x4000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x8000, 0x0000, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x8000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	/* 16bit */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_START2 )
INPUT_PORTS_END

/**********************************************************************************/

static struct GfxLayout tile_8x8_layout =
{
	8,8,
	RGN_FRAC(1,2),
	4,
	{ RGN_FRAC(1,2)+8,RGN_FRAC(1,2)+0,RGN_FRAC(0,2)+8,RGN_FRAC(0,2)+0 },
	{ 0, 1, 2, 3, 4, 5, 6, 7 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16 },
	8*16
};

static struct GfxLayout tile_16x16_layout_5bpp =
{
	16,16,
	RGN_FRAC(1,3),
	5,
	{ RGN_FRAC(2,3), RGN_FRAC(1,3)+8,RGN_FRAC(1,3)+0,RGN_FRAC(0,3)+8,RGN_FRAC(0,3)+0 },
	{ 256,257,258,259,260,261,262,263,0, 1, 2, 3, 4, 5, 6, 7 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16,8*16,9*16,10*16,11*16,12*16,13*16,14*16,15*16 },
	32*16
};

static struct GfxLayout tile_16x16_layout =
{
	16,16,
	RGN_FRAC(1,2),
	4,
	{ RGN_FRAC(1,2)+8,RGN_FRAC(1,2)+0,RGN_FRAC(0,2)+8,RGN_FRAC(0,2)+0 },
	{ 256,257,258,259,260,261,262,263,0, 1, 2, 3, 4, 5, 6, 7 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16,8*16,9*16,10*16,11*16,12*16,13*16,14*16,15*16 },
	32*16
};

static struct GfxLayout spritelayout =
{
	16,16,
	RGN_FRAC(1,1),
	4,
	{ 24,8,16,0 },
	{ 512,513,514,515,516,517,518,519, 0, 1, 2, 3, 4, 5, 6, 7 },
	{ 0*32, 1*32, 2*32, 3*32, 4*32, 5*32, 6*32, 7*32,
	  8*32, 9*32,10*32,11*32,12*32,13*32,14*32,15*32},
	32*32
};


static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &tile_8x8_layout,            0, 16 },	/* Tiles (8x8) */
	{ REGION_GFX2, 0, &tile_16x16_layout_5bpp, 0x100, 16 },	/* Tiles (16x16) */
	{ REGION_GFX3, 0, &tile_16x16_layout,      0x300, 32 },	/* Tiles (16x16) */
	{ REGION_GFX4, 0, &spritelayout,           0x500, 32 },	/* Sprites (16x16) */
	{ REGION_GFX5, 0, &spritelayout,           0x700, 16 },	/* Sprites (16x16) */
	{ -1 } /* end of array */
};

/**********************************************************************************/

static void sound_irq(int state)
{
	cpu_set_irq_line(1,1,state); /* IRQ 2 */
}

static WRITE_HANDLER( sound_bankswitch_w )
{
	OKIM6295_set_bank_base(1, ((data & 2)>>1) * 0x40000);
	OKIM6295_set_bank_base(0, (data & 1) * 0x40000);
}

static struct YM2151interface ym2151_interface =
{
	1,
	32220000/9, /* Accurate, audio section crystal is 32.220 MHz */
	{ YM3012_VOL(80,MIXER_PAN_LEFT,80,MIXER_PAN_RIGHT) },
	{ sound_irq },
	{ sound_bankswitch_w }
};

static struct OKIM6295interface okim6295_interface =
{
	2,              /* 2 chips */
	{ 32220000/32/132, 32220000/16/132 },/* Frequency */
	{ REGION_SOUND1, REGION_SOUND2 },
	{ 100, 45 } /* Note!  Keep chip 1 (voices) louder than chip 2 */
};


static MACHINE_DRIVER_START( boogwing )
	/* basic machine hardware */
	MDRV_CPU_ADD(M68000, 14000000)	/* DE102 */
	MDRV_CPU_MEMORY(boogwing_readmem,boogwing_writemem)
	MDRV_CPU_VBLANK_INT(irq6_line_hold,1)

	MDRV_CPU_ADD(H6280,32220000/4)
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(sound_readmem, sound_writemem)

	MDRV_FRAMES_PER_SECOND(58)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_BUFFERS_SPRITERAM | VIDEO_RGB_DIRECT)
	MDRV_SCREEN_SIZE(40*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 40*8-1, 1*8, 31*8-1)

	MDRV_PALETTE_LENGTH(2048)
	MDRV_GFXDECODE(gfxdecodeinfo)

	MDRV_VIDEO_START(boogwing)
	MDRV_VIDEO_UPDATE(boogwing)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM2151, ym2151_interface)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

/**********************************************************************************/

ROM_START( boogwing )
	ROM_REGION( 0x100000, REGION_CPU1, 0 ) /* DE102 code (encrypted) */
	ROM_LOAD16_BYTE( "kn00-2.2b",    0x000000, 0x040000, 0xe38892b9 )
	ROM_LOAD16_BYTE( "kn02-2.2e",    0x000001, 0x040000, 0x8426efef )
	ROM_LOAD16_BYTE( "kn01-2.4b",    0x080000, 0x040000, 0x3ad4b54c )
	ROM_LOAD16_BYTE( "kn03-2.4e",    0x080001, 0x040000, 0x10b61f4a )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "kn06.18p",    0x00000, 0x10000, 0x3e8bc4e1 )

	ROM_REGION( 0x20000, REGION_GFX1, ROMREGION_DISPOSE ) /* Tiles 1 */
	ROM_LOAD16_BYTE( "kn05.9e",   0x00000, 0x010000, 0xd10aef95 )
	ROM_LOAD16_BYTE( "kn04.8e",   0x00001, 0x010000, 0x329323a8 )

	ROM_REGION( 0x300000, REGION_GFX2, ROMREGION_DISPOSE ) /* Tiles 2 */
	ROM_LOAD( "mbd-01.9b", 0x000000, 0x100000, 0xd7de4f4b )
	ROM_LOAD( "mbd-00.8b", 0x100000, 0x100000, 0xadb20ba9 )
	/* 0x100000 bytes expanded from mbd-02.10e copied here later */

	ROM_REGION( 0x200000, REGION_GFX3, ROMREGION_DISPOSE ) /* Tiles 3 */
	ROM_LOAD( "mbd-03.13b",   0x000000, 0x100000, 0xcf798f2c )
	ROM_LOAD( "mbd-04.14b",   0x100000, 0x100000, 0xd9764d0b )

	ROM_REGION( 0x400000, REGION_GFX4, ROMREGION_DISPOSE ) /* Sprites 1 */
	ROM_LOAD16_BYTE( "mbd-05.16b",    0x000001, 0x200000, 0x1768c66a )
	ROM_LOAD16_BYTE( "mbd-06.17b",    0x000000, 0x200000, 0x7750847a )

	ROM_REGION( 0x400000, REGION_GFX5, ROMREGION_DISPOSE ) /* Sprites 2 */
	ROM_LOAD16_BYTE( "mbd-07.18b",    0x000001, 0x200000, 0x241faac1 )
	ROM_LOAD16_BYTE( "mbd-08.19b",    0x000000, 0x200000, 0xf13b1e56 )

	ROM_REGION( 0x0100000, REGION_GFX6, ROMREGION_DISPOSE ) /* 1bpp graphics */
	ROM_LOAD16_BYTE( "mbd-02.10e",    0x000000, 0x080000, 0xb25aa721 )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 ) /* Oki samples 1 */
	ROM_LOAD( "mbd-10.17p",    0x000000, 0x080000, 0xf159f76a )

	ROM_REGION( 0x80000, REGION_SOUND2, 0 ) /* Oki samples 1 */
	ROM_LOAD( "mbd-09.16p",    0x000000, 0x080000, 0xf44f2f87 )

	ROM_REGION( 0x000400, REGION_PROMS, ROMREGION_DISPOSE ) /* Priority (not used) */
	ROM_LOAD( "kj-00.15n",    0x000000, 0x00400, 0xadd4d50b )
ROM_END

ROM_START( boogwina )
	ROM_REGION( 0x100000, REGION_CPU1, 0 ) /* DE102 code (encrypted) */
	ROM_LOAD16_BYTE( "km00-2.2b",    0x000000, 0x040000, 0x71ab71c6 )
	ROM_LOAD16_BYTE( "km02-2.2e",    0x000001, 0x040000, 0xe90f07f9 )
	ROM_LOAD16_BYTE( "km01-2.4b",    0x080000, 0x040000, 0x7fdce2d3 )
	ROM_LOAD16_BYTE( "km03-2.4e",    0x080001, 0x040000, 0x0b582de3 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "km06.18p",    0x00000, 0x10000, 0x3e8bc4e1 )

	ROM_REGION( 0x20000, REGION_GFX1, ROMREGION_DISPOSE ) /* Tiles 1 */
	ROM_LOAD16_BYTE( "kn05.9e",   0x00000, 0x010000, 0xd10aef95 )
	ROM_LOAD16_BYTE( "kn04.8e",   0x00001, 0x010000, 0x329323a8 )

	ROM_REGION( 0x300000, REGION_GFX2, ROMREGION_DISPOSE ) /* Tiles 2 */
	ROM_LOAD( "mbd-01.9b", 0x000000, 0x100000, 0xd7de4f4b )
	ROM_LOAD( "mbd-00.8b", 0x100000, 0x100000, 0xadb20ba9 )
	/* 0x100000 bytes expanded from mbd-02.10e copied here later */

	ROM_REGION( 0x200000, REGION_GFX3, ROMREGION_DISPOSE ) /* Tiles 3 */
	ROM_LOAD( "mbd-03.13b",   0x000000, 0x100000, 0xcf798f2c )
	ROM_LOAD( "mbd-04.14b",   0x100000, 0x100000, 0xd9764d0b )

	ROM_REGION( 0x400000, REGION_GFX4, ROMREGION_DISPOSE ) /* Sprites 1 */
	ROM_LOAD16_BYTE( "mbd-05.16b",    0x000001, 0x200000, 0x1768c66a )
	ROM_LOAD16_BYTE( "mbd-06.17b",    0x000000, 0x200000, 0x7750847a )

	ROM_REGION( 0x400000, REGION_GFX5, ROMREGION_DISPOSE ) /* Sprites 2 */
	ROM_LOAD16_BYTE( "mbd-07.18b",    0x000001, 0x200000, 0x241faac1 )
	ROM_LOAD16_BYTE( "mbd-08.19b",    0x000000, 0x200000, 0xf13b1e56 )

	ROM_REGION( 0x0100000, REGION_GFX6, ROMREGION_DISPOSE ) /* 1bpp graphics */
	ROM_LOAD16_BYTE( "mbd-02.10e",    0x000000, 0x080000, 0xb25aa721 )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 ) /* Oki samples 1 */
	ROM_LOAD( "mbd-10.17p",    0x000000, 0x080000, 0xf159f76a )

	ROM_REGION( 0x80000, REGION_SOUND2, 0 ) /* Oki samples 1 */
	ROM_LOAD( "mbd-09.16p",    0x000000, 0x080000, 0xf44f2f87 )

	ROM_REGION( 0x000400, REGION_PROMS, ROMREGION_DISPOSE ) /* Priority (not used) */
	ROM_LOAD( "kj-00.15n",    0x000000, 0x00400, 0xadd4d50b )
ROM_END

ROM_START( ragtime )
	ROM_REGION( 0x100000, REGION_CPU1, 0 ) /* DE102 code (encrypted) */
	ROM_LOAD16_BYTE( "kh00-1.2b",    0x000000, 0x040000, 0x553e179f )
	ROM_LOAD16_BYTE( "kh02-1.2e",    0x000001, 0x040000, 0x6c759ec0 )
	ROM_LOAD16_BYTE( "kh01-1.4b",    0x080000, 0x040000, 0x12dfee70 )
	ROM_LOAD16_BYTE( "kh03-1.4e",    0x080001, 0x040000, 0x076fea18 )
 
	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "km06.18p",    0x00000, 0x10000, 0x3e8bc4e1 )

	ROM_REGION( 0x20000, REGION_GFX1, ROMREGION_DISPOSE ) /* Tiles 1 */
	ROM_LOAD16_BYTE( "kn05.9e",   0x00000, 0x010000, 0xd10aef95 )
	ROM_LOAD16_BYTE( "kn04.8e",   0x00001, 0x010000, 0x329323a8 )

	ROM_REGION( 0x300000, REGION_GFX2, ROMREGION_DISPOSE ) /* Tiles 2 */
	ROM_LOAD( "mbd-01.9b", 0x000000, 0x100000, 0xd7de4f4b )
	ROM_LOAD( "mbd-00.8b", 0x100000, 0x100000, 0xadb20ba9 )
	/* 0x100000 bytes expanded from mbd-02.10e copied here later */

	ROM_REGION( 0x200000, REGION_GFX3, ROMREGION_DISPOSE ) /* Tiles 3 */
	ROM_LOAD( "mbd-03.13b",   0x000000, 0x100000, 0xcf798f2c )
	ROM_LOAD( "mbd-04.14b",   0x100000, 0x100000, 0xd9764d0b )

	ROM_REGION( 0x400000, REGION_GFX4, ROMREGION_DISPOSE ) /* Sprites 1 */
	ROM_LOAD16_BYTE( "mbd-05.16b",    0x000001, 0x200000, 0x1768c66a )
	ROM_LOAD16_BYTE( "mbd-06.17b",    0x000000, 0x200000, 0x7750847a )

	ROM_REGION( 0x400000, REGION_GFX5, ROMREGION_DISPOSE ) /* Sprites 2 */
	ROM_LOAD16_BYTE( "mbd-07.18b",    0x000001, 0x200000, 0x241faac1 )
	ROM_LOAD16_BYTE( "mbd-08.19b",    0x000000, 0x200000, 0xf13b1e56 )

	ROM_REGION( 0x0100000, REGION_GFX6, ROMREGION_DISPOSE ) /* 1bpp graphics */
	ROM_LOAD16_BYTE( "mbd-02.10e",    0x000000, 0x080000, 0xb25aa721 )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 ) /* Oki samples 1 */
	ROM_LOAD( "mbd-10.17p",    0x000000, 0x080000, 0xf159f76a )

	ROM_REGION( 0x80000, REGION_SOUND2, 0 ) /* Oki samples 1 */
	ROM_LOAD( "mbd-09.16p",    0x000000, 0x080000, 0xf44f2f87 )

	ROM_REGION( 0x000400, REGION_PROMS, ROMREGION_DISPOSE ) /* Priority (not used) */
	ROM_LOAD( "kj-00.15n",    0x000000, 0x00400, 0xadd4d50b )
ROM_END

static DRIVER_INIT( boogwing )
{
	const UINT8* src=memory_region(REGION_GFX6);
	UINT8* dst=memory_region(REGION_GFX2) + 0x200000;

	deco56_decrypt(REGION_GFX1);
	deco56_decrypt(REGION_GFX2);
	deco56_decrypt(REGION_GFX3);
	deco56_remap(REGION_GFX6);
	deco102_decrypt(REGION_CPU1, 0x42ba, 0x00, 0x18);
	memcpy(dst, src, 0x100000);
}

GAME( 1992, boogwing, 0,        boogwing, boogwing,  boogwing,  ROT0, "Data East Corporation", "Boogie Wings (Euro v1.5, 92.12.07)" )
GAME( 1992, boogwina, boogwing, boogwing, boogwing,  boogwing,  ROT0, "Data East Corporation", "Boogie Wings (Asia v1.5, 92.12.07)" )
GAME( 1992, ragtime,  boogwing, boogwing, boogwing,  boogwing,  ROT0, "Data East Corporation", "The Great Ragtime Show (Japan v1.5, 92.12.07)" )
