// Object_Image

#include "Object_Image.h"
#include "Object_Matrix.h"
#include "Object_List.h"
#include "Expr.h"
#include "PackedVar.h"

namespace AScript {

struct BitmapFileHeader {
	XPackedUShort_LE(bfType);
	XPackedULong_LE(bfSize);
	XPackedUShort_LE(bfReserved1);
	XPackedUShort_LE(bfReserved2);
	XPackedULong_LE(bfOffBits);
};

struct BitmapInfoHeader {
	XPackedULong_LE(biSize);
	XPackedLong_LE(biWidth);
	XPackedLong_LE(biHeight);
	XPackedUShort_LE(biPlanes);
	XPackedUShort_LE(biBitCount);
	XPackedULong_LE(biCompression);
	XPackedULong_LE(biSizeImage);
	XPackedLong_LE(biXPelsPerMeter);
	XPackedLong_LE(biYPelsPerMeter);
	XPackedULong_LE(biClrUsed);
	XPackedULong_LE(biClrImportant);
};

//-----------------------------------------------------------------------------
// Object_Image (common)
//-----------------------------------------------------------------------------
Value Object_Image::EvalGetter(Signal sig, const Symbol *pSymbol, bool &evaluatedFlag)
{
	evaluatedFlag = true;
	if (pSymbol->IsIdentical(AScript_Symbol(format))) {
		return Value(FormatToSymbol(_format));
	} else if (pSymbol->IsIdentical(AScript_Symbol(width))) {
		return Value(static_cast<unsigned int>(_width));
	} else if (pSymbol->IsIdentical(AScript_Symbol(height))) {
		return Value(static_cast<unsigned int>(_height));
	}
	evaluatedFlag = false;
	return Value::Null;
}

Value Object_Image::EvalSetter(Signal sig, const Symbol *pSymbol,
									const Value &value, bool &evaluatedFlag)
{
	return EvalGetter(sig, pSymbol, evaluatedFlag);
}

String Object_Image::ToString(Signal sig, bool exprFlag)
{
	String rtn;
	rtn = "<image:";
	rtn += FormatToSymbol(_format)->GetName();
	rtn += ":";
	if (IsValid()) {
		char buff[64];
		::sprintf(buff, "%dx%d", GetWidth(), GetHeight());
		rtn += buff;
	} else {
		rtn += "invalid";
	}
	rtn += ">";
	return rtn;
}

bool Object_Image::CheckEmpty(Signal sig) const
{
	if (!IsValid()) return true;
	sig.SetError(ERR_ValueError, "image has already been initialized with a buffer");
	return false;
}

bool Object_Image::CheckValidity(Signal sig) const
{
	if (IsValid()) return true;
	sig.SetError(ERR_ValueError, "image does not have a buffer");
	return false;
}

size_t Object_Image::SymbolToPixelOffset(Signal sig, const Symbol *pSymbol) const
{
	if (pSymbol->IsIdentical(AScript_Symbol(red))) {
		return IsWindowDIB()? 2 : 0;
	} else if (pSymbol->IsIdentical(AScript_Symbol(green))) {
		return 1;
	} else if (pSymbol->IsIdentical(AScript_Symbol(blue))) {
		return IsWindowDIB()? 0 : 2;
	} else if (pSymbol->IsIdentical(AScript_Symbol(alpha))) {
		if (GetFormat() != FORMAT_RGBA) {
			sig.SetError(ERR_ValueError, "only RGBA format contains alpha element");
			return 0;
		}
		return 3;
	}
	sig.SetError(ERR_ValueError, "unsupported image element: %s", pSymbol->GetName());
	return 0;
}

Object_Image::Format Object_Image::SymbolToFormat(Signal sig, const Symbol *pSymbol)
{
	if (pSymbol->IsIdentical(AScript_Symbol(rgb))) {
		return FORMAT_RGB;
	} else if (pSymbol->IsIdentical(AScript_Symbol(rgba))) {
		return FORMAT_RGBA;
	} else {
		sig.SetError(ERR_ValueError, "unsupported image format: %s", pSymbol->GetName());
		return FORMAT_None;
	}
}

const Symbol *Object_Image::FormatToSymbol(Format format)
{
	if (format == FORMAT_RGB) {
		return AScript_Symbol(rgb);
	} else if (format == FORMAT_RGBA) {
		return AScript_Symbol(rgba);
	} else {
		return AScript_Symbol(nil);
	}
}

void Object_Image::InitMetrics()
{
	_metrics.bitsPerPixel =
			(_format == FORMAT_RGB)? 24 :
			(_format == FORMAT_RGBA)? 32 : 0;
	_metrics.bytesPerPixel = _metrics.bitsPerPixel / 8;
	_metrics.bytesPerLine =
			(_format == FORMAT_RGB)?	((_width * 3 + 3) / 4) * 4 :
			(_format == FORMAT_RGBA)?	_width * 4 : 0;
}

bool Object_Image::CheckCoord(Signal sig, size_t x, size_t y) const
{
	if (CheckCoord(static_cast<int>(x), static_cast<int>(y))) return true;
	sig.SetError(ERR_ValueError, "coordinate is out of range");
	return false;
}

bool Object_Image::PutPixel(Signal sig, unsigned char *buff, const ValueList &valList)
{
	unsigned char *p = buff;
	if (_format == FORMAT_RGB) {
		if (valList.size() != 3) {
			sig.SetError(ERR_ValueError, "format RGB requires three elements in color");
			return false;
		}
#if defined(HAVE_WINDOWS_H)
		ValueList::const_reverse_iterator pValue = valList.rbegin();
#else
		ValueList::const_iterator pValue = valList.begin();
#endif
		*p++ = pValue->GetUChar(); pValue++;
		*p++ = pValue->GetUChar(); pValue++;
		*p++ = pValue->GetUChar(); pValue++;
	} else if (_format == FORMAT_RGBA) {
		if (valList.size() != 4) {
			sig.SetError(ERR_ValueError, "format RGBA requires four elements in color");
			return false;
		}
#if defined(HAVE_WINDOWS_H)
		ValueList::const_reverse_iterator pValue = valList.rbegin() + 1;
#else
		ValueList::const_iterator pValue = valList.begin();
#endif
		*p++ = pValue->GetUChar(); pValue++;
		*p++ = pValue->GetUChar(); pValue++;
		*p++ = pValue->GetUChar(); pValue++;
		*p = valList.back().GetUChar();
	}
	return true;
}

bool Object_Image::GetPixel(Signal sig, unsigned char *buff, ValueList &valList)
{
	unsigned char *p = buff;
	if (_format == FORMAT_RGB) {
#if defined(HAVE_WINDOWS_H)
		unsigned char *p = buff + 2;
		for (int i = 0; i < 3; i++) {
			valList.push_back(Value(*(p - i)));
		}
#else
		unsigned char *p = buff;
		for (int i = 0; i < 3; i++) {
			valList.push_back(Value(*(p + i)));
		}
#endif
	} else if (_format == FORMAT_RGBA) {
#if defined(HAVE_WINDOWS_H)
		unsigned char *p = buff + 2;
		for (int i = 0; i < 3; i++) {
			valList.push_back(Value(*(p - i)));
		}
#else
		unsigned char *p = buff;
		for (int i = 0; i < 3; i++) {
			valList.push_back(Value(*(p + i)));
		}
#endif
		valList.push_back(Value(*(buff + 3)));
	}
	return true;
}

bool Object_Image::Store(Signal sig, size_t x, size_t y, size_t width, size_t height,
					const Symbol *pSymbol, const Object_Matrix *pObjMat)
{
	if (pObjMat->RowSize() < height || pObjMat->ColSize() < width) {
		sig.SetError(ERR_ValueError, "matrix size is too small");
		return false;
	}
	size_t bytesPerLine = GetBytesPerLine();
	size_t bytesPerPixel = GetBytesPerPixel();
	unsigned char *pLine = GetPointer(x, y);
	size_t offPixel = SymbolToPixelOffset(sig, pSymbol);
	if (sig.IsSignalled()) return false;
	for (size_t iLine = 0; iLine < height; iLine++, pLine += bytesPerLine) {
		unsigned char *pPixel = pLine + offPixel;
		ValueList::const_iterator pValueElem = pObjMat->GetPointer(iLine, 0);
		for (size_t iPixel = 0; iPixel < width;
							iPixel++, pPixel += bytesPerPixel, pValueElem++) {
			*pPixel = pValueElem->GetUChar();
		}
	}
	return true;
}

bool Object_Image::Store(Signal sig, size_t x, size_t y, size_t width, size_t height,
					const Symbol *pSymbol, Iterator *pIterator)
{
	size_t bytesPerLine = GetBytesPerLine();
	size_t bytesPerPixel = GetBytesPerPixel();
	unsigned char *pLine = GetPointer(x, y);
	size_t offPixel = SymbolToPixelOffset(sig, pSymbol);
	if (sig.IsSignalled()) return false;
	for (size_t iLine = 0; iLine < height; iLine++, pLine += bytesPerLine) {
		unsigned char *pPixel = pLine + offPixel;
		for (size_t iPixel = 0; iPixel < width; iPixel++, pPixel += bytesPerPixel) {
			Value value;
			if (!pIterator->Next(sig, value)) return false;
			*pPixel = value.GetUChar();
		}
	}
	return true;
}

bool Object_Image::Extract(Signal sig, size_t x, size_t y, size_t width, size_t height,
					const Symbol *pSymbol, Object_Matrix *pObjMat)
{
	if (pObjMat->RowSize() < height || pObjMat->ColSize() < width) {
		sig.SetError(ERR_ValueError, "matrix size is too small");
		return false;
	}
	size_t bytesPerLine = GetBytesPerLine();
	size_t bytesPerPixel = GetBytesPerPixel();
	const unsigned char *pLine = GetPointer(x, y);
	size_t offPixel = SymbolToPixelOffset(sig, pSymbol);
	if (sig.IsSignalled()) return false;
	for (size_t iLine = 0; iLine < height; iLine++, pLine += bytesPerLine) {
		const unsigned char *pPixel = pLine + offPixel;
		ValueList::iterator pValueElem = pObjMat->GetPointer(iLine, 0);
		for (size_t iPixel = 0; iPixel < width;
							iPixel++, pPixel += bytesPerPixel, pValueElem++) {
			*pValueElem = Value(*pPixel);
		}
	}
	return true;
}

bool Object_Image::Extract(Signal sig, size_t x, size_t y, size_t width, size_t height,
					const Symbol *pSymbol, Object_List *pObjList)
{
	if (pObjList->GetList().size() < height * width) {
		sig.SetError(ERR_ValueError, "list size is too small");
		return false;
	}
	size_t bytesPerLine = GetBytesPerLine();
	size_t bytesPerPixel = GetBytesPerPixel();
	const unsigned char *pLine = GetPointer(x, y);
	size_t offPixel = SymbolToPixelOffset(sig, pSymbol);
	if (sig.IsSignalled()) return false;
	ValueList::iterator pValueElem = pObjList->GetList().begin();
	for (size_t iLine = 0; iLine < height; iLine++, pLine += bytesPerLine) {
		const unsigned char *pPixel = pLine + offPixel;
		for (size_t iPixel = 0; iPixel < width;
							iPixel++, pPixel += bytesPerPixel, pValueElem++) {
			*pValueElem = Value(*pPixel);
		}
	}
	return true;
}

bool Object_Image::FillRect(Signal sig, size_t x, size_t y,
						size_t width, size_t height, const ValueList &color)
{
	unsigned char buff[8];
	if (!PutPixel(sig, buff, color)) return false;
	size_t bytesPerLine = GetBytesPerLine();
	size_t bytesPerPixel = GetBytesPerPixel();
	unsigned char *pLine = GetPointer(x, y);
	for (size_t iLine = 0; iLine < height; iLine++, pLine += bytesPerLine) {
		unsigned char *pPixel = pLine;
		for (size_t iPixel = 0; iPixel < width; iPixel++, pPixel += bytesPerPixel) {
			::memcpy(pPixel, buff, bytesPerPixel);
		}
	}
	return true;
}

void SetError_InvalidBMPFormat(Signal sig)
{
	sig.SetError(ERR_IOError, "invalid BMP format");
}

bool Object_Image::ReadBMP(Signal sig, File &file)
{
	BitmapFileHeader bfh;
	BitmapInfoHeader bih;
	if (file.Read(&bfh, 14) < 14) {
		SetError_InvalidBMPFormat(sig);
		return false;
	}
	if (XUnpackUShort(bfh.bfType) != 0x4d42) {
		SetError_InvalidBMPFormat(sig);
		return false;
	}
	if (file.Read(&bih, 40) < 40) {
		SetError_InvalidBMPFormat(sig);
		return false;
	}
	unsigned long bfOffBits = XUnpackULong(bfh.bfOffBits);
	int biWidth = XUnpackLong(bih.biWidth);
	int biHeight = XUnpackLong(bih.biHeight);
	bool vertRevFlag = true;
	if (biHeight < 0) {
		biHeight = -biHeight;
		vertRevFlag = false;
	}
	if (!AllocBuffer(sig, biWidth, biHeight)) return false;
	unsigned short biBitCount = XUnpackUShort(bih.biBitCount);
	if (biBitCount == 1 || biBitCount == 4 || biBitCount == 8) {
		sig.SetError(ERR_IOError,
			"sorry, %d-bit depth of BMP format is not yet supported", biBitCount);
		return false;
	} else if (biBitCount == 24) {
		size_t bytesAlign = (3 * biWidth) % 4;
		if (bfOffBits != 0) file.Seek(bfOffBits, SEEK_SET);
		unsigned char buff[3];
		int iLine = 0, iPixel = 0;
		unsigned char *pLine = GetPointer(0, vertRevFlag? biHeight - 1 : 0);
		unsigned char *pPixel = pLine;
		int bytesPerLine = static_cast<int>(GetBytesPerLine());
		if (vertRevFlag) bytesPerLine = -bytesPerLine;
		while (file.Read(buff, 3) == 3) {
			if (iPixel >= biWidth) {
				iLine++;
				if (iLine >= biHeight) break;
				iPixel = 0;
				pLine += bytesPerLine;
				pPixel = pLine;
				file.Seek(static_cast<long>(bytesAlign), SEEK_CUR);
			}
#if defined(HAVE_WINDOWS_H)
			*(pPixel + 0) = buff[0];
			*(pPixel + 1) = buff[1];
			*(pPixel + 2) = buff[2];
#else
			*(pPixel + 0) = buff[2];
			*(pPixel + 1) = buff[1];
			*(pPixel + 2) = buff[0];
#endif
			pPixel += GetBytesPerPixel();
			iPixel++;
		}
	} else if (biBitCount == 32) {
		if (bfOffBits != 0) file.Seek(bfOffBits, SEEK_SET);
		unsigned char buff[4];
		int iLine = 0, iPixel = 0;
		unsigned char *pLine = GetPointer(0, vertRevFlag? biHeight - 1 : 0);
		unsigned char *pPixel = pLine;
		int bytesPerLine = static_cast<int>(GetBytesPerLine());
		if (vertRevFlag) bytesPerLine = -bytesPerLine;
		while (file.Read(buff, 4) == 4) {
			if (iPixel >= biWidth) {
				iLine++;
				if (iLine >= biHeight) break;
				iPixel = 0;
				pLine += bytesPerLine;
				pPixel = pLine;
			}
#if defined(HAVE_WINDOWS_H)
			*(pPixel + 0) = buff[0];
			*(pPixel + 1) = buff[1];
			*(pPixel + 2) = buff[2];
#else
			*(pPixel + 0) = buff[2];
			*(pPixel + 1) = buff[1];
			*(pPixel + 2) = buff[0];
#endif
			pPixel += GetBytesPerPixel();
			iPixel++;
		}
	} else {
		SetError_InvalidBMPFormat(sig);
		return false;
	}
	return true;
}

bool Object_Image::WriteBMP(Signal sig, File &file)
{
	BitmapFileHeader bfh;
	BitmapInfoHeader bih;
	::memset(&bfh, 0x00, 14);
	::memset(&bih, 0x00, 40);
	unsigned long bfOffBits = 14 + 40;
	int biWidth = static_cast<int>(GetWidth());
	int biHeight = static_cast<int>(GetHeight());
	int biBitCount = static_cast<int>(GetBitsPerPixel());
	unsigned long bfSize = static_cast<unsigned long>(GetBufferSize() + bfOffBits);
	XPackUShort(bfh.bfType,			0x4d42);
	XPackULong(bfh.bfSize,			bfSize);
	XPackUShort(bfh.bfOffBits,		bfOffBits);
	XPackULong(bih.biSize,			40);
	XPackULong(bih.biWidth,			biWidth);
	XPackULong(bih.biHeight,		biHeight);
	XPackUShort(bih.biPlanes,		1);
	XPackUShort(bih.biBitCount,		biBitCount);
	XPackULong(bih.biCompression,	0);
	XPackULong(bih.biSizeImage,		0);
	XPackULong(bih.biXPelsPerMeter,	3780);
	XPackULong(bih.biYPelsPerMeter,	3780);
	XPackULong(bih.biClrUsed,		0);
	XPackULong(bih.biClrImportant,	0);
	if (file.Write(&bfh, 14) < 14) {
		sig.SetError(ERR_IOError, "failed to write bitmap data");
		return false;
	}
	if (file.Write(&bih, 40) < 40) {
		sig.SetError(ERR_IOError, "failed to write bitmap data");
		return false;
	}
	if (biBitCount == 1 || biBitCount == 4 || biBitCount == 8) {
		sig.SetError(ERR_IOError,
			"sorry, %d-bit depth of BMP format is not yet supported", biBitCount);
		return false;
	} else if (biBitCount == 24) {
		size_t bytesAlign = (3 * biWidth) % 4;
		unsigned char buff[3];
		int iLine = 0, iPixel = 0;
		unsigned char *pLine = GetPointer(0, biHeight - 1);
		unsigned char *pPixel = pLine;
		int bytesPerLine = -static_cast<int>(GetBytesPerLine());
		for (;;) {
			if (iPixel >= biWidth) {
				iLine++;
				if (iLine >= biHeight) break;
				iPixel = 0;
				pLine += bytesPerLine;
				pPixel = pLine;
				if (file.Write("\x00\x00\x00\x00", bytesAlign) < bytesAlign) {
					sig.SetError(ERR_IOError, "failed to write bitmap data");
					return false;
				}
			}
#if defined(HAVE_WINDOWS_H)
			buff[0] = *(pPixel + 0);
			buff[1] = *(pPixel + 1);
			buff[2] = *(pPixel + 2);
#else
			buff[2] = *(pPixel + 0);
			buff[1] = *(pPixel + 1);
			buff[0] = *(pPixel + 2);
#endif
			if (file.Write(buff, 3) < 3) {
				sig.SetError(ERR_IOError, "failed to write bitmap data");
				return false;
			}
			pPixel += GetBytesPerPixel();
			iPixel++;
		}
	} else if (biBitCount == 32) {
		unsigned char buff[4];
		int iLine = 0, iPixel = 0;
		unsigned char *pLine = GetPointer(0, biHeight - 1);
		unsigned char *pPixel = pLine;
		int bytesPerLine = -static_cast<int>(GetBytesPerLine());
		for (;;) {
			if (iPixel >= biWidth) {
				iLine++;
				if (iLine >= biHeight) break;
				iPixel = 0;
				pLine += bytesPerLine;
				pPixel = pLine;
			}
#if defined(HAVE_WINDOWS_H)
			buff[0] = *(pPixel + 0);
			buff[1] = *(pPixel + 1);
			buff[2] = *(pPixel + 2);
#else
			buff[2] = *(pPixel + 0);
			buff[1] = *(pPixel + 1);
			buff[0] = *(pPixel + 2);
#endif
			buff[3] = *(pPixel + 3);
			if (file.Write(buff, 4) < 4) {
				sig.SetError(ERR_IOError, "failed to write bitmap data");
				return false;
			}
			pPixel += GetBytesPerPixel();
			iPixel++;
		}
	} else {
		SetError_InvalidBMPFormat(sig);
		return false;
	}
	return false;
}

#if defined(HAVE_WINDOWS_H)
//-----------------------------------------------------------------------------
// Object_Image (Windows DIB section)
//-----------------------------------------------------------------------------
Object_Image::Object_Image(Class *pClass, Format format) : Object(pClass),
		_format(format), _width(0), _height(0), _buff(NULL), _hBmp(NULL)
{
	InitMetrics();
}

Object_Image::Object_Image(const Object_Image &obj) : Object(obj),
		_format(obj._format), _width(0), _height(0), _buff(NULL), _hBmp(NULL)
{
	Signal sig;
	if (AllocBuffer(sig, obj._width, obj._height)) {
		::memcpy(_buff, obj._buff, GetBufferSize());
	}
}

Object_Image::~Object_Image()
{
	FreeBuffer();
}

Object *Object_Image::Clone() const
{
	return new Object_Image(*this);
}

bool Object_Image::IsWindowDIB() const
{
	return true;
}

bool Object_Image::AllocBuffer(Signal sig, size_t width, size_t height)
{
	FreeBuffer();
	if (width == 0 || height == 0) {
		sig.SetError(ERR_MemoryError, "failed to allocate image buffer");
		return false;
	}
	BITMAPINFO bmi;
	BITMAPINFOHEADER &hdr = bmi.bmiHeader;
	::memset(&bmi, 0, sizeof(BITMAPINFO));
	hdr.biSize = sizeof(BITMAPINFOHEADER);
	hdr.biWidth = static_cast<int>(width);
	hdr.biHeight = -static_cast<int>(height);
	hdr.biPlanes = 1;
	hdr.biBitCount =
		(_format == FORMAT_RGB)? 24 :
		(_format == FORMAT_RGBA)? 32 : 32;
	hdr.biCompression = BI_RGB;
	hdr.biSizeImage = 0;
	hdr.biXPelsPerMeter = 0;
	hdr.biYPelsPerMeter = 0;
	hdr.biClrUsed = 0;
	hdr.biClrImportant = 0;
	void *buff = NULL;
	_hBmp = ::CreateDIBSection(NULL, &bmi, DIB_RGB_COLORS, &buff, NULL, 0);
	if (_hBmp == NULL || buff == NULL) {
		sig.SetError(ERR_MemoryError, "failed to allocate image buffer");
		return false;
	}
	_width = width, _height = height;
	_buff = reinterpret_cast<unsigned char *>(buff);
	InitMetrics();
	return true;
}

void Object_Image::FreeBuffer()
{
	if (_hBmp != NULL) ::DeleteObject(_hBmp);
	_width = 0, _height = 0;
	_buff = NULL;
	_hBmp = NULL;
}
#else
//-----------------------------------------------------------------------------
// Object_Image (on memory)
//-----------------------------------------------------------------------------
Object_Image::Object_Image(Class *pClass, Format format) : Object(pClass),
		_format(format), _width(0), _height(0), _buff(NULL)
{
	InitMetrics();
}

Object_Image::Object_Image(const Object_Image &obj) : Object(obj),
		_format(obj._format), _width(0), _height(0), _buff(NULL)
{
	Signal sig;
	if (AllocBuffer(sig, obj._width, obj._height)) {
		::memcpy(_buff, obj._buff, GetBufferSize());
	}
}

Object_Image::~Object_Image()
{
	FreeBuffer();
}

Object *Object_Image::Clone() const
{
	return new Object_Image(*this);
}

bool Object_Image::IsWindowDIB() const
{
	return false;
}

bool Object_Image::AllocBuffer(Signal sig, size_t width, size_t height)
{
	FreeBuffer();
	if (width == 0 || height == 0) {
		sig.SetError(ERR_MemoryError, "failed to allocate image buffer");
		return false;
	}
	_width = width, _height = height;
	InitMetrics();
	_buff = reinterpret_cast<unsigned char *>(OAL::AllocLargeMemory(GetBufferSize()));
	return true;
}

void Object_Image::FreeBuffer()
{
	if (_buff != NULL) OAL::FreeLargeMemory(_buff);
	_width = 0, _height = 0;
	_buff = NULL;
}
#endif

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Image
//-----------------------------------------------------------------------------
// image#allocbuff(width:number, height:number, color[]?:number):void
AScript_DeclareMethod(Image, allocbuff)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "width", VTYPE_Number);
	DeclareArg(env, "height", VTYPE_Number);
	DeclareArg(env, "color", VTYPE_Number, OCCUR_ZeroOrOnce, true);
}

AScript_ImplementMethod(Image, allocbuff)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	if (!pSelf->CheckEmpty(sig)) return Value::Null;
	pSelf->AllocBuffer(sig, context.GetSizeT(0), context.GetSizeT(1));
	if (context.IsList(2)) pSelf->Fill(sig, context.GetList(2));
	return Value::Null;
}

// image#putpixel(x:number, y:number, color[]:number):map:void
AScript_DeclareMethod(Image, putpixel)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "x", VTYPE_Number);
	DeclareArg(env, "y", VTYPE_Number);
	DeclareArg(env, "color", VTYPE_Number, OCCUR_Once, true);
}

AScript_ImplementMethod(Image, putpixel)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	int x = context.GetInt(0), y = context.GetInt(1);
	if (!pSelf->CheckCoord(sig, x, y)) return Value::Null;
	unsigned char *p = pSelf->GetPointer(x, y);
	pSelf->PutPixel(sig, p, context.GetList(2));
	return Value::Null;
}

// image#getpixel(x:number, y:number):map
AScript_DeclareMethod(Image, getpixel)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "x", VTYPE_Number);
	DeclareArg(env, "y", VTYPE_Number);
}

AScript_ImplementMethod(Image, getpixel)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	int x = context.GetInt(0), y = context.GetInt(1);
	if (!pSelf->CheckCoord(sig, x, y)) return Value::Null;
	unsigned char *p = pSelf->GetPointer(x, y);
	Value result;
	ValueList &valList = result.InitAsList(env);
	pSelf->GetPixel(sig, p, valList);
	return result;
}

// image#store(x:number, y:number, width:number, height:number, element:symbol, src):void
AScript_DeclareMethod(Image, store)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "x", VTYPE_Number);
	DeclareArg(env, "y", VTYPE_Number);
	DeclareArg(env, "width", VTYPE_Number);
	DeclareArg(env, "height", VTYPE_Number);
	DeclareArg(env, "element", VTYPE_Symbol);
	DeclareArg(env, "src", VTYPE_Any);
}

AScript_ImplementMethod(Image, store)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	size_t x = context.GetSizeT(0), y = context.GetSizeT(1);
	size_t width = context.GetSizeT(2), height = context.GetSizeT(3);
	if (!pSelf->CheckCoord(sig, x, y)) return Value::Null;
	if (!pSelf->CheckCoord(sig, x + width - 1, y + height - 1)) return Value::Null;
	const Symbol *pSymbol = context.GetSymbol(4);
	if (context.IsMatrix(5)) {
		pSelf->Store(sig, x, y, width, height, pSymbol, context.GetMatrixObj(5));
	} else if (context.IsList(5) || context.IsIterator(5)) {
		Iterator *pIterator = context.GetValue(5).CreateIterator(sig);
		pSelf->Store(sig, x, y, width, height, pSymbol, pIterator);
		Iterator::Delete(pIterator);
	} else {
		sig.SetError(ERR_ValueError, "invalid object for image's source");
		return Value::Null;
	}
	return Value::Null;
}

// image#extract(x:number, y:number, width:number, height:number, element:symbol, dst):void
AScript_DeclareMethod(Image, extract)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "x", VTYPE_Number);
	DeclareArg(env, "y", VTYPE_Number);
	DeclareArg(env, "width", VTYPE_Number);
	DeclareArg(env, "height", VTYPE_Number);
	DeclareArg(env, "element", VTYPE_Symbol);
	DeclareArg(env, "dst", VTYPE_Any);
}

AScript_ImplementMethod(Image, extract)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	size_t x = context.GetSizeT(0), y = context.GetSizeT(1);
	size_t width = context.GetSizeT(2), height = context.GetSizeT(3);
	if (!pSelf->CheckCoord(sig, x, y)) return Value::Null;
	if (!pSelf->CheckCoord(sig, x + width - 1, y + height - 1)) return Value::Null;
	const Symbol *pSymbol = context.GetSymbol(4);
	if (context.IsMatrix(5)) {
		pSelf->Extract(sig, x, y, width, height, pSymbol, context.GetMatrixObj(5));
	} else if (context.IsList(5)) {
		pSelf->Extract(sig, x, y, width, height, pSymbol, context.GetListObj(5));
	} else {
		sig.SetError(ERR_ValueError, "invalid object for image's destination");
		return Value::Null;
	}
	return Value::Null;
}

// image#fill(color[]:number):void
AScript_DeclareMethod(Image, fill)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "color", VTYPE_Number, OCCUR_Once, true);
}

AScript_ImplementMethod(Image, fill)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	pSelf->Fill(sig, context.GetList(4));
	return Value::Null;
}

// image#fillrect(x:number, y:number, width:number, height:number, color[]:number):map:void
AScript_DeclareMethod(Image, fillrect)
{
	SetMode(RSLTMODE_Void, MAP_On, FLAT_Off);
	DeclareArg(env, "x", VTYPE_Number);
	DeclareArg(env, "y", VTYPE_Number);
	DeclareArg(env, "width", VTYPE_Number);
	DeclareArg(env, "height", VTYPE_Number);
	DeclareArg(env, "color", VTYPE_Number, OCCUR_Once, true);
}

AScript_ImplementMethod(Image, fillrect)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	size_t x = context.GetSizeT(0), y = context.GetSizeT(1);
	size_t width = context.GetSizeT(2), height = context.GetSizeT(3);
	if (!pSelf->CheckCoord(sig, x, y)) return Value::Null;
	if (!pSelf->CheckCoord(sig, x + width - 1, y + height - 1)) return Value::Null;
	pSelf->FillRect(sig, x, y, width, height, context.GetList(4));
	return Value::Null;
}

// image#readbmp(file:file)
AScript_DeclareMethod(Image, readbmp)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	SetHelp("Reads an image file in BMP format.");
}

AScript_ImplementMethod(Image, readbmp)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	File &file = context.GetFile(0);
	if (!pSelf->CheckEmpty(sig)) return Value::Null;
	if (!pSelf->ReadBMP(sig, file)) return Value::Null;
	return context.GetSelf();
}

// image#writebmp(file:file)
AScript_DeclareMethod(Image, writebmp)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "file", VTYPE_File);
	SetHelp("Writes an image file in BMP format.");
}

AScript_ImplementMethod(Image, writebmp)
{
	Object_Image *pSelf = Object_Image::GetSelfObj(context);
	File &file = context.GetFile(0);
	if (!pSelf->CheckValidity(sig)) return Value::Null;
	if (!pSelf->WriteBMP(sig, file)) return Value::Null;
	return context.GetSelf();
}

// assignment
Class_Image::Class_Image(Environment &env) : Class(env.LookupClass(VTYPE_Object))
{
	AScript_AssignMethod(Image, allocbuff);
	AScript_AssignMethod(Image, putpixel);
	AScript_AssignMethod(Image, getpixel);
	AScript_AssignMethod(Image, store);
	AScript_AssignMethod(Image, extract);
	AScript_AssignMethod(Image, fill);
	AScript_AssignMethod(Image, fillrect);
	AScript_AssignMethod(Image, readbmp);
	AScript_AssignMethod(Image, writebmp);
}

}
