//-----------------------------------------------------------------------------
// AScript glu module
// see http://www.opengl.org/sdk/docs/man/ for detailed information of each API
//-----------------------------------------------------------------------------
#include <ascript.h>
#include <Gl/glu.h>

AScript_BeginModule(glu)

AScript_DeclarePrivClass(Quadric);
AScript_DeclarePrivClass(Tesselator);
AScript_DeclarePrivClass(Nurbs);

//-----------------------------------------------------------------------------
// helper
//-----------------------------------------------------------------------------
#define ToArrayTemplate(type, getFunc) \
bool ToArray(Signal sig, type v[], const ValueList &valList, size_t n, \
			bool checkFlag, const char *funcName, const char *paramName) \
{ \
	if (checkFlag && valList.size() != n) { \
		sig.SetError(ERR_ValueError, \
				"%s's %s parameter takes a list with %d elements", \
				funcName, paramName, n); \
		return false; \
	} \
	size_t i = 0; \
	foreach_const (ValueList, pValue, valList) { \
		if (i >= n) break; \
		v[i++] = pValue->getFunc(); \
	} \
	return true; \
}

#define DeclareArray(type, varName, n, iParam) \
type varName[n]; \
if (!ToArray(sig, varName, context.GetList(iParam), n, true, GetName(), #varName)) { \
	return Value::Null; \
}

#define DeclareNewArray(type, varName, iParam) \
type *varName = new type[context.GetList(iParam).size()]; \
do { \
	size_t i = 0; \
	foreach_const (ValueList, pValue, context.GetList(iParam)) { \
		varName[i++] = static_cast<type>(pValue->GetNumber()); \
	} \
} while (0);

ToArrayTemplate(GLbyte,		GetChar)
ToArrayTemplate(GLubyte,	GetUChar)
ToArrayTemplate(GLshort,	GetShort)
ToArrayTemplate(GLushort,	GetUShort)
ToArrayTemplate(GLint,		GetInt)
ToArrayTemplate(GLuint,		GetUInt)
ToArrayTemplate(GLfloat,	GetFloat)
ToArrayTemplate(GLdouble,	GetDouble)

GLenum GetImageFormat(Signal sig, Object_Image *pObjImage)
{
	GLenum format = 0;
	Object_Image::Format fmt = pObjImage->GetFormat();
	if (pObjImage->IsWindowDIB()) {
		format =
			(fmt == Object_Image::FORMAT_RGB)? GL_BGR_EXT :
			(fmt == Object_Image::FORMAT_RGBA)? GL_BGRA_EXT : 0;
	} else {
		format =
			(fmt == Object_Image::FORMAT_RGB)? GL_RGB :
			(fmt == Object_Image::FORMAT_RGBA)? GL_RGBA : 0;
	}
	if (format == 0) {
		sig.SetError(ERR_ValueError, "unsupported image type");
	}
	return format;
}

//-----------------------------------------------------------------------------
// Object_Quadric
//-----------------------------------------------------------------------------
class Object_Quadric : public Object {
public:
	inline static Object_Quadric *GetSelfObj(Context &context) {
		return dynamic_cast<Object_Quadric *>(context.GetSelfObj());
	}
private:
	GLUquadric *_qobj;
	static Function *_pFunc_QuadricErrorProc;
public:
	inline Object_Quadric(GLUquadric *qobj) :
			Object(AScript_PrivClass(Quadric)), _qobj(qobj) {}
	inline Object_Quadric(const Object_Quadric &obj) : Object(obj) {}
	virtual ~Object_Quadric();
	virtual Object *Clone() const;
	virtual String ToString(Signal sig, bool exprFlag);
	inline GLUquadric *GetQuadric() { return _qobj; }
	inline static Value CreateValue(GLUquadric *qobj) {
		return Value(new Object_Quadric(qobj), AScript_PrivVTYPE(Quadric));
	}
	void SetQuadricErrorProc(Function *pFunc);
	static void Callback_QuadricErrorProc(GLenum err);
};

Function *Object_Quadric::_pFunc_QuadricErrorProc = NULL;

Object_Quadric::~Object_Quadric()
{
	::gluDeleteQuadric(_qobj);
}

Object *Object_Quadric::Clone() const
{
	return NULL;
}

String Object_Quadric::ToString(Signal sig, bool exprFlag)
{
	return String("<quadric>");
}

void Object_Quadric::SetQuadricErrorProc(Function *pFunc)
{
	Function::Delete(_pFunc_QuadricErrorProc);
	_pFunc_QuadricErrorProc = pFunc;
}

void Object_Quadric::Callback_QuadricErrorProc(GLenum err)
{
	if (_pFunc_QuadricErrorProc == NULL) return;
	Environment &env = _pFunc_QuadricErrorProc->GetEnvScope();
	Signal sig;
	ValueList valListArg(Value(static_cast<int>(err)));
	Context context(valListArg);
	_pFunc_QuadricErrorProc->Eval(env, sig, context);
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Quadric
//-----------------------------------------------------------------------------
// glu.quadric#gluQuadricNormals(normals:number):void
AScript_DeclareMethod(Quadric, gluQuadricNormals)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "normals", VTYPE_Number);
}

AScript_ImplementMethod(Quadric, gluQuadricNormals)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
    GLenum normals = context.GetUInt(0);
	::gluQuadricNormals(qobj, normals);
	return Value::Null;
}

// glu.quadric#gluQuadricTexture(textureCoords:boolean):void
AScript_DeclareMethod(Quadric, gluQuadricTexture)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "textureCoords", VTYPE_Boolean);
}

AScript_ImplementMethod(Quadric, gluQuadricTexture)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
    GLboolean textureCoords = context.GetBoolean(0)? 1 : 0;
	::gluQuadricTexture(qobj, textureCoords);
	return Value::Null;
}

// glu.quadric#gluQuadricOrientation(orientation:number):void
AScript_DeclareMethod(Quadric, gluQuadricOrientation)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "normals", VTYPE_Number);
}

AScript_ImplementMethod(Quadric, gluQuadricOrientation)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
    GLenum normals = context.GetUInt(0);
	::gluQuadricOrientation(qobj, normals);
	return Value::Null;
}

// glu.quadric#gluQuadricDrawStyle(drawStyle:number):void
AScript_DeclareMethod(Quadric, gluQuadricDrawStyle)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "drawStyle", VTYPE_Number);
}

AScript_ImplementMethod(Quadric, gluQuadricDrawStyle)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
    GLenum drawStyle = context.GetUInt(0);
	::gluQuadricDrawStyle(qobj, drawStyle);
	return Value::Null;
}

// glu.quadric#gluCylinder(baseRadius:number, topRadius:number, height:number,
//                         slices:number, stacks:number):void
AScript_DeclareMethod(Quadric, gluCylinder)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "baseRadius", VTYPE_Number);
	DeclareArg(env, "topRadius", VTYPE_Number);
	DeclareArg(env, "height", VTYPE_Number);
	DeclareArg(env, "slices", VTYPE_Number);
	DeclareArg(env, "stacks", VTYPE_Number);
}

AScript_ImplementMethod(Quadric, gluCylinder)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
	GLdouble baseRadius	= context.GetDouble(0);
	GLdouble topRadius	= context.GetDouble(1);
	GLdouble height		= context.GetDouble(2);
	GLint    slices		= context.GetInt(3);
	GLint    stacks		= context.GetInt(4);
	::gluCylinder(qobj, baseRadius, topRadius, height, slices, stacks);
	return Value::Null;
}

// glu.quadric#gluDisk(innerRadius:number, outerRadius:number,
//                     slices:number, loops:number):void
AScript_DeclareMethod(Quadric, gluDisk)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "innerRadius", VTYPE_Number);
	DeclareArg(env, "outerRadius", VTYPE_Number);
	DeclareArg(env, "slices", VTYPE_Number);
	DeclareArg(env, "loops", VTYPE_Number);
}

AScript_ImplementMethod(Quadric, gluDisk)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
	GLdouble innerRadius	= context.GetDouble(0);
	GLdouble outerRadius	= context.GetDouble(1);
	GLint    slices			= context.GetInt(2);
	GLint    loops			= context.GetInt(3);
	::gluDisk(qobj, innerRadius, outerRadius, slices, loops);
	return Value::Null;
}

// glu.quadric#gluPartialDisk(innerRadius:number, outerRadius:number,
//     slices:number, loops:number, startAngle:number, sweepAngle:number):void
AScript_DeclareMethod(Quadric, gluPartialDisk)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "innerRadius", VTYPE_Number);
	DeclareArg(env, "outerRadius", VTYPE_Number);
	DeclareArg(env, "slices", VTYPE_Number);
	DeclareArg(env, "loops", VTYPE_Number);
	DeclareArg(env, "startAngle", VTYPE_Number);
	DeclareArg(env, "sweepAngle", VTYPE_Number);
}

AScript_ImplementMethod(Quadric, gluPartialDisk)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
	GLdouble innerRadius	= context.GetDouble(0);
	GLdouble outerRadius	= context.GetDouble(1);
	GLint    slices			= context.GetInt(2);
	GLint    loops			= context.GetInt(3);
	GLdouble startAngle		= context.GetDouble(4);
	GLdouble sweepAngle		= context.GetDouble(5);
	::gluPartialDisk(qobj, innerRadius, outerRadius, slices, loops, startAngle, sweepAngle);
	return Value::Null;
}

// glu.quadric#gluSphere(radius:number, slices:number, stacks:number):void
AScript_DeclareMethod(Quadric, gluSphere)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "radius", VTYPE_Number);
	DeclareArg(env, "slices", VTYPE_Number);
	DeclareArg(env, "stacks", VTYPE_Number);
}

AScript_ImplementMethod(Quadric, gluSphere)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
	GLdouble radius = context.GetDouble(0);
	GLint    slices = context.GetInt(1);
	GLint    stacks = context.GetInt(2);
	::gluSphere(qobj, radius, slices, stacks);
	return Value::Null;
}

// glu.quadric#gluQuadricCallback(which:number, fn:function):void
AScript_DeclareMethod(Quadric, gluQuadricCallback)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "which", VTYPE_Number);
	DeclareArg(env, "fn", VTYPE_Function);
}

AScript_ImplementMethod(Quadric, gluQuadricCallback)
{
	GLUquadric *qobj = Object_Quadric::GetSelfObj(context)->GetQuadric();
    GLenum which = context.GetUInt(0);
	Function *fn = context.GetFunction(1)->IncRef();
	Object_Quadric::GetSelfObj(context)->SetQuadricErrorProc(fn);
	::gluQuadricCallback(qobj, which,
		reinterpret_cast<void (__stdcall *)()>(Object_Quadric::Callback_QuadricErrorProc));
	return Value::Null;
}

// implementation of class Quadric
AScript_ImplementPrivClass(Quadric)
{
	AScript_AssignMethod(Quadric, gluQuadricNormals);
	AScript_AssignMethod(Quadric, gluQuadricTexture);
	AScript_AssignMethod(Quadric, gluQuadricOrientation);
	AScript_AssignMethod(Quadric, gluQuadricDrawStyle);
	AScript_AssignMethod(Quadric, gluCylinder);
	AScript_AssignMethod(Quadric, gluDisk);
	AScript_AssignMethod(Quadric, gluPartialDisk);
	AScript_AssignMethod(Quadric, gluSphere);
	AScript_AssignMethod(Quadric, gluQuadricCallback);
}

//-----------------------------------------------------------------------------
// Object_Tesselator
//-----------------------------------------------------------------------------
class Object_Tesselator : public Object {
public:
	inline static Object_Tesselator *GetSelfObj(Context &context) {
		return dynamic_cast<Object_Tesselator *>(context.GetSelfObj());
	}
private:
	GLUtesselator *_tess;
public:
	inline Object_Tesselator(GLUtesselator *tess) :
			Object(AScript_PrivClass(Tesselator)), _tess(tess) {}
	inline Object_Tesselator(const Object_Tesselator &obj) : Object(obj) {}
	virtual ~Object_Tesselator();
	virtual Object *Clone() const;
	virtual String ToString(Signal sig, bool exprFlag);
	inline GLUtesselator *GetTesselator() { return _tess; }
	inline static Value CreateValue(GLUtesselator *tess) {
		return Value(new Object_Tesselator(tess), AScript_PrivVTYPE(Tesselator));
	}
};

Object_Tesselator::~Object_Tesselator()
{
	::gluDeleteTess(_tess);
}

Object *Object_Tesselator::Clone() const
{
	return NULL;
}

String Object_Tesselator::ToString(Signal sig, bool exprFlag)
{
	return String("<tesselator>");
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Tesselator
//-----------------------------------------------------------------------------
// glu.tesselator#gluTessBeginPolygon():void {block?}
AScript_DeclareMethod(Tesselator, gluTessBeginPolygon)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Tesselator, gluTessBeginPolygon)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	::gluTessBeginPolygon(tess, tess); // pass this object pointer to callback
	if (context.IsBlockSpecified()) {
		const Expr_Block *pExprBlock = context.GetBlock(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		pExprBlock->Exec(env, sig);
		::gluTessEndPolygon(tess);
	}
	return Value::Null;
}

// glu.tesselator#gluTessBeginContour():void {block?}
AScript_DeclareMethod(Tesselator, gluTessBeginContour)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Tesselator, gluTessBeginContour)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	::gluTessBeginContour(tess);
	if (context.IsBlockSpecified()) {
		const Expr_Block *pExprBlock = context.GetBlock(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		pExprBlock->Exec(env, sig);
		::gluTessEndContour(tess);
	}
	return Value::Null;
}

// glu.tesselator#gluTessVertex(coords[]:number):void
AScript_DeclareMethod(Tesselator, gluTessVertex)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "coords", VTYPE_Number, OCCUR_Once, true);
}

AScript_ImplementMethod(Tesselator, gluTessVertex)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	DeclareArray(GLdouble, coords, 3, 0);
	::gluTessVertex(tess, coords, tess); // pass this object pointer to callback
	return Value::Null;
}

// glu.tesselator#gluTessEndContour():void
AScript_DeclareMethod(Tesselator, gluTessEndContour)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Tesselator, gluTessEndContour)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	::gluTessEndContour(tess);
	return Value::Null;
}

// glu.tesselator#gluTessEndPolygon():void
AScript_DeclareMethod(Tesselator, gluTessEndPolygon)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Tesselator, gluTessEndPolygon)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	::gluTessEndPolygon(tess);
	return Value::Null;
}

// glu.tesselator#gluTessProperty(which:number, value:number):void
AScript_DeclareMethod(Tesselator, gluTessProperty)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "which", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_Number);
}

AScript_ImplementMethod(Tesselator, gluTessProperty)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	GLenum   which = context.GetUInt(0);
	GLdouble value = context.GetDouble(1);
	::gluTessProperty(tess, which, value);
	return Value::Null;
}

// glu.tesselator#gluTessNormal(x:number, y:number, z:number):void
AScript_DeclareMethod(Tesselator, gluTessNormal)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "x", VTYPE_Number);
	DeclareArg(env, "y", VTYPE_Number);
	DeclareArg(env, "z", VTYPE_Number);
}

AScript_ImplementMethod(Tesselator, gluTessNormal)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	GLdouble x = context.GetDouble(0);
	GLdouble y = context.GetDouble(1);
	GLdouble z = context.GetDouble(2);
	::gluTessNormal(tess, x, y, z);
	return Value::Null;
}

// glu.tesselator#gluTessCallback(which:number, fn:function):void
AScript_DeclareMethod(Tesselator, gluTessCallback)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "which", VTYPE_Number);
	DeclareArg(env, "fn", VTYPE_Function);
}

AScript_ImplementMethod(Tesselator, gluTessCallback)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
#if 0
    GLenum which = context.GetUInt(0);
	Function *fn = context.GetFunction(1)->IncRef();
	Object_Quadric::GetSelfObj(context)->SetQuadricErrorProc(fn);
	::gluTessCallback(tess, which,
		reinterpret_cast<void (__stdcall *)()>(Object_Tesselator::Callback));
#endif
	sig.SetError(ERR_SystemError, "not implemented");
	return Value::Null;
}

// glu.tesselator#gluGetTessProperty(which:number)
AScript_DeclareMethod(Tesselator, gluGetTessProperty)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "which", VTYPE_Number);
}

AScript_ImplementMethod(Tesselator, gluGetTessProperty)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	GLenum   which = context.GetUInt(0);
	GLdouble value = 0;
	::gluGetTessProperty(tess, which, &value);
	return Value(static_cast<Number>(value));
}

// glu.tesselator#gluBeginPolygon():void {block?}
AScript_DeclareMethod(Tesselator, gluBeginPolygon)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Tesselator, gluBeginPolygon)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	::gluBeginPolygon(tess);
	if (context.IsBlockSpecified()) {
		const Expr_Block *pExprBlock = context.GetBlock(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		pExprBlock->Exec(env, sig);
		::gluEndPolygon(tess);
	}
	return Value::Null;
}

// glu.tesselator#gluNextContour(type:number):void
AScript_DeclareMethod(Tesselator, gluNextContour)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "type", VTYPE_Number);
}

AScript_ImplementMethod(Tesselator, gluNextContour)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	GLenum type = context.GetUInt(0);
	::gluNextContour(tess, type);
	return Value::Null;
}

// glu.tesselator#gluEndPolygon():void
AScript_DeclareMethod(Tesselator, gluEndPolygon)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Tesselator, gluEndPolygon)
{
	GLUtesselator *tess = Object_Tesselator::GetSelfObj(context)->GetTesselator();
	::gluEndPolygon(tess);
	return Value::Null;
}

// implementation of class Tesselator
AScript_ImplementPrivClass(Tesselator)
{
	AScript_AssignMethod(Tesselator, gluTessBeginPolygon);
	AScript_AssignMethod(Tesselator, gluTessBeginContour);
	AScript_AssignMethod(Tesselator, gluTessVertex);
	AScript_AssignMethod(Tesselator, gluTessEndContour);
	AScript_AssignMethod(Tesselator, gluTessEndPolygon);
	AScript_AssignMethod(Tesselator, gluTessProperty);
	AScript_AssignMethod(Tesselator, gluTessNormal);
	AScript_AssignMethod(Tesselator, gluTessCallback);
	AScript_AssignMethod(Tesselator, gluGetTessProperty);
	AScript_AssignMethod(Tesselator, gluBeginPolygon);
	AScript_AssignMethod(Tesselator, gluNextContour);
	AScript_AssignMethod(Tesselator, gluEndPolygon);
}

//-----------------------------------------------------------------------------
// Object_Nurbs
//-----------------------------------------------------------------------------
class Object_Nurbs : public Object {
public:
	inline static Object_Nurbs *GetSelfObj(Context &context) {
		return dynamic_cast<Object_Nurbs *>(context.GetSelfObj());
	}
private:
	GLUnurbs *_nobj;
public:
	inline Object_Nurbs(GLUnurbs *nobj) :
			Object(AScript_PrivClass(Nurbs)), _nobj(nobj) {}
	inline Object_Nurbs(const Object_Nurbs &obj) : Object(obj) {}
	virtual ~Object_Nurbs();
	virtual Object *Clone() const;
	virtual String ToString(Signal sig, bool exprFlag);
	inline GLUnurbs *GetNurbs() { return _nobj; }
	inline static Value CreateValue(GLUnurbs *nobj) {
		return Value(new Object_Nurbs(nobj), AScript_PrivVTYPE(Nurbs));
	}
};

Object_Nurbs::~Object_Nurbs()
{
	::gluDeleteNurbsRenderer(_nobj);
}

Object *Object_Nurbs::Clone() const
{
	return NULL;
}

String Object_Nurbs::ToString(Signal sig, bool exprFlag)
{
	return String("<nurbs>");
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Nurbs
//-----------------------------------------------------------------------------
// glu.nurbs#gluBeginSurface():void {block?}
AScript_DeclareMethod(Nurbs, gluBeginSurface)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Nurbs, gluBeginSurface)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	::gluBeginSurface(nobj);
	if (context.IsBlockSpecified()) {
		const Expr_Block *pExprBlock = context.GetBlock(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		pExprBlock->Exec(env, sig);
		::gluEndSurface(nobj);
	}
	return Value::Null;
}

// glu.nurbs#gluBeginCurve():void {block?}
AScript_DeclareMethod(Nurbs, gluBeginCurve)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Nurbs, gluBeginCurve)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	::gluBeginCurve(nobj);
	if (context.IsBlockSpecified()) {
		const Expr_Block *pExprBlock = context.GetBlock(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		pExprBlock->Exec(env, sig);
		::gluEndCurve(nobj);
	}
	return Value::Null;
}

// glu.nurbs#gluEndCurve():void
AScript_DeclareMethod(Nurbs, gluEndCurve)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Nurbs, gluEndCurve)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	::gluEndCurve(nobj);
	return Value::Null;
}

// glu.nurbs#gluEndSurface():void
AScript_DeclareMethod(Nurbs, gluEndSurface)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Nurbs, gluEndSurface)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	::gluEndSurface(nobj);
	return Value::Null;
}

// glu.nurbs#gluBeginTrim():void {block?}
AScript_DeclareMethod(Nurbs, gluBeginTrim)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Nurbs, gluBeginTrim)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	::gluBeginTrim(nobj);
	if (context.IsBlockSpecified()) {
		const Expr_Block *pExprBlock = context.GetBlock(env, sig);
		if (sig.IsSignalled()) return Value::Null;
		pExprBlock->Exec(env, sig);
		::gluEndTrim(nobj);
	}
	return Value::Null;
}

// glu.nurbs#gluEndTrim():void
AScript_DeclareMethod(Nurbs, gluEndTrim)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Nurbs, gluEndTrim)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	::gluEndTrim(nobj);
	return Value::Null;
}

// glu.nurbs#gluPwlCurve(data[]:number, stride:number, type:number):void
AScript_DeclareMethod(Nurbs, gluPwlCurve)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "data", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "stride", VTYPE_Number);
	DeclareArg(env, "type", VTYPE_Number);
}

AScript_ImplementMethod(Nurbs, gluPwlCurve)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	size_t n = context.GetList(0).size();
	DeclareNewArray(GLfloat, array, 0);
	GLint stride = context.GetInt(1);
	GLenum type = context.GetUInt(2);
	GLint count = 0;
	if (type == GLU_MAP1_TRIM_2) {
		count = static_cast<GLint>(n / 2);
	} else if (type == GLU_MAP1_TRIM_3) {
		count = static_cast<GLint>(n / 3);
	} else {
		sig.SetError(ERR_ValueError, "invalid type");
		return Value::Null;
	}
	::gluPwlCurve(nobj, count, array, stride, type);
	delete[] array;
	return Value::Null;
}

// glu.nurbs#gluNurbsCurve(knot[]:number, stride:number,
//                         ctlarray[]:number, order:number, type:number):void
AScript_DeclareMethod(Nurbs, gluNurbsCurve)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "knot", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "stride", VTYPE_Number);
	DeclareArg(env, "ctlarray", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "order", VTYPE_Number);
	DeclareArg(env, "type", VTYPE_Number);
}

AScript_ImplementMethod(Nurbs, gluNurbsCurve)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	GLint   nknots = static_cast<GLint>(context.GetList(0).size());
	DeclareNewArray(GLfloat, knot, 0);
	GLint   stride = context.GetInt(1);
	DeclareNewArray(GLfloat, ctlarray, 2);
	GLint   order = context.GetInt(3);
	GLenum  type = context.GetUInt(4);
	::gluNurbsCurve(nobj, nknots, knot, stride, ctlarray, order, type);
	delete[] knot;
	delete[] ctlarray;
	return Value::Null;
}

// glu.gluNurbsSurface(sknot[]:number, tknot[]:number, s_stride:number, t_stride:number,
//                     ctlarray[]:number, order:number, torder:number, type:number):void
AScript_DeclareMethod(Nurbs, gluNurbsSurface)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "sknot", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "tknot", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "s_stride", VTYPE_Number);
	DeclareArg(env, "t_stride", VTYPE_Number);
	DeclareArg(env, "ctlarray", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "order", VTYPE_Number);
	DeclareArg(env, "torder", VTYPE_Number);
	DeclareArg(env, "type", VTYPE_Number);
}

AScript_ImplementMethod(Nurbs, gluNurbsSurface)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	GLint   sknot_count = static_cast<GLint>(context.GetList(0).size());
	DeclareNewArray(GLfloat, sknot, 0);
	GLint   tknot_count = static_cast<GLint>(context.GetList(1).size());
	DeclareNewArray(GLfloat, tknot, 1);
	GLint   s_stride = context.GetInt(2);
	GLint   t_stride = context.GetInt(3);
	DeclareNewArray(GLfloat, ctlarray, 4);
	GLint   sorder = context.GetInt(5);
	GLint   torder = context.GetInt(6);
	GLenum  type = context.GetUInt(7);
	::gluNurbsSurface(nobj, sknot_count, sknot, tknot_count,
					tknot, s_stride, t_stride, ctlarray, sorder, torder, type);
	delete[] sknot;
	delete[] tknot;
	delete[] ctlarray;
	return Value::Null;
}

// glu.gluLoadSamplingMatrices(model[]:number, perspective[]:number, view[]:number):void
AScript_DeclareMethod(Nurbs, gluLoadSamplingMatrices)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "model", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "perspective", VTYPE_Number, OCCUR_Once, true);
	DeclareArg(env, "view", VTYPE_Number, OCCUR_Once, true);
}

AScript_ImplementMethod(Nurbs, gluLoadSamplingMatrices)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	DeclareArray(GLfloat, model, 16, 0);
	DeclareArray(GLfloat, perspective, 16, 1);
	DeclareArray(GLint, view, 4, 2);
	::gluLoadSamplingMatrices(nobj, model, perspective, view);
	return Value::Null;
}

// glu.gluNurbsProperty(property:number, value:number):void
AScript_DeclareMethod(Nurbs, gluNurbsProperty)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "property", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_Number);
}

AScript_ImplementMethod(Nurbs, gluNurbsProperty)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	GLenum  property = context.GetUInt(0);
	GLfloat value = context.GetFloat(1);
	::gluNurbsProperty(nobj, property, value);
	return Value::Null;
}

// glu.gluGetNurbsProperty(property:number)
AScript_DeclareMethod(Nurbs, gluGetNurbsProperty)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "property", VTYPE_Number);
}

AScript_ImplementMethod(Nurbs, gluGetNurbsProperty)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	GLenum  property = context.GetUInt(0);
	GLfloat value = 0;
	::gluGetNurbsProperty(nobj, property, &value);
	return Value(static_cast<Number>(value));
}

// glu.gluNurbsCallback():void
AScript_DeclareMethod(Nurbs, gluNurbsCallback)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Nurbs, gluNurbsCallback)
{
	GLUnurbs *nobj = Object_Nurbs::GetSelfObj(context)->GetNurbs();
	sig.SetError(ERR_SystemError, "not implemented");
	return Value::Null;
}

// implementation of class Nurbs
AScript_ImplementPrivClass(Nurbs)
{
	AScript_AssignMethod(Nurbs, gluBeginSurface);
	AScript_AssignMethod(Nurbs, gluBeginCurve);
	AScript_AssignMethod(Nurbs, gluEndCurve);
	AScript_AssignMethod(Nurbs, gluEndSurface);
	AScript_AssignMethod(Nurbs, gluBeginTrim);
	AScript_AssignMethod(Nurbs, gluEndTrim);
	AScript_AssignMethod(Nurbs, gluPwlCurve);
	AScript_AssignMethod(Nurbs, gluNurbsCurve);
	AScript_AssignMethod(Nurbs, gluNurbsSurface);
	AScript_AssignMethod(Nurbs, gluLoadSamplingMatrices);
	AScript_AssignMethod(Nurbs, gluNurbsProperty);
	AScript_AssignMethod(Nurbs, gluGetNurbsProperty);
	AScript_AssignMethod(Nurbs, gluNurbsCallback);
}

//-----------------------------------------------------------------------------
// AScript module functions: glu
//-----------------------------------------------------------------------------
// glu.gluErrorStringWIN():void
AScript_DeclareFunction(gluErrorStringWIN)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(gluErrorStringWIN)
{
	sig.SetError(ERR_SystemError, "not implemented");
	return Value::Null;
}

// glu.gluErrorString():void
AScript_DeclareFunction(gluErrorString)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(gluErrorString)
{
	sig.SetError(ERR_SystemError, "not implemented");
	return Value::Null;
}

// glu.gluErrorUnicodeStringEXT():void
AScript_DeclareFunction(gluErrorUnicodeStringEXT)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(gluErrorUnicodeStringEXT)
{
	sig.SetError(ERR_SystemError, "not implemented");
	return Value::Null;
}

// glu.gluGetString():void
AScript_DeclareFunction(gluGetString)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(gluGetString)
{
	sig.SetError(ERR_SystemError, "not implemented");
	return Value::Null;
}

// glu.gluOrtho2D(left:number, right:number, bottom:number, top:number):void
AScript_DeclareFunction(gluOrtho2D)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "left", VTYPE_Number);
	DeclareArg(env, "right", VTYPE_Number);
	DeclareArg(env, "bottom", VTYPE_Number);
	DeclareArg(env, "top", VTYPE_Number);
}

AScript_ImplementFunction(gluOrtho2D)
{
	GLdouble left = context.GetDouble(0);
	GLdouble right = context.GetDouble(1);
	GLdouble bottom = context.GetDouble(2);
	GLdouble top = context.GetDouble(3);
	::gluOrtho2D(left, right, bottom, top);
	return Value::Null;
}

// glu.gluPerspective(fovy:number, aspect:number, zNear:number, zFar:number):void
AScript_DeclareFunction(gluPerspective)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "fovy", VTYPE_Number);
	DeclareArg(env, "aspect", VTYPE_Number);
	DeclareArg(env, "zNear", VTYPE_Number);
	DeclareArg(env, "zFar", VTYPE_Number);
}

AScript_ImplementFunction(gluPerspective)
{
	GLdouble fovy = context.GetDouble(0);
	GLdouble aspect = context.GetDouble(1);
	GLdouble zNear = context.GetDouble(2);
	GLdouble zFar = context.GetDouble(3);
	::gluPerspective(fovy, aspect, zNear, zFar);
	return Value::Null;
}

// glu.gluPickMatrix(x:number, y:number, width:number, height:number, viewport[]:number):void
AScript_DeclareFunction(gluPickMatrix)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "x", VTYPE_Number);
	DeclareArg(env, "y", VTYPE_Number);
	DeclareArg(env, "width", VTYPE_Number);
	DeclareArg(env, "height", VTYPE_Number);
	DeclareArg(env, "viewport", VTYPE_Number, OCCUR_Once, true);
}

AScript_ImplementFunction(gluPickMatrix)
{
	GLdouble x = context.GetDouble(0);
	GLdouble y = context.GetDouble(1);
	GLdouble width = context.GetDouble(2);
	GLdouble height = context.GetDouble(3);
	GLint viewport[4];
	do {
		const ValueList &valList = context.GetList(4);
		if (valList.size() != 4) {
			sig.SetError(ERR_ValueError,
						"viewport must have be a list with four elements");
			return Value::Null;
		}
		int i = 0;
		foreach_const (ValueList, pValue, valList) {
			viewport[i++] = pValue->GetInt();
		}
	} while (0);
	::gluPickMatrix(x, y, width, height, viewport);
	return Value::Null;
}

// glu.gluLookAt(eyex:number, eyey:number, eyez:number,
//               centerx:number, centery:number, centerz:number,
//               upx:number, upy:number, upz:number):void
AScript_DeclareFunction(gluLookAt)
{
	SetMode(RSLTMODE_Void, MAP_Off, FLAT_Off);
	DeclareArg(env, "eyex", VTYPE_Number);
	DeclareArg(env, "eyey", VTYPE_Number);
	DeclareArg(env, "eyez", VTYPE_Number);
	DeclareArg(env, "centerx", VTYPE_Number);
	DeclareArg(env, "centery", VTYPE_Number);
	DeclareArg(env, "centerz", VTYPE_Number);
	DeclareArg(env, "upx", VTYPE_Number);
	DeclareArg(env, "upy", VTYPE_Number);
	DeclareArg(env, "upz", VTYPE_Number);
}

AScript_ImplementFunction(gluLookAt)
{
	GLdouble eyex = context.GetDouble(0);
	GLdouble eyey = context.GetDouble(1);
	GLdouble eyez = context.GetDouble(2);
	GLdouble centerx = context.GetDouble(3);
	GLdouble centery = context.GetDouble(4);
	GLdouble centerz = context.GetDouble(5);
	GLdouble upx = context.GetDouble(6);
	GLdouble upy = context.GetDouble(7);
	GLdouble upz = context.GetDouble(8);
	::gluLookAt(eyex, eyey, eyez, centerx, centery, centerz, upx, upy, upz);
	return Value::Null;
}

// glu.gluProject()
AScript_DeclareFunction(gluProject)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "objX", VTYPE_Number);
	DeclareArg(env, "objY", VTYPE_Number);
	DeclareArg(env, "objZ", VTYPE_Number);
	DeclareArg(env, "model", VTYPE_Number);
	DeclareArg(env, "proj", VTYPE_Number);
	DeclareArg(env, "view", VTYPE_Number);
}

AScript_ImplementFunction(gluProject)
{
	GLdouble objX = context.GetDouble(0);
	GLdouble objY = context.GetDouble(1);
	GLdouble objZ = context.GetDouble(2);
	DeclareArray(GLdouble, model, 16, 3);
	DeclareArray(GLdouble, proj, 16, 4);
	DeclareArray(GLint, view, 4, 5);
	GLdouble winX, winY, winZ;
	::gluProject(objX, objY, objZ, model, proj, view, &winX, &winY, &winZ);
	Value result;
	ValueList &valList = result.InitAsList(env);
	valList.push_back(Value(winX));
	valList.push_back(Value(winY));
	valList.push_back(Value(winZ));
	return result;
}

// glu.gluUnProject()
AScript_DeclareFunction(gluUnProject)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "winX", VTYPE_Number);
	DeclareArg(env, "winY", VTYPE_Number);
	DeclareArg(env, "winZ", VTYPE_Number);
	DeclareArg(env, "model", VTYPE_Number);
	DeclareArg(env, "proj", VTYPE_Number);
	DeclareArg(env, "view", VTYPE_Number);
}

AScript_ImplementFunction(gluUnProject)
{
	GLdouble winX = context.GetDouble(0);
	GLdouble winY = context.GetDouble(1);
	GLdouble winZ = context.GetDouble(2);
	DeclareArray(GLdouble, model, 16, 3);
	DeclareArray(GLdouble, proj, 16, 4);
	DeclareArray(GLint, view, 4, 5);
	GLdouble objX, objY, objZ;
	::gluUnProject(winX, winY, winZ, model, proj, view, &objX, &objY, &objZ);
	Value result;
	ValueList &valList = result.InitAsList(env);
	valList.push_back(Value(objX));
	valList.push_back(Value(objY));
	valList.push_back(Value(objZ));
	return result;
}

// glu.gluScaleImage(image:image, w:number, h:number):void
AScript_DeclareFunction(gluScaleImage)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "image", VTYPE_Image);
	DeclareArg(env, "w", VTYPE_Number);
	DeclareArg(env, "h", VTYPE_Number);
	//DeclareArg(env, "format", VTYPE_Symbol);
}

AScript_ImplementFunction(gluScaleImage)
{
	Object_Image *pObjImageIn = context.GetImageObj(0);
	GLenum format = GetImageFormat(sig, pObjImageIn);
	if (sig.IsSignalled()) return Value::Null;
	GLsizei wIn = static_cast<GLsizei>(pObjImageIn->GetWidth());
	GLsizei hIn = static_cast<GLsizei>(pObjImageIn->GetHeight());
	GLenum typeIn = GL_UNSIGNED_BYTE;
	const void *dataIn = pObjImageIn->GetBuffer();
	GLsizei wOut = context.GetInt(1);
	GLsizei hOut = context.GetInt(2);
	//Object_Image::Format fmtOut =
	//			Object_Image::SymbolToFormat(sig, context.GetSymbol(3));
	GLenum typeOut = GL_UNSIGNED_BYTE;
	if (sig.IsSignalled()) return Value::Null;
	Object_Image *pObjImageOut =
		new Object_Image(env.LookupClass(VTYPE_Image), pObjImageIn->GetFormat());
	if (!pObjImageOut->AllocBuffer(sig, wOut, hOut)) {
		delete pObjImageOut;
		return Value::Null;
	}
	GLvoid *dataOut = pObjImageOut->GetBuffer();
	GLint rtn = ::gluScaleImage(format, wIn, hIn, typeIn, dataIn,
										 	wOut, hOut, typeOut, dataOut);
	if (rtn != 0) {
		sig.SetError(ERR_RuntimeError, "gluScaleImage error");
		return Value::Null;
	}
	return Value(pObjImageOut, VTYPE_Image);
}

// glu.gluBuild1DMipmaps(target:number, internalFormat:number, image:image):map
AScript_DeclareFunction(gluBuild1DMipmaps)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "target", VTYPE_Number);
	DeclareArg(env, "internalFormat", VTYPE_Number);
	DeclareArg(env, "image", VTYPE_Image);
}

AScript_ImplementFunction(gluBuild1DMipmaps)
{
	GLenum target = context.GetUInt(0);
	GLint internalFormat = context.GetInt(1);
	Object_Image *pObjImage = context.GetImageObj(2);
	GLsizei width = static_cast<GLsizei>(pObjImage->GetWidth());
	GLenum format = static_cast<GLsizei>(GetImageFormat(sig, pObjImage));
	if (sig.IsSignalled()) return Value::Null;
	GLenum type = GL_UNSIGNED_BYTE;
	const void *data = pObjImage->GetBuffer();
	GLint rtn = ::gluBuild1DMipmaps(target,
					internalFormat, width, format, type, data);
	return Value(rtn);
}

// glu.gluBuild2DMipmaps(target:number, internalFormat:number, image:image):map
AScript_DeclareFunction(gluBuild2DMipmaps)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "target", VTYPE_Number);
	DeclareArg(env, "internalFormat", VTYPE_Number);
	DeclareArg(env, "image", VTYPE_Image);
}

AScript_ImplementFunction(gluBuild2DMipmaps)
{
	GLenum target = context.GetUInt(0);
	GLint internalFormat = context.GetInt(1);
	Object_Image *pObjImage = context.GetImageObj(2);
	Object_Image::Format fmt = pObjImage->GetFormat();
	GLsizei width = static_cast<GLsizei>(pObjImage->GetWidth());
	GLsizei height = static_cast<GLsizei>(pObjImage->GetHeight());
	GLenum format = GetImageFormat(sig, pObjImage);
	if (sig.IsSignalled()) return Value::Null;
	GLenum type = GL_UNSIGNED_BYTE;
	const void *data = pObjImage->GetBuffer();
	GLint rtn = ::gluBuild2DMipmaps(target,
					internalFormat, width, height, format, type, data);
	return Value(rtn);
}

// glu.gluNewQuadric()
AScript_DeclareFunction(gluNewQuadric)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(gluNewQuadric)
{
	GLUquadric *qobj = ::gluNewQuadric();
	return Object_Quadric::CreateValue(qobj);
}

// glu.gluNewTess()
AScript_DeclareFunction(gluNewTess)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(gluNewTess)
{
	GLUtesselator *tess = ::gluNewTess();
	return Object_Tesselator::CreateValue(tess);
}

// glu.gluNewNurbsRenderer()
AScript_DeclareFunction(gluNewNurbsRenderer)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementFunction(gluNewNurbsRenderer)
{
	GLUnurbs *nobj = ::gluNewNurbsRenderer();
	return Object_Nurbs::CreateValue(nobj);
}

// Module entry
AScript_ModuleEntry()
{
	// class realization
	AScript_RealizePrivClass(Quadric,		"quadric");
	AScript_RealizePrivClass(Tesselator,	"tesselator");
	AScript_RealizePrivClass(Nurbs,			"nurbs");
	// value assignment
	AScript_AssignValue(GLU_VERSION_1_1,                Value(GLU_VERSION_1_1));
	AScript_AssignValue(GLU_VERSION_1_2,                Value(GLU_VERSION_1_2));
	AScript_AssignValue(GLU_INVALID_ENUM,               Value(GLU_INVALID_ENUM));
	AScript_AssignValue(GLU_INVALID_VALUE,              Value(GLU_INVALID_VALUE));
	AScript_AssignValue(GLU_OUT_OF_MEMORY,              Value(GLU_OUT_OF_MEMORY));
	AScript_AssignValue(GLU_INCOMPATIBLE_GL_VERSION,    Value(GLU_INCOMPATIBLE_GL_VERSION));
	AScript_AssignValue(GLU_VERSION,                    Value(GLU_VERSION));
	AScript_AssignValue(GLU_EXTENSIONS,                 Value(GLU_EXTENSIONS));
	AScript_AssignValue(GLU_TRUE,                       Value(GLU_TRUE));
	AScript_AssignValue(GLU_FALSE,                      Value(GLU_FALSE));
	AScript_AssignValue(GLU_SMOOTH,                     Value(GLU_SMOOTH));
	AScript_AssignValue(GLU_FLAT,                       Value(GLU_FLAT));
	AScript_AssignValue(GLU_NONE,                       Value(GLU_NONE));
	AScript_AssignValue(GLU_POINT,                      Value(GLU_POINT));
	AScript_AssignValue(GLU_LINE,                       Value(GLU_LINE));
	AScript_AssignValue(GLU_FILL,                       Value(GLU_FILL));
	AScript_AssignValue(GLU_SILHOUETTE,                 Value(GLU_SILHOUETTE));
	AScript_AssignValue(GLU_OUTSIDE,                    Value(GLU_OUTSIDE));
	AScript_AssignValue(GLU_INSIDE,                     Value(GLU_INSIDE));
	AScript_AssignValue(GLU_TESS_MAX_COORD,             Value(GLU_TESS_MAX_COORD));
	AScript_AssignValue(GLU_TESS_WINDING_RULE,          Value(GLU_TESS_WINDING_RULE));
	AScript_AssignValue(GLU_TESS_BOUNDARY_ONLY,         Value(GLU_TESS_BOUNDARY_ONLY));
	AScript_AssignValue(GLU_TESS_TOLERANCE,             Value(GLU_TESS_TOLERANCE));
	AScript_AssignValue(GLU_TESS_WINDING_ODD,           Value(GLU_TESS_WINDING_ODD));
	AScript_AssignValue(GLU_TESS_WINDING_NONZERO,       Value(GLU_TESS_WINDING_NONZERO));
	AScript_AssignValue(GLU_TESS_WINDING_POSITIVE,      Value(GLU_TESS_WINDING_POSITIVE));
	AScript_AssignValue(GLU_TESS_WINDING_NEGATIVE,      Value(GLU_TESS_WINDING_NEGATIVE));
	AScript_AssignValue(GLU_TESS_WINDING_ABS_GEQ_TWO,   Value(GLU_TESS_WINDING_ABS_GEQ_TWO));
	AScript_AssignValue(GLU_TESS_BEGIN,                 Value(GLU_TESS_BEGIN));
	AScript_AssignValue(GLU_TESS_VERTEX,                Value(GLU_TESS_VERTEX));
	AScript_AssignValue(GLU_TESS_END,                   Value(GLU_TESS_END));
	AScript_AssignValue(GLU_TESS_ERROR,                 Value(GLU_TESS_ERROR));
	AScript_AssignValue(GLU_TESS_EDGE_FLAG,             Value(GLU_TESS_EDGE_FLAG));
	AScript_AssignValue(GLU_TESS_COMBINE,               Value(GLU_TESS_COMBINE));
	AScript_AssignValue(GLU_TESS_BEGIN_DATA,            Value(GLU_TESS_BEGIN_DATA));
	AScript_AssignValue(GLU_TESS_VERTEX_DATA,           Value(GLU_TESS_VERTEX_DATA));
	AScript_AssignValue(GLU_TESS_END_DATA,              Value(GLU_TESS_END_DATA));
	AScript_AssignValue(GLU_TESS_ERROR_DATA,            Value(GLU_TESS_ERROR_DATA));
	AScript_AssignValue(GLU_TESS_EDGE_FLAG_DATA,        Value(GLU_TESS_EDGE_FLAG_DATA));
	AScript_AssignValue(GLU_TESS_COMBINE_DATA,          Value(GLU_TESS_COMBINE_DATA));
	AScript_AssignValue(GLU_TESS_ERROR1,                Value(GLU_TESS_ERROR1));
	AScript_AssignValue(GLU_TESS_ERROR2,                Value(GLU_TESS_ERROR2));
	AScript_AssignValue(GLU_TESS_ERROR3,                Value(GLU_TESS_ERROR3));
	AScript_AssignValue(GLU_TESS_ERROR4,                Value(GLU_TESS_ERROR4));
	AScript_AssignValue(GLU_TESS_ERROR5,                Value(GLU_TESS_ERROR5));
	AScript_AssignValue(GLU_TESS_ERROR6,                Value(GLU_TESS_ERROR6));
	AScript_AssignValue(GLU_TESS_ERROR7,                Value(GLU_TESS_ERROR7));
	AScript_AssignValue(GLU_TESS_ERROR8,                Value(GLU_TESS_ERROR8));
	AScript_AssignValue(GLU_TESS_MISSING_BEGIN_POLYGON, Value(GLU_TESS_MISSING_BEGIN_POLYGON));
	AScript_AssignValue(GLU_TESS_MISSING_BEGIN_CONTOUR, Value(GLU_TESS_MISSING_BEGIN_CONTOUR));
	AScript_AssignValue(GLU_TESS_MISSING_END_POLYGON,   Value(GLU_TESS_MISSING_END_POLYGON));
	AScript_AssignValue(GLU_TESS_MISSING_END_CONTOUR,   Value(GLU_TESS_MISSING_END_CONTOUR));
	AScript_AssignValue(GLU_TESS_COORD_TOO_LARGE,       Value(GLU_TESS_COORD_TOO_LARGE));
	AScript_AssignValue(GLU_TESS_NEED_COMBINE_CALLBACK, Value(GLU_TESS_NEED_COMBINE_CALLBACK));
	AScript_AssignValue(GLU_AUTO_LOAD_MATRIX,           Value(GLU_AUTO_LOAD_MATRIX));
	AScript_AssignValue(GLU_CULLING,                    Value(GLU_CULLING));
	AScript_AssignValue(GLU_SAMPLING_TOLERANCE,         Value(GLU_SAMPLING_TOLERANCE));
	AScript_AssignValue(GLU_DISPLAY_MODE,               Value(GLU_DISPLAY_MODE));
	AScript_AssignValue(GLU_PARAMETRIC_TOLERANCE,       Value(GLU_PARAMETRIC_TOLERANCE));
	AScript_AssignValue(GLU_SAMPLING_METHOD,            Value(GLU_SAMPLING_METHOD));
	AScript_AssignValue(GLU_U_STEP,                     Value(GLU_U_STEP));
	AScript_AssignValue(GLU_V_STEP,                     Value(GLU_V_STEP));
	AScript_AssignValue(GLU_PATH_LENGTH,                Value(GLU_PATH_LENGTH));
	AScript_AssignValue(GLU_PARAMETRIC_ERROR,           Value(GLU_PARAMETRIC_ERROR));
	AScript_AssignValue(GLU_DOMAIN_DISTANCE,            Value(GLU_DOMAIN_DISTANCE));
	AScript_AssignValue(GLU_MAP1_TRIM_2,                Value(GLU_MAP1_TRIM_2));
	AScript_AssignValue(GLU_MAP1_TRIM_3,                Value(GLU_MAP1_TRIM_3));
	AScript_AssignValue(GLU_OUTLINE_POLYGON,            Value(GLU_OUTLINE_POLYGON));
	AScript_AssignValue(GLU_OUTLINE_PATCH,              Value(GLU_OUTLINE_PATCH));
	AScript_AssignValue(GLU_NURBS_ERROR1,               Value(GLU_NURBS_ERROR1));
	AScript_AssignValue(GLU_NURBS_ERROR2,               Value(GLU_NURBS_ERROR2));
	AScript_AssignValue(GLU_NURBS_ERROR3,               Value(GLU_NURBS_ERROR3));
	AScript_AssignValue(GLU_NURBS_ERROR4,               Value(GLU_NURBS_ERROR4));
	AScript_AssignValue(GLU_NURBS_ERROR5,               Value(GLU_NURBS_ERROR5));
	AScript_AssignValue(GLU_NURBS_ERROR6,               Value(GLU_NURBS_ERROR6));
	AScript_AssignValue(GLU_NURBS_ERROR7,               Value(GLU_NURBS_ERROR7));
	AScript_AssignValue(GLU_NURBS_ERROR8,               Value(GLU_NURBS_ERROR8));
	AScript_AssignValue(GLU_NURBS_ERROR9,               Value(GLU_NURBS_ERROR9));
	AScript_AssignValue(GLU_NURBS_ERROR10,              Value(GLU_NURBS_ERROR10));
	AScript_AssignValue(GLU_NURBS_ERROR11,              Value(GLU_NURBS_ERROR11));
	AScript_AssignValue(GLU_NURBS_ERROR12,              Value(GLU_NURBS_ERROR12));
	AScript_AssignValue(GLU_NURBS_ERROR13,              Value(GLU_NURBS_ERROR13));
	AScript_AssignValue(GLU_NURBS_ERROR14,              Value(GLU_NURBS_ERROR14));
	AScript_AssignValue(GLU_NURBS_ERROR15,              Value(GLU_NURBS_ERROR15));
	AScript_AssignValue(GLU_NURBS_ERROR16,              Value(GLU_NURBS_ERROR16));
	AScript_AssignValue(GLU_NURBS_ERROR17,              Value(GLU_NURBS_ERROR17));
	AScript_AssignValue(GLU_NURBS_ERROR18,              Value(GLU_NURBS_ERROR18));
	AScript_AssignValue(GLU_NURBS_ERROR19,              Value(GLU_NURBS_ERROR19));
	AScript_AssignValue(GLU_NURBS_ERROR20,              Value(GLU_NURBS_ERROR20));
	AScript_AssignValue(GLU_NURBS_ERROR21,              Value(GLU_NURBS_ERROR21));
	AScript_AssignValue(GLU_NURBS_ERROR22,              Value(GLU_NURBS_ERROR22));
	AScript_AssignValue(GLU_NURBS_ERROR23,              Value(GLU_NURBS_ERROR23));
	AScript_AssignValue(GLU_NURBS_ERROR24,              Value(GLU_NURBS_ERROR24));
	AScript_AssignValue(GLU_NURBS_ERROR25,              Value(GLU_NURBS_ERROR25));
	AScript_AssignValue(GLU_NURBS_ERROR26,              Value(GLU_NURBS_ERROR26));
	AScript_AssignValue(GLU_NURBS_ERROR27,              Value(GLU_NURBS_ERROR27));
	AScript_AssignValue(GLU_NURBS_ERROR28,              Value(GLU_NURBS_ERROR28));
	AScript_AssignValue(GLU_NURBS_ERROR29,              Value(GLU_NURBS_ERROR29));
	AScript_AssignValue(GLU_NURBS_ERROR30,              Value(GLU_NURBS_ERROR30));
	AScript_AssignValue(GLU_NURBS_ERROR31,              Value(GLU_NURBS_ERROR31));
	AScript_AssignValue(GLU_NURBS_ERROR32,              Value(GLU_NURBS_ERROR32));
	AScript_AssignValue(GLU_NURBS_ERROR33,              Value(GLU_NURBS_ERROR33));
	AScript_AssignValue(GLU_NURBS_ERROR34,              Value(GLU_NURBS_ERROR34));
	AScript_AssignValue(GLU_NURBS_ERROR35,              Value(GLU_NURBS_ERROR35));
	AScript_AssignValue(GLU_NURBS_ERROR36,              Value(GLU_NURBS_ERROR36));
	AScript_AssignValue(GLU_NURBS_ERROR37,              Value(GLU_NURBS_ERROR37));
	AScript_AssignValue(GLU_CW,                         Value(GLU_CW));
	AScript_AssignValue(GLU_CCW,                        Value(GLU_CCW));
	AScript_AssignValue(GLU_INTERIOR,                   Value(GLU_INTERIOR));
	AScript_AssignValue(GLU_EXTERIOR,                   Value(GLU_EXTERIOR));
	AScript_AssignValue(GLU_UNKNOWN,                    Value(GLU_UNKNOWN));
	AScript_AssignValue(GLU_BEGIN,                      Value(GLU_BEGIN));
	AScript_AssignValue(GLU_VERTEX,                     Value(GLU_VERTEX));
	AScript_AssignValue(GLU_END,                        Value(GLU_END));
	AScript_AssignValue(GLU_ERROR,                      Value(GLU_ERROR));
	AScript_AssignValue(GLU_EDGE_FLAG,                  Value(GLU_EDGE_FLAG));
	// function assignment
	AScript_AssignFunction(gluErrorStringWIN);
	AScript_AssignFunction(gluErrorString);
	AScript_AssignFunction(gluErrorUnicodeStringEXT);
	AScript_AssignFunction(gluGetString);
	AScript_AssignFunction(gluOrtho2D);
	AScript_AssignFunction(gluPerspective);
	AScript_AssignFunction(gluPickMatrix);
	AScript_AssignFunction(gluLookAt);
	AScript_AssignFunction(gluProject);
	AScript_AssignFunction(gluUnProject);
	AScript_AssignFunction(gluScaleImage);
	AScript_AssignFunction(gluBuild1DMipmaps);
	AScript_AssignFunction(gluBuild2DMipmaps);
	AScript_AssignFunction(gluNewQuadric);
	AScript_AssignFunction(gluNewTess);
	AScript_AssignFunction(gluNewNurbsRenderer);
}

AScript_ModuleTerminate()
{
}

AScript_EndModule(glu)

AScript_RegisterModule(glu)
