#include "Iterator.h"
#include "Object.h"
#include "Expr.h"
#include "OAL.h"
#include "Function.h"
#include "Object_Function.h"
#include "Object_Bytes.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Iterator
//-----------------------------------------------------------------------------
Iterator::~Iterator()
{
	delete _pShare;
}

bool Iterator::IsSequence() const { return false; }
bool Iterator::IsSequenceInf() const { return false; }

Iterator *Iterator::Clone()
{
	if (_pShare == NULL) _pShare = new Share();
	int id = _pShare->Register();
	return new Iterator_GenericClone(IncRef(), id);
}

bool Iterator::NextShared(int id, Signal sig, Value &value)
{
	if (_pShare == NULL) {
		for (;;) {
			if (!DoNext(sig, value)) return false;
			if (!IsSkipInvalid() || value.IsValid()) break;
		}
		return true;
	}
	if (!_pShare->Next(id, value)) {
		if (_pShare->IsDone()) return false;
		for (;;) {
			if (!DoNext(sig, value)) {
				_pShare->SetDone();
				return false;
			}
			if (!IsSkipInvalid() || value.IsValid()) break;
		}
		_pShare->Store(id, value);
	}
	return true;
}

Value Iterator::ToList(Environment &env, Signal sig)
{
	if (IsInfinite()) {
		SetError_InfiniteNotAllowed(sig);
		return Value::Null;
	}
	Value result;
	ValueList &valList = result.InitAsList(env);
	Value value;
	while (Next(sig, value)) {
		valList.push_back(value);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Eval(Environment &env, Signal sig, Context &context)
{
	if (IsInfinite()) {
		SetError_InfiniteNotAllowed(sig);
		return Value::Null;
	}
	Value value, result;
	Function::ResultListComposer resultListComposer(env, context, result);
	while (Next(sig, value)) {
		resultListComposer.Store(value);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Eval(Environment &env, Signal sig,
								Context &context, const Function *pFuncBlock)
{
	if (IsInfinite()) {
		SetError_InfiniteNotAllowed(sig);
		return Value::Null;
	}
	Value value, result;
	std::auto_ptr<Function::ResultListComposer> pResultListComposer;
	if (context.IsRsltList() || context.IsRsltXList() ||
								context.IsRsltSet() || context.IsRsltXSet()) {
		pResultListComposer.reset(
					new Function::ResultListComposer(env, context, result));
	}
	for (int idx = 0; Next(sig, value); idx++) {
		ValueList valListArg(value, Value(static_cast<Number>(idx)));
		Context contextSub(valListArg);
		if (pResultListComposer.get() == NULL) {
			result = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, result, Value::Null)
		} else {
			Value resultElem = pFuncBlock->Eval(env, sig, contextSub);
			AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
			pResultListComposer->Store(resultElem);
		}
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Reduce(Environment &env, Signal sig,
									Value valueAccum, const Function *pFuncBlock)
{
	if (IsInfinite()) {
		SetError_InfiniteNotAllowed(sig);
		return Value::Null;
	}
	Value value;
	for (int idx = 0; Next(sig, value); idx++) {
		ValueList valListArg(value, valueAccum);
		Context contextSub(valListArg);
		valueAccum = pFuncBlock->Eval(env, sig, contextSub);
		AScript_BlockSignalHandlerInLoop(sig, valueAccum, Value::Null)
	}
	if (sig.IsSignalled()) return Value::Null;
	return valueAccum;
}

Value Iterator::MinMax(Environment &env, Signal sig,
									bool maxFlag, const SymbolSet &attrs)
{
	Value valueHit;
	if (!Next(sig, valueHit)) return Value::Null;
	Value result;
	if (attrs.IsSet(AScript_Symbol(index))) {
		size_t idxHit = 0;
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
				idxHit = idx;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result.SetNumber(idxHit);
	} else if (attrs.IsSet(AScript_Symbol(last_index))) {
		size_t idxHit = 0;
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp >= 0) {
				valueHit = value;
				idxHit = idx;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result.SetNumber(idxHit);
	} else if (attrs.IsSet(AScript_Symbol(indices))) {
		ValueList &resultList = result.InitAsList(env);
		resultList.push_back(Value(static_cast<Number>(0)));
		Value value;
		for (size_t idx = 1; Next(sig, value); idx++) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
				resultList.clear();
				resultList.push_back(Value(static_cast<Number>(idx)));
			} else if (cmp == 0) {
				resultList.push_back(Value(static_cast<Number>(idx)));
			}
		}
		if (sig.IsSignalled()) return Value::Null;
	} else {
		Value value;
		while (Next(sig, value)) {
			int cmp = Value::Compare(valueHit, value);
			if (maxFlag) cmp = -cmp;
			if (cmp > 0) {
				valueHit = value;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		result = valueHit;
	}
	return result;
}

void Iterator::SetError_InvalidDataTypeOfElement(Signal sig)
{
	sig.SetError(ERR_ValueError, "invalid data type of element");
}

void Iterator::SetError_InfiniteNotAllowed(Signal sig)
{
	sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
}

Value Iterator::Sum(Signal sig, size_t &cnt)
{
	cnt = 0;
	Value value;
	if (!Next(sig, value)) return Value::Null;
	cnt++;
	if (value.IsNumber()) {
		Number result = value.GetNumber();
		while (Next(sig, value)) {
			cnt++;
			if (value.IsNumber()) {
				result += value.GetNumber();
			} else if (value.IsComplex()) {
				Complex resultEx = result;
				while (Next(sig, value)) {
					cnt++;
					if (value.IsNumberOrComplex()) {
						resultEx += value.GetComplex();
					} else {
						SetError_InvalidDataTypeOfElement(sig);
						return Value::Null;
					}
				}
				if (sig.IsSignalled()) return Value::Null;
				return Value(resultEx);
			} else {
				SetError_InvalidDataTypeOfElement(sig);
				return Value::Null;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		return Value(result);
	} else if (value.IsComplex()) {
		Complex result = value.GetComplex();
		while (Next(sig, value)) {
			cnt++;
			if (value.IsNumberOrComplex()) {
				result += value.GetComplex();
			} else {
				SetError_InvalidDataTypeOfElement(sig);
				return Value::Null;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		return Value(result);
	} else {
		SetError_InvalidDataTypeOfElement(sig);
		return Value::Null;
	}
}

Value Iterator::Average(Signal sig, size_t &cnt)
{
	Value valueSum = Clone()->Sum(sig, cnt);
	if (valueSum.IsNumber()) {
		return Value(valueSum.GetNumber() / static_cast<Number>(cnt));
	} else if (valueSum.IsComplex()) {
		return Value(valueSum.GetComplex() / static_cast<Number>(cnt));
	} else {
		return Value::Null;
	}
}

Value Iterator::Variance(Signal sig, size_t &cnt)
{
	// this doesn't use a variance calculation formula V(x) = E(x**2) - E(x)**2
	// to minimize calculation error.
	Value valueAve = Clone()->Average(sig, cnt);
	if (!valueAve.IsNumberOrComplex()) return Value::Null;
	Value value;
	if (!Next(sig, value)) return Value::Null;
	if (value.IsNumber() && valueAve.IsNumber()) {
		Number result;
		Number average = valueAve.GetNumber();
		do {
			Number tmp = value.GetNumber() - average;
			result = tmp * tmp;
		} while (0);
		while (Next(sig, value)) {
			if (value.IsNumber()) {
				Number tmp = value.GetNumber() - average;
				result += tmp * tmp;
			} else if (value.IsComplex()) {
				while (Next(sig, value)) {
					if (value.IsNumberOrComplex()) {
						Complex tmp = value.GetComplex() - average;
						result += std::norm(tmp);
					} else {
						SetError_InvalidDataTypeOfElement(sig);
						return Value::Null;
					}
				}
			} else {
				SetError_InvalidDataTypeOfElement(sig);
				return Value::Null;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		return Value(result / static_cast<Number>(cnt));
	} else if (value.IsNumberOrComplex()) {
		Number result;
		Complex average = valueAve.GetComplex();
		do {
			Complex tmp = value.GetComplex() - average;
			result = std::norm(tmp);
		} while (0);
		while (Next(sig, value)) {
			if (value.IsNumberOrComplex()) {
				Complex tmp = value.GetComplex() - average;
				result += std::norm(tmp);
			} else {
				SetError_InvalidDataTypeOfElement(sig);
				return Value::Null;
			}
		}
		if (sig.IsSignalled()) return Value::Null;
		return Value(result / static_cast<Number>(cnt));
	} else {
		SetError_InvalidDataTypeOfElement(sig);
		return Value::Null;
	}
}

Value Iterator::StandardDeviation(Signal sig, size_t &cnt)
{
	Value valueVar = Clone()->Variance(sig, cnt);
	if (!valueVar.IsNumber()) return Value::Null;
	return Value(::sqrt(valueVar.GetNumber()));
}

Value Iterator::And(Signal sig)
{
	Value value;
	if (!Next(sig, value)) return Value::Null;
	if (!value.GetBoolean()) return Value(false);
	while (Next(sig, value)) {
		if (!value.GetBoolean()) return Value(false);
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(true);
}

Value Iterator::Or(Signal sig)
{
	Value value;
	if (!Next(sig, value)) return Value::Null;
	if (value.GetBoolean()) return Value(true);
	while (Next(sig, value)) {
		if (value.GetBoolean()) return Value(true);
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(false);
}

size_t Iterator::Count(Environment &env, Signal sig, const Value &criteria)
{
	size_t cnt = 0;
	if (criteria.IsFunction()) {
		const Function *pFunc = criteria.GetFunction();
		Value value;
		while (Next(sig, value)) {
			ValueList valListArg(value);
			Context context(valListArg);
			Value valueFlag = pFunc->Eval(env, sig, context);
			if (sig.IsSignalled()) return 0;
			if (valueFlag.GetBoolean()) cnt++;
		}
		if (sig.IsSignalled()) return 0;
	} else {
		Value value;
		while (Next(sig, value)) {
			if (Value::Compare(value, criteria) == 0) cnt++;
		}
	}
	return cnt;
}

Iterator *Iterator::Filter(Environment &env, Signal sig, const Value &criteria)
{
	if (criteria.IsFunction()) {
		Object_Function *pFuncObjCriteria =
			dynamic_cast<Object_Function *>(criteria.GetFunctionObj()->IncRef());
		return new Iterator_FilterWithFunc(env, this, pFuncObjCriteria);
	} else if (criteria.IsList() || criteria.IsIterator()) {
		Iterator *pIteratorCriteria = criteria.CreateIterator(sig);
		if (sig.IsSignalled()) return NULL;
		return new Iterator_FilterWithIter(this, pIteratorCriteria);
	} else {
		sig.SetError(ERR_ValueError, "invalid type of criteria for filter");
		return NULL;
	}
}

Iterator *Iterator::While(Environment &env, Signal sig, const Value &criteria)
{
	if (criteria.IsFunction()) {
		Object_Function *pFuncObjCriteria = 
			dynamic_cast<Object_Function *>(criteria.GetFunctionObj()->IncRef());
		return new Iterator_WhileWithFunc(env, this, pFuncObjCriteria);
	} else if (criteria.IsList() || criteria.IsIterator()) {
		Iterator *pIteratorCriteria = criteria.CreateIterator(sig);
		if (sig.IsSignalled()) return NULL;
		return new Iterator_WhileWithIter(this, pIteratorCriteria);
	} else {
		sig.SetError(ERR_ValueError, "invalid type of criteria for while");
		return NULL;
	}
}

Iterator *Iterator::Since(Environment &env, Signal sig, const Value &criteria)
{
	if (criteria.IsFunction()) {
		Object_Function *pFuncObjCriteria = 
			dynamic_cast<Object_Function *>(criteria.GetFunctionObj()->IncRef());
		return new Iterator_SinceWithFunc(env, this, pFuncObjCriteria);
	} else if (criteria.IsList() || criteria.IsIterator()) {
		Iterator *pIteratorCriteria = criteria.CreateIterator(sig);
		if (sig.IsSignalled()) return NULL;
		return new Iterator_SinceWithIter(this, pIteratorCriteria);
	} else {
		sig.SetError(ERR_ValueError, "invalid type of criteria for since");
		return NULL;
	}
}

bool Iterator::IsContain(Signal sig, const Value &value)
{
	Value valueToFind;
	while (Next(sig, valueToFind)) {
		if (Value::Compare(value, valueToFind) == 0) return true;
	}
	return false;
}

String Iterator::ToString(Signal sig) const
{
	return String("<iterator>");
}

bool Iterator::PrepareRepeaterIterators(Environment &env, Signal sig,
						const ValueList &valListArg, ExprList &exprLeftList,
						IteratorOwner &iteratorOwner)
{
	foreach_const (ValueList, pValue, valListArg) {
		ASSUME(env, pValue->IsExpr());
		const Expr *pExpr = pValue->GetExpr();
		Value value;
		if (!(pExpr->IsBinaryOp() && dynamic_cast<const Expr_BinaryOp *>(pExpr)->
												GetFunction().IsContainCheck())) {
			sig.SetError(ERR_TypeError, "iteratable must be specified as arguments");
			return false;
		}
		const Expr_BinaryOp *pExprBin =
						dynamic_cast<const Expr_BinaryOp *>(pExpr);
		value = pExprBin->GetRight()->Exec(env, sig);
		if (sig.IsSignalled()) return false;
		Iterator *pIterator = value.CreateIterator(sig);
		if (pIterator == NULL) return false;
		exprLeftList.push_back(const_cast<Expr *>(pExprBin->GetLeft()));
		iteratorOwner.push_back(pIterator);
	}
	return true;
}

//-----------------------------------------------------------------------------
// Iterator::Share
//-----------------------------------------------------------------------------
Iterator::Share::Share() : _doneFlag(false), _indexMin(0)
{
	_indexList.push_back(_indexMin);
}

Iterator::Share::~Share()
{
}

int Iterator::Share::Register()
{
	int id = static_cast<int>(_indexList.size());
	_indexList.push_back(_indexMin);
	return id;
}

bool Iterator::Share::Next(int id, Value &value)
{
	size_t iElem = _indexList[id] - _indexMin;
	if (iElem >= _valueDeque.size()) return false;
	value = _valueDeque[iElem];
	_indexList[id] = static_cast<unsigned int>(_indexMin + iElem + 1);
	size_t indexMinCur = _indexMin;
	foreach (IndexList, pIndex, _indexList) {
		if (indexMinCur > *pIndex) indexMinCur = *pIndex;
	}
	for (size_t cnt = indexMinCur - _indexMin; cnt > 0; cnt--) {
		_valueDeque.pop_front();
	}
	_indexMin = indexMinCur;
	return true;
}

void Iterator::Share::Store(int id, const Value &value)
{
	_valueDeque.push_back(value);
	_indexList[id] = static_cast<unsigned int>(_indexMin + _valueDeque.size());
}

//-----------------------------------------------------------------------------
// Iterator_GenericClone
//-----------------------------------------------------------------------------
Iterator_GenericClone::~Iterator_GenericClone()
{
}

bool Iterator_GenericClone::DoNext(Signal sig, Value &value)
{
	return _pIterator->NextShared(_id, sig, value);
}

String Iterator_GenericClone::ToString(Signal sig) const
{
	return _pIterator->ToString(sig);
}

//-----------------------------------------------------------------------------
// Iterator_Constant
//-----------------------------------------------------------------------------
Iterator_Constant::~Iterator_Constant()
{
}

Iterator *Iterator_Constant::Clone()
{
	return new Iterator_Constant(*this);
}

bool Iterator_Constant::DoNext(Signal sig, Value &value)
{
	value = _value;
	return true;
}

String Iterator_Constant::ToString(Signal sig) const
{
	String rtn = "<iterator:constant:";
	rtn += _value.ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_OneShot
//-----------------------------------------------------------------------------
Iterator_OneShot::~Iterator_OneShot()
{
}

Iterator *Iterator_OneShot::Clone()
{
	return new Iterator_OneShot(*this);
}

bool Iterator_OneShot::DoNext(Signal sig, Value &value)
{
	if (_doneFlag) return false;
	value = _value;
	_doneFlag = true;
	return true;
}

String Iterator_OneShot::ToString(Signal sig) const
{
	String rtn = "<iterator:oneshot:";
	rtn += _value.ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Fill
//-----------------------------------------------------------------------------
Iterator_Fill::~Iterator_Fill()
{
}

Iterator *Iterator_Fill::Clone()
{
	return new Iterator_Fill(*this);
}

bool Iterator_Fill::DoNext(Signal sig, Value &value)
{
	if (_idx >= _cnt) return false;
	if (_cnt > 0) _idx++;
	value = _value;
	return true;
}

String Iterator_Fill::ToString(Signal sig) const
{
	String rtn = "<iterator:fill(";
	rtn += NumberToString(_cnt);
	if (_value.IsValid()) {
		rtn += ", ";
		rtn += _value.ToString(sig, true);
	}
	rtn += ")>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Rand
//-----------------------------------------------------------------------------
Iterator_Rand::~Iterator_Rand()
{
}

bool Iterator_Rand::DoNext(Signal sig, Value &value)
{
	if (_cnt > 0) {
		if (_idx >= _cnt) return false;
		_idx++;
	}
	value = Value((_range > 0)?
		static_cast<Number>(static_cast<int>(RandomGenerator::Real2() * _range)) :
		static_cast<Number>(RandomGenerator::Real2()));
	return true;
}

String Iterator_Rand::ToString(Signal sig) const
{
	String rtn = "<iterator:rands(";
	if (_cnt > 0) {
		rtn += NumberToString(_cnt);
	} else {
		rtn += "infinite";
	}
	if (_range > 0) {
		rtn += ", ";
		rtn += NumberToString(_range);
	}
	rtn += ")>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Range
//-----------------------------------------------------------------------------
Iterator_Range::~Iterator_Range()
{
}

Iterator *Iterator_Range::Clone()
{
	return new Iterator_Range(*this);
}

bool Iterator_Range::DoNext(Signal sig, Value &value)
{
	if (!((_numStep > 0)? (_num < _numEnd) : (_num > _numEnd))) return false;
	value = Value(_num);
	_num += _numStep;
	return true;
}

String Iterator_Range::ToString(Signal sig) const
{
	String rtn;
	if (_numStep == 1 || _numStep == -1) {
		if (_numBegin == 0) {
			rtn = "<iterator:range(";
			rtn += NumberToString(_numEnd);
			rtn += ")>";
		} else {
			rtn = "<iterator:range(";
			rtn += NumberToString(_numBegin);
			rtn += ", ";
			rtn += NumberToString(_numEnd);
			rtn += ")>";
		}
	} else {
		rtn = "<iterator:range(";
		rtn += NumberToString(_numBegin);
		rtn += ", ";
		rtn += NumberToString(_numEnd);
		rtn += ", ";
		rtn += NumberToString(_numStep);
		rtn += ")>";
	}
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Sequence
//-----------------------------------------------------------------------------
bool Iterator_Sequence::IsSequence() const { return true; }
Iterator_Sequence::~Iterator_Sequence()
{
}

Iterator *Iterator_Sequence::Clone()
{
	return new Iterator_Sequence(*this);
}

bool Iterator_Sequence::DoNext(Signal sig, Value &value)
{
	if (!((_numStep > 0)? (_num <= _numEnd) : (_num >= _numEnd))) return false;
	value = Value(_num);
	_num += _numStep;
	return true;
}

String Iterator_Sequence::ToString(Signal sig) const
{
	String rtn;
	if (_numStep == 1. || _numStep == -1.) {
		rtn = "<iterator:";
		rtn += NumberToString(_numBegin);
		rtn += "..";
		rtn += NumberToString(_numEnd);
		rtn += ">";
	} else {
		rtn = "<iterator:seq(";
		rtn += NumberToString(_numBegin);
		rtn += ", ";
		rtn += NumberToString(_numEnd);
		rtn += ", ";
		rtn += NumberToString(_numStep);
		rtn += ")>";
	}
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_SequenceInf
//-----------------------------------------------------------------------------
bool Iterator_SequenceInf::IsSequenceInf() const { return true; }
Iterator_SequenceInf::~Iterator_SequenceInf()
{
}

Iterator *Iterator_SequenceInf::Clone()
{
	return new Iterator_SequenceInf(*this);
}

bool Iterator_SequenceInf::DoNext(Signal sig, Value &value)
{
	value = Value(_num);
	_num += 1;
	return true;
}

String Iterator_SequenceInf::ToString(Signal sig) const
{
	String rtn;
	rtn = "<iterator:";
	rtn += NumberToString(_numBegin);
	rtn += "..>";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Interval
//-----------------------------------------------------------------------------
Iterator_Interval::~Iterator_Interval()
{
}

Iterator *Iterator_Interval::Clone()
{
	return new Iterator_Interval(*this);
}

bool Iterator_Interval::DoNext(Signal sig, Value &value)
{
	Number num;
	if (!_DoNext(num)) return false;
	value = Value(num);
	return true;
}

String Iterator_Interval::ToString(Signal sig) const
{
	return String("<iterator:interval>");
}

#if 0
//-----------------------------------------------------------------------------
// Iterator_Fork
//-----------------------------------------------------------------------------
Iterator_Fork::Iterator_Fork(Environment &env, Signal sig,
		Function *pFunc, const Value &valueSelf, const ValueList &valListArg) :
	Iterator(false), _env(env), _pFunc(pFunc), _valueSelf(valueSelf), _doneFlag(false)
{
	_iterOwner.PrepareForMap(sig, pFunc->GetDeclList(), valListArg);
	_pValListToWrite = &_valListA;
	_pValListToRead = &_valListB;
	_pValueRead = _pValListToRead->begin();
	_readBlock.blockedFlag = false;
	_writeBlock.blockedFlag = false;
}

Iterator_Fork::~Iterator_Fork()
{
	Function::Delete(_pFunc);
}

bool Iterator_Fork::DoNext(Signal sig, Value &value)
{
	if (_pValueRead == _pValListToRead->end()) {
		for (;;) {
			SwapList();
			if (_pValueRead != _pValListToRead->end()) {
				break;
			} else if (_doneFlag) {
				return false;
			} else {
				_readBlock.blockedFlag = true;
				_readBlock.event.Wait();
			}
		}
	}
	value = *_pValueRead++;
	return true;
}

String Iterator_Fork::ToString(Signal sig) const
{
	return String("<iterator:fork>");
}

void Iterator_Fork::SwapList()
{
	_semaphore.Wait();
	ValueList *pValList = _pValListToWrite;
	_pValListToWrite = _pValListToRead;
	_pValListToRead = pValList;
	_pValListToWrite->clear();
	_pValueRead = _pValListToRead->begin();
	if (_writeBlock.blockedFlag) {
		_writeBlock.blockedFlag = false;
		_writeBlock.event.Notify();
	}
	_semaphore.Release();
}

void Iterator_Fork::ForkProcess()
{
	Clone();
	Start();
}

void Iterator_Fork::Run()
{
	Signal sig;
	ValueList valList;
	while (_iterOwner.Next(sig, valList)) {
		Context context(valList);
		context.SetSelf(_valueSelf);
		Value value = _pFunc->Eval(_env, sig, context);
		if (sig.IsSignalled()) break;
		_semaphore.Wait();
		_pValListToWrite->push_back(value);
		_semaphore.Release();
		if (_readBlock.blockedFlag) {
			_readBlock.blockedFlag = false;
			_readBlock.event.Notify();
		}
	}
	_doneFlag = true;
}
#endif

//-----------------------------------------------------------------------------
// Iterator_ExplicitMap
//-----------------------------------------------------------------------------
Iterator_ExplicitMap::Iterator_ExplicitMap(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
		Iterator(pIterator->IsInfinite()), _env(env),
		_pIterator(pIterator), _pObjFunc(pObjFunc)
{
}

Iterator_ExplicitMap::Iterator_ExplicitMap(const Iterator_ExplicitMap &iter) :
		Iterator(iter), _env(iter._env), _pIterator(iter._pIterator->Clone()),
		_pObjFunc(dynamic_cast<Object_Function *>(iter._pObjFunc->IncRef()))
{
}

Iterator_ExplicitMap::~Iterator_ExplicitMap()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

Iterator *Iterator_ExplicitMap::Clone()
{
	return new Iterator_ExplicitMap(*this);
}

bool Iterator_ExplicitMap::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	ValueList valList(value);
	value = _pObjFunc->Eval(_env, sig, valList);
	return true;
}

String Iterator_ExplicitMap::ToString(Signal sig) const
{
	return String("<iterator:explicit_map>");
}

//-----------------------------------------------------------------------------
// Iterator_ImplicitMap
//-----------------------------------------------------------------------------
Iterator_ImplicitMap::Iterator_ImplicitMap(Environment &env, Signal sig, Function *pFunc,
		const Value &valueSelf, const ValueList &valListArg, bool skipInvalidFlag) :
	Iterator(false, skipInvalidFlag), _env(env), _pFunc(pFunc),
	_valueSelf(valueSelf)
{
	_iterOwner.PrepareForMap(sig, pFunc->GetDeclList(), valListArg);
	if (sig.IsSignalled()) return;
	SetInfiniteFlag(_iterOwner.IsInfinite());
}

Iterator_ImplicitMap::~Iterator_ImplicitMap()
{
	Function::Delete(_pFunc);
}

bool Iterator_ImplicitMap::DoNext(Signal sig, Value &value)
{
	ValueList valList;
	if (!_iterOwner.Next(sig, valList)) return false;
	Context context(valList);
	context.SetSelf(_valueSelf);
	value = _pFunc->Eval(_env, sig, context);
	if (sig.IsSignalled()) return false;
	return true;
}

String Iterator_ImplicitMap::ToString(Signal sig) const
{
	String str;
	str += "<iterator:implicit_map:";
	str += _pFunc->ToString();
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// Iterator_MemberMap
//-----------------------------------------------------------------------------
Iterator_MemberMap::~Iterator_MemberMap()
{
	Expr::Delete(_pExpr);
	Iterator::Delete(_pIterator);
}

bool Iterator_MemberMap::DoNext(Signal sig, Value &value)
{
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	ObjectBase *pObjEach = valueSelfEach.ExtractObject(sig);
	if (sig.IsSignalled()) return false;
	Environment &env = *pObjEach;
	value = _pExpr->Exec(env, sig);
	if (value.IsFunction()) {
		Object *pObj = new Object_WrappedMethod(env.LookupClass(VTYPE_Function),
							value.GetFunction()->IncRef(), valueSelfEach);
		value = Value(pObj, VTYPE_Function, true);
	}
	return true;
}

String Iterator_MemberMap::ToString(Signal sig) const
{
	return String("<iterator:member_map>");
}

//-----------------------------------------------------------------------------
// Iterator_MethodMap
//-----------------------------------------------------------------------------
Iterator_MethodMap::~Iterator_MethodMap()
{
	Expr::Delete(_pExprCaller);
	Iterator::Delete(_pIterator);
}

bool Iterator_MethodMap::DoNext(Signal sig, Value &value)
{
	const Function *pFuncSuccRequester = NULL;
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	value = _pExprCaller->EvalEach(_env, sig, valueSelfEach, &pFuncSuccRequester);
	return true;
}

String Iterator_MethodMap::ToString(Signal sig) const
{
	return String("<iterator:method_map>");
}

//-----------------------------------------------------------------------------
// Iterator_FuncBinder
//-----------------------------------------------------------------------------
Iterator_FuncBinder::Iterator_FuncBinder(Environment &env,
				Function *pFunc, const Value &valueSelf, Iterator *pIterator) :
		Iterator(false), _env(env), _pFunc(pFunc),
		_valueSelf(valueSelf), _pIterator(pIterator)
{
}

Iterator_FuncBinder::~Iterator_FuncBinder()
{
	Function::Delete(_pFunc);
	Iterator::Delete(_pIterator);
}

bool Iterator_FuncBinder::DoNext(Signal sig, Value &value)
{
	Value valueArg;
	if (!_pIterator->Next(sig, valueArg)) return false;
	if (valueArg.IsList()) {
		const Function *pFuncSuccRequester = NULL;
		Context context(SymbolSet::Null, SymbolSet::Null,
					NULL, &pFuncSuccRequester, Value::Null, valueArg.GetList());
		value = _pFunc->Eval(_env, sig, context);
		if (sig.IsSignalled()) return false;
	} else {
		sig.SetError(ERR_TypeError, "invalid structure of arguments");
		return false;
	}
	return true;
}

String Iterator_FuncBinder::ToString(Signal sig) const
{
	String str;
	str += "<iterator:func_binder:";
	str += _pFunc->ToString();
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// Iterator_Delay
//-----------------------------------------------------------------------------
Iterator_Delay::~Iterator_Delay()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Delay::DoNext(Signal sig, Value &value)
{
	OAL::Sleep(_delay);
	return _pIterator->Next(sig, value);
}

String Iterator_Delay::ToString(Signal sig) const
{
	return String("<iterator:delay>");
}

//-----------------------------------------------------------------------------
// Iterator_Skip
//-----------------------------------------------------------------------------
Iterator_Skip::~Iterator_Skip()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Skip::DoNext(Signal sig, Value &value)
{
	bool flag = _pIterator->Next(sig, value);
	for (int i = 0; i < _nSkip; i++) {
		Value valueTmp;
		if (!_pIterator->Next(sig, valueTmp)) break;
	}
	return flag;
}

String Iterator_Skip::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:skip:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_SkipInvalid
//-----------------------------------------------------------------------------
Iterator_SkipInvalid::~Iterator_SkipInvalid()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_SkipInvalid::DoNext(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		if (value.IsValid()) return true;
	}
	return false;
}

String Iterator_SkipInvalid::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:skip_invalid:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_FilterWithFunc
//-----------------------------------------------------------------------------
Iterator_FilterWithFunc::~Iterator_FilterWithFunc()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_FilterWithFunc::DoNext(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		ValueList valList(value);
		Value valueCriteria = _pObjFunc->Eval(_env, sig, valList);
		if (sig.IsSignalled()) return false;
		if (valueCriteria.GetBoolean()) return true;
	}
	return false;
}

String Iterator_FilterWithFunc::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:filter_with_func:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_FilterWithIter
//-----------------------------------------------------------------------------
Iterator_FilterWithIter::~Iterator_FilterWithIter()
{
	Iterator::Delete(_pIterator);
	Iterator::Delete(_pIteratorCriteria);
}

bool Iterator_FilterWithIter::DoNext(Signal sig, Value &value)
{
	while (_pIterator->Next(sig, value)) {
		Value valueCriteria;
		if (!_pIteratorCriteria->Next(sig, valueCriteria)) break;
		if (valueCriteria.GetBoolean()) return true;
	}
	return false;
}

String Iterator_FilterWithIter::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:filter_with_iter:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_WhileWithFunc
//-----------------------------------------------------------------------------
Iterator_WhileWithFunc::~Iterator_WhileWithFunc()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_WhileWithFunc::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	ValueList valList(value);
	Value valueCriteria = _pObjFunc->Eval(_env, sig, valList);
	if (sig.IsSignalled()) return false;
	return valueCriteria.GetBoolean();
}

String Iterator_WhileWithFunc::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:while_with_func:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_WhileWithIter
//-----------------------------------------------------------------------------
Iterator_WhileWithIter::~Iterator_WhileWithIter()
{
	Iterator::Delete(_pIterator);
	Iterator::Delete(_pIteratorCriteria);
}

bool Iterator_WhileWithIter::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	Value valueCriteria;
	if (!_pIteratorCriteria->Next(sig, valueCriteria)) return false;
	return valueCriteria.GetBoolean();
}

String Iterator_WhileWithIter::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:while_with_iter:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_SinceWithFunc
//-----------------------------------------------------------------------------
Iterator_SinceWithFunc::~Iterator_SinceWithFunc()
{
	Iterator::Delete(_pIterator);
	Object::Delete(_pObjFunc);
}

bool Iterator_SinceWithFunc::DoNext(Signal sig, Value &value)
{
	for (;;) {
		if (!_pIterator->Next(sig, value)) return false;
		if (_pObjFunc == NULL) break;
		ValueList valList(value);
		Value valueCriteria = _pObjFunc->Eval(_env, sig, valList);
		if (sig.IsSignalled()) return false;
		if (valueCriteria.GetBoolean()) {
			Object::Delete(_pObjFunc);
			_pObjFunc = NULL;
			break;
		}
	}
	return true;
}

String Iterator_SinceWithFunc::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:since_with_func:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_SinceWithIter
//-----------------------------------------------------------------------------
Iterator_SinceWithIter::~Iterator_SinceWithIter()
{
	Iterator::Delete(_pIterator);
	Iterator::Delete(_pIteratorCriteria);
}

bool Iterator_SinceWithIter::DoNext(Signal sig, Value &value)
{
	for (;;) {
		if (!_pIterator->Next(sig, value)) return false;
		if (_pIteratorCriteria == NULL) break;
		Value valueCriteria;
		if (!_pIteratorCriteria->Next(sig, valueCriteria)) return false;
		if (valueCriteria.GetBoolean()) {
			Iterator::Delete(_pIteratorCriteria);
			_pIteratorCriteria = NULL;
			break;
		}
	}
	return true;
}

String Iterator_SinceWithIter::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:since_with_iter:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Replace
//-----------------------------------------------------------------------------
Iterator_Replace::~Iterator_Replace()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Replace::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	if (Value::Compare(value, _value) == 0) value = _valueReplace;
	return true;
}

String Iterator_Replace::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:replace:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_ReplaceInvalid
//-----------------------------------------------------------------------------
Iterator_ReplaceInvalid::~Iterator_ReplaceInvalid()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_ReplaceInvalid::DoNext(Signal sig, Value &value)
{
	if (!_pIterator->Next(sig, value)) return false;
	if (value.IsInvalid()) value = _valueReplace;
	return true;
}

String Iterator_ReplaceInvalid::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:replace_invalid:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Format
//-----------------------------------------------------------------------------
Iterator_Format::~Iterator_Format()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Format::DoNext(Signal sig, Value &value)
{
	Value valueSrc;
	if (!_pIterator->Next(sig, valueSrc)) return false;
	String str;
	if (valueSrc.IsList()) {
		str = Formatter::Format(sig, _format.c_str(), valueSrc.GetList());
	} else {
		ValueList valList(valueSrc);
		str = Formatter::Format(sig, _format.c_str(), valList);
	}
	value = Value(_env, str.c_str());
	return true;
}

String Iterator_Format::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:format:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Pack
//-----------------------------------------------------------------------------
Iterator_Pack::~Iterator_Pack()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Pack::DoNext(Signal sig, Value &value)
{
	Value valueSrc;
	if (!_pIterator->Next(sig, valueSrc)) return false;
	Object_Bytes *pObjBytes = value.InitAsBytes(_env);
	size_t offset = 0;
	if (valueSrc.IsList()) {
		pObjBytes->Pack(sig, offset, _format.c_str(), valueSrc.GetList());
	} else {
		ValueList valList(valueSrc);
		pObjBytes->Pack(sig, offset, _format.c_str(), valList);
	}
	if (sig.IsSignalled()) {
		value = Value::Null;
		return false;
	}
	return true;
}

String Iterator_Pack::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:pack:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Zip
//-----------------------------------------------------------------------------
Iterator_Zip::~Iterator_Zip()
{
}

bool Iterator_Zip::DoNext(Signal sig, Value &value)
{
	ValueList &valList = value.InitAsList(_env);
	return _iterOwner.Next(sig, valList);
}

String Iterator_Zip::ToString(Signal sig) const
{
	return String("<iterator:zip>");
}

//-----------------------------------------------------------------------------
// Iterator_Align
//-----------------------------------------------------------------------------
Iterator_Align::~Iterator_Align()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Align::DoNext(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	if (!_pIterator->Next(sig, value)) value = _valueFill;
	return true;
}

String Iterator_Align::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:align:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Head
//-----------------------------------------------------------------------------
Iterator_Head::~Iterator_Head()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Head::DoNext(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	if (_cnt > 0) _cnt--;
	return _pIterator->Next(sig, value);
}

String Iterator_Head::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:head:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Fold
//-----------------------------------------------------------------------------
Iterator_Fold::~Iterator_Fold()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Fold::DoNext(Signal sig, Value &value)
{
	Value valueNext;
	if (!_pIterator->Next(sig, valueNext)) return false;
	value.InitAsIterator(_env,
				new Iterator_FoldSeg(_pIterator->IncRef(), _cnt, valueNext));
	return true;
}

String Iterator_Fold::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:fold:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_FoldSeg
//-----------------------------------------------------------------------------
Iterator_FoldSeg::~Iterator_FoldSeg()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_FoldSeg::DoNext(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	value = _valueNext;
	_cnt--;
	if (_cnt > 0 && !_pIterator->Next(sig, _valueNext)) _cnt = 0;
	return true;
}

String Iterator_FoldSeg::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:fold_seg:";
	rtn += _pIterator->ToString(sig);
	rtn += ">";
	return rtn;
}

//-----------------------------------------------------------------------------
// Iterator_Concat
//-----------------------------------------------------------------------------
Iterator_Concat::~Iterator_Concat()
{
}

bool Iterator_Concat::DoNext(Signal sig, Value &value)
{
	for ( ; _ppIterator != _iterOwner.end(); _ppIterator++) {
		Iterator *pIterator = *_ppIterator;
		if (pIterator->Next(sig, value)) return true;
	}
	return false;
}

String Iterator_Concat::ToString(Signal sig) const
{
	String rtn;
	rtn += "<iterator:concat>";
	return rtn;
}

void Iterator_Concat::Add(Iterator *pIterator)
{
	_iterOwner.push_back(pIterator);
	_ppIterator = _iterOwner.begin();
	if (pIterator->IsInfinite()) _infiniteFlag = true;
}


//-----------------------------------------------------------------------------
// Iterator_repeat
//-----------------------------------------------------------------------------
Iterator_repeat::Iterator_repeat(Environment &env, Signal sig, Function *pFuncBlock,
					bool skipInvalidFlag, bool standaloneFlag, int cnt) :
		Iterator(cnt < 0, skipInvalidFlag), _env(env), _pFuncBlock(pFuncBlock),
		_standaloneFlag(standaloneFlag),
		_pIteratorSub(NULL), _cnt(cnt), _idx(0)
{
}

Iterator_repeat::~Iterator_repeat()
{
	Function::Delete(_pFuncBlock);
	Iterator::Delete(_pIteratorSub);
}

bool Iterator_repeat::DoNext(Signal sig, Value &value)
{
	for (;;) {
		if (_pIteratorSub == NULL) {
			if (_cnt >= 0 && _idx >= _cnt) return false;
			ValueList valListArg(Value(static_cast<Number>(_idx)));
			Context context(valListArg);
			value = _pFuncBlock->Eval(_env, sig, context);
			if (sig.IsBreak()) {
				sig.ClearSignal();
				return false;
			} else if (sig.IsReturn()) {
				if (_standaloneFlag) {
					sig.ClearSignal();
				}
				return false;
			} else if (sig.IsContinue()) {
				value = sig.GetValue();
				sig.ClearSignal();
			} else if ((sig).IsSignalled()) {
				return false;
			}
			_idx++;
			if (_standaloneFlag && value.IsIterator()) {
				_pIteratorSub = value.GetIterator()->IncRef();
				continue;
			}
		} else if (!_pIteratorSub->Next(sig, value)) {
			Iterator::Delete(_pIteratorSub);
			_pIteratorSub = NULL;
			if (sig.IsSignalled()) return false;
			continue;
		}
		break;
	}
	return true;
}

String Iterator_repeat::ToString(Signal sig) const
{
	return String("<iterator:repeat>");
}

//-----------------------------------------------------------------------------
// Iterator_while
//-----------------------------------------------------------------------------
Iterator_while::Iterator_while(Environment &env, Signal sig, Function *pFuncBlock,
					bool skipInvalidFlag, bool standaloneFlag, Expr *pExpr) :
		Iterator(false, skipInvalidFlag), _env(env), _pFuncBlock(pFuncBlock),
		_standaloneFlag(standaloneFlag),
		_pIteratorSub(NULL), _pExpr(pExpr->IncRef()), _idx(0)
{
}

Iterator_while::~Iterator_while()
{
	Function::Delete(_pFuncBlock);
	Iterator::Delete(_pIteratorSub);
	Expr::Delete(_pExpr);
}

bool Iterator_while::DoNext(Signal sig, Value &value)
{
	for (;;) {
		if (_pIteratorSub == NULL) {
			if (!_pExpr->Exec(_env, sig).GetBoolean()) return false;
			ValueList valListArg(Value(static_cast<Number>(_idx)));
			Context context(valListArg);
			value = _pFuncBlock->Eval(_env, sig, context);
			if (sig.IsBreak()) {
				sig.ClearSignal();
				return false;
			} else if (sig.IsReturn()) {
				if (_standaloneFlag) {
					sig.ClearSignal();
				}
				return false;
			} else if (sig.IsContinue()) {
				value = sig.GetValue();
				sig.ClearSignal();
			} else if ((sig).IsSignalled()) {
				return false;
			}
			_idx++;
			if (_standaloneFlag && value.IsIterator()) {
				_pIteratorSub = value.GetIterator()->IncRef();
				continue;
			}
		} else if (!_pIteratorSub->Next(sig, value)) {
			Iterator::Delete(_pIteratorSub);
			_pIteratorSub = NULL;
			if (sig.IsSignalled()) return false;
			continue;
		}
		break;
	}
	return true;
}

String Iterator_while::ToString(Signal sig) const
{
	return String("<iterator:while>");
}

//-----------------------------------------------------------------------------
// Iterator_for
//-----------------------------------------------------------------------------
Iterator_for::Iterator_for(Environment &env, Signal sig, Function *pFuncBlock,
			bool skipInvalidFlag, bool standaloneFlag, const ValueList &valListArg) :
		Iterator(false, skipInvalidFlag), _env(env), _pFuncBlock(pFuncBlock),
		_standaloneFlag(standaloneFlag),
		_pIteratorSub(NULL), _idx(0), _doneFlag(false)
{
	PrepareRepeaterIterators(_env, sig, valListArg, _exprLeftList, _iteratorOwner);
}

Iterator_for::~Iterator_for()
{
	Function::Delete(_pFuncBlock);
	Iterator::Delete(_pIteratorSub);
}

bool Iterator_for::DoNext(Signal sig, Value &value)
{
	for (;;) {
		if (_pIteratorSub == NULL) {
			if (_doneFlag) return false;
			ExprList::iterator ppExprLeft = _exprLeftList.begin();
			foreach_const (IteratorOwner, ppIterator, _iteratorOwner) {
				Iterator *pIterator = *ppIterator;
				Value valueVar;
				if (!pIterator->Next(sig, valueVar)) {
					if (sig.IsSignalled()) return false;
					_doneFlag = true;
					break;
				}
				(*ppExprLeft)->DoAssign(_env, sig, valueVar, NULL, false);
				if (sig.IsSignalled()) return false;
				ppExprLeft++;
			}
			if (_doneFlag) return false;
			ValueList valListArg(Value(static_cast<Number>(_idx)));
			Context context(valListArg);
			value = _pFuncBlock->Eval(_env, sig, context);
			if (sig.IsBreak()) {
				value = sig.GetValue();
				sig.ClearSignal();
				return false;
			} else if (sig.IsReturn()) {
				if (_standaloneFlag) {
					value = sig.GetValue();
					sig.ClearSignal();
				}
				return false;
			} else if (sig.IsContinue()) {
				value = sig.GetValue();
				sig.ClearSignal();
			} else if ((sig).IsSignalled()) {
				return false;
			}
			_idx++;
			if (_standaloneFlag && value.IsIterator()) {
				_pIteratorSub = value.GetIterator()->IncRef();
				continue;
			}
		} else if (!_pIteratorSub->Next(sig, value)) {
			Iterator::Delete(_pIteratorSub);
			_pIteratorSub = NULL;
			if (sig.IsSignalled()) return false;
			continue;
		}
		break;
	}
	return true;
}

String Iterator_for::ToString(Signal sig) const
{
	return String("<iterator:for>");
}

//-----------------------------------------------------------------------------
// Iterator_cross
//-----------------------------------------------------------------------------
Iterator_cross::Iterator_cross(Environment &env, Signal sig, Function *pFuncBlock,
			bool skipInvalidFlag, bool standaloneFlag, const ValueList &valListArg) :
		Iterator(false, skipInvalidFlag), _env(env), _pFuncBlock(pFuncBlock),
		_standaloneFlag(standaloneFlag),
		_pIteratorSub(NULL), _idx(0), _doneFlag(true)
{
	if (!PrepareRepeaterIterators(_env, sig, valListArg,
									_exprLeftList, _iteratorOwnerOrg)) return;
	_valListArg.reserve(_iteratorOwnerOrg.size() + 1);
	_iteratorOwner.reserve(_iteratorOwnerOrg.size());
	_valListArg.push_back(Value(0));
	ExprList::iterator ppExprLeft = _exprLeftList.begin();
	foreach (IteratorOwner, ppIteratorOrg, _iteratorOwnerOrg) {
		Iterator *pIterator = (*ppIteratorOrg)->Clone();
		Value valueVar;
		bool rtn = pIterator->Next(sig, valueVar);
		if (sig.IsSignalled()) return;
		if (rtn) {
			_doneFlag = false;
			_valListArg.push_back(Value(0));
			_iteratorOwner.push_back(pIterator);
		} else {
			_iteratorOwner.push_back(NULL);
			_valListArg.push_back(Value::Null);
			Iterator::Delete(pIterator);
		}
		(*ppExprLeft)->DoAssign(_env, sig, valueVar, NULL, false);
		if (sig.IsSignalled()) return;
		ppExprLeft++;
	}
}

Iterator_cross::~Iterator_cross()
{
	Function::Delete(_pFuncBlock);
	Iterator::Delete(_pIteratorSub);
}

bool Iterator_cross::DoNext(Signal sig, Value &value)
{
	for (;;) {
		if (_pIteratorSub == NULL) {
			if (_doneFlag) return false;
			_valListArg[0] = Value(_idx);
			Context context(_valListArg);
			value = _pFuncBlock->Eval(_env, sig, context);
			if (sig.IsBreak()) {
				value = sig.GetValue();
				sig.ClearSignal();
				return false;
			} else if (sig.IsReturn()) {
				if (_standaloneFlag) {
					value = sig.GetValue();
					sig.ClearSignal();
				}
				return false;
			} else if (sig.IsContinue()) {
				value = sig.GetValue();
				sig.ClearSignal();
			} else if ((sig).IsSignalled()) {
				return false;
			}
			_idx++;
			if (_standaloneFlag && value.IsIterator()) {
				_pIteratorSub = value.GetIterator()->IncRef();
				continue;
			}
			if (!AdvanceIterators(sig)) return false;
		} else if (!_pIteratorSub->Next(sig, value)) {
			Iterator::Delete(_pIteratorSub);
			_pIteratorSub = NULL;
			if (sig.IsSignalled()) return false;
			if (!AdvanceIterators(sig)) return false;
			continue;
		}
		break;
	}
	return true;
}

String Iterator_cross::ToString(Signal sig) const
{
	return String("<iterator:cross>");
}

bool Iterator_cross::AdvanceIterators(Signal sig)
{
	ExprList::reverse_iterator ppExprLeft = _exprLeftList.rbegin();
	ValueList::reverse_iterator pValueArg = _valListArg.rbegin();
	IteratorOwner::reverse_iterator ppIteratorOrg = _iteratorOwnerOrg.rbegin();
	foreach_reverse (IteratorOwner, ppIterator, _iteratorOwner) {
		Iterator *pIterator = *ppIterator;
		Value valueVar;
		if (pIterator != NULL) {
			if (pIterator->Next(sig, valueVar)) {
				*pValueArg = Value(pValueArg->GetNumber() + 1);
				(*ppExprLeft)->DoAssign(_env, sig, valueVar, NULL, false);
				if (sig.IsSignalled()) return false;
				break;
			}
			if (sig.IsSignalled()) return false;
			Iterator::Delete(pIterator);
			pIterator = (*ppIteratorOrg)->Clone();
			pIterator->Next(sig, valueVar);
			if (sig.IsSignalled()) return false;
			(*ppExprLeft)->DoAssign(_env, sig, valueVar, NULL, false);
			if (sig.IsSignalled()) return false;
			*pValueArg = Value(0);
			*ppIterator = pIterator;
		}
		if (ppIterator + 1 == _iteratorOwner.rend()) {
			_doneFlag = true;
			break;
		}
		ppExprLeft++;
		pValueArg++;
		ppIteratorOrg++;
	}
	return true;
}

//-----------------------------------------------------------------------------
// IteratorOwner
//-----------------------------------------------------------------------------
IteratorOwner::IteratorOwner(const IteratorOwner &iterOwner)
{
	reserve(iterOwner.size());
	foreach_const (IteratorOwner, ppIterator, iterOwner) {
		push_back((*ppIterator)->Clone());
	}
}

IteratorOwner::~IteratorOwner()
{
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator::Delete(*ppIterator);
	}
}

bool IteratorOwner::Next(Signal sig, ValueList &valList)
{
	valList.clear();
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator *pIterator = *ppIterator;
		Value value;
		if (!pIterator->Next(sig, value)) return false;
		valList.push_back(value);
	}
	return true;
}

bool IteratorOwner::IsInfinite() const
{
	foreach_const (IteratorOwner, ppIterator, *this) {
		if (!(*ppIterator)->IsInfinite()) return false;
	}
	return true;
}

bool IteratorOwner::PrepareForMap(Signal sig,
				const DeclarationList &declList, const ValueList &valListArg)
{
	//const DeclarationList &declList = GetDeclList();
	ValueList::const_iterator pValue = valListArg.begin();
	DeclarationList::const_iterator ppDecl = declList.begin();
	for ( ; pValue != valListArg.end() && ppDecl != declList.end(); pValue++) {
		const Declaration *pDecl = *ppDecl;
		Iterator *pIterator = NULL;
		if (pDecl->IsApplicable(*pValue)) {
			pIterator = new Iterator_Constant(*pValue);
		} else {
			pIterator = pValue->CreateIterator(sig);
			if (pIterator == NULL) return false;
		}
		push_back(pIterator);
		if (!pDecl->IsVariableLength()) ppDecl++;
	}
	return true;
}


}
