#include "DateTime.h"

namespace AScript {

//-----------------------------------------------------------------------------
// DateTime
//-----------------------------------------------------------------------------
#if defined(HAVE_WINDOWS_H)
DateTime::DateTime(const SYSTEMTIME &st)
{
	SetSystemTime(st);
}

DateTime::DateTime(const FILETIME &ft, bool toLocalFlag)
{
	SYSTEMTIME stUTC;
	::FileTimeToSystemTime(&ft, &stUTC);
	if (toLocalFlag) {
		SYSTEMTIME stLocal;
		::SystemTimeToTzSpecificLocalTime(NULL, &stUTC, &stLocal);
		SetSystemTime(stLocal);
	} else {
		SetSystemTime(stUTC);
	}
}

void DateTime::SetSystemTime(const SYSTEMTIME &st)
{
	_year	= static_cast<short>(st.wYear);
	_month	= static_cast<char>(st.wMonth);
	_day	= static_cast<char>(st.wDay);
	_sec	= static_cast<long>(st.wHour) * 3600 +
				static_cast<long>(st.wMinute) * 60 + st.wSecond;
	_usec	= static_cast<long>(st.wMilliseconds) * 1000;
}
#else
DateTime::DateTime(const struct tm &tm)
{
	SetTime(tm);
}

DateTime::DateTime(time_t t, bool toLocalFlag)
{
	struct tm tm;
	if (toLocalFlag) {
		localtime_r(&t, &tm);
	} else {
		gmtime_r(&t, &tm);
	}
	SetTime(tm);
}

void DateTime::SetTime(const struct tm &tm)
{
	_year	= static_cast<short>(tm.tm_year) + 1900;
	_month	= static_cast<char>(tm.tm_mon) + 1;
	_day	= static_cast<char>(tm.tm_mday);
	_sec	= static_cast<long>(tm.tm_hour) * 3600 +
				static_cast<long>(tm.tm_min) * 60 + tm.tm_sec;
	_usec	= 0;
}
#endif

void DateTime::Plus(const TimeDelta &td)
{
	long dayOfYear = GetDayOfYear();
	_usec += td.GetUSecs();
	if (_usec >= 1000000) {
		_usec -= 1000000;
		_sec++;
	}
	_sec += td.GetSecsRaw();
	if (_sec >= 3600 * 24) {
		_sec -= 3600 * 24;
		dayOfYear++;
	}
	dayOfYear += td.GetDays();
	if (dayOfYear >= 0) {
		while (dayOfYear >= _GetDaysOfYear(_year)) {
			dayOfYear -= _GetDaysOfYear(_year);
			_year++;
		}
	} else {
		_year--;
		while (dayOfYear < 0) {
			dayOfYear += _GetDaysOfYear(_year);
			_year--;
		}
	}
	_GetMonthDay(_year, static_cast<short>(dayOfYear), _month, _day);
}

void DateTime::Minus(const TimeDelta &td)
{
	Plus(TimeDelta(-td.GetDays(), -td.GetSecsRaw(), -td.GetUSecs()));
}

TimeDelta DateTime::Minus(const DateTime &dt) const
{
	long daysDiff = GetDayOfYear() - dt.GetDayOfYear();
	if (GetYear() < dt.GetYear()) {
		for (short year = GetYear(); year < dt.GetYear(); year++) {
			daysDiff -= _GetDaysOfYear(year);
		}
	} else if (GetYear() > dt.GetYear()) {
		for (short year = GetYear() - 1; year >= dt.GetYear(); year--) {
			daysDiff += _GetDaysOfYear(year);
		}
	}
	long secsDiff = GetSecRaw() - dt.GetSecRaw();
	long usecsDiff = GetUSec() - dt.GetUSec();
	return TimeDelta(daysDiff, secsDiff, usecsDiff);
}

String DateTime::ToString(const char *format) const
{
	String rtn;
	for (const char *p = format; *p != '\0'; p++) {
		if (*p != '%') {
			rtn.push_back(*p);
			continue;
		}
		char buff[32];
		p++;
		char ch = *p;
		if (ch == 'a') {			// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'A') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'b') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'B') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'c') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'd') {
			::sprintf(buff, "%02d", GetDay());
			rtn += buff;
		} else if (ch == 'H') {
			::sprintf(buff, "%02d", GetHour());
			rtn += buff;
		} else if (ch == 'I') {
			char hour = GetHour();
			::sprintf(buff, "%02d", (hour == 0)? 12 : (hour > 12)? hour - 12 : hour);
			rtn += buff;
		} else if (ch == 'j') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'm') {
			::sprintf(buff, "%02d", GetMonth());
			rtn += buff;
		} else if (ch == 'M') {
			::sprintf(buff, "%02d", GetMin());
			rtn += buff;
		} else if (ch == 'p') {		// locale
			
		} else if (ch == 'S') {
			::sprintf(buff, "%02d", GetSec());
			rtn += buff;
		} else if (ch == 'U') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'w') {
			::sprintf(buff, "%d", GetDayOfWeek());
			rtn += buff;
		} else if (ch == 'W') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'x') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'X') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'y') {
			::sprintf(buff, "%02d", GetYear() % 100);
			rtn += buff;
		} else if (ch == 'Y') {
			::sprintf(buff, "%04d", GetYear());
			rtn += buff;
		} else if (ch == 'Z') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == '%') {
			rtn.push_back(ch);
		} else {
			rtn.push_back(ch);
		}
	}
	return rtn;
}

char DateTime::_GetDaysOfMonth(short year, char month)
{
	static const char daysTbl_Normal[] = {
		0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31,
	};
	static const char daysTbl_Leap[] = {
		0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31,
	};
	if (month < 1 || month > 12) return -1;
	return (_IsLeapYear(year)? daysTbl_Leap : daysTbl_Normal)[month];
}

short DateTime::_GetDayOfYear(short year, char month, char day)
{
	static const short offsetTbl_Normal[] = {
		0, 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334,
	};
	static const short offsetTbl_Leap[] = {
		0, 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335,
	};
	if (day < 1 || month < 1 || month > 12) return -1;
	return (_IsLeapYear(year)? offsetTbl_Leap : offsetTbl_Normal)[month] + day - 1;
}

void DateTime::_GetMonthDay(short year, short dayOfYear, char &month, char &day)
{
	static const short offsetTbl_Normal[] = {
		0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334,
	};
	static const short offsetTbl_Leap[] = {
		0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335,
	};
	const short *offsetTbl = _IsLeapYear(year)? offsetTbl_Leap : offsetTbl_Normal;
	int i = 0;
	for ( ; i < 12 && dayOfYear >= offsetTbl[i]; i++) ;
	i--;
	month = static_cast<char>(i + 1);
	day = static_cast<char>(dayOfYear - offsetTbl[i] + 1);
}

// Zeller's congruence. treat 0 as Sunday
char DateTime::_GetDayOfWeek(short year, char month, char day)
{
	if (day < 1 || month < 1 || month > 12) return -1;
	if (month <= 2) {
		month += 12, year -= 1;
	}
	short yearH = year / 100, yearL = year % 100;
	int rtn = yearL + yearL / 4 + yearH / 4 - 2 * yearH +
							13 * (static_cast<int>(month) + 1) / 5 + day;
	rtn %= 7;
	if (rtn < 0) rtn += 7;
	return (rtn + 6) % 7;
}

int DateTime::Compare(const DateTime &dt1, const DateTime &dt2)
{
	long result = 0;
	if ((result = dt1._year - dt2._year) != 0) {
	} else if ((result = dt1._month - dt2._month) != 0) {
	} else if ((result = dt1._day - dt2._day) != 0) {
	} else if ((result = dt1._sec - dt2._sec) != 0) {
	} else if ((result = dt1._usec - dt2._usec) != 0) {
	}
	return (result < 0)? -1 : (result > 0)? +1 : 0;
}

//-----------------------------------------------------------------------------
// TimeDelta
//-----------------------------------------------------------------------------
TimeDelta::TimeDelta(long days, long secs, long usecs) :
								_days(days), _secs(secs), _usecs(usecs)
{
	_secs += _usecs / 1000000;
	_usecs %= 1000000;
	_days += _secs / (3600 * 24);
	_secs %= 3600 * 24;
	if (_usecs < 0) {
		_usecs += 1000000;
		_secs--;
	}
	if (_secs < 0) {
		_secs += 3600 * 24;
		_days--;
	}
}

String TimeDelta::ToString(const char *format) const
{
	char buff[80];
	::sprintf(buff, "%lddays,%ldsecs(%02d:%02d:%02d),%ldusecs",
		GetDays(), GetSecsRaw(), GetHours(), GetMins(), GetSecs(), GetUSecs());
	return String(buff);
}

int TimeDelta::Compare(const TimeDelta &td1, const TimeDelta &td2)
{
	long result = 0;
	if ((result = td1._days - td2._days) != 0) {
	} else if ((result = td1._secs - td2._secs) != 0) {
	} else if ((result = td1._usecs - td2._usecs) != 0) {
	}
	return (result < 0)? -1 : (result > 0)? +1 : 0;
}

}
