// OAL: OS Abstraction Layer
#include "OAL.h"

namespace AScript {
namespace OAL {

#if defined(HAVE_WINDOWS_H)

//=============================================================================
// Windows API
//=============================================================================
const char *getenv(const char *name)
{
	return ::getenv(name);
}
	
void rename(const char *src, const char *dst)
{
	::MoveFileEx(src, dst, MOVEFILE_REPLACE_EXISTING);
}

void remove(const char *pathName)
{
	::DeleteFile(pathName);
}

bool mkdir(const char *pathName, bool recursiveFlag)
{
	return ::CreateDirectory(pathName, NULL)? true : false;
}

bool chdir(const char *pathName)
{
	return ::SetCurrentDirectory(pathName)? true : false;
}

String getcwd()
{
	char pathName[MAX_PATH];
	::GetCurrentDirectory(MAX_PATH, pathName);
	return String(pathName);
}

void exec(const char *pathName, const char **argv)
{
#if 0
	size_t lenSum = 1;
	for (const char **ppArg = argv + 1; *ppArg != NULL; ppArg++) {
		const char *pArg = *ppArg;
		lenSum += ::strlen(pArg) + 1;
	}
	char *commandLine = new char[lenSum];
	char *p = commandLine;
	*p = '\0';
	for (const char **ppArg = argv + 1; *ppArg != NULL; ppArg++) {
		const char *pArg = *ppArg;
		if (p != commandLine) *p++ = ' ';
		::strcpy(p, pArg);
		p += ::strlen(pArg);
	}
	char *environment = "\0";
	STARTUPINFO si;
	PROCESS_INFORMATION ps;
	::memset(&si, 0x00, sizeof(si));
	::CreateProcess(pathName, commandLine,
						NULL, NULL, FALSE, 0, environment, NULL, &si, &ps);
	delete[] commandLine;
#endif
}

void Sleep(Number delay)	// unit: sec
{
	::Sleep(static_cast<long>(delay * 1000));	// unit: msec
}

//-----------------------------------------------------------------------------
// Thread
//-----------------------------------------------------------------------------
static DWORD WINAPI ThreadProc(LPVOID lpParameter)
{
	reinterpret_cast<Thread *>(lpParameter)->Run();
	return 0;
}

void Thread::Start()
{
	DWORD threadId;
	::CreateThread(NULL, 0, ThreadProc, this, 0, &threadId);
}

//-----------------------------------------------------------------------------
// Semaphore
//-----------------------------------------------------------------------------
Semaphore::Semaphore()
{
	_hMutex = ::CreateMutex(NULL, FALSE, NULL);
}

Semaphore::~Semaphore()
{
	::DeleteObject(_hMutex);
}

void Semaphore::Wait()
{
	::WaitForSingleObject(_hMutex, INFINITE);
}

void Semaphore::Release()
{
	::ReleaseMutex(_hMutex);
}

//-----------------------------------------------------------------------------
// Event
//-----------------------------------------------------------------------------
Event::Event()
{
	_hEvent = ::CreateEvent(NULL, FALSE, FALSE, NULL);
}

Event::~Event()
{
	::DeleteObject(_hEvent);
}

void Event::Wait()
{
	::WaitForSingleObject(_hEvent, INFINITE);
}

void Event::Notify()
{
	::SetEvent(_hEvent);
}

#else
//=============================================================================
// POSIX
//=============================================================================
const char *getenv(const char *name)
{
	return ::getenv(name);
}
	
void rename(const char *src, const char *dst)
{
	::rename(src, dst);
}

void remove(const char *pathName)
{
	::unlink(pathName);
}

bool mkdir(const char *pathName, bool recursiveFlag)
{
	return ::mkdir(pathName, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) == 0;
}

bool chdir(const char *pathName)
{
	return ::chdir(pathName) == 0;
}

String getcwd()
{
	char *pathName = ::get_current_dir_name();
	String rtn(pathName);
	::free(pathName);
	return rtn;
}

void exec(const char *pathName, const char **argv)
{
#if 0
	::execv(pathName, args);
#endif
}

void Sleep(Number delay)	// unit: sec
{
	::usleep(static_cast<unsigned long>(delay * 1000000));	// unit: usec
}

//-----------------------------------------------------------------------------
// Thread
//-----------------------------------------------------------------------------
void Thread::Start()
{
}

//-----------------------------------------------------------------------------
// Semaphore
//-----------------------------------------------------------------------------
#if defined(HAVE_SEMAPHORE_H)
// should use mutex instead?
Semaphore::Semaphore()
{
	::sem_init(&_sem, 0, 1);
}

Semaphore::~Semaphore()
{
	::sem_destroy(&_sem);
}

void Semaphore::Wait()
{
	::sem_wait(&_sem);
}

void Semaphore::Release()
{
	::sem_post(&_sem);
}
#else
#endif

//-----------------------------------------------------------------------------
// Event
//-----------------------------------------------------------------------------
#if defined(HAVE_SEMAPHORE_H)
Event::Event()
{
}

Event::~Event()
{
}

void Event::Wait()
{
}

void Event::Notify()
{
}
#else
#endif

#endif

}
}
