//-----------------------------------------------------------------------------
// AScript re module
//-----------------------------------------------------------------------------
#include "Module.h"
#include "oniguruma.h"
#include "Object_String.h"

AScript_BeginModule(re)

AScript_DeclarePrivSymbol(re);
AScript_DeclarePrivSymbol(string);
AScript_DeclarePrivSymbol(multiline);

AScript_DeclarePrivClass(Match);
AScript_DeclarePrivClass(RegEx);

static regex_t *CreateRegEx(Signal sig, const char *pattern, const SymbolSet &attrs);
static Value DoMatch(Environment &env, Signal sig, regex_t *pRegEx, const char *str);
static Value DoSub(Environment &env, Signal sig, regex_t *pRegEx,
							const char *repl, const char *str, int cnt);
static Value DoSubWithFunc(Environment &env, Signal sig, regex_t *pRegEx,
							const Function *pFunc, const char *str, int cnt);
static void SetError_OnigurumaError(Signal sig, int rtn);
static void SetError_FailInOniguruma(Signal sig);

//-----------------------------------------------------------------------------
// Object_Match class declaration
//-----------------------------------------------------------------------------
class Object_Match : public Object {
public:
	class Group {
	private:
		int _posBegin, _posEnd;
	public:
		inline Group(int posBegin, int posEnd) :
						_posBegin(posBegin), _posEnd(posEnd) {}
		inline Group(const Group &group) :
						_posBegin(group._posBegin), _posEnd(group._posEnd) {}
		inline void operator=(const Group &group) {
			_posBegin = group._posBegin, _posEnd = group._posEnd;
		}
		inline int GetPosBegin() const { return _posBegin; }
		inline int GetPosEnd() const { return _posEnd; }
		inline int GetLength() const { return _posEnd - _posBegin; }
	};
	typedef std::vector<Group> GroupList;
	typedef std::map<String, size_t> GroupNameDict;
public:
	inline static Object_Match *GetSelfObj(Context &context) {
		return dynamic_cast<Object_Match *>(context.GetSelfObj());
	}
private:
	String _str;
	GroupList _groupList;
	GroupNameDict _groupNameDict;
public:
	inline Object_Match(Environment &env) : Object(AScript_PrivClass(Match)) {}
	inline Object_Match(const Object_Match &obj) : Object(obj),
							_str(obj._str), _groupList(obj._groupList) {}
	virtual ~Object_Match();
	virtual Object *Clone() const;
	virtual String ToString(Signal sig, bool exprFlag);
	bool SetMatchInfo(const char *str, regex_t *pRegEx, const OnigRegion *pRegion);
	String GetGroup(Signal sig, size_t index) const;
	String GetGroup(GroupList::const_iterator pGroup) const {
		return String(_str, pGroup->GetPosBegin(), pGroup->GetLength());
	}
	String GetGroupByName(Signal sig, const char *name) const;
	const GroupList &GetGroupList() const { return _groupList; }
private:
	int ForeachNameCallback(const String &name, int nGroups,
											int *idxGroupTbl, regex_t *pRegEx);
	static int ForeachNameCallbackStub(
				const UChar *nameRaw, const UChar *nameRawEnd,
				int nGroups, int *idxGroupTbl, regex_t *pRegEx, void *pArg);
};

//-----------------------------------------------------------------------------
// Object_RegEx class declaration
//-----------------------------------------------------------------------------
class Object_RegEx : public Object {
private:
	String _pattern;
	regex_t *_pRegEx;
public:
	inline static Object_RegEx *GetSelfObj(Context &context) {
		return dynamic_cast<Object_RegEx *>(context.GetSelfObj());
	}
public:
	inline Object_RegEx(Environment &env) :
						Object(AScript_PrivClass(RegEx)), _pRegEx(NULL) {}
	virtual ~Object_RegEx();
	virtual Object *Clone() const;
	virtual String ToString(Signal sig, bool exprFlag);
	inline bool SetPattern(Signal sig, const char *pattern, const Context &context) {
		_pattern = pattern;
		_pRegEx = CreateRegEx(sig, pattern, context.GetAttrs());
		return _pRegEx != NULL;
	}
	inline regex_t *GetRegEx() { return _pRegEx; }
};

//-----------------------------------------------------------------------------
// IteratorSplit class declaration
//-----------------------------------------------------------------------------
class IteratorSplit : public Iterator {
private:
	Object_RegEx *_pObjRegEx;
	Object_String *_pObjStr;
	int _cnt, _cntMax;
	int _pos;
	int _len;
	bool _doneFlag;
	OnigRegion *_pRegion;
public:
	IteratorSplit(Object_RegEx *pObjRegEx, Object_String *pObjStr, int cntMax);
	virtual ~IteratorSplit();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// IteratorScan class declaration
//-----------------------------------------------------------------------------
class IteratorScan : public Iterator {
private:
	Object_RegEx *_pObjRegEx;
	Object_String *_pObjStr;
	int _cnt, _cntMax;
	int _pos;
	int _len;
	OnigRegion *_pRegion;
public:
	IteratorScan(Object_RegEx *pObjRegEx, Object_String *pObjStr, int cntMax);
	virtual ~IteratorScan();
	virtual bool DoNext(Signal sig, Value &value);
	virtual String ToString(Signal sig) const;
};

//-----------------------------------------------------------------------------
// IteratorSplit
//-----------------------------------------------------------------------------
IteratorSplit::IteratorSplit(Object_RegEx *pObjRegEx, Object_String *pObjStr, int cntMax) :
		Iterator(false), _pObjRegEx(pObjRegEx), _pObjStr(pObjStr),
		_cnt(cntMax), _cntMax(cntMax), _pos(0),
		_len(static_cast<int>(pObjStr->_GetString().size())), _doneFlag(false),
		_pRegion(::onig_region_new())
{
}

IteratorSplit::~IteratorSplit()
{
	Object::Delete(_pObjRegEx);
	Object::Delete(_pObjStr);
	::onig_region_free(_pRegion, 1); // 1:free self, 0:free contents only
}

bool IteratorSplit::DoNext(Signal sig, Value &value)
{
	Environment &env = *_pObjStr;
	const char *str = _pObjStr->GetString();
	if (_doneFlag) return false;
	if (_cnt == 0) {
		value = Value(env, str + _pos);
		_pos = _len;
		return true;
	} else if (_pos >= _len) {
		value = Value(env, "");
		_doneFlag = true;
		return true;
	}
	int rtn = ::onig_search(_pObjRegEx->GetRegEx(),
					reinterpret_cast<const OnigUChar *>(str),
					reinterpret_cast<const OnigUChar *>(str + _len),
					reinterpret_cast<const OnigUChar *>(str + _pos),
					reinterpret_cast<const OnigUChar *>(str + _len),
					_pRegion, ONIG_OPTION_NONE);
	if (rtn >= 0) {
		if (rtn < _pos || _pRegion->num_regs == 0 || _pRegion->end[0] < _pos) {
			SetError_FailInOniguruma(sig);
			return false;
		}
		if (_pRegion->end[0] == _pos) {
			value = Value(env, str + _pos);
			_doneFlag = true;
			return true;
		}
		value = Value(env, String(str + _pos, rtn - _pos).c_str());
		_pos = _pRegion->end[0];
	} else if (rtn == ONIG_MISMATCH) {
		value = Value(env, str + _pos);
		_pos = _len;
		_doneFlag = true;
	} else { // error
		SetError_OnigurumaError(sig, rtn);
		return false;
	}
	if (_cnt > 0) _cnt--;
	return true;
}

String IteratorSplit::ToString(Signal sig) const
{
	return String("<iterator:re.split>");
}

//-----------------------------------------------------------------------------
// IteratorScan
//-----------------------------------------------------------------------------
IteratorScan::IteratorScan(Object_RegEx *pObjRegEx, Object_String *pObjStr, int cntMax) :
		Iterator(false), _pObjRegEx(pObjRegEx), _pObjStr(pObjStr),
		_cnt(cntMax), _cntMax(cntMax), _pos(0),
		_len(static_cast<int>(pObjStr->_GetString().size())), _pRegion(::onig_region_new())
{
}

IteratorScan::~IteratorScan()
{
	Object::Delete(_pObjRegEx);
	Object::Delete(_pObjStr);
	::onig_region_free(_pRegion, 1); // 1:free self, 0:free contents only
}

bool IteratorScan::DoNext(Signal sig, Value &value)
{
	Environment &env = *_pObjStr;
	const char *str = _pObjStr->GetString();
	if (_cnt == 0) return false;
	int rtn = ::onig_search(_pObjRegEx->GetRegEx(),
					reinterpret_cast<const OnigUChar *>(str),
					reinterpret_cast<const OnigUChar *>(str + _len),
					reinterpret_cast<const OnigUChar *>(str + _pos),
					reinterpret_cast<const OnigUChar *>(str + _len),
					_pRegion, ONIG_OPTION_NONE);
	if (rtn >= 0) {
		if (rtn < _pos || _pRegion->num_regs == 0 || _pRegion->end[0] < _pos) {
			SetError_FailInOniguruma(sig);
			return false;
		}
		if (_pRegion->end[0] == _pos) {
			return false;
		}
		Object_Match *pObj = new Object_Match(env);
		if (!pObj->SetMatchInfo(str, _pObjRegEx->GetRegEx(), _pRegion)) {
			SetError_FailInOniguruma(sig);
			delete pObj;
			return false;
		}
		value = Value(pObj, AScript_PrivVTYPE(Match));
		_pos = _pRegion->end[0];
	} else if (rtn == ONIG_MISMATCH) {
		value = Value(env, str + _pos);
		_pos = _len;
		return false;
	} else { // error
		SetError_OnigurumaError(sig, rtn);
		return false;
	}
	if (_cnt > 0) _cnt--;
	return true;
}

String IteratorScan::ToString(Signal sig) const
{
	return String("<iterator:re.scan>");
}

//-----------------------------------------------------------------------------
// Object_RegEx
//-----------------------------------------------------------------------------
Object_RegEx::~Object_RegEx()
{
	if (_pRegEx != NULL) {
		::onig_free(_pRegEx);
	}
}

Object *Object_RegEx::Clone() const
{
	return NULL;
}

String Object_RegEx::ToString(Signal sig, bool exprFlag)
{
	String rtn;
	rtn += "<RegEx:'";
	rtn += _pattern;
	rtn += "'>";
	return rtn;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_RegEx
//-----------------------------------------------------------------------------
// m = re.regex#match(str:string):map
AScript_DeclareMethod(RegEx, match)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	SetHelp(
	"Applies a pattern matching to a string and returns a match object.");
}

AScript_ImplementMethod(RegEx, match)
{
	Object_RegEx *pObj = Object_RegEx::GetSelfObj(context);
	return DoMatch(env, sig, pObj->GetRegEx(), context.GetString(0));
}


// str = re.regex#sub(repl, str:string, count?:number):map
AScript_DeclareMethod(RegEx, sub)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "repl", VTYPE_Any);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "count", VTYPE_Number, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(RegEx, sub)
{
	Object_RegEx *pObj = Object_RegEx::GetSelfObj(context);
	int cnt = context.IsNumber(2)? static_cast<int>(context.GetNumber(2)) : -1;
	if (context.IsString(0)) {
		return DoSub(env, sig, pObj->GetRegEx(),
						context.GetString(0), context.GetString(1), cnt);
	} else if (context.IsFunction(0)) {
		return DoSubWithFunc(env, sig, pObj->GetRegEx(),
						context.GetFunction(0), context.GetString(1), cnt);
	}
	SetError_ArgumentTypeByIndex(sig, 0, context.GetValue(0));
	return Value::Null;
}

// re.regex#split(str:string, count?:number):map {block?}
AScript_DeclareMethod(RegEx, split)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "count", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(RegEx, split)
{
	Object_RegEx *pSelf = Object_RegEx::GetSelfObj(context);
	Object_RegEx *pObjRegEx = dynamic_cast<Object_RegEx *>(pSelf->IncRef());
	Object_String *pObjStr = dynamic_cast<Object_String *>(context.GetStringObj(0)->IncRef());
	int cntMax = context.IsNumber(1)? static_cast<int>(context.GetNumber(1)) : -1;
	return ReturnIterator(env, sig, context,
							new IteratorSplit(pObjRegEx, pObjStr, cntMax));
}

// re.regex#scan(str:string, count?:number):map {block?}
AScript_DeclareMethod(RegEx, scan)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "count", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(RegEx, scan)
{
	Object_RegEx *pSelf = Object_RegEx::GetSelfObj(context);
	Object_RegEx *pObjRegEx = dynamic_cast<Object_RegEx *>(pSelf->IncRef());
	Object_String *pObjStr = dynamic_cast<Object_String *>(context.GetStringObj(0)->IncRef());
	int cntMax = context.IsNumber(1)? static_cast<int>(context.GetNumber(1)) : -1;
	return ReturnIterator(env, sig, context,
							new IteratorScan(pObjRegEx, pObjStr, cntMax));
}

// implementation of class RegEx
AScript_ImplementPrivClass(RegEx)
{
	AScript_AssignMethod(RegEx, match);
	AScript_AssignMethod(RegEx, sub);
	AScript_AssignMethod(RegEx, split);
	AScript_AssignMethod(RegEx, scan);
}

//-----------------------------------------------------------------------------
// Object_Match
//-----------------------------------------------------------------------------
Object_Match::~Object_Match()
{
}

Object *Object_Match::Clone() const
{
	return new Object_Match(*this);
}

String Object_Match::ToString(Signal sig, bool exprFlag)
{
	String rtn;
	rtn += "<match:";
	foreach_const (GroupList, pGroup, _groupList) {
		if (pGroup != _groupList.begin()) rtn += ",";
		char str[80];
		::sprintf(str, "%d-%d", pGroup->GetPosBegin(), pGroup->GetPosEnd());
		rtn += str;
	}
	rtn += ">";
	return rtn;
}

bool Object_Match::SetMatchInfo(const char *str,
								regex_t *pRegEx, const OnigRegion *pRegion)
{
	if (pRegion->num_regs == 0) return false;
	::onig_foreach_name(pRegEx, &ForeachNameCallbackStub, this);
	_str = str;
	AssignValue(AScript_Symbol(string), Value(*this, str), false);
	for (int iGroup = 0; iGroup < pRegion->num_regs; iGroup++) {
		int posBegin = pRegion->beg[iGroup];
		int posEnd = pRegion->end[iGroup];
		if (posBegin > posEnd) return false;
		_groupList.push_back(Group(posBegin, posEnd));
	}
	return true;
}

String Object_Match::GetGroup(Signal sig, size_t index) const
{
	if (index >= _groupList.size()) {
		sig.SetError(ERR_IndexError, "index is out of range");
		return "";
	}
	const Group &group = _groupList[index];
	return String(_str, group.GetPosBegin(), group.GetLength());
}

String Object_Match::GetGroupByName(Signal sig, const char *name) const
{
	GroupNameDict::const_iterator iter = _groupNameDict.find(name);
	if (iter == _groupNameDict.end()) {
		sig.SetError(ERR_IndexError,
			"regular expression doesn't have a group named '%s%", name);
		return "";
	}
	return GetGroup(sig, iter->second);
}

int Object_Match::ForeachNameCallback(const String &name, int nGroups,
											int *idxGroupTbl, regex_t *pRegEx)
{
	if (nGroups > 0) _groupNameDict[name] = idxGroupTbl[0];
	return 0;
}

int Object_Match::ForeachNameCallbackStub(
			const UChar *nameRaw, const UChar *nameRawEnd,
			int nGroups, int *idxGroupTbl, regex_t *pRegEx, void *pArg)
{
	String name(reinterpret_cast<const char *>(nameRaw), nameRawEnd - nameRaw);
	return reinterpret_cast<Object_Match *>(pArg)->
					ForeachNameCallback(name, nGroups, idxGroupTbl, pRegEx);
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Match
//-----------------------------------------------------------------------------
// str = re.match_t#group(index):map
AScript_DeclareMethod(Match, group)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "index", VTYPE_Any);
}

AScript_ImplementMethod(Match, group)
{
	Object_Match *pObj = Object_Match::GetSelfObj(context);
	Value result;
	if (context.IsNumber(0)) {
		size_t index = static_cast<size_t>(context.GetNumber(0));
		String str = pObj->GetGroup(sig, index);
		if (sig.IsSignalled()) return Value::Null;
		result = Value(env, str.c_str());
	} else if (context.IsString(0)) {
		String str = pObj->GetGroupByName(sig, context.GetString(0));
		if (sig.IsSignalled()) return Value::Null;
		result = Value(env, str.c_str());
	} else {
		sig.SetError(ERR_TypeError, "invalid argument type");
	}
	return result;
}

// list = re.match_t#groups()
AScript_DeclareMethod(Match, groups)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Match, groups)
{
	Object_Match *pObj = Object_Match::GetSelfObj(context);
	Value result;
	ValueList &valList = result.InitAsList(env);
	const Object_Match::GroupList &groupList = pObj->GetGroupList();
	Object_Match::GroupList::const_iterator pGroup = groupList.begin();
	if (pGroup != groupList.end()) pGroup++;
	for ( ; pGroup != groupList.end(); pGroup++) {
		valList.push_back(Value(env, pObj->GetGroup(pGroup).c_str()));
	}
	return result;
}

// implementation of class Match
AScript_ImplementPrivClass(Match)
{
	AScript_AssignMethod(Match, group);
	AScript_AssignMethod(Match, groups);
}

//-----------------------------------------------------------------------------
// AScript module functions: re
//-----------------------------------------------------------------------------
// regex = re.compile(pat:string):map:[icase]
AScript_DeclareFunction(compile)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pattern", VTYPE_String);
	DeclareAttr(AScript_Symbol(icase));
	DeclareAttr(AScript_PrivSymbol(multiline));
}

AScript_ImplementFunction(compile)
{
	Object_RegEx *pObjRegEx = new Object_RegEx(env);
	if (!pObjRegEx->SetPattern(sig, context.GetString(0), context)) {
		delete pObjRegEx;
		return Value::Null;
	}
	return Value(pObjRegEx, VTYPE_Object);
}

// m = re.match(pat:string, str:string):map:[icase]
AScript_DeclareFunction(match)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pattern", VTYPE_String);
	DeclareArg(env, "str", VTYPE_String);
	DeclareAttr(AScript_Symbol(icase));
	DeclareAttr(AScript_PrivSymbol(multiline));
}

AScript_ImplementFunction(match)
{
	regex_t *pRegEx = CreateRegEx(sig, context.GetString(0), context.GetAttrs());
	if (pRegEx == NULL) return Value::Null;
	Value result = DoMatch(env, sig, pRegEx, context.GetString(1));
	::onig_free(pRegEx);
	return result;
}

// str = re.sub(pat:string, repl, str:string, count?:number):map:[icase]
AScript_DeclareFunction(sub)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pattern", VTYPE_String);
	DeclareArg(env, "repl", VTYPE_Any);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "count", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
	DeclareAttr(AScript_PrivSymbol(multiline));
}

AScript_ImplementFunction(sub)
{
	regex_t *pRegEx = CreateRegEx(sig, context.GetString(0), context.GetAttrs());
	if (pRegEx == NULL) return Value::Null;
	int cnt = context.IsNumber(3)? static_cast<int>(context.GetNumber(3)) : -1;
	Value result;
	if (context.IsString(1)) {
		result = DoSub(env, sig, pRegEx,
						context.GetString(1), context.GetString(2), cnt);
	} else if (context.IsFunction(1)) {
		result = DoSubWithFunc(env, sig, pRegEx,
						context.GetFunction(1), context.GetString(2), cnt);
	} else {
		SetError_ArgumentTypeByIndex(sig, 1, context.GetValue(1));
	}
	::onig_free(pRegEx);
	return result;
}

// iter = re.split(pattern:string, str:string, count?:number):map:[icase] {block?}
AScript_DeclareFunction(split)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pattern", VTYPE_String);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "count", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
	DeclareAttr(AScript_PrivSymbol(multiline));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(split)
{
	Object_RegEx *pObjRegEx = new Object_RegEx(env);
	if (!pObjRegEx->SetPattern(sig, context.GetString(0), context)) {
		delete pObjRegEx;
		return Value::Null;
	}
	Object_String *pObjStr =
				dynamic_cast<Object_String *>(context.GetStringObj(1)->IncRef());
	int cntMax = context.IsNumber(2)? static_cast<int>(context.GetNumber(2)) : -1;
	return ReturnIterator(env, sig, context,
							new IteratorSplit(pObjRegEx, pObjStr, cntMax));
}

// iter = re.scan(pattern:string, str:string, count?:number):map:[icase] {block?}
AScript_DeclareFunction(scan)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "pattern", VTYPE_String);
	DeclareArg(env, "str", VTYPE_String);
	DeclareArg(env, "count", VTYPE_Number, OCCUR_ZeroOrOnce);
	DeclareAttr(AScript_Symbol(icase));
	DeclareAttr(AScript_PrivSymbol(multiline));
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(scan)
{
	Object_RegEx *pObjRegEx = new Object_RegEx(env);
	if (!pObjRegEx->SetPattern(sig, context.GetString(0), context)) {
		delete pObjRegEx;
		return Value::Null;
	}
	Object_String *pObjStr =
				dynamic_cast<Object_String *>(context.GetStringObj(1)->IncRef());
	int cntMax = context.IsNumber(2)? static_cast<int>(context.GetNumber(2)) : -1;
	return ReturnIterator(env, sig, context,
							new IteratorScan(pObjRegEx, pObjStr, cntMax));
}

// Module entry
AScript_ModuleEntry()
{
	// symbol realization
	AScript_RealizePrivSymbol(re);
	AScript_RealizePrivSymbol(string);
	AScript_RealizePrivSymbol(multiline);
	// class realization
	AScript_RealizePrivClass(Match, "match_t");
	AScript_RealizePrivClass(RegEx, "regex");
	// function assignment
	AScript_AssignFunction(compile);
	AScript_AssignFunction(match);
	AScript_AssignFunction(sub);
	AScript_AssignFunction(split);
	AScript_AssignFunction(scan);
}

AScript_ModuleTerminate()
{
}

//-----------------------------------------------------------------------------
// Utilities
//-----------------------------------------------------------------------------
static regex_t *CreateRegEx(Signal sig, const char *pattern, const SymbolSet &attrs)
{
	// ::onig_end() call may be necessary when module is destroyed
	regex_t *pRegEx = NULL;
	OnigOptionType option = ONIG_OPTION_CAPTURE_GROUP; //ONIG_OPTION_NONE
	OnigEncoding enc = ONIG_ENCODING_UTF8; //ONIG_ENCODING_SJIS;
	OnigErrorInfo errInfo;
	size_t len = ::strlen(pattern);
	if (attrs.IsSet(AScript_Symbol(icase))) {
		option |= ONIG_OPTION_IGNORECASE;
	}
	if (attrs.IsSet(AScript_PrivSymbol(multiline))) {
		option |= ONIG_OPTION_MULTILINE;
	}
	int rtn = ::onig_new(&pRegEx,
				reinterpret_cast<const OnigUChar *>(pattern),
				reinterpret_cast<const OnigUChar *>(pattern + len),
				option, enc, ONIG_SYNTAX_DEFAULT, &errInfo);
	if (rtn != ONIG_NORMAL) {
		SetError_OnigurumaError(sig, rtn);
		return NULL;
	}
	return pRegEx;
}

static Value DoMatch(Environment &env, Signal sig, regex_t *pRegEx, const char *str)
{
	Value result;
	size_t len = ::strlen(str);
	const char *start = str;
	OnigRegion *pRegion = ::onig_region_new();
	int rtn = ::onig_search(pRegEx,
				reinterpret_cast<const OnigUChar *>(str),
				reinterpret_cast<const OnigUChar *>(str + len),
				reinterpret_cast<const OnigUChar *>(start),
				reinterpret_cast<const OnigUChar *>(start + len),
				pRegion, ONIG_OPTION_NONE);
	if (rtn >= 0) {
		Object_Match *pObj = new Object_Match(env);
		if (pObj->SetMatchInfo(str, pRegEx, pRegion)) {
			result.InitAsObject(pObj, AScript_PrivVTYPE(Match));
		} else {
			SetError_FailInOniguruma(sig);
			delete pObj;
		}
	} else if (rtn == ONIG_MISMATCH) {
		// nothing to do
	} else { // error
		SetError_OnigurumaError(sig, rtn);
	}
	::onig_region_free(pRegion, 1); // 1:free self, 0:free contents only
	return result;
}

static Value DoSub(Environment &env, Signal sig, regex_t *pRegEx,
							const char *repl, const char *str, int cnt)
{
	enum Stat { STAT_Start, STAT_Escape };
	size_t len = ::strlen(str);
	String result;
	OnigRegion *pRegion = ::onig_region_new();
	int pos = 0;
	for ( ; cnt != 0; cnt--) {
		int rtn = ::onig_search(pRegEx,
						reinterpret_cast<const OnigUChar *>(str),
						reinterpret_cast<const OnigUChar *>(str + len),
						reinterpret_cast<const OnigUChar *>(str + pos),
						reinterpret_cast<const OnigUChar *>(str + len),
						pRegion, ONIG_OPTION_NONE);
		if (rtn >= 0) {
			if (rtn < pos || pRegion->num_regs == 0 || pRegion->end[0] <= pos) {
				SetError_FailInOniguruma(sig);
				goto error_done;
			}
			result += String(str + pos, rtn - pos);
			Stat stat = STAT_Start;
			for (const char *p = repl; *p != '\0'; p++) {
				char ch = *p;
				if (stat == STAT_Start) {
					if (ch == '\\') {
						stat = STAT_Escape;
					} else {
						result.push_back(*p);
					}
				} else if (stat == STAT_Escape) {
					if (IsDigit(ch)) {
						int iGroup = ch - '0';
						if (iGroup < pRegion->num_regs) {
							int posBegin = pRegion->beg[iGroup];
							int posEnd = pRegion->end[iGroup];
							result += String(str + posBegin, posEnd - posBegin);
						}
						stat = STAT_Start;
					} else {
						result.push_back(GetEscaped(ch));
						stat = STAT_Start;
					}
				}
			}
			pos = pRegion->end[0];
		} else if (rtn == ONIG_MISMATCH) {
			break;
		} else { // error
			SetError_OnigurumaError(sig, rtn);
			goto error_done;
		}
	}
	::onig_region_free(pRegion, 1); // 1:free self, 0:free contents only
	result += String(str + pos);
	return Value(env, result.c_str());
error_done:
	::onig_region_free(pRegion, 1); // 1:free self, 0:free contents only
	return Value::Null;
}

static Value DoSubWithFunc(Environment &env, Signal sig, regex_t *pRegEx,
						const Function *pFunc, const char *str, int cnt)
{
	enum Stat { STAT_Start, STAT_Escape };
	size_t len = ::strlen(str);
	String result;
	OnigRegion *pRegion = ::onig_region_new();
	int pos = 0;
	for ( ; cnt != 0; cnt--) {
		int rtn = ::onig_search(pRegEx,
					reinterpret_cast<const OnigUChar *>(str),
					reinterpret_cast<const OnigUChar *>(str + len),
					reinterpret_cast<const OnigUChar *>(str + pos),
					reinterpret_cast<const OnigUChar *>(str + len),
					pRegion, ONIG_OPTION_NONE);
		if (rtn >= 0) {
			Object_Match *pObj = new Object_Match(env);
			if (!pObj->SetMatchInfo(str, pRegEx, pRegion)) {
				SetError_FailInOniguruma(sig);
				delete pObj;
				goto error_done;
			}
			Value value(pObj, AScript_PrivVTYPE(Match));
			ValueList valListArg(value);
			Context context(valListArg);
			Value resultFunc = pFunc->Eval(env, sig, context);
			if (sig.IsSignalled()) goto error_done;
			result += String(str + pos, rtn - pos);
			result += resultFunc.ToString(sig, false);
			if (sig.IsSignalled()) goto error_done;
			pos = pRegion->end[0];
		} else if (rtn == ONIG_MISMATCH) {
			break;
		} else { // error
			SetError_OnigurumaError(sig, rtn);
			goto error_done;
		}
	}
	::onig_region_free(pRegion, 1); // 1:free self, 0:free contents only
	result += String(str + pos);
	return Value(env, result.c_str());
error_done:
	::onig_region_free(pRegion, 1); // 1:free self, 0:free contents only
	return Value::Null;
}

void SetError_OnigurumaError(Signal sig, int errCode)
{
	char errMsg[ONIG_MAX_ERROR_MESSAGE_LEN];
	::onig_error_code_to_str(reinterpret_cast<OnigUChar *>(errMsg), errCode);
	sig.SetError(ERR_ValueError, "oniguruma: %s", errMsg);
}

void SetError_FailInOniguruma(Signal sig)
{
	sig.SetError(ERR_SystemError,
				"something's wrong in the process of Oniguruma library");
}

AScript_EndModule(re)

AScript_RegisterModule(re)
