#include "Symbol.h"
#include "Environment.h"
#include "Iterator.h"
#include "Parser.h"
#include "Operator.h"
#include "Object_Error.h"
#include "Module.h"

#if defined(HAVE_LIBDL)
#include <dlfcn.h>
#endif

AScript_IncludeModule(__builtins__)
AScript_IncludeModuleBegin(sys)
void Setup(Module *pModule, Signal sig, int argc, const char *argv[]);
AScript_IncludeModuleEnd()
AScript_IncludeModule(os)
AScript_IncludeModule(time)
AScript_IncludeModule(string)
AScript_IncludeModule(math)

namespace AScript {

//-----------------------------------------------------------------------------
// EnvType
//-----------------------------------------------------------------------------
const char *GetEnvTypeName(EnvType envType)
{
	static const struct {
		EnvType envType;
		const char *name;
	} tbl[] = {
		{ ENVTYPE_Invalid,	"invalid",	},
		{ ENVTYPE_Root,		"root",		},
		{ ENVTYPE_Local,	"local",	},
		{ ENVTYPE_Block,	"block",	},
		{ ENVTYPE_Module,	"module",	},
		{ ENVTYPE_Class,	"class",	},
		{ ENVTYPE_Instance,	"instance",	},
		{ ENVTYPE_Method,	"method",	},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].envType == envType) return tbl[i].name;
	}
	return "unknown";
}

//-----------------------------------------------------------------------------
// IntegratedModule
//-----------------------------------------------------------------------------
bool IntegratedModule::IsMatch(const SymbolList &symbolOfModule) const
{
	return symbolOfModule.size() == 1 &&
			::strcmp(symbolOfModule.back()->GetName(), _name.c_str()) == 0;
}

//-----------------------------------------------------------------------------
// IntegratedModuleOwner
//-----------------------------------------------------------------------------
IntegratedModuleOwner::~IntegratedModuleOwner()
{
	foreach (IntegratedModuleOwner, ppIntegratedModule, *this) {
		delete *ppIntegratedModule;
	}
}


//-----------------------------------------------------------------------------
// ModuleIntegrator
//-----------------------------------------------------------------------------
ModuleIntegrator::ModuleIntegrator(const char *name,
			ModuleEntryType moduleEntry, ModuleTerminateType moduleTerminate)
{
	Environment::IntegrateModule(name, moduleEntry, moduleTerminate);
}

//-----------------------------------------------------------------------------
// Environment
//-----------------------------------------------------------------------------
IntegratedModuleOwner *Environment::_pIntegratedModuleOwner = NULL;

Environment::Environment(const Environment &env) : _pFrameCache(NULL)
{
	// _pFrameCache will be initialized when the program reads some variable at first
	foreach_const (FrameList, ppFrame, env._frameList) {
		Frame *pFrame = *ppFrame;
		_frameList.push_back(pFrame->IncRef());
	}
}

Environment::Environment(Environment *pEnvOuter, EnvType envType) : _pFrameCache(NULL)
{
	// _pFrameCache will be initialized when the program reads some variable at first
	if (pEnvOuter == NULL) {
		SymbolPool::Initialize();
		Global *pGlobal = new Global();
		pGlobal->_pSymbolPool = SymbolPool::GetInstance();
		_frameList.push_back(new Frame(ENVTYPE_Root, pGlobal));
	} else if (envType == ENVTYPE_Block) {
		_frameList.push_back(new Frame(envType, pEnvOuter->GetGlobal()));
		foreach (FrameList, ppFrame, pEnvOuter->_frameList) {
			Frame *pFrame = *ppFrame;
			if (!pFrame->IsType(ENVTYPE_Method)) {
				_frameList.push_back(pFrame->IncRef());
			}
		}
	} else if (envType == ENVTYPE_Outer) {
		FrameList &frameList = pEnvOuter->_frameList;
		FrameList::iterator ppFrame = frameList.begin();
		if (frameList.size() > 1) ppFrame++;
		for ( ; ppFrame != frameList.end(); ppFrame++) {
			Frame *pFrame = *ppFrame;
			_frameList.push_back(pFrame->IncRef());
		}
	} else {
		_frameList.push_back(new Frame(envType, pEnvOuter->GetGlobal()));
		foreach (FrameList, ppFrame, pEnvOuter->_frameList) {
			Frame *pFrame = *ppFrame;
			_frameList.push_back(pFrame->IncRef());
		}
	}
}

Environment::~Environment()
{
	delete _pFrameCache;
	foreach (FrameList, ppFrame, _frameList) {
		Frame::Delete(*ppFrame);
	}
}

void Environment::CacheFrame(const Symbol *pSymbol, Frame *pFrame)
{
	if (_pFrameCache == NULL) _pFrameCache = new FrameCache();
	(*_pFrameCache)[pSymbol] = pFrame;
}

void Environment::AssignValue(const Symbol *pSymbol, const Value &value, bool escalateFlag)
{
	if (escalateFlag) {
		if (_pFrameCache != NULL) {
			FrameCache::iterator iter = _pFrameCache->find(pSymbol);
			if (iter != _pFrameCache->end()) {
				Frame *pFrame = iter->second;
				pFrame->AssignValue(pSymbol, value);
				return;
			}
		}
		foreach (FrameList, ppFrame, _frameList) {
			Frame *pFrame = *ppFrame;
			if (!pFrame->IsType(ENVTYPE_Block)) {
				pFrame->AssignValue(pSymbol, value);
				break;
			}
		}
	} else {
		GetTopFrame().AssignValue(pSymbol, value);
	}
}

void Environment::RemoveValue(const Symbol *pSymbol)
{
	GetTopFrame().RemoveValue(pSymbol);
}

Value *Environment::LookupValue(const Symbol *pSymbol, bool escalateFlag)
{
	Value *pValue = NULL;
	Frame *pFrame = NULL;
	EnvType envType = GetTopFrame().GetType();
	if (!escalateFlag) {// || envType == ENVTYPE_Module) {
		pFrame = &GetTopFrame();
		pValue = pFrame->LookupValue(pSymbol);
		if (pValue != NULL) {
			CacheFrame(pSymbol, pFrame);
			return pValue;
		}
	} else if (envType == ENVTYPE_Method) {
		foreach (FrameList, ppFrame, _frameList) {
			pFrame = *ppFrame;
			if (!(pFrame->IsType(ENVTYPE_Instance) ||
								pFrame->IsType(ENVTYPE_Class))) {
				pValue = pFrame->LookupValue(pSymbol);
				if (pValue != NULL) {
					CacheFrame(pSymbol, pFrame);
					return pValue;
				}
			}
		}
	} else if (envType == ENVTYPE_Instance || envType == ENVTYPE_Class) {
		foreach (FrameList, ppFrame, _frameList) {
			pFrame = *ppFrame;
			if (pFrame->IsType(ENVTYPE_Instance) ||
								pFrame->IsType(ENVTYPE_Class)) {
				pValue = pFrame->LookupValue(pSymbol);
				if (pValue != NULL) {
					CacheFrame(pSymbol, pFrame);
					return pValue;
				}
			}
		}
	} else {
		foreach (FrameList, ppFrame, _frameList) {
			pFrame = *ppFrame;
			pValue = pFrame->LookupValue(pSymbol);
			if (pValue != NULL) {
				CacheFrame(pSymbol, pFrame);
				return pValue;
			}
		}
	}
	return NULL;
}

//****** reference problem of FunctionCustom and Environment still remains ******
Function *Environment::AssignFunction(Function *pFunc)
{
	Value value;
	value.InitAsFunction(*this, pFunc);
	GetTopFrame().AssignValue(pFunc->GetSymbol(), value);
	return pFunc;
}

Function *Environment::LookupFunction(const Symbol *pSymbol, bool escalateFlag)
{
	EnvType envType = GetTopFrame().GetType();
	if (!escalateFlag) {
		Frame *pFrame = &GetTopFrame();
		Value *pValue = pFrame->LookupValue(pSymbol);
		if (pValue != NULL && pValue->IsFunction()) {
			return pValue->GetFunction();
		}
	} else if (envType == ENVTYPE_Instance || envType == ENVTYPE_Class) {
		foreach (FrameList, ppFrame, _frameList) {
			Frame *pFrame = *ppFrame;
			if (pFrame->IsType(ENVTYPE_Instance) ||
									pFrame->IsType(ENVTYPE_Class)) {
				Value *pValue = pFrame->LookupValue(pSymbol);
				if (pValue != NULL && pValue->IsFunction()) {
					return pValue->GetFunction();
				}
			}
		}
	} else {
		foreach (FrameList, ppFrame, _frameList) {
			Frame *pFrame = *ppFrame;
			Value *pValue = pFrame->LookupValue(pSymbol);
			if (pValue != NULL && pValue->IsFunction()) {
				return pValue->GetFunction();
			}
		}
	}
	return NULL;
}

void Environment::AssignValueType(const ValueTypeInfo *pValueTypeInfo)
{
	GetTopFrame().AssignValueType(pValueTypeInfo);
}

const ValueTypeInfo *Environment::LookupValueType(const SymbolList &symbolList) const
{
	bool escalateFlag = true;
	const Environment *pEnv = this;
	SymbolList::const_iterator ppSymbol = symbolList.begin();
	for ( ; ppSymbol + 1 != symbolList.end(); ppSymbol++) {
		const Value *pValue = pEnv->LookupValue(*ppSymbol, escalateFlag);
		if (pValue == NULL || !pValue->IsModule()) return NULL;
		pEnv = pValue->GetModule();
		escalateFlag = false;
	}
	return pEnv->LookupValueType(*ppSymbol);
}

const ValueTypeInfo *Environment::LookupValueType(const Symbol *pSymbol) const
{
	foreach_const (FrameList, ppFrame, _frameList) {
		const Frame *pFrame = *ppFrame;
		const ValueTypeInfo *pValueTypeInfo = pFrame->LookupValueType(pSymbol);
		if (pValueTypeInfo != NULL) return pValueTypeInfo;
	}
	return NULL;
}

Value Environment::EvalGetter(Signal sig, const Symbol *pSymbol, bool &evaluatedFlag)
{
	return Value::Null;
}

Value Environment::EvalSetter(Signal sig, const Symbol *pSymbol,
										const Value &value, bool &evaluatedFlag)
{
	return Value::Null;
}

ICallable *Environment::GetCallable(Signal sig, const Symbol *pSymbol)
{
	return NULL;
}

void Environment::AssignModule(Module *pModule)
{
	Value value;
	value.InitAsModule(pModule);
	GetTopFrame().AssignValue(pModule->GetSymbol(), value);
}

bool Environment::ImportModule(Signal sig, const Expr *pExpr,
	const Symbol *pSymbolOfModule, const SymbolSet *pSymbolsToMixIn, bool forceFlag)
{
	SymbolList symbolOfModule;
	if (!pExpr->GetChainedSymbolList(symbolOfModule)) {
		sig.SetError(ERR_SyntaxError, "wrong format for module name");
		return false;
	}
	Module *pModule = ImportIntegratedModule(sig, symbolOfModule);
	if (sig.IsSignalled()) return false;
	if (pModule == NULL) {
		pModule = ImportSeparatedModule(sig, symbolOfModule);
		if (sig.IsSignalled()) return false;
	}
	if (pSymbolsToMixIn == NULL) {
		if (pSymbolOfModule == NULL) {
			if (LookupValue(symbolOfModule.front(), false) != NULL) {
				sig.SetError(ERR_ValueError,
						"module symbol conflicts with an existing variable '%s'",
						symbolOfModule.front()->GetName());
				return false;
			}
			Environment *pEnvDst = this;
			for (SymbolList::const_iterator ppSymbol = symbolOfModule.begin();
									ppSymbol + 1 != symbolOfModule.end(); ppSymbol++) {
				const Symbol *pSymbol = *ppSymbol;
				Value *pValue = pEnvDst->LookupValue(pSymbol, false);
				if (pValue == NULL) {
					Module *pModuleParent = new Module(pEnvDst, pSymbol, NULL);
					Value valueOfModule;
					valueOfModule.InitAsModule(pModuleParent);
					AssignValue(pSymbol, valueOfModule, false);
					pEnvDst = pModuleParent;
				} else if (pValue->IsModule()) {
					pEnvDst = pValue->GetModule();
				} else {
					sig.SetError(ERR_IOError,
							"module symbol conflicts with an existing variable '%s'",
							symbolOfModule.Join('.').c_str());
					return false;
				}
			}
			pSymbolOfModule = symbolOfModule.back();
			Value valueOfModule;
			valueOfModule.InitAsModule(pModule->IncRef());
			pEnvDst->AssignValue(pSymbolOfModule, valueOfModule, false);
		} else {
			if (LookupValue(pSymbolOfModule, false) != NULL) {
				sig.SetError(ERR_ValueError,
						"module symbol conflicts with an existing variable '%s'",
						pSymbolOfModule->GetName());
				return false;
			}
			Value valueOfModule;
			valueOfModule.InitAsModule(pModule->IncRef());
			AssignValue(pSymbolOfModule, valueOfModule, false);
		}
	} else if (pSymbolsToMixIn->IsSet(AScript_Symbol(Char_Multiply))) {
		foreach_const (ValueMap, iter, pModule->GetTopFrame().GetValueMap()) {
			const Symbol *pSymbol = iter->first;
			const Value &value = iter->second;
			if (pSymbol->IsPrivateName()) continue;
			if (!forceFlag && LookupValue(pSymbol, false) != NULL) {
				sig.SetError(ERR_IOError,
						"imported variable name conflicts with an existing one '%s'",
						pSymbol->GetName());
				return false;
			}
			AssignValue(pSymbol, value, false);
		}
	} else {
		foreach_const (SymbolSet, ppSymbol, *pSymbolsToMixIn) {
			const Symbol *pSymbol = *ppSymbol;
			if (!forceFlag && LookupValue(pSymbol, false) != NULL) {
				sig.SetError(ERR_IOError,
						"imported variable name conflicts with an existing one '%s'",
												pSymbol->GetName());
				return false;
			}
			Value *pValue = pModule->LookupValue(pSymbol, false);
			if (pValue != NULL) {
				AssignValue(pSymbol, *pValue, false);
			}
		}
	}
	return true;
}

Module *Environment::ImportIntegratedModule(Signal sig, const SymbolList &symbolOfModule)
{
	int id = 0;
	IntegratedModule *pIntegratedModule = NULL;
	if (_pIntegratedModuleOwner != NULL) {
		foreach (IntegratedModuleOwner, ppIntegratedModule, *_pIntegratedModuleOwner) {
			if ((*ppIntegratedModule)->IsMatch(symbolOfModule)) {
				pIntegratedModule = *ppIntegratedModule;
				break;
			}
			id++;
		}
	}
	if (pIntegratedModule == NULL) return NULL;
	Module *pModule = GetGlobal()->LookupIntegratedModule(id);
	if (pModule == NULL) {
		pModule = new Module(this, symbolOfModule.back(), NULL);
		pIntegratedModule->ModuleEntry(*pModule, sig);
		if (sig.IsSignalled()) {
			delete pModule;
			return NULL;
		}
		GetGlobal()->RegisterIntegratedModule(id, pModule);
	}
	return pModule;
}

Module *Environment::ImportSeparatedModule(Signal sig,
											const SymbolList &symbolOfModule)
{
	String pathName;
	if (!SearchSeparatedModuleFile(sig, pathName, symbolOfModule)) return NULL;
	Module *pModule = GetGlobal()->LookupSeparatedModule(pathName.c_str());
	if (pModule != NULL) return pModule;
	if (IsBinaryModule(pathName.c_str())) {
		pModule = ImportSeparatedModule_Binary(sig, this,
								pathName.c_str(), symbolOfModule);
	} else {
		pModule = ImportSeparatedModule_Script(sig, this,
								pathName.c_str(), symbolOfModule);
	}
	if (pModule == NULL) return NULL;
	GetGlobal()->RegisterSeparatedModule(pathName.c_str(), pModule);
	return pModule;
}

bool Environment::SearchSeparatedModuleFile(Signal sig, String &pathName,
											const SymbolList &symbolOfModule)
{
	const Value *pValDirNameList =
					GetModule_sys()->LookupValue(AScript_Symbol(path), false);
	if (pValDirNameList == NULL) {
		sig.SetError(ERR_ValueError, "variable path is not specified");
		return false;
	} else if (!pValDirNameList->IsList()) {
		sig.SetError(ERR_ValueError, "wrong value of variable path");
		return false;
	}
	StringList extNameList;
	extNameList.push_back("azd");
	extNameList.push_back("azc");
	extNameList.push_back("az");
	String baseName = symbolOfModule.Join(File::Separator);
	foreach_const (ValueList, pValue, pValDirNameList->GetList()) {
		if (!pValue->IsString()) {
			sig.SetError(ERR_ValueError, "wrong element value of variable path");
			return false;
		}
		do {
			String pathNameBase = pValue->GetString();
			pathNameBase += File::Separator;
			pathNameBase += baseName;
			pathNameBase += '.';
			foreach_const (StringList, pExtName, extNameList) {
				pathName = pathNameBase + *pExtName;
				if (File::IsExist(pathName.c_str())) return true;
			}
		} while (0);
		do {
			String pathNameBase = pValue->GetString();
			pathNameBase += File::Separator;
			pathNameBase += baseName;
			if (File::IsDir(pathNameBase.c_str())) {
				pathNameBase += File::Separator;
				pathNameBase += "__init__.";
				foreach_const (StringList, pExtName, extNameList) {
					pathName = pathNameBase + *pExtName;
					if (File::IsExist(pathName.c_str())) return true;
				}
			}
		} while (0);
	}
	sig.SetError(ERR_IOError, "can't find a module named '%s'",
										symbolOfModule.Join('.').c_str());
	return false;
}

Module *Environment::ImportSeparatedModule_Script(Signal sig, Environment *pEnvOuter,
						const char *pathName, const SymbolList &symbolOfModule)
{
	File file;
	file.Open(sig, pathName, "r", NULL);
	if (sig.IsError()) return NULL;
	Expr *pExpr = Parser().ParseFile(*pEnvOuter, sig, file);
	if (sig.IsSignalled()) return NULL;
	Module *pModule = new Module(pEnvOuter, symbolOfModule.back(), pExpr);
	bool echoFlagSaved = pModule->GetEchoFlag();
	pModule->SetEchoFlag(false);
	//Parser().ExecFile(*pModule, sig, file, true);
	pExpr->Exec(*pModule, sig);
	pModule->SetEchoFlag(echoFlagSaved);
	if (sig.IsSignalled()) {
		delete pModule;
		return NULL;
	}
	return pModule;
}

#if defined(HAVE_WINDOWS_H)
Module *Environment::ImportSeparatedModule_Binary(Signal sig, Environment *pEnvOuter,
						const char *pathName, const SymbolList &symbolOfModule)
{
	HMODULE hModule = ::LoadLibrary(pathName);
	if (hModule == NULL) {
		sig.SetError(ERR_ImportError, "can't open module file '%s'",
							File::ExtractBaseName(pathName).c_str());
		return NULL;
	}
	FARPROC pFunc = ::GetProcAddress(hModule, "AScriptModuleEntry");
	if (pFunc == NULL) {
		pFunc = ::GetProcAddress(hModule, "_AScriptModuleEntry");
	}
	if (pFunc == NULL) {
		sig.SetError(ERR_ImportError, "can't find entry function '%s'",
												"AScriptModuleEntry");
		::FreeLibrary(hModule);
		return NULL;
	}
	ModuleEntryType moduleEntry = reinterpret_cast<ModuleEntryType>(pFunc);
	Module *pModule = new Module(pEnvOuter, symbolOfModule.back(), NULL);
	//FARPROC pFunc = ::GetProcAddress(hModule, "_AScriptModuleTerminate");
	(*moduleEntry)(*pModule, sig);
	if (sig.IsSignalled()) {
		delete pModule;
		return NULL;
	}
	return pModule;
}
#elif defined(HAVE_LIBDL)
Module *Environment::ImportSeparatedModule_Binary(Signal sig, Environment *pEnvOuter,
						const char *pathName, const SymbolList &symbolOfModule)
{
	void *hLibrary = dlopen(pathName, RTLD_LAZY);
	if (hLibrary == NULL) {
		sig.SetError(ERR_ImportError, "can't open module file '%s'",
							File::ExtractBaseName(pathName).c_str());
		return NULL;
	}
	void *pFunc = dlsym(hLibrary, "AScriptModuleEntry");
	if (pFunc == NULL) {
		sig.SetError(ERR_ImportError, "can't find entry function '%s'",
												"AScriptModuleEntry");
		dlclose(hLibrary);
		return NULL;
	}
	ModuleEntryType moduleEntry = reinterpret_cast<ModuleEntryType>(pFunc);
	Module *pModule = new Module(pEnvOuter, symbolOfModule.back(), NULL);
	//void *pFunc = dlsym(hLibrary "AScriptModuleTerminate");
	(*moduleEntry)(*pModule, sig);
	if (sig.IsSignalled()) {
		dlclose(hLibrary);
		return NULL;
	}
	return pModule;
}
#else
Module *Environment::ImportSeparatedModule_Binary(Signal sig,
						const char *pathName, const SymbolList &symbolOfModule)
{
	sig.SetError(ERR_SystemError, "module import is not supported");
	return NULL;
}
#endif

bool Environment::IsBinaryModule(const char *pathName)
{
	return ::strcasecmp(File::ExtractExtName(pathName), "azd") == 0;
}

void Environment::DbgPrint() const
{
	int idx = 0;
	foreach_const (FrameList, ppFrame, _frameList) {
		idx++;
		::printf("frame#%d ", idx);
		(*ppFrame)->DbgPrint();
	}
}

const char *Environment::GetPrompt(bool indentFlag)
{
	Value *pValue = GetModule_sys()->LookupValue(
				indentFlag? AScript_Symbol(ps2) : AScript_Symbol(ps1), false);
	return (pValue == NULL || !pValue->IsString())? "" : pValue->GetString();
}

void Environment::SetConsole(bool errorOutputFlag, Console *pConsole)
{
	if (errorOutputFlag) {
		GetGlobal()->_pConsoleError = pConsole;
	} else {
		GetGlobal()->_pConsole = pConsole;
	}
}

Console *Environment::GetConsole(bool errorOutputFlag)
{
	if (errorOutputFlag) {
		if (GetGlobal()->_pConsoleError != NULL) return GetGlobal()->_pConsoleError;
	} else {
		if (GetGlobal()->_pConsole != NULL) return GetGlobal()->_pConsole;
	}
	const Symbol *pSymbol = errorOutputFlag?
							AScript_Symbol(stderr) : AScript_Symbol(stdout);
	Value *pValue = GetModule_sys()->LookupValue(pSymbol, false);
	if (pValue == NULL || !pValue->IsFile()) return NULL;
	return &pValue->GetFile();
}

// this function is called in a context before main() function.
void Environment::IntegrateModule(const char *name,
			ModuleEntryType moduleEntry, ModuleTerminateType moduleTerminate)
{
	if (_pIntegratedModuleOwner == NULL) {
		_pIntegratedModuleOwner = new IntegratedModuleOwner();
	}
	_pIntegratedModuleOwner->push_back(
					new IntegratedModule(name, moduleEntry, moduleTerminate));
}

bool Environment::IsModule() const { return false; }
bool Environment::IsClass() const { return false; }
bool Environment::IsObject() const { return false; }

//-----------------------------------------------------------------------------
// Environment::Global
//-----------------------------------------------------------------------------
Environment::Global::Global() : _echoFlag(false),
	_pFunc_Pos(NULL),
	_pFunc_Neg(NULL),
	_pFunc_Invert(NULL),
	_pFunc_Not(NULL),
	_pFunc_Plus(NULL),
	_pFunc_Minus(NULL),
	_pFunc_Multiply(NULL),
	_pFunc_Divide(NULL),
	_pFunc_Modulo(NULL),
	_pFunc_format(NULL),
	_pFunc_Power(NULL),
	_pFunc_Equal(NULL),
	_pFunc_NotEqual(NULL),
	_pFunc_Greater(NULL),
	_pFunc_Less(NULL),
	_pFunc_GreaterEq(NULL),
	_pFunc_LessEq(NULL),
	_pFunc_Compare(NULL),
	_pFunc_ContainCheck(NULL),
	_pFunc_Or(NULL),
	_pFunc_And(NULL),
	_pFunc_Xor(NULL),
	_pFunc_ShiftL(NULL),
	_pFunc_ShiftR(NULL),
	_pFunc_OrOr(NULL),
	_pFunc_AndAnd(NULL),
	_pFunc_Sequence(NULL),
	_pFunc_SequenceInf(NULL),
	_pConsole(NULL),
	_pConsoleError(NULL)
{
}

Environment::Global::~Global()
{
	foreach_const (SeparatedModuleMap, iter, _separatedModuleMap) {
		ObjectBase::Delete(iter->second);
	}
	Function::Delete(_pFunc_Pos);
	Function::Delete(_pFunc_Neg);
	Function::Delete(_pFunc_Invert);
	Function::Delete(_pFunc_Not);
	Function::Delete(_pFunc_Plus);
	Function::Delete(_pFunc_Minus);
	Function::Delete(_pFunc_Multiply);
	Function::Delete(_pFunc_Divide);
	Function::Delete(_pFunc_Modulo);
	Function::Delete(_pFunc_format);
	Function::Delete(_pFunc_Power);
	Function::Delete(_pFunc_Equal);
	Function::Delete(_pFunc_NotEqual);
	Function::Delete(_pFunc_Greater);
	Function::Delete(_pFunc_Less);
	Function::Delete(_pFunc_GreaterEq);
	Function::Delete(_pFunc_LessEq);
	Function::Delete(_pFunc_Compare);
	Function::Delete(_pFunc_ContainCheck);
	Function::Delete(_pFunc_Or);
	Function::Delete(_pFunc_And);
	Function::Delete(_pFunc_Xor);
	Function::Delete(_pFunc_ShiftL);
	Function::Delete(_pFunc_ShiftR);
	Function::Delete(_pFunc_OrOr);
	Function::Delete(_pFunc_AndAnd);
	Function::Delete(_pFunc_Sequence);
	Function::Delete(_pFunc_SequenceInf);
}

void Environment::Global::Prepare()
{
	_workingDirList.push_back(OAL::getcwd());
	_pValueTypePool = ValueTypePool::GetInstance();
}

Class *Environment::Global::LookupClass(ValueType valType) const
{
	return ValueTypePool::GetInstance()->Lookup(valType)->GetClass();
}

Module *Environment::Global::LookupIntegratedModule(int id) const
{
	IntegratedModuleMap::const_iterator iter = _integratedModuleMap.find(id);
	return (iter == _integratedModuleMap.end())? NULL : iter->second;
}

void Environment::Global::RegisterIntegratedModule(int id, Module *pModule)
{
	_integratedModuleMap[id] = pModule;
}

Module *Environment::Global::LookupSeparatedModule(const char *pathName) const
{
	SeparatedModuleMap::const_iterator iter = _separatedModuleMap.find(pathName);
	return (iter == _separatedModuleMap.end())? NULL : iter->second;
}

void Environment::Global::RegisterSeparatedModule(const char *pathName, Module *pModule)
{
	_separatedModuleMap[pathName] = pModule;
}

//-----------------------------------------------------------------------------
// Environment::Frame
//-----------------------------------------------------------------------------
Environment::Frame::Frame(const Frame &frame) :
			_cntRef(1), _envType(frame._envType), _pGlobal(frame._pGlobal)
{
	if (frame._pValueMap.get() != NULL) {
		_pValueMap.reset(new ValueMap(*frame._pValueMap));
	}
	if (frame._pValueTypeMap.get() != NULL) {
		_pValueTypeMap.reset(new ValueTypeMap(*frame._pValueTypeMap));
	}
}

Environment::Frame::Frame(EnvType envType, Global *pGlobal) :
			_cntRef(1), _envType(envType), _pGlobal(pGlobal)
{
}

Environment::Frame::~Frame()
{
}

void Environment::Frame::AssignValue(const Symbol *pSymbol, const Value &value)
{
	if (_pValueMap.get() == NULL) _pValueMap.reset(new ValueMap());
	(*_pValueMap)[pSymbol] = value;
}

void Environment::Frame::RemoveValue(const Symbol *pSymbol)
{
	if (_pValueMap.get() == NULL) return;
	_pValueMap->erase(pSymbol);
}

Value *Environment::Frame::LookupValue(const Symbol *pSymbol)
{
	if (_pValueMap.get() == NULL) return NULL;
	ValueMap::iterator iter = _pValueMap->find(pSymbol);
	return (iter == _pValueMap->end())? NULL : &iter->second;
}

void Environment::Frame::AssignValueType(const ValueTypeInfo *pValueTypeInfo)
{
	if (_pValueTypeMap.get() == NULL) _pValueTypeMap.reset(new ValueTypeMap());
	(*_pValueTypeMap)[pValueTypeInfo->GetSymbol()] = pValueTypeInfo;
}

const ValueTypeInfo *Environment::Frame::LookupValueType(const Symbol *pSymbol) const
{
	if (_pValueTypeMap.get() == NULL) return NULL;
	ValueTypeMap::iterator iter = _pValueTypeMap->find(pSymbol);
	return (iter == _pValueTypeMap->end())? NULL : iter->second;
}

void Environment::Frame::DbgPrint() const
{
	::printf("%p %-10s _pValueMap=%p\n",
						this, GetEnvTypeName(GetType()), _pValueMap.get());
	::printf("[Values]\n");
	if (_pValueMap.get() != NULL && !_pValueMap->empty()) {
		foreach_const (ValueMap, iter, *_pValueMap) {
			::printf(" %s", iter->first->GetName());
		}
		::printf("\n");
	}
	::printf("[Value Types]\n");
	if (_pValueTypeMap.get() != NULL && !_pValueTypeMap->empty()) {
		foreach_const (ValueTypeMap, iter, *_pValueTypeMap) {
			::printf(" %s", iter->first->GetName());
		}
		::printf("\n");
	}
}

//-----------------------------------------------------------------------------
// Environment::FrameList
//-----------------------------------------------------------------------------
Environment::FrameList::~FrameList()
{
}

//-----------------------------------------------------------------------------
// EnvironmentRoot
//-----------------------------------------------------------------------------
EnvironmentRoot::EnvironmentRoot(int argc, const char *argv[]) :
											Environment(NULL, ENVTYPE_Root)
{
	Signal sig;
	Environment &env = *this;
	RandomGenerator::Initialize(1234);	// initialize random generator SFMT
	ValueTypePool::Initialize(env);
	GetGlobal()->Prepare();
	AssignErrorTypes(env);	// Signal.cpp
	AssignOperators(env);	// Operators.cpp
	do { // import(__builtins__) { * }
		AScript_Module(__builtins__)::MixIn(env, sig);
		if (sig.IsSignalled()) return;
	} while (0);
	do { // import(sys)
		Module *pModule = AScript_Module(sys)::Import(env, sig);
		if (sig.IsSignalled()) return;
		AScript_Module(sys)::Setup(pModule, sig, argc, argv);
		if (sig.IsSignalled()) return;
	} while (0);
	do { // import(os)
		AScript_Module(os)::Import(env, sig);
		if (sig.IsSignalled()) return;
	} while (0);
	do { // import(time)
		AScript_Module(time)::Import(env, sig);
		if (sig.IsSignalled()) return;
	} while (0);
	do { // import(string)
		AScript_Module(string)::Import(env, sig);
		if (sig.IsSignalled()) return;
	} while (0);
	do { // import(math)
		AScript_Module(math)::Import(env, sig);
		if (sig.IsSignalled()) return;
	} while (0);
}

EnvironmentRoot::~EnvironmentRoot()
{
	Global *pGlobal = GetGlobal();
	Global::Delete(pGlobal);
}

}
