#include "Declaration.h"
#include "Expr.h"
#include "Environment.h"
#include "Object.h"
#include "Object_Matrix.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Declaration
//-----------------------------------------------------------------------------
Declaration::Declaration(const Declaration &decl) :
	_pSymbol(decl._pSymbol), _valType(decl._valType),
	_occurPattern(decl._occurPattern), _listFlag(decl._listFlag), _nomapFlag(decl._nomapFlag),
	_pExprDefault((decl._pExprDefault == NULL)? NULL : decl._pExprDefault->IncRef())
{
}

Declaration::Declaration(const Symbol *pSymbol, ValueType valType,
	OccurPattern occurPattern, bool listFlag, bool nomapFlag, Expr *pExprDefault) :
	_pSymbol(pSymbol), _valType(valType),
	_occurPattern(occurPattern), _listFlag(listFlag), _nomapFlag(nomapFlag),
	_pExprDefault(pExprDefault)
{
}

Declaration::~Declaration()
{
	Expr::Delete(_pExprDefault);
}

Declaration *Declaration::Create(Environment &env, Signal sig, const Expr *pExpr)
{
	bool listFlag = false;
	bool nomapFlag = false;
	OccurPattern occurPattern = OCCUR_Once;
	ValueType valType = VTYPE_Any;
	Expr *pExprDefault = NULL;
	if (pExpr->IsDictAssign()) {
		const Expr_DictAssign *pExprDictAssign =
								dynamic_cast<const Expr_DictAssign *>(pExpr);
		pExpr = pExprDictAssign->GetLeft();
		const Expr *pExprRight = pExprDictAssign->GetRight();
		if (pExprRight->IsForce()) {
			Value value = pExprRight->Exec(env, sig);
			if (sig.IsSignalled()) return NULL;
			pExprDefault = new Expr_Value(value);
		} else {
			pExprDefault = pExprRight->IncRef();
		}
	}
	if (pExpr->IsSuffix()) {
		const Expr_Suffix *pExprSuffix = dynamic_cast<const Expr_Suffix *>(pExpr);
		pExpr = pExprSuffix->GetChild();
		occurPattern = pExprSuffix->GetOccurPattern();
		if (occurPattern == OCCUR_Invalid) {
			sig.SetError(ERR_SyntaxError, "invalid argument expression");
			return NULL;
		}
		if (pExprDefault) {
			sig.SetError(ERR_SyntaxError, "optional arguments cannot take default value");
			return NULL;
		}
	}
	if (pExpr->IsQuote()) {
		pExpr = pExpr->Unquote();
		valType = VTYPE_Quote;
	}
	if (pExpr->IsIndexer()) {
		pExpr = dynamic_cast<const Expr_Indexer *>(pExpr)->GetCar();
		listFlag = true;
	}
	if (!pExpr->IsSymbol()) {
		sig.SetError(ERR_SyntaxError, "invalid argument expression");
		return NULL;
	}
	const Expr_Symbol *pExprSym = dynamic_cast<const Expr_Symbol *>(pExpr);
	const SymbolList &attrFront = pExprSym->GetAttrFront();
	const Symbol *pSymbolForType = NULL;
	if (!attrFront.empty()) {
		if (valType != VTYPE_Any) {
			sig.SetError(ERR_SyntaxError, "incompatible type declaration");
			return NULL;
		}
		const ValueTypeInfo *pValueTypeInfo = env.LookupValueType(attrFront);
		if (pValueTypeInfo != NULL) {
			pSymbolForType = attrFront.front();
			valType = pValueTypeInfo->GetValueType();
		}
	}
	foreach_const (SymbolSet, ppSymbol, pExprSym->GetAttrs()) {
		const Symbol *pSymbol = *ppSymbol;
		if (pSymbol->IsIdentical(AScript_Symbol(nomap))) {
			nomapFlag = true;
		} else if (pSymbolForType == NULL || !pSymbol->IsIdentical(pSymbolForType)) {
			sig.SetError(ERR_SyntaxError,
				"cannot accept a symbol %s in argument declaration", pSymbol->GetName());
		}
	}
	return new Declaration(pExprSym->GetSymbol(), valType,
							occurPattern, listFlag, nomapFlag, pExprDefault);
}

// value will be casted only when that is valid for declaration.
// it will not be casted if validation fails.
bool Declaration::ValidateAndCast(Environment &env, Signal sig,
						Value &value, bool &exprFlag, bool listElemFlag) const
{
	if (!listElemFlag && GetListFlag()) {
		if (value.IsIterator()) {
			Iterator *pIterator = value.CreateIterator(sig);
			if (sig.IsSignalled()) return false;
			Value result = pIterator->ToList(env, sig);
			Iterator::Delete(pIterator);
			if (sig.IsSignalled()) return false;
			value = result;
		} else if (value.IsMatrix()) {
			const Object_Matrix *pObjMat = value.GetMatrixObj();
			if (pObjMat->RowSize() == 1) {
				Value result = pObjMat->GetRow(env, sig, 0);
				value = result;
			} else if (pObjMat->ColSize() == 1) {
				Value result = pObjMat->GetCol(env, sig, 0);
				value = result;
			}
		}
		if (value.IsList()) {
			foreach (ValueList, pValue, value.GetList()) {
				if (!ValidateAndCast(env, sig, *pValue, exprFlag, true)) {
					SetError_ArgumentType(sig, *pValue);
					return false;
				}
			}
			return true;
		}
		if (IsOptional() && value.IsInvalid()) {
			return true;
		} else {
			SetError_ArgumentMustBeList(sig, value);
			return false;
		}
	}
#if 0
	if (GetValueType() != VTYPE_Expr && GetValueType() != VTYPE_Quote &&
												GetValueType() != VTYPE_Symbol) {
		if (value.IsExpr()) {
			exprFlag = true;
			return true;
		} else if (value.IsSymbol()) {
			const Symbol *pSymbol = value.GetSymbol();
			value.InitAsExpr(env, new Expr_Symbol(pSymbol));
			exprFlag = true;
			return true;
		}
	}
#endif
	if (IsOptional() && value.IsInvalid()) {
		return true;
	} else if (GetValueType() == VTYPE_Any ||
							value.IsInstanceOf(GetValueType())) {
		if (value.IsIterator()) {
			// make a clone of the iterator
			Iterator *pIterator = value.CreateIterator(sig);
			if (pIterator == NULL) return false;
			value = Value(env, pIterator);
		}
		return true;
	} else if (GetValueType() == VTYPE_Iterator) {
		Iterator *pIterator = value.CreateIterator(sig);
		if (pIterator != NULL) {
			value = Value(env, pIterator);
			return true;
		}
	} else if (GetValueType() == VTYPE_List) {
		if (value.IsList()) {
			return true;
		} else if (value.IsIterator()) {
			Iterator *pIterator = value.CreateIterator(sig);
			if (sig.IsSignalled()) return false;
			Value result = pIterator->ToList(env, sig);
			Iterator::Delete(pIterator);
			if (sig.IsSignalled()) return false;
			value = result;
			return true;
		} else if (value.IsMatrix()) {
			const Object_Matrix *pObjMat = value.GetMatrixObj();
			if (pObjMat->RowSize() == 1) {
				Value result = pObjMat->GetRow(env, sig, 0);
				value = result;
			} else if (pObjMat->ColSize() == 1) {
				Value result = pObjMat->GetCol(env, sig, 0);
				value = result;
			}
			return true;
		}
	} else if (GetValueType() == VTYPE_Object) {
		// type check of different Objects must be in consideration
		if (!value.IsList() && value.IsObject()) {
			return true;
		}
	} else if (GetValueType() == VTYPE_Number) {
		bool allowPartFlag = false;
		bool successFlag;
		Number num = value.ToNumber(allowPartFlag, successFlag);
		if (successFlag) {
			value = Value(num);
			return true;
		} else {
			sig.SetError(ERR_ValueError, "failed to convert to a number");
			return false;
		}
	} else if (GetValueType() == VTYPE_String) {
		value = Value(env, value.ToString(sig, false).c_str());
		return !sig.IsSignalled();
	} else if (GetValueType() == VTYPE_Complex) {
		if (value.IsNumber()) {		// cast number to complex
			return true;
		}
	} else if (GetValueType() == VTYPE_Expr) {
		if (value.IsSymbol()) {		// cast Symbol to Expr
			const Symbol *pSymbol = value.GetSymbol();
			value.InitAsExpr(env, new Expr_Symbol(pSymbol));
			return true;
		}
	} else if (GetValueType() == VTYPE_Quote) {
		return true;
	} else if (GetValueType() == VTYPE_Boolean) {
		if (value.IsList()) {
			return true;	// ?????
		} else {
			value = Value(value.GetBoolean());
			return true;
		}
	}
	SetError_ArgumentType(sig, value);
	return false;
}

void Declaration::SetError_ArgumentType(Signal sig, const Value &value) const
{
	sig.SetError(ERR_TypeError, "argument '%s' cannot take %s value in '%s'",
				GetSymbol()->GetName(), value.GetTypeName(), ToString().c_str());
}

void Declaration::SetError_ArgumentMustBeList(Signal sig, const Value &value) const
{
	sig.SetError(ERR_TypeError, "argument '%s' can only take a list in '%s'",
				GetSymbol()->GetName(), ToString().c_str());
}

String Declaration::ToString() const
{
	String str;
	if (_valType == VTYPE_Quote) {
		str += "`";
	}
	str += _pSymbol->GetName();
	if (_listFlag) str += "[]";
	str += GetOccurPatternSymbol(_occurPattern)->GetName();
	if (_valType != VTYPE_Invalid &&
					_valType != VTYPE_Any && _valType != VTYPE_Quote) {
		str += ":";
		str += GetValueTypeSymbol(_valType)->GetName();
	}
	if (_nomapFlag) {
		str += ":";
		str += AScript_Symbol(nomap)->GetName();
	}
	if (_pExprDefault != NULL) {
		str += " ";
		str += "=>";
		str += " ";
		str += _pExprDefault->ToString();
	}
	return str;
}

//-----------------------------------------------------------------------------
// DeclarationList
//-----------------------------------------------------------------------------
const DeclarationList DeclarationList::Null;

DeclarationList::~DeclarationList()
{
}

bool DeclarationList::IsVariableLength() const
{
	foreach_const (DeclarationList, ppDecl, *this) {
		if ((*ppDecl)->IsVariableLength()) return true;
	}
	return false;
}

bool DeclarationList::IsApplicable(const ValueList &valList) const
{
	ValueList::const_iterator pValue = valList.begin();
	const_iterator ppDecl = begin();
	for ( ; pValue != valList.end() && ppDecl != end(); pValue++) {
		const Declaration *pDecl = *ppDecl;
		if (!pDecl->IsApplicable(*pValue)) return false;
		if (!pDecl->IsVariableLength()) ppDecl++;
	}
	return true;
}

void DeclarationList::SetAsLoose()
{
	foreach (DeclarationList, ppDecl, *this) {
		Declaration *pDecl = *ppDecl;
		if (pDecl->GetOccurPattern() == OCCUR_Once) {
			pDecl->SetOccurPattern(OCCUR_ZeroOrOnce);
		} else if (pDecl->GetOccurPattern() == OCCUR_OnceOrMore) {
			pDecl->SetOccurPattern(OCCUR_ZeroOrMore);
		}
	}
}

bool DeclarationList::ValidateAndCast(Environment &env, Signal sig,
	const Function *pFunc, const ValueList &valList, ValueList &valListCasted,
	bool &exprFlag) const
{
	ValueList::const_iterator pValue = valList.begin();
	DeclarationList::const_iterator ppDecl = begin();
	for ( ; ppDecl != end(); ppDecl++) {
		const Declaration *pDecl = *ppDecl;
		OccurPattern occurPattern = pDecl->GetOccurPattern();
		if (occurPattern == OCCUR_ZeroOrMore || occurPattern == OCCUR_OnceOrMore) {
			Value value;
			ValueList &valListElem = value.InitAsList(env);
			valListCasted.push_back(value);
			for ( ; pValue != valList.end(); pValue++) {
				Value value = *pValue;
				if (!pDecl->ValidateAndCast(env, sig, value, exprFlag)) return false;
				valListElem.push_back(value);
			}
			if (occurPattern == OCCUR_OnceOrMore && valListElem.empty()) {
				SetError_NotEnoughArguments(sig, pFunc);
				return false;
			}
			break;
		} else if (pValue == valList.end()) {
			if (occurPattern == OCCUR_ZeroOrOnce) {
				valListCasted.push_back(Value::Null);
			} else {
				SetError_NotEnoughArguments(sig, pFunc);
				return false;
			}
		} else {
			Value value = *pValue;
			if (!pDecl->ValidateAndCast(env, sig, value, exprFlag)) return false;
			valListCasted.push_back(value);
			pValue++;
		}
	}
	if (pValue != valList.end() && !pFunc->IsAllowTooManyArgs()) {
		SetError_TooManyArguments(sig, pFunc);
		return false;
	}
	return true;
}

bool DeclarationList::Compensate(Environment &env, Signal sig,
									const Function *pFunc, ValueList &valList)
{
	if (size() <= valList.size()) return true;
	DeclarationList::const_iterator ppDecl = begin() + valList.size();
	for ( ; ppDecl != end(); ppDecl++) {
		const Declaration *pDecl = *ppDecl;
		const Expr *pExprArg = pDecl->GetExprDefault();
		Value value;
		if (pExprArg == NULL) {
			if (pDecl->GetOccurPattern() == OCCUR_ZeroOrOnce) {
				value = Value::Null;
			} else if (pDecl->GetOccurPattern() == OCCUR_ZeroOrMore) {
				break;
			} else {
				SetError_NotEnoughArguments(sig, pFunc);
				return false;
			}
		} else if (pDecl->IsQuote()) {
			value.InitAsExpr(env, pExprArg->IncRef());
			valList.push_back(value);
		} else if (pDecl->IsType(VTYPE_Symbol)) {
			const Expr *pExpr = pExprArg;
			if (pExpr->IsQuote()) {
				pExpr = dynamic_cast<const Expr_Quote *>(pExpr)->GetChild();
			}
			if (!pExpr->IsSymbol()) {
				sig.SetError(ERR_TypeError, "symbol is expected");
				return false;
			}
			const Symbol *pSymbol =
						dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
			value = Value(pSymbol);
		} else {
			value = pExprArg->Exec(env, sig);
			if (sig.IsSignalled()) return false;
		}
		valList.push_back(value);
	}
	return true;
}

bool DeclarationList::ShouldGenerateIterator(const ValueList &valListArg) const
{
	ValueList::const_iterator pValue = valListArg.begin();
	DeclarationList::const_iterator ppDecl = begin();
	for ( ; pValue != valListArg.end() && ppDecl != end(); pValue++) {
		const Declaration *pDecl = *ppDecl;
		if (pValue->IsIterator() &&
						pDecl->GetValueType() != VTYPE_Iterator) return true;
		if (!pDecl->IsVariableLength()) ppDecl++;
	}
	return false;
}

void DeclarationList::AssignAsDeclaration(Environment &env, Signal sig,
												const ValueList &valList) const
{
	ValueList::const_iterator pValue = valList.begin();
	DeclarationList::const_iterator ppDecl = begin();
	for ( ; pValue != valList.end() && ppDecl != end(); pValue++, ppDecl++) {
		const Declaration *pDecl = *ppDecl;
		env.AssignValue(pDecl->GetSymbol(), *pValue, false);
	}
}

String DeclarationList::ToString() const
{
	String str;
	foreach_const (DeclarationList, ppDecl, *this) {
		if (ppDecl != begin()) str += ", ";
		str += (*ppDecl)->ToString();
	}
	return str;
}

void DeclarationList::SetError_InvalidArgument(Signal sig, const Function *pFunc) const
{
	sig.SetError(ERR_SyntaxError, "invalid argument");
}

void DeclarationList::SetError_NotEnoughArguments(Signal sig, const Function *pFunc) const
{
	sig.SetError(ERR_TypeError, "not enough arguments for '%s'",
												pFunc->ToString().c_str());
}

void DeclarationList::SetError_TooManyArguments(Signal sig, const Function *pFunc) const
{
	sig.SetError(ERR_TypeError, "too many arguments for '%s'",
												pFunc->ToString().c_str());
}

void DeclarationList::SetError_InvalidArgumentName(Signal sig,
							const Function *pFunc, const ExprMap &exprMap) const
{
	String str("function '");
	str += pFunc->GetName();
	str += "' doesn't have arguments named ";
	foreach_const (ExprMap, iter, exprMap) {
		if (iter != exprMap.begin()) str += ", ";
		str += iter->first->GetName();
	}
	sig.SetError(ERR_ValueError, "%s", str.c_str());
}

//-----------------------------------------------------------------------------
// DeclarationOwner
//-----------------------------------------------------------------------------
DeclarationOwner::DeclarationOwner(const DeclarationOwner &declOwner) :
										_pSymbolDict(declOwner._pSymbolDict)
{
	foreach_const (DeclarationList, ppDecl, declOwner) {
		push_back((*ppDecl)->Clone());
	}
}

DeclarationOwner::~DeclarationOwner()
{
	foreach (DeclarationOwner, ppDecl, *this) {
		Declaration::Delete(*ppDecl);
	}
}

void DeclarationOwner::operator=(const DeclarationOwner &declOwner)
{
	_pSymbolDict = declOwner._pSymbolDict;
	foreach_const (DeclarationList, ppDecl, declOwner) {
		push_back((*ppDecl)->Clone());
	}
}

bool DeclarationOwner::Declare(Environment &env, Signal sig, const ExprList &exprList)
{
	foreach_const (ExprList, ppExpr, exprList) {
		const Expr *pExpr = *ppExpr;
		if (pExpr->IsSuffix()) {
			const Expr_Suffix *pExprSuffix =
									dynamic_cast<const Expr_Suffix *>(pExpr);
			const Symbol *pSymbol = pExprSuffix->GetSymbol();
			if (pSymbol->IsIdentical(AScript_Symbol(Char_Modulo))) {
				const Expr *pExprChild = pExprSuffix->GetChild();
				if (!pExprChild->IsSymbol()) {
					sig.SetError(ERR_SyntaxError,
									"invalid expression for declaration");
					return false;
				}
				SetSymbolDict(
						dynamic_cast<const Expr_Symbol *>(pExprChild)->GetSymbol());
				continue;
			}
		}
		Declaration *pDecl = Declaration::Create(env, sig, pExpr);
		if (pDecl == NULL) return false;
		if (IsVariableLength()) {
			sig.SetError(ERR_TypeError,
				"any parameters cannot follow after a parameter with variable length");
			Declaration::Delete(pDecl);
			return false;
		}
		if (pDecl->IsMandatory() && pDecl->GetExprDefault() == NULL &&
											!empty() && back()->IsOptional()) {
			sig.SetError(ERR_TypeError,
				"mandatory parameters cannot follow after a parameter with variable length");
			Declaration::Delete(pDecl);
			return false;
		}
		push_back(pDecl);
	}
	return true;
}

bool DeclarationOwner::PrepareArgsForUnary(Environment &env, Signal sig, const Function *pFunc,
		const ExprList &exprArgs, ValueList &valListArg, Value &valueWithDict) const
{
	Value value;
	ValueList &valList = value.InitAsList(env);
	ValueDict &valDict = valueWithDict.GetDict();
	const Declaration *pDecl = front();
	foreach_const (ExprList, ppExprArg, exprArgs) {
		const Expr *pExprArg = *ppExprArg;
		size_t nElems = 0;
		if (pExprArg->IsDictAssign()) {
			if (GetSymbolDict() == NULL) {
				sig.SetError(ERR_ValueError, "dictionary assignment is not effective");
				return false;
			}
			const Expr_DictAssign *pExprDictAssign =
						dynamic_cast<const Expr_DictAssign *>(pExprArg);
			Value valueKey = pExprDictAssign->GetKey(sig);
			if (sig.IsSignalled()) return false;
			Value value = pExprDictAssign->GetRight()->Exec(env, sig);
			if (sig.IsSignalled()) return false;
			valDict[valueKey] = value;
		} else if (!pExprArg->ExecInArg(env, sig, valList, nElems, pDecl->IsQuote())) {
			return false;
		}
	}
	valListArg.push_back(value);
	return true;
}

bool DeclarationOwner::PrepareArgs(Environment &env, Signal sig, const Function *pFunc,
		const ExprList &exprArgs, ValueList &valListArg, Value &valueWithDict) const
{
	ExprMap exprMap;
	ValueDict &valDict = valueWithDict.GetDict();
	DeclarationList::const_iterator ppDecl = begin();
	bool stayDeclPointerFlag = false;
	foreach_const (ExprList, ppExprArg, exprArgs) {
		const Expr *pExprArg = *ppExprArg;
		bool quoteFlag = ppDecl != end() && (*ppDecl)->IsQuote();
		if (!quoteFlag && pExprArg->IsDictAssign()) {
			const Expr_DictAssign *pExprDictAssign =
							dynamic_cast<const Expr_DictAssign *>(pExprArg);
			Value valueKey = pExprDictAssign->GetKey(sig);
			if (sig.IsSignalled()) return false;
			if (valueKey.IsSymbol()) {
				exprMap[valueKey.GetSymbol()] = pExprDictAssign->GetRight();
			} else {
				Value value = pExprDictAssign->GetRight()->Exec(env, sig);
				if (sig.IsSignalled()) return false;
				valDict[valueKey] = value;
			}
		} else if (ppDecl != end()) {
			const Declaration *pDecl = *ppDecl;
			if (exprMap.find(pDecl->GetSymbol()) != exprMap.end()) {
				sig.SetError(ERR_ValueError, "argument confliction");
				return false;
			}
			size_t nElems = 0;
			if (!pExprArg->ExecInArg(env, sig, valListArg, nElems, pDecl->IsQuote())) {
				return false;
			}
			if (pDecl->IsVariableLength()) {
				stayDeclPointerFlag = true;
			} else {
				ppDecl += nElems;
			}
		} else if (pFunc->IsAllowTooManyArgs()) {
			break;
		} else {
			SetError_TooManyArguments(sig, pFunc);
			return false;
		}
	}
	if (stayDeclPointerFlag) ppDecl = end();
	for ( ; ppDecl != end(); ppDecl++) {
		const Declaration *pDecl = *ppDecl;
		const Expr *pExprArg = pDecl->GetExprDefault();
		ExprMap::iterator iter = exprMap.find(pDecl->GetSymbol());
		if (iter != exprMap.end()) {
			exprMap.erase(iter);
			pExprArg = iter->second;
		}
		Value value;
		if (pExprArg == NULL) {
			if (pDecl->GetOccurPattern() == OCCUR_ZeroOrOnce) {
				value = Value::Null;
			} else if (pDecl->GetOccurPattern() == OCCUR_ZeroOrMore) {
				break;
			} else {
				SetError_NotEnoughArguments(sig, pFunc);
				return false;
			}
		} else if (pDecl->IsQuote()) {
			value.InitAsExpr(env, pExprArg->IncRef());
			valListArg.push_back(value);
		} else if (pDecl->IsType(VTYPE_Symbol)) {
			const Expr *pExpr = pExprArg;
			if (pExpr->IsQuote()) {
				pExpr = dynamic_cast<const Expr_Quote *>(pExpr)->GetChild();
			}
			if (!pExpr->IsSymbol()) {
				sig.SetError(ERR_TypeError, "symbol is expected");
				return false;
			}
			const Symbol *pSymbol =
						dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
			value = Value(pSymbol);
		} else {
			value = pExprArg->Exec(env, sig);
			if (sig.IsSignalled()) return false;
		}
		valListArg.push_back(value);
	}
	if (GetSymbolDict() != NULL) {
		foreach (ExprMap, iter, exprMap) {
			const Symbol *pSymbol = iter->first;
			const Expr *pExprArg = iter->second;
			Value value = pExprArg->Exec(env, sig);
			if (sig.IsSignalled()) return false;
			valDict[Value(pSymbol)] = value;
		}
	} else if (!exprMap.empty()) {
		SetError_InvalidArgumentName(sig, pFunc, exprMap);
		return false;
	}
	return true;
}

String DeclarationOwner::ToString() const
{
	String str = DeclarationList::ToString();
	if (_pSymbolDict != NULL) {
		if (!empty()) str += ", ";
		str += AScript_Symbol(Char_Modulo)->GetName();
		str += _pSymbolDict->GetName();
	}
	return str;
}

}
