#include "Operator.h"
#include "Iterator.h"
#include "Expr.h"
#include "Object_Function.h"
#include "Object_Bytes.h"

namespace AScript {

//-----------------------------------------------------------------------------
// basic operator
//-----------------------------------------------------------------------------
// -v
bool Func_Neg::IsNeg() const { return true; }

Func_Neg::Func_Neg(Environment &env) : Function("__neg__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("-");
	DeclareArg(env, "value", VTYPE_AnyType);
}

Value Func_Neg::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &value = context.GetValue(0);
	Value result;
	if (value.IsNumber()) {
		result.SetNumber(-value.GetNumber());
		return result;
	} else if (value.IsComplex()) {
		result.SetComplex(-value.GetComplex());
		return result;
	}
	SetError_InvalidValType(sig, value);
	return result;
}

// ~v
Func_Invert::Func_Invert(Environment &env) : Function("__invert__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("~");
	DeclareArg(env, "value", VTYPE_AnyType);
}

Value Func_Invert::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &value = context.GetValue(0);
	Value result;
	if (value.IsNumber()) {
		unsigned long num = ~static_cast<unsigned long>(value.GetNumber());
		result.SetNumber(static_cast<Number>(num));
		return result;
	}
	SetError_InvalidValType(sig, value);
	return result;
}

// !v
Func_Not::Func_Not(Environment &env) : Function("__not__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("!");
	DeclareArg(env, "flag", VTYPE_Boolean);
}

Value Func_Not::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	result.SetBoolean(!context.GetBoolean(0));
	return result;
}

// v = v + v
bool Func_Plus::IsPlus() const { return true; }

Func_Plus::Func_Plus(Environment &env) : Function("__add__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("+");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Plus::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		result.SetNumber(valueLeft.GetNumber() + valueRight.GetNumber());
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsComplex()) {
		result.SetComplex(valueLeft.GetComplex() + valueRight.GetComplex());
		return result;
	} else if (valueLeft.IsNumber() && valueRight.IsComplex()) {
		result.SetComplex(valueLeft.GetNumber() + valueRight.GetComplex());
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsNumber()) {
		result.SetComplex(valueLeft.GetComplex() + valueRight.GetNumber());
		return result;
	} else if (valueLeft.IsString() && valueRight.IsString()) {
		String str(valueLeft.GetString());
		str += valueRight.GetString();
		result = Value(env, str.c_str());
		return result;
	} else if (valueLeft.IsBytes() && valueRight.IsBytes()) {
		Bytes buff(valueLeft.GetBytes());
		buff += valueRight.GetBytes();
		result.InitAsBytes(env, buff.data(), buff.size());
		return result;
	} else if (valueLeft.IsBytes() && valueRight.IsString()) {
		Bytes buff(valueLeft.GetBytes());
		buff += valueRight.GetString();
		result.InitAsBytes(env, buff.data(), buff.size());
		return result;
	} else if (valueLeft.IsString() && valueRight.IsBytes()) {
		Bytes buff;
		buff += valueLeft.GetString();
		buff += valueRight.GetBytes();
		result.InitAsBytes(env, buff.data(), buff.size());
		return result;
	} else if (valueLeft.IsBytesPtr() && valueRight.IsNumber()) {
		Object_BytesPtr *pObj =
			dynamic_cast<Object_BytesPtr *>(valueLeft.GetBytesPtr()->Clone());
		pObj->UnpackForward(sig,
							static_cast<int>(valueRight.GetNumber()), true);
		if (sig.IsSignalled()) return Value::Null;
		Value result;
		result.InitAsObject(pObj, VTYPE_BytesPtr);
		return result;
	} else if (valueLeft.IsString()) {
		String str(valueLeft.GetString());
		str += valueRight.ToString(sig);
		result = Value(env, str.c_str());
		return result;
	} else if (valueRight.IsString()) {
		String str(valueLeft.ToString(sig));
		str += valueRight.GetString();
		result = Value(env, str.c_str());
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return Value::Null;
}

// v = v - v
bool Func_Minus::IsMinus() const { return true; }

Func_Minus::Func_Minus(Environment &env) : Function("__sub__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("-");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Minus::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		result.SetNumber(valueLeft.GetNumber() - valueRight.GetNumber());
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsComplex()) {
		result.SetComplex(valueLeft.GetComplex() - valueRight.GetComplex());
		return result;
	} else if (valueLeft.IsNumber() && valueRight.IsComplex()) {
		result.SetComplex(valueLeft.GetNumber() - valueRight.GetComplex());
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsNumber()) {
		result.SetComplex(valueLeft.GetComplex() - valueRight.GetNumber());
		return result;
	} else if (valueLeft.IsBytesPtr() && valueRight.IsNumber()) {
		Object_BytesPtr *pObj =
			dynamic_cast<Object_BytesPtr *>(valueLeft.GetBytesPtr()->Clone());
		pObj->UnpackForward(sig,
							-static_cast<int>(valueRight.GetNumber()), true);
		if (sig.IsSignalled()) return Value::Null;
		Value result;
		result.InitAsObject(pObj, VTYPE_BytesPtr);
		return result;
	} else if (valueLeft.IsBytesPtr() && valueRight.IsBytesPtr()) {
		const Object_BytesPtr *pObj1 = valueLeft.GetBytesPtr();
		const Object_BytesPtr *pObj2 = valueRight.GetBytesPtr();
		if (&pObj1->GetBytes() != &pObj2->GetBytes()) {
			sig.SetError(ERR_ValueError,
				"cannot calculate difference between pointers of different bytes");
			return Value::Null;
		}
		int offset1 = static_cast<int>(pObj1->GetOffset());
		int offset2 = static_cast<int>(pObj2->GetOffset());
		return Value(static_cast<Number>(offset1 - offset2));
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v = v * v
bool Func_Multiply::IsMultiply() const { return true; }

Func_Multiply::Func_Multiply(Environment &env) : Function("__mul__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("*");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Multiply::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		result.SetNumber(valueLeft.GetNumber() * valueRight.GetNumber());
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsComplex()) {
		result.SetComplex(valueLeft.GetComplex() * valueRight.GetComplex());
		return result;
	} else if (valueLeft.IsNumber() && valueRight.IsComplex()) {
		result.SetComplex(valueLeft.GetNumber() * valueRight.GetComplex());
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsNumber()) {
		result.SetComplex(valueLeft.GetComplex() * valueRight.GetNumber());
		return result;
	} else if (valueLeft.IsFunction()) {
		const Object_Function *pObj = valueLeft.GetFunctionObj();
		if (pObj->GetFunction()->IsUnary()) {
			ValueList valListArg(valueRight);
			Context context(valListArg);
			result = pObj->Eval(env, sig, context);
			if (sig.IsSignalled()) return Value::Null;
		} else {
			sig.SetError(ERR_TypeError, "unary function is expected for multiplier-form applier");
			return Value::Null;
		}
		return result;
	} else if (valueLeft.IsString() && valueRight.IsNumber()) {
		String str;
		for (int cnt = static_cast<int>(valueRight.GetNumber()); cnt > 0; cnt--) {
			str += valueLeft.GetString();
		}
		result = Value(env, str.c_str());
		return result;
	} else if (valueLeft.IsNumber() && valueRight.IsString()) {
		String str;
		for (int cnt = static_cast<int>(valueLeft.GetNumber()); cnt > 0; cnt--) {
			str += valueRight.GetString();
		}
		result = Value(env, str.c_str());
		return result;
	} else if (valueLeft.IsBytes() && valueRight.IsNumber()) {
		Bytes buff;
		for (int cnt = static_cast<int>(valueRight.GetNumber()); cnt > 0; cnt--) {
			buff += valueLeft.GetBytes();
		}
		result.InitAsBytes(env, buff.c_str(), buff.size());
		return result;
	} else if (valueLeft.IsNumber() && valueRight.IsBytes()) {
		Bytes buff;
		for (int cnt = static_cast<int>(valueLeft.GetNumber()); cnt > 0; cnt--) {
			buff += valueRight.GetBytes();
		}
		result.InitAsBytes(env, buff.c_str(), buff.size());
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v = v / v
bool Func_Divide::IsDivide() const { return true; }

Func_Divide::Func_Divide(Environment &env) : Function("__div__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("/");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Divide::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		Number numRight = valueRight.GetNumber();
		if (numRight == 0) {
			SetError_DivideByZero(sig);
			return Value::Null;
		}
		result.SetNumber(valueLeft.GetNumber() / numRight);
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsComplex()) {
		Complex numRight = valueRight.GetComplex();
		if (numRight == Complex(0.)) {
			SetError_DivideByZero(sig);
			return Value::Null;
		}
		result.SetComplex(valueLeft.GetComplex() / valueRight.GetComplex());
		return result;
	} else if (valueLeft.IsNumber() && valueRight.IsComplex()) {
		Complex numRight = valueRight.GetComplex();
		if (numRight == Complex(0.)) {
			SetError_DivideByZero(sig);
			return Value::Null;
		}
		result.SetComplex(valueLeft.GetNumber() / numRight);
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsNumber()) {
		Number numRight = valueRight.GetNumber();
		if (numRight == 0) {
			SetError_DivideByZero(sig);
			return Value::Null;
		}
		result.SetComplex(valueLeft.GetComplex() / numRight);
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// num = mod(num, num):map
bool Func_Modulo::IsModulo() const { return true; }

Func_Modulo::Func_Modulo(Environment &env) : Function("mod", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("%");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Modulo::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		result.SetNumber(::fmod(valueLeft.GetNumber(), valueRight.GetNumber()));
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValType(sig, valueLeft, valueRight);
	return result;
}

// str = format(format, value*):map
Func_format::Func_format(Environment &env) : Function("format", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
	DeclareArg(env, "values", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

Value Func_format::DoEval(Environment &env, Signal sig, Context &context) const
{
	return Value(env, Formatter::Format(sig,
						context.GetString(0), context.GetList(1)).c_str());
}

// v = v ** v
bool Func_Power::IsPower() const { return true; }

Func_Power::Func_Power(Environment &env) : Function("__pow__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("**");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Power::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		result.SetNumber(::pow(valueLeft.GetNumber(), valueRight.GetNumber()));
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsComplex()) {
		result.SetComplex(std::pow(valueLeft.GetComplex(), valueRight.GetComplex()));
		return result;
	} else if (valueLeft.IsNumber() && valueRight.IsComplex()) {
		result.SetComplex(std::pow(valueLeft.GetNumber(), valueRight.GetComplex()));
		return result;
	} else if (valueLeft.IsComplex() && valueRight.IsNumber()) {
		result.SetComplex(std::pow(valueLeft.GetComplex(), valueRight.GetNumber()));
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v == v
Func_Equal::Func_Equal(Environment &env) : Function("__eq__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("==");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Equal::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	do {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	} while (0);
	int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
	result.SetBoolean(cmp == 0);
	return result;
}

// v != v
Func_NotEqual::Func_NotEqual(Environment &env) : Function("__ne__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("!=");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_NotEqual::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	do {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	} while (0);
	int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
	result.SetBoolean(cmp != 0);
	return result;
}

// v > v
Func_Greater::Func_Greater(Environment &env) : Function("__gt__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol(">");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Greater::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	do {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	} while (0);
	int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
	result.SetBoolean(cmp > 0);
	return result;
}

// v < v
Func_Less::Func_Less(Environment &env) : Function("__lt__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("<");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Less::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	do {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	} while (0);
	int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
	result.SetBoolean(cmp < 0);
	return result;
}

// v >= v
Func_GreaterEq::Func_GreaterEq(Environment &env) : Function("__ge__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol(">=");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_GreaterEq::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	do {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	} while (0);
	int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
	result.SetBoolean(cmp >= 0);
	return result;
}

// v <= v
Func_LessEq::Func_LessEq(Environment &env) : Function("__le__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("<=");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_LessEq::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	do {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	} while (0);
	int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
	result.SetBoolean(cmp <= 0);
	return result;
}

// v <=> v
Func_Compare::Func_Compare(Environment &env) : Function("__cmp__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("<=>");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Compare::DoEval(Environment &env, Signal sig, Context &context) const
{
	do {
		bool evaluatedFlag = false;
		Value result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	} while (0);
	int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
	return Value(static_cast<Number>(cmp));
}

// v in v
bool Func_ContainCheck::IsContainCheck() const { return true; }

Func_ContainCheck::Func_ContainCheck(Environment &env) : Function("in", FUNCTYPE_Function)
{
	SetMathSymbol("in");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_ContainCheck::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	if (context.IsList(0)) {
		ValueList &valList = result.InitAsList(env);
		if (context.IsList(1)) {
			const ValueList &valListToFind = context.GetList(1);
			foreach_const (ValueList, pValue, context.GetList(0)) {
				valList.push_back(valListToFind.IsContain(*pValue));
			}
		} else {
			const Value &valueToCompare = context.GetValue(1);
			foreach_const (ValueList, pValue, context.GetList(0)) {
				valList.push_back(Value::Compare(*pValue, valueToCompare) == 0);
			}
		}
	} else {
		if (context.IsList(1)) {
			result.SetBoolean(context.GetList(1).IsContain(context.GetValue(0)));
		} else {
			int cmp = Value::Compare(context.GetValue(0), context.GetValue(1));
			result.SetBoolean(cmp == 0);
		}
	}
	return result;
}

// v | v
Func_Or::Func_Or(Environment &env) : Function("__or__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("|");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Or::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		unsigned long num = static_cast<unsigned long>(valueLeft.GetNumber()) |
							static_cast<unsigned long>(valueRight.GetNumber());
		result.SetNumber(num);
		return result;
	} else if (valueLeft.IsBoolean() && valueRight.IsBoolean()) {
		return valueLeft.GetBoolean() || valueRight.GetBoolean();
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v & v
Func_And::Func_And(Environment &env) : Function("__and__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("&");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_And::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		unsigned long num = static_cast<unsigned long>(valueLeft.GetNumber()) &
							static_cast<unsigned long>(valueRight.GetNumber());
		result.SetNumber(num);
		return result;
	} else if (valueLeft.IsBoolean() && valueRight.IsBoolean()) {
		return valueLeft.GetBoolean() && valueRight.GetBoolean();
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v ^ v
Func_Xor::Func_Xor(Environment &env) : Function("__xor__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("^");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Xor::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		unsigned long num = static_cast<unsigned long>(valueLeft.GetNumber()) ^
							static_cast<unsigned long>(valueRight.GetNumber());
		result.SetNumber(num);
		return result;
	} else if (valueLeft.IsBoolean() && valueRight.IsBoolean()) {
		bool flagLeft = valueLeft.GetBoolean();
		bool flagRight = valueRight.GetBoolean();
		return (flagLeft && !flagRight) || (!flagLeft && flagRight);
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v << v
Func_ShiftL::Func_ShiftL(Environment &env) : Function("__shl__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("^");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_ShiftL::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		unsigned long num = static_cast<unsigned long>(valueLeft.GetNumber()) <<
							static_cast<unsigned long>(valueRight.GetNumber());
		result.SetNumber(num);
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v >> v
Func_ShiftR::Func_ShiftR(Environment &env) : Function("__shr__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("^");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_ShiftR::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		unsigned long num = static_cast<unsigned long>(valueLeft.GetNumber()) >>
							static_cast<unsigned long>(valueRight.GetNumber());
		result.SetNumber(num);
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v || v
// this function takes quoted values as its arguments to implement
// a short-circuit evaluation.
Func_OrOr::Func_OrOr(Environment &env) : Function("__oror__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("||");
	DeclareArg(env, "valueLeft", VTYPE_Quote);
	DeclareArg(env, "valueRight", VTYPE_Quote);
}

Value Func_OrOr::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	result = context.GetExpr(0)->Exec(env, sig);
	if (sig.IsSignalled()) return Value::Null;
	if (result.GetBoolean()) return result;
	result = context.GetExpr(1)->Exec(env, sig);
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// v && v
// this function takes quoted values as its arguments to implement
// a short-circuit evaluation.
Func_AndAnd::Func_AndAnd(Environment &env) : Function("__andand__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("&&");
	DeclareArg(env, "valueLeft", VTYPE_Quote);
	DeclareArg(env, "valueRight", VTYPE_Quote);
}

Value Func_AndAnd::DoEval(Environment &env, Signal sig, Context &context) const
{
	Value result;
	result = context.GetExpr(0)->Exec(env, sig);
	if (sig.IsSignalled()) return Value::Null;
	if (!result.GetBoolean()) return result;
	result = context.GetExpr(1)->Exec(env, sig);
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

// v .. v
Func_Sequence::Func_Sequence(Environment &env) : Function("__seq__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("..");
	DeclareArg(env, "valueLeft", VTYPE_AnyType);
	DeclareArg(env, "valueRight", VTYPE_AnyType);
}

Value Func_Sequence::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &valueLeft = context.GetValue(0);
	const Value &valueRight = context.GetValue(1);
	Value result;
	if (valueLeft.IsNumber() && valueRight.IsNumber()) {
		Number numBegin = valueLeft.GetNumber();
		Number numEnd = valueRight.GetNumber();
		Number numStep = (numEnd >= numBegin)? +1 : -1;
		result.InitAsIterator(env, new Iterator_Sequence(numBegin, numEnd, numStep));
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValTypeM(sig, valueLeft, valueRight);
	return result;
}

// v ..
Func_SequenceInf::Func_SequenceInf(Environment &env) : Function("__seqinf__", FUNCTYPE_Function)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	SetMathSymbol("..");
	DeclareArg(env, "value", VTYPE_AnyType);
}

Value Func_SequenceInf::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Value &value = context.GetValue(0);
	Value result;
	if (value.IsNumber()) {
		Number numBegin = value.GetNumber();
		result.InitAsIterator(env, new Iterator_SequenceInf(numBegin));
		return result;
	} else {
		bool evaluatedFlag = false;
		result = EvalOverride(sig, context, evaluatedFlag);
		if (evaluatedFlag) return result;
	}
	SetError_InvalidValType(sig, value);
	return result;
}

}
