#ifndef __FILE_H__
#define __FILE_H__
#include "Common.h"
#include "String.h"
#include "Formatter.h"
#include "Codec.h"

namespace AScript {

//-----------------------------------------------------------------------------
// DateTime
//-----------------------------------------------------------------------------
class DateTime {
private:
	short _year;
	char _month, _day;
	char _hour, _minute, _second;
	long _microsec;
public:
	inline DateTime() : _year(0), _month(0),
					_day(0), _hour(0), _minute(0), _second(0), _microsec(0) {}
	inline DateTime(short year, char month, char day,
					char hour, char minute, char second, long microsec) :
		_year(year), _month(month), _day(day),
		_hour(hour), _minute(minute), _second(second), _microsec(microsec) {}
	inline DateTime(const DateTime &dt) :
		_year(dt._year), _month(dt._month), _day(dt._day),
		_hour(dt._hour), _minute(dt._minute), _second(dt._second), _microsec(dt._microsec) {}
#if defined(HAVE_WINDOWS_H)
	DateTime(const SYSTEMTIME &st);
	DateTime(const FILETIME &ft, bool toLocalFlag);
	void SetSystemTime(const SYSTEMTIME &st);
#else
	DateTime(const struct tm &tm);
	DateTime(time_t t, bool toLocalFlag);
	void SetTime(const struct tm &tm);
#endif
	String ToString(const char *format) const;
	short GetYear() const		{ return _year; }
	char GetMonth() const		{ return _month; }
	char GetDay() const			{ return _day; }
	char GetHour() const		{ return _hour; }
	char GetMinute() const		{ return _minute; }
	char GetSecond() const		{ return _second; }
	long GetMicroSec() const	{ return _microsec; }
	static int Compare(const DateTime &dt1, const DateTime &dt2);
};

//-----------------------------------------------------------------------------
// FileStat
//-----------------------------------------------------------------------------
class FileStat {
public:
	enum {
		ATTR_Dir		= (1 << 22),
		ATTR_Chr		= (1 << 21),
		ATTR_Blk		= (1 << 20),
		ATTR_Reg		= (1 << 19),
		ATTR_Fifo		= (1 << 18),
		ATTR_Lnk		= (1 << 17),
		ATTR_Sock		= (1 << 16),
	};
private:
	String _pathName;
	unsigned long _attr;
	unsigned long _bytes;
	DateTime _atime;
	DateTime _mtime;
	DateTime _ctime;
	long _uid;
	long _gid;
public:
	inline FileStat() : _attr(0), _bytes(0), _uid(0), _gid(0) {}
	inline FileStat(const char *pathName, unsigned long attr, unsigned long bytes,
				const DateTime &atime, const DateTime &mtime, const DateTime &ctime,
				long uid, long gid) :
		_pathName(pathName), _attr(attr), _bytes(bytes),
		_atime(atime), _mtime(mtime), _ctime(ctime),
		_uid(uid), _gid(gid) {}
	inline FileStat(const FileStat &fileStat) :
		_pathName(fileStat._pathName), _attr(fileStat._attr), _bytes(fileStat._bytes),
		_atime(fileStat._atime), _mtime(fileStat._mtime), _ctime(fileStat._ctime),
		_uid(fileStat._uid), _gid(fileStat._gid) {}
#if defined(HAVE_WINDOWS_H)
	FileStat(const char *pathName, const BY_HANDLE_FILE_INFORMATION &attrData);
	FileStat(const char *pathName, const WIN32_FILE_ATTRIBUTE_DATA &attrData);
#else
	FileStat(const char *pathName, const struct stat &stat);
#endif
	inline const char *GetPathName() const { return _pathName.c_str(); }
	inline unsigned long GetSize() const { return _bytes; }
	inline const DateTime &GetATime() const { return _atime; }
	inline const DateTime &GetMTime() const { return _mtime; }
	inline const DateTime &GetCTime() const { return _ctime; }
	inline long GetUid() const { return _uid; }
	inline long GetGid() const { return _gid; }
	inline long GetMode() const { return _attr & 0777; }
	inline bool IsDir() const { return (_attr & ATTR_Dir)? true : false; }
	inline bool IsChr() const { return (_attr & ATTR_Chr)? true : false; }
	inline bool IsBlk() const { return (_attr & ATTR_Blk)? true : false; }
	inline bool IsReg() const { return (_attr & ATTR_Reg)? true : false; }
	inline bool IsFifo() const { return (_attr & ATTR_Fifo)? true : false; }
	inline bool IsLnk() const { return (_attr & ATTR_Lnk)? true : false; }
	inline bool IsSock() const { return (_attr & ATTR_Sock)? true : false; }
	//inline bool IsReadOnly() const { return _attr & ATTR_ReadOnly; }
};

//-----------------------------------------------------------------------------
// FileEntry
//-----------------------------------------------------------------------------
class FileEntry {
private:
	String _fileName;
	bool _dirFlag;
public:
	inline FileEntry() : _dirFlag(false) {}
	inline FileEntry(const String &fileName, bool dirFlag) :
						_fileName(fileName), _dirFlag(dirFlag) {}
	inline FileEntry(const FileEntry &entry) :
						_fileName(entry._fileName), _dirFlag(entry._dirFlag) {}
	inline const char *GetFileName() const { return _fileName.c_str(); }
	inline bool IsDirectory() const { return _dirFlag; }
};

//-----------------------------------------------------------------------------
// FileLister
//-----------------------------------------------------------------------------
class FileLister {
protected:
	String _dirName;
protected:
	inline FileLister(const char *dirName) : _dirName(dirName) {}
public:
	virtual ~FileLister();
	virtual bool Next(FileEntry &entry) = 0;
	inline const char *GetDirName() const { return _dirName.c_str(); }
	static FileLister *CreateInstance(const char *dirName);
};

//-----------------------------------------------------------------------------
// File
//-----------------------------------------------------------------------------
class File {
public:
	class Formatter : public FormatterBase {
	private:
		File &_file;
	public:
		inline Formatter(File &file) : _file(file) {}
		inline void Format(Signal sig, const char *format, const ValueList &valList) {
			DoFormat(sig, format, valList);
		}
		virtual void PutChar(char ch);
	};
public:
	static const char SeparatorWin;
	static const char SeparatorUnix;
	static const char Separator;
public:
	enum AccessMode {
		ACCESS_Read, ACCESS_Write, ACCESS_ReadWrite,
	};
	class Entry {
	private:
		String _fileName;
		bool _dirFlag;
	public:
		inline Entry(const String &fileName, bool dirFlag) :
								_fileName(fileName), _dirFlag(dirFlag) {}
		inline const char *GetFileName() const { return _fileName.c_str(); }
		inline bool IsDirectory() const { return _dirFlag; }
	};
	typedef std::vector<Entry> EntryList;
private:
#if defined(HAVE_WINDOWS_H)
	HANDLE _hFile;
#else
	// FILE instance cannot be passed to DLL's context (see KB190799 in MSDN)
	FILE *_fp;
#endif
	bool _needCloseFlag;
	String _fileName;
	AccessMode _accessMode;
	bool _textFlag;
	String _encoding;
	Codec *_pCodecRead;
	Codec *_pCodecWrite;
	static String *_pWorkingDir;
	static String *_pBaseDir;
public:
	File();
	~File();
	bool Open(Signal sig, const char *fileName,
								const char *mode, const char *encoding);
	bool OpenStdin();
	bool OpenStdout();
	bool OpenStderr();
	void Close();
	inline const char *GetFileName() const { return _fileName.c_str(); }
	inline bool IsReadable() const {
		return _accessMode == ACCESS_Read || _accessMode == ACCESS_ReadWrite;
	}
	inline bool IsWritable() const {
		return _accessMode == ACCESS_Write || _accessMode == ACCESS_ReadWrite;
	}
	inline bool GetTextFlag() const { return _textFlag; }
	inline const char *GetEncoding() const { return _encoding.c_str(); }
	int Seek(long offset, int origin);
	int GetChar();
	int _GetChar();
	void PutChar(int ch);
	void _PutChar(int ch);
	void Flush();
	bool GetFileStatByHandle(Signal sig, FileStat &fileStat) const;
	static bool GetFileStat(Signal sig, const char *fileName, FileStat &fileStat);
	static bool IsExist(const char *pathName);
	void PutString(const char *str);
	size_t Read(void *buff, size_t bytes);
	bool InstallCodec(const char *encoding);
	void ReleaseCodec();
	inline static bool IsSeparator(char ch) {
		return ch == SeparatorWin || ch == SeparatorUnix;
	}
	static String ExtractDirName(const char *pathName);
	static String ExtractBaseName(const char *pathName);
	static void Split(const char *pathName, StringList &pathNameSepList);
	static bool HasWildCard(const char *pathName);
	static String MakeAbsPath(const char *fileName, const char *dirName = NULL);
	static const String &GetWorkingDir();
	static void InitBaseDir(const char *argv0);
	static inline const String &GetBaseDir() { return *_pBaseDir; }
	static String &RegulateName(String &result, const char *name);
	static String &RegulateName(String &result, const char *name, size_t cnt);
	static bool IsMatchName(const char *pattern,
							const char *fileName, bool ignoreCaseFlag);
private:
	inline File(const File &file) {}
};

}

#endif
