#ifndef __STRING_H__
#define __STRING_H__

#include "Common.h"
#include "Signal.h"

namespace AScript {

class Symbol;
class SymbolSet;
class Value;
class ValueList;
class Environment;

//-----------------------------------------------------------------------------
// String operations
//-----------------------------------------------------------------------------
int GetEscaped(int ch);
const char *FindString(const char *str, const char *sub, bool ignoreCaseFlag);
String::const_iterator FindString(String::const_iterator str,
		String::const_iterator strEnd, const String &sub, bool ignoreCaseFlag);

String EscapeString(const char *str);
String Capitalize(const char *str);
String Lower(const char *str);
String Upper(const char *str);
String Strip(const char *str, const SymbolSet &attrs);
String Left(const char *str, size_t len);
String Right(const char *str, size_t len);
String Middle(const char *str, int start, int len);
String Join(const ValueList &valList, const char *str);
String Replace(const char *str, const char *sub, const char *replace,
									int nMaxReplace, const SymbolSet &attrs);
void SplitPathList(Environment &env, const char *str, ValueList &valList);
Value FindString(Environment &env, Signal sig,
		const char *str, const char *sub, int start, const SymbolSet &attrs);

inline bool IsSJISFirst(int ch) {
	return 0x81 <= ch && ch <= 0x9f || 0xe0 <= ch && ch <= 0xef;
}

inline bool IsSJISSecond(int ch) {
	return 0x40 <= ch && ch <= 0x7e || 0x80 <= ch && ch <= 0xfc;
}

inline bool IsDigit(int ch) {
	return '0' <= ch && ch <= '9';
}

inline bool IsHexDigit(int ch) {
	return
		'0' <= ch && ch <= '9' ||
		'A' <= ch && ch <= 'F' ||
		'a' <= ch && ch <= 'f';
}

inline int ConvHexDigit(int ch) {
	return
		('0' <= ch && ch <= '9')? ch - '0' :
		('A' <= ch && ch <= 'F')? ch - 'A' + 10 :
		('a' <= ch && ch <= 'f')? ch - 'a' + 10 : 0;
}

inline bool IsOctDigit(int ch) {
	return '0' <= ch && ch <= '7';
}

inline int ConvOctDigit(int ch) {
	return ('0' <= ch && ch <= '7')? ch - '0' : 0;
}

inline bool IsWhite(int ch) {
	return ch == ' ' || ch == '\t';
}

inline bool IsSpace(int ch) {
	return ch == ' ' || ch == '\t' || ch == '\r' || ch == '\n';
}

inline bool IsSymbolFirstChar(int ch) {
	return
		('a' <= ch && ch <= 'z') ||
		('A' <= ch && ch <= 'Z') ||
		ch == '_' || ch == '$' || ch == '@' || IsSJISFirst(ch);
}

inline bool IsSymbolChar(int ch) {
	return
		('a' <= ch && ch <= 'z') ||
		('A' <= ch && ch <= 'Z') ||
		ch == '_' || ch == '$' || ch == '@' || ('0' <= ch && ch <= '9');
}

inline char ToUpper(char ch) {
	return ('a' <= ch && ch <= 'z')? ch - 'a' + 'A' : ch;
}

inline char ToLower(char ch) {
	return ('A' <= ch && ch <= 'Z')? ch - 'A' + 'a' : ch;
}

inline int CompareChar(char ch1, char ch2, bool ignoreCaseFlag) {
	return ignoreCaseFlag?
		static_cast<int>(ToUpper(ch1)) - static_cast<int>(ToUpper(ch2)) :
		static_cast<int>(ch1) - static_cast<int>(ch2);
}

#if defined(HAVE_WINDOWS_H)
inline bool IsPathSeparator(char ch) { return ch == ';'; }
#else
inline bool IsPathSeparator(char ch) { return ch == ';' || ch == ':'; }
#endif

//-----------------------------------------------------------------------------
// StringList
//-----------------------------------------------------------------------------
class StringList : public std::vector<String> {
public:
	static const StringList Null;
	inline StringList() {}
private:
	inline StringList(const StringList &stringList) {}
	inline void operator=(const StringList &stringList) {}
};

}

#endif
