#include "String.h"
#include "Symbol.h"
#include "Value.h"

namespace AScript {

//-----------------------------------------------------------------------------
// String
//-----------------------------------------------------------------------------
const StringList StringList::Null;

int GetEscaped(int ch)
{
	static const struct {
		int ch;
		int chConv;
	} tbl[] = {
		{ 'a',	'\a'	},
		{ 'b',	'\b'	},
		{ 'f',	'\f'	},
		{ 'n',	'\n'	},
		{ 'r',	'\r'	},
		{ 't',	'\t'	},
		{ 'v',	'\v'	},
		{ '\\',	'\\'	},
		{ '\'',	'\''	},
		{ '"',	'"'		},
	};
	for (int i = 0; i < NUMBEROF(tbl); i++) {
		if (tbl[i].ch == ch) return tbl[i].chConv;
	}
	return ch;
}

String EscapeString(const char *str)
{
	String rtn;
	bool shiftFlag = false;
	for (const char *p = str; *p != '\0'; p++) {
		char ch = *p;
		bool sjisFirstFlag = IsSJISFirst(static_cast<unsigned char>(ch));
		if (ch == '\n') {
			rtn += "\\n";
		} else if (ch == '\r') {
			rtn += "\\r";
		} else if (ch == '\t') {
			rtn += "\\t";
		} else if (ch == '"') {
			rtn += "\\\"";
		} else if (ch == '\\' && !shiftFlag) {
			rtn += "\\\\";
		} else if (sjisFirstFlag || shiftFlag) {
			rtn += ch;
		} else if (ch < 0x20 || ch >= 0x7f) {
			char tmp[16];
			::sprintf(tmp, "\\x%02x", ch);
			rtn += tmp;
		} else {
			rtn += ch;
		}
		shiftFlag = shiftFlag? false : sjisFirstFlag;
	}
	return rtn;
}

const char *FindString(const char *str, const char *sub, bool ignoreCaseFlag)
{
	for ( ; *str != '\0'; str++) {
		const char *p1 = str, *p2 = sub;
		for ( ; *p2 != '\0'; p1++, p2++) {
			if (CompareChar(*p1, *p2, ignoreCaseFlag) != 0) break;
		}
		if (*p2 == '\0') return str;
	}
	return NULL;
}

String::const_iterator FindString(String::const_iterator str,
		String::const_iterator strEnd, const String &sub, bool ignoreCaseFlag)
{
	for ( ; str != strEnd; str++) {
		String::const_iterator p1 = str;
		String::const_iterator p2 = sub.begin();
		for ( ; p2 != sub.end(); p1++, p2++) {
			if (CompareChar(*p1, *p2, ignoreCaseFlag) != 0) break;
		}
		if (p2 == sub.end()) return str;
	}
	return strEnd;
}

String Capitalize(const char *str)
{
	String rtn;
	const char *p = str;
	if (*p != '\0') {
		rtn += ('a' <= *p && *p <= 'z')? *p - 'a' + 'A' : *p;
		p++;
	}
	for ( ; *p != '\0'; p++) {
		rtn += ('A' <= *p && *p <= 'Z')? *p - 'A' + 'a' : *p;
	}
	return rtn;
}

String Lower(const char *str)
{
	String rtn;
	for (const char *p = str; *p != '\0'; p++) {
		rtn += ('A' <= *p && *p <= 'Z')? *p - 'A' + 'a' : *p;
	}
	return rtn;
}

String Upper(const char *str)
{
	String rtn;
	for (const char *p = str; *p != '\0'; p++) {
		rtn += ('a' <= *p && *p <= 'z')? *p - 'a' + 'A' : *p;
	}
	return rtn;
}

String Strip(const char *str, const SymbolSet &attrs)
{
	bool stripLeftFlag = true, stripRightFlag = true;
	if (attrs.IsSet(AScript_Symbol(left)) && !attrs.IsSet(AScript_Symbol(right))) {
		stripRightFlag = false;
	} else if (!attrs.IsSet(AScript_Symbol(left)) && attrs.IsSet(AScript_Symbol(right))) {
		stripLeftFlag = false;
	}
	size_t len = ::strlen(str);
	if (len == 0) return String("");
	const char *p1 = str;
	const char *p2 = str + len - 1;
	if (stripLeftFlag) {
		for ( ; IsSpace(*p1); p1++) ;
	}
	if (stripRightFlag) {
		for ( ; p2 != p1 && IsSpace(*p2); p2--) ;
	}
	return String(p1, p2 - p1 + 1);
}

String Left(const char *str, size_t len)
{
	return (len < ::strlen(str))? String(str, 0, len) : String(str);
}

String Right(const char *str, size_t len)
{
	size_t lenSrc = ::strlen(str);
	return (len < lenSrc)? String(str + lenSrc - len) : String(str);
}

String Middle(const char *str, int start, int len)
{
	int lenSrc = static_cast<int>(::strlen(str));
	if (start < 0) {
		start += lenSrc;
		if (start < 0) start = 0;
	}
	if (start >= lenSrc || len == 0) {
		return String("");
	} else if (len > 0 && start + len < lenSrc) {
		return String(str + start, len);
	} else {
		return String(str + start);
	}
}

String Join(const ValueList &valList, const char *str)
{
	Signal sig;
	ValueList::const_iterator pValue = valList.begin();
	if (pValue == valList.end()) return "";
	String rtn = pValue->ToString(sig, false);
	pValue++;
	for ( ; pValue != valList.end(); pValue++) {
		rtn += str;
		rtn += pValue->ToString(sig, false);
	}
	return rtn;
}

Value FindString(Environment &env, Signal sig,
		const char *str, const char *sub, int start, const SymbolSet &attrs)
{
	bool ignoreCaseFlag = attrs.IsSet(AScript_Symbol(icase));
	do {
		int len = static_cast<int>(::strlen(str));
		if (start < 0) {
			start += len;
			if (start < 0) start = 0;
		}
		if (start > len) return Value::Null;
	} while (0);
	if (attrs.IsSet(AScript_Symbol(rev))) {
		const char *p = FindString(str + start, sub, ignoreCaseFlag);
		if (attrs.IsSet(AScript_Symbol(list))) {
			ValueList valListOrg;
			for ( ; p != NULL; p = FindString(p + 1, sub, ignoreCaseFlag)) {
				valListOrg.push_back(Value(static_cast<Number>(p - str)));
			}
			Value result;
			ValueList &valList = result.InitAsList(env);
			foreach_reverse (ValueList, pValue, valListOrg) {
				valList.push_back(*pValue);
			}
			return result;
		} else {
			const char *pLast = NULL;
			for ( ; p != NULL; p = FindString(p + 1, sub, ignoreCaseFlag)) {
				pLast = p;
			}
			return (pLast == NULL)? Value::Null :
							Value(static_cast<Number>(pLast - str));
		}
	} else {
		const char *p = FindString(str + start, sub, ignoreCaseFlag);
		if (attrs.IsSet(AScript_Symbol(list))) {
			Value result;
			ValueList &valList = result.InitAsList(env);
			for ( ; p != NULL; p = FindString(p + 1, sub, ignoreCaseFlag)) {
				valList.push_back(Value(static_cast<Number>(p - str)));
			}
			return result;
		} else {
			return (p == NULL)? Value::Null :
							Value(static_cast<Number>(p - str));
		}
	}
}

String Replace(const char *str, const char *sub, const char *replace,
										int nMaxReplace, const SymbolSet &attrs)
{
	bool ignoreCaseFlag = attrs.IsSet(AScript_Symbol(icase));
	String result;
	int lenSub = ::strlen(sub);
	if (lenSub == 0) {
		if (nMaxReplace != 0) {
			result += replace;
			nMaxReplace--;
		}
		const char *p = str;
		for ( ; *p != '\0' && nMaxReplace != 0; p++, nMaxReplace--) {
			result += *p;
			result += replace;
		}
		result += p;
	} else {
		const char *pLeft = str;
		const char *pRight = FindString(pLeft, sub, ignoreCaseFlag);
		for ( ; pRight != NULL && nMaxReplace != 0;
					pRight = FindString(pLeft, sub, ignoreCaseFlag), nMaxReplace--) {
			result.append(pLeft, pRight - pLeft);
			result += replace;
			pLeft = pRight + lenSub;
		}
		result += pLeft;
	}
	return result;
}

void SplitPathList(Environment &env, const char *str, ValueList &valList)
{
	enum {
		STAT_SkipSpace, STAT_Field,
	} stat = STAT_SkipSpace;
	char ch = '\0';
	String field;
	int cntSpace = 0;
	bool eatNextFlag = true;
	do {
		if (eatNextFlag) ch = *str++;
		eatNextFlag = true;
		if (stat == STAT_SkipSpace) {
			if (IsSpace(ch)) {
				// nothing to do
			} else {
				field.clear();
				cntSpace = 0;
				eatNextFlag = false;
				stat = STAT_Field;
			}
		} else if (stat == STAT_Field) {
			if (IsPathSeparator(ch) || ch == '\0') {
				valList.push_back(Value(env, field.c_str()));
				stat = STAT_SkipSpace;
			} else if (ch == ' ') {
				cntSpace++;
			} else {
				while (cntSpace-- > 0) field.push_back(' ');
				field.push_back(ch);
			}
		}
	} while (ch != '\0');
}

}
