//
// Object_List
//

#include "Object_List.h"
#include "Expr.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_List
//-----------------------------------------------------------------------------
bool Object_List::IsList() const { return true; }

Object *Object_List::Clone() const
{
	return new Object_List(*this);
}

Value Object_List::GetByIndex(Signal sig, const Value &valueIdx)
{
	if (valueIdx.IsNumber()) {
		size_t idx = static_cast<size_t>(valueIdx.GetNumber());
		if (idx < GetList().size()) {
			return GetList()[idx];
		} else {
			sig.SetError(ERR_IndexError, "index is out of range");
		}
	} else {
		sig.SetError(ERR_IndexError, "index must be a number for List");
	}
	return Value::Null;
}

void Object_List::SetByIndex(Signal sig, const Value &valueIdx, const Value &value)
{
	if (valueIdx.IsNumber()) {
		size_t idx = static_cast<size_t>(valueIdx.GetNumber());
		if (idx < GetList().size()) {
			GetList()[idx] = value;
		} else {
			sig.SetError(ERR_IndexError, "index is out of range");
		}
	} else {
		sig.SetError(ERR_IndexError, "index must be a number for List");
	}
}

Iterator *Object_List::CreateIterator(Signal sig)
{
	return new IteratorNormal(dynamic_cast<Object_List *>(IncRef()));
}

String Object_List::ToString(Signal sig, bool exprFlag)
{
	String str;
	const ValueList &valList = GetList();
	str += "[";
	foreach_const (ValueList, pValue, valList) {
		if (pValue != valList.begin()) str += ", ";
		str += pValue->ToString(sig);
	}
	str += "]";
	return str;
}

void Object_List::DoPadding(Signal sig, ValueList &valList, const Value &value)
{
	if (sig.IsSignalled()) return;	// detect stack overflow
	int sizeMax = -1;
	foreach (ValueList, pValue, valList) {
		if (pValue->IsList()) {
			int size = static_cast<int>(pValue->GetList().size());
			if (sizeMax < size) sizeMax = size;
		}
	}
	if (sizeMax < 0) return;
	foreach (ValueList, pValue, valList) {
		if (pValue->IsList()) {
			ValueList &valListElem = pValue->GetList();
			int sizeToPadding = sizeMax - static_cast<int>(valListElem.size());
			while (sizeToPadding-- > 0) {
				valListElem.push_back(value);
			}
			DoPadding(sig, valListElem, value);
		}
	}
}

void Object_List::ValueVisitorEx::Visit(Signal sig, const Value &value)
{
	ASSUME(_env, value.IsNumber());
	size_t idx = static_cast<size_t>(value.GetNumber());
	if (std::find(_indexList.begin(), _indexList.end(), idx) != _indexList.end()) {
		// nothing to do
	} else if (idx < _valList.size()) {
		_indexList.push_back(idx);
	} else {
		sig.SetError(ERR_IndexError, "index is out of range");
	}
}

// Constructor: List(func?:Function) {block?}
Object_List::Constructor::Constructor(Environment &env, const char *name) :
							ConstructorBase(name, env.GetClass_List())
{
	DeclareArg(env, "func", VTYPE_Function, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

Value Object_List::Constructor::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Expr_Block *pExprBlock = context.GetBlock();
	const Value &valueFunc = context.GetValue(0);
	Value result;
	if (pExprBlock == NULL) {
		result.InitAsList(env);
	} else if (valueFunc.IsFunction()) {
		const Function *pFunc = valueFunc.GetFunction();
		size_t cntArgs = pFunc->GetDeclList().size();
		if (cntArgs == 0) {
			sig.SetError(ERR_TypeError, "function '%s' needs no argument", pFunc->GetName());
			return Value::Null;
		}
		Environment envLister(&env, ENVTYPE_Lister);
		Value valueRaw = pExprBlock->GetExprList().ExecForList(envLister, sig, false);
		if (sig.IsSignalled() || !valueRaw.IsList()) return Value::Null;
		ValueList &valList = result.InitAsList(env);
		foreach_const (ValueList, pValue, valueRaw.GetList()) {
			if (!pValue->IsList()) {
				sig.SetError(ERR_SyntaxError, "invalid format in list initializer");
				return Value::Null;
			}
			Context contextSub(pValue->GetList());
			Value valueElem = pFunc->Eval(env, sig, contextSub);
			valList.push_back(valueElem);
		}
	} else {
		Environment envLister(&env, ENVTYPE_Lister);
		result = pExprBlock->GetExprList().ExecForList(envLister, sig, false);
	}
	return result;
}

//-----------------------------------------------------------------------------
// Object_List::IteratorNormal
//-----------------------------------------------------------------------------
Object_List::IteratorNormal::~IteratorNormal()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorNormal::Next(Signal sig, Value &value)
{
	ValueList &valList = _pObj->GetList();
	if (_pValue == valList.end()) return false;
	value = *_pValue;
	_pValue++;
	return true;
}

//-----------------------------------------------------------------------------
// Object_List::IteratorReverse
//-----------------------------------------------------------------------------
Object_List::IteratorReverse::~IteratorReverse()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorReverse::Next(Signal sig, Value &value)
{
	ValueList &valList = _pObj->GetList();
	if (_pValue == valList.rend()) return false;
	value = *_pValue;
	_pValue++;
	return true;
}

//-----------------------------------------------------------------------------
// Object_List::IteratorRound
//-----------------------------------------------------------------------------
Object_List::IteratorRound::~IteratorRound()
{
	Object::Delete(_pObj);
}

bool Object_List::IteratorRound::Next(Signal sig, Value &value)
{
	ValueList &valList = _pObj->GetList();
	if (_pValue == valList.end() || _cnt == 0) return false;
	value = *_pValue;
	_pValue++;
	if (_cnt > 0) _cnt--;
	if (_pValue == valList.end()) _pValue = valList.begin();
	return true;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_List
//-----------------------------------------------------------------------------
// List#len()
AScript_DeclareMethod(List, len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, len)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	size_t cnt = pSelf->GetList().size();
	return Value(static_cast<Number>(cnt));
}

// List#count(value)
AScript_DeclareMethod(List, count)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
}

AScript_ImplementMethod(List, count)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	size_t cnt = 0;
	const Value &value = context.GetValue(0);
	if (value.IsFunction()) {
		const Function *pFunc = value.GetFunction();
		foreach_const (ValueList, pValue, pSelf->GetList()) {
			ValueList valListArg(*pValue);
			Context contextSub(context, valListArg, Value::Null);
			Value valueFlag = pFunc->Eval(env, sig, contextSub);
			if (sig.IsSignalled()) return Value::Null;
			if (valueFlag.GetBoolean()) cnt++;
		}
	} else {
		foreach_const (ValueList, pValue, pSelf->GetList()) {
			if (Value::Compare(*pValue, value) == 0) cnt++;
		}
	}
	return Value(static_cast<Number>(cnt));
}

// List#get(index):map
AScript_DeclareMethod(List, get)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "index", VTYPE_AnyType);
}

AScript_ImplementMethod(List, get)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	return pSelf->GetByIndex(sig, context.GetValue(0));
}

// List#first()
AScript_DeclareMethod(List, first)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, first)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	if (valList.empty()) {
		sig.SetError(ERR_ValueError, "list is empty");
		return Value::Null;
	}
	return valList.front();
}

// List#last()
AScript_DeclareMethod(List, last)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, last)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	if (valList.empty()) {
		sig.SetError(ERR_ValueError, "list is empty");
		return Value::Null;
	}
	return valList.back();
}

// List#join(sep?:string)
AScript_DeclareMethod(List, join)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "sep", VTYPE_String, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, join)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	return Value(env, Join(valList,
			context.IsString(0)? context.GetString(0) : "").c_str());
}

// List#format(format:string)
AScript_DeclareMethod(List, format)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "format", VTYPE_String);
}

AScript_ImplementMethod(List, format)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	return Value(env, Formatter::Format(sig,
						context.GetString(0), pSelf->GetList()).c_str());
}

// iter = List#each() {block?}
AScript_DeclareMethod(List, each)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_Once);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, each)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorNormal(pObj));
}

// iter = List#offset(n:number):map {block?}
AScript_DeclareMethod(List, offset)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareBlock(OCCUR_Once);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, offset)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	size_t offset = static_cast<size_t>(context.GetNumber(0));
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorNormal(pObj, offset));
}

// iter = List#skip(n:number):map {block?}
AScript_DeclareMethod(List, skip)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, skip)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	Iterator *pIterator = new Object_List::IteratorNormal(pObj);
	pIterator = new Iterator_Skip(pIterator,
			static_cast<size_t>(context.GetNumber(0)));
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = List#align(n:number, value?):map {block?}
AScript_DeclareMethod(List, align)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, align)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	Iterator *pIterator = new Object_List::IteratorNormal(pObj);
	pIterator = new Iterator_Align(pIterator,
			context.IsNumber(0)? static_cast<int>(context.GetNumber(0)) : -1,
			context.GetValue(1));
	return ReturnIterator(env, sig, context, pIterator);
}

// iter = List#reverse() {block?}
AScript_DeclareMethod(List, reverse)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, reverse)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorReverse(pObj));
}

// iter = List#round(n?:number) {block?}
AScript_DeclareMethod(List, round)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, round)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	int cnt = context.IsNumber(0)? static_cast<int>(context.GetNumber(0)) : -1;
	Object_List *pObj = dynamic_cast<Object_List *>(pSelf->IncRef());
	return ReturnIterator(env, sig, context,
							new Object_List::IteratorRound(pObj, cnt));
}

// iter = List#permutation(n?:number) {block?}
class IteratorPermutation : public Iterator {
private:
	Environment _env;
	ValueList _valList;
	bool _validFlag;
public:
	IteratorPermutation(Environment &env, const ValueList &valList);
	~IteratorPermutation();
	virtual bool Next(Signal sig, Value &value);
};

IteratorPermutation::IteratorPermutation(Environment &env, const ValueList &valList) :
					Iterator(false), _env(env), _valList(valList), _validFlag(true)
{
}

IteratorPermutation::~IteratorPermutation()
{
}

bool IteratorPermutation::Next(Signal sig, Value &value)
{
	if (!_validFlag) return false;
	ValueList &valList = value.InitAsList(_env);
	foreach (ValueList, pValue, _valList) {
		valList.push_back(*pValue);
	}
	_validFlag = std::next_permutation(_valList.begin(), _valList.end());
	return true;
}

AScript_DeclareMethod(List, permutation)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, permutation)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	return ReturnIterator(env, sig, context,
							new IteratorPermutation(env, pSelf->GetList()));
}

// iter = List#combination(n?:number) {block?}
AScript_DeclareMethod(List, combination)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(List, combination)
{
	return Value::Null;
}

// List#clear!()
AScript_DeclareMethod(List, clear_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, clear_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	pSelf->GetList().clear();
	return context.GetSelf();
}

// List#shuffle!()
AScript_DeclareMethod(List, shuffle_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(List, shuffle_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	RandomNumberGenerator randomNumberGenerator;
	std::random_shuffle(valList.begin(), valList.end(), randomNumberGenerator);
	return context.GetSelf();
}

// List#add!(elem+)
AScript_DeclareMethod(List, add_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, false, OCCUR_OnceOrMore);
}

AScript_ImplementMethod(List, add_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		valList.push_back(*pValue);
	}
	return context.GetSelf();
}

// List#append!(elem*)
AScript_DeclareMethod(List, append_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "elem", VTYPE_AnyType, false, OCCUR_ZeroOrMore);
}

AScript_ImplementMethod(List, append_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	foreach_const (ValueList, pValue, context.GetList(0)) {
		if (pValue->IsList()) {
			foreach_const (ValueList, pValueSub, pValue->GetList()) {
				valList.push_back(*pValueSub);
			}
		} else {
			valList.push_back(*pValue);
		}
	}
	return context.GetSelf();
}

// List#erase!(idx+:number)
AScript_DeclareMethod(List, erase_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "idx", VTYPE_Number, false, OCCUR_OnceOrMore);
}

AScript_ImplementMethod(List, erase_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	ValueList &valList = pSelf->GetList();
	Object_List::ValueVisitorEx visitor(env, valList);
	foreach_const (ValueList, pValue, context.GetList(0)) {
		pValue->Accept(sig, visitor);
	}
	Object_List::IndexList &indexList = visitor.GetIndexList();
	if (!indexList.empty()) {
		std::sort(indexList.begin(), indexList.end());
		size_t offset = 0;
		foreach_const (Object_List::IndexList, pIdx, indexList) {
			size_t idx = *pIdx;
			valList.erase(valList.begin() + idx - offset);
			offset++;
		}
	}
	return context.GetSelf();
}

// List#padding!(value)
AScript_DeclareMethod(List, padding_X)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "value", VTYPE_AnyType);
}

AScript_ImplementMethod(List, padding_X)
{
	Object_List *pSelf = Object_List::GetSelfObj(context);
	Object_List::DoPadding(sig, pSelf->GetList(), context.GetValue(0));
	if (sig.IsSignalled()) return Value::Null;
	return context.GetSelf();
}

// assignment
Class_List::Class_List(Class *pClassSuper, const Symbol *pSymbol) :
												Class(pClassSuper, pSymbol)
{
	AScript_AssignMethod(List, len);
	AScript_AssignMethod(List, count);
	AScript_AssignMethod(List, get);
	AScript_AssignMethod(List, first);
	AScript_AssignMethod(List, last);
	AScript_AssignMethod(List, join);
	AScript_AssignMethod(List, format);
	AScript_AssignMethod(List, each);
	AScript_AssignMethod(List, offset);
	AScript_AssignMethod(List, skip);
	AScript_AssignMethod(List, align);
	AScript_AssignMethod(List, reverse);
	AScript_AssignMethod(List, round);
	AScript_AssignMethod(List, permutation);
	AScript_AssignMethod(List, combination);
	AScript_AssignMethodEx(List, clear_X, "clear!");
	AScript_AssignMethodEx(List, shuffle_X, "shuffle!");
	AScript_AssignMethodEx(List, add_X, "add!");
	AScript_AssignMethodEx(List, append_X, "append!");
	AScript_AssignMethodEx(List, erase_X, "erase!");
	AScript_AssignMethodEx(List, padding_X, "padding!");
}

Object *Class_List::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	return new Object_List((pClass == NULL)? this : pClass);
}

}
