//-----------------------------------------------------------------------------
// AScript sqlite3 module
//-----------------------------------------------------------------------------
#include "Module.h"
#include "Function.h"
extern "C" {
#include "sqlite3.h"
}

AScript_BeginModule(sqlite3)
//-----------------------------------------------------------------------------
// Object_SQLite3
//-----------------------------------------------------------------------------
AScript_DeclareClass(SQLite3);

class Object_SQLite3 : public Object {
public:
	class IteratorEx : public Iterator {
	private:
		Object_SQLite3 *_pObj;
		sqlite3_stmt *_pStmt;
	public:
		inline IteratorEx(Object_SQLite3 *pObj, sqlite3_stmt *pStmt) :
							Iterator(false), _pObj(pObj), _pStmt(pStmt) {}
		virtual ~IteratorEx();
		virtual bool Next(Signal sig, Value &value);
	};
private:
	sqlite3 *_db;
	String _fileName;
public:
	inline static Object_SQLite3 *GetSelfObj(Context &context) {
		return dynamic_cast<Object_SQLite3 *>(context.GetSelfObj());
	}
public:
	inline Object_SQLite3(Class *pClass) : Object(pClass), _db(NULL) {}
	virtual ~Object_SQLite3();
	virtual Object *Clone() const;
	virtual String ToString(Signal sig, bool exprFlag);
	bool Open(Signal sig, const char *fileName);
	void Close();
	Value Exec(Signal sig, const char *sql, Context &context);
	bool ExecNoResult(Signal sig, const char *sql);
	IteratorEx *Query(Signal sig, const char *sql);
public:
	void SetError_NotOpened(Signal sig);
private:
	static int Callback(void *user, int argc, char **argv, char **azColName);
};

Object_SQLite3::~Object_SQLite3()
{
	Close();
}

Object *Object_SQLite3::Clone() const
{
	return NULL;
}

bool Object_SQLite3::Open(Signal sig, const char *fileName)
{
	int rc = ::sqlite3_open(fileName, &_db);
	if (rc != SQLITE_OK) {
		sig.SetError(ERR_RuntimeError, "SQLite3 failed to create SQLite3 object");
		return false;
	}
	_fileName = fileName;
	return true;
}

void Object_SQLite3::Close()
{
	if (_db != NULL) {
		::sqlite3_close(_db);
		_db = NULL;
		_fileName = "";
	}
}

Value Object_SQLite3::Exec(Signal sig, const char *sql, Context &context)
{
	if (_db == NULL) {
		SetError_NotOpened(sig);
		return Value::Null;
	}
	char *errMsg;
	Value result;
	Function::ResultListComposer resultListComposer(*this, context, result);
	int rc = ::sqlite3_exec(_db, sql, Callback, &resultListComposer, &errMsg); 
	if (rc != SQLITE_OK) {
		sig.SetError(ERR_RuntimeError, "SQLite3 %s", errMsg);
		return Value::Null;
	}
	return result;
}

bool Object_SQLite3::ExecNoResult(Signal sig, const char *sql)
{
	if (_db == NULL) {
		SetError_NotOpened(sig);
		return false;
	}
	char *errMsg;
	int rc = ::sqlite3_exec(_db, sql, NULL, NULL, &errMsg); 
	if (rc != SQLITE_OK) {
		sig.SetError(ERR_RuntimeError, "SQLite3 %s", errMsg);
		return false;
	}
	return true;
}

Object_SQLite3::IteratorEx *Object_SQLite3::Query(Signal sig, const char *sql)
{
	if (_db == NULL) {
		SetError_NotOpened(sig);
		return NULL;
	}
	sqlite3_stmt *pStmt;
	const char *pzTail;
	::sqlite3_prepare(_db, sql, -1, &pStmt, &pzTail);
	return new IteratorEx(dynamic_cast<Object_SQLite3 *>(IncRef()), pStmt);
}

String Object_SQLite3::ToString(Signal sig, bool exprFlag)
{
	String str;
	str += "<sqlite3:'";
	str += _fileName;
	str += "'>";
	return str;
}

int Object_SQLite3::Callback(void *user, int argc, char **argv, char **azColName)
{
	Function::ResultListComposer *pResultListComposer =
						reinterpret_cast<Function::ResultListComposer *>(user);
	Environment &env = pResultListComposer->GetEnv();
	Value value;
	ValueList &valList = value.InitAsList(env);
	for (int i = 0; i < argc; i++) {
		valList.push_back(Value(env, argv[i]));
	}
	pResultListComposer->Store(value);
	return SQLITE_OK;
}

void Object_SQLite3::SetError_NotOpened(Signal sig)
{
	sig.SetError(ERR_RuntimeError, "database not opened");
}

//-----------------------------------------------------------------------------
// Object_SQLite3::IteratorEx
//-----------------------------------------------------------------------------
Object_SQLite3::IteratorEx::~IteratorEx()
{
	Object::Delete(_pObj);
	::sqlite3_finalize(_pStmt);
}


bool Object_SQLite3::IteratorEx::Next(Signal sig, Value &value)
{
	if (::sqlite3_step(_pStmt) != SQLITE_ROW) {
		return false;
	}
	ValueList &valList = value.InitAsList(*_pObj);
	int nCols = ::sqlite3_column_count(_pStmt);
	for (int iCol = 0; iCol < nCols; iCol++) {
		int type = ::sqlite3_column_type(_pStmt, iCol);
		if (type == SQLITE_INTEGER) {
			valList.push_back(Value(static_cast<Number>(::sqlite3_column_int(_pStmt, iCol))));
		} else if (type == SQLITE_FLOAT) {
			valList.push_back(Value(static_cast<Number>(::sqlite3_column_double(_pStmt, iCol))));
		} else if (type == SQLITE_TEXT) {
			valList.push_back(Value(*_pObj,
				reinterpret_cast<const char *>(::sqlite3_column_text(_pStmt, iCol))));
		} else if (type == SQLITE_BLOB) {
			//::sqlite3_column_blob(_pStmt, iCol);
			valList.push_back(Value::Null);
		} else if (type == SQLITE_NULL) {
			valList.push_back(Value::Null);
		} else {
			sig.SetError(ERR_RuntimeError, "somthing's wrong in SQLite3");
			return false;
		}
	}
	return true;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_SQLite3
//-----------------------------------------------------------------------------
// SQLite3#exec(sql:string):map
AScript_DeclareMethod(SQLite3, exec)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "sql", VTYPE_String);
}

AScript_ImplementMethod(SQLite3, exec)
{
	Object_SQLite3 *pObj = Object_SQLite3::GetSelfObj(context);
	return pObj->Exec(sig, context.GetString(0), context);
}

// SQLite3#query(sql:string):map {block?}
AScript_DeclareMethod(SQLite3, query)
{
	SetMode(RSLTMODE_Normal, MAP_On, FLAT_Off);
	DeclareArg(env, "sql", VTYPE_String);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(SQLite3, query)
{
	Object_SQLite3 *pObj = Object_SQLite3::GetSelfObj(context);
	return ReturnIterator(env, sig, context, pObj->Query(sig, context.GetString(0)));
}

// SQLite3#transaction() {block}
AScript_DeclareMethod(SQLite3, transaction)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementMethod(SQLite3, transaction)
{
	Object_SQLite3 *pObj = Object_SQLite3::GetSelfObj(context);
	const Function *pFuncBlock = GetBlockFunction(env, sig, context);
	if (sig.IsSignalled()) return Value::Null;
	if (!pObj->ExecNoResult(sig, "BEGIN TRANSACTION")) return Value::Null;
	Environment envBlock(&env, ENVTYPE_Block);
	Context contextSub(ValueList::Null);
	Value result = pFuncBlock->Eval(envBlock, sig, contextSub);
	// "END TRANSACTION" has the same effect as "COMMIT"
	if (!pObj->ExecNoResult(sig, "END TRANSACTION")) return Value::Null;
	return result;
}

// SQLite3#close()
AScript_DeclareMethod(SQLite3, close)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(SQLite3, close)
{
	Object_SQLite3 *pObj = Object_SQLite3::GetSelfObj(context);
	pObj->Close();
	return Value::Null;
}

// assignment
AScript_ImplementClass(SQLite3)
{
	AScript_AssignMethod(SQLite3, exec);
	AScript_AssignMethod(SQLite3, query);
	AScript_AssignMethod(SQLite3, transaction);
	AScript_AssignMethod(SQLite3, close);
}

//-----------------------------------------------------------------------------
// Module functions: sqlite3
//-----------------------------------------------------------------------------
// db = sqlite3.open(filename:string) {block?}
AScript_DeclareFunction(open)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "filename", VTYPE_String);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementFunction(open)
{
	Object_SQLite3 *pObj = new Object_SQLite3(AScript_Class(SQLite3, env));
	if (!pObj->Open(sig, context.GetString(0))) {
		delete pObj;
		return Value::Null;
	}
	Value result(pObj, VTYPE_Object);
	if (context.IsBlockSpecified()) {
		const Function *pFuncBlock = GetBlockFunction(env, sig, context);
		if (pFuncBlock == NULL) return Value::Null;
		ValueList valListArg(result);
		Context contextSub(valListArg);
		pFuncBlock->Eval(env, sig, contextSub);
		result = Value::Null; // object is destroyed here
	}
	return result;
}

// Module entry
AScript_ModuleEntry()
{
	AScript_AssignFunction(open);
}

AScript_ModuleTerminate()
{
}

AScript_EndModule(sqlite3)

AScript_DLLModuleEntry(sqlite3)
