#include "Parser.h"
#include "Option.h"
#include "Module_builtins.h"
#include "Module_os.h"
#include "Module_time.h"
#include "Module_string.h"
#if defined(HAVE_LIBREADLINE)
#include <readline/readline.h>
#include <readline/history.h>
#endif

namespace AScript {

void AcceptChar(Environment &env, Signal sig, Parser &parser, int ch);

void PrintVersion(FILE *fp)
{
	::fprintf(fp, "AScript %s copyright (c) 2010- Dan-san\n", VERSION);
}

void PrintHelp(FILE *fp)
{
	::fprintf(fp,
"usage: ascript [option] [file] [arg] ...\n"
"available options:\n"
"-h    print this help\n"
"-v    print version string\n"
"-i    interactive mode after running script file if specified\n"
	);
}

//-----------------------------------------------------------------------------
// Main entry
//-----------------------------------------------------------------------------
int Main(int argc, const char *argv[])
{
	File::InitBaseDir(argv[0]);
	static const Option::Info optInfoTbl[] = {
		{ "help",			'h', false	},
		{ "interactive",	'i', false	},
		{ "command",		'c', true	},
		{ "version",		'v', false	},
	};
	Signal sig;
	Option opt(optInfoTbl, NUMBEROF(optInfoTbl));
	if (!opt.Parse(sig, argc, argv)) {
		::fprintf(stderr, "%s\n", sig.GetErrString().c_str());
		return 1;
	}
	if (opt.IsSet("version")) {
		PrintVersion(stderr);
		return 0;
	}
	if (opt.IsSet("help")) {
		PrintVersion(stderr);
		PrintHelp(stderr);
		return 0;
	}
	::init_gen_rand(1234);	// initialize random generator SFMT
	Parser parser;
	EnvironmentRoot env;
	env.SetupBuiltIn(sig, argc, argv);
	Module___builtins__::MixIn(env, sig);
	Module_os::Import(env, sig);
	Module_time::Import(env, sig);
	Module_string::Import(env, sig);
	if (argc >= 2) {
		File file;
		file.Open(sig, argv[1], "rt");
		if (sig.IsSignalled()) {
			env.PutString(sig.GetErrString().c_str());
			env.PutString("\n");
			return 1;
		}
		parser.ExecFile(env, sig, file, true);
		if (sig.IsSignalled()) {
			env.PutString(sig.GetErrString().c_str());
			env.PutString("\n");
			sig.ClearSignal();
		}
		file.Close();
		if (!opt.IsSet("interactive")) return 0;
	}
	PrintVersion(stdout);
	env.SetEchoFlag(true);
#if defined(HAVE_LIBREADLINE)
	char *lineBuff = NULL;
	while (lineBuff = readline(env.GetPrompt(parser.IsContinued()))) {
		for (char *p = lineBuff; ; p++) {
			int ch = (*p == '\0')? '\n' : *p;
			AcceptChar(env, sig, parser, ch);
			if (ch == '\n') break;
		}
		if (lineBuff[0] != '\0') {
			add_history(lineBuff);
		}
		free(lineBuff);
	}
#else
	env.PutPrompt(false);
	for (;;) {
		int ch = ::fgetc(stdin);
		AcceptChar(env, sig, parser, ch);
		if (ch < 0) break;
		if (ch == '\n') env.PutPrompt(parser.IsContinued());
	}
#endif
	return 0;
}

void AcceptChar(Environment &env, Signal sig, Parser &parser, int ch)
{
	Expr *pExpr = parser.ParseChar(env, sig, ch);
	if (sig.IsSignalled()) {
		env.PutString(sig.GetErrString().c_str());
		env.PutString("\n");
		sig.ClearSignal();
	} else if (pExpr != NULL) {
		Value result = pExpr->Exec(env, sig);
		if (sig.IsSignalled()) {
			env.PutString(sig.GetErrString().c_str());
			env.PutString("\n");
			sig.ClearSignal();
		} else if (!env.GetEchoFlag()) {
			// nothing to do
		} else if (result.IsValid()) {
			env.PutString(result.ToString(sig).c_str());
			env.PutString("\n");
		}
		delete pExpr;
	}
}

}

int main(int argc, const char *argv[])
{
	return AScript::Main(argc, argv);
}
