#include "Iterator.h"
#include "Object.h"
#include "Expr.h"
#include "OAL.h"
#include "Function.h"
#include "Object_Function.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Iterator
//-----------------------------------------------------------------------------
Iterator::~Iterator()
{
}

Value Iterator::Eval(Environment &env, Signal sig, Context &context)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value value, result;
	Function::ResultListComposer resultListComposer(env, context, result);
	while (Next(sig, value)) {
		resultListComposer.Store(value);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

Value Iterator::Eval(Environment &envBlock, Signal sig, Context &context,
												const Function *pFuncBlock)
{
	if (IsInfinite()) {
		sig.SetError(ERR_IteratorError, "cannot evaluate infinite iterator");
		return Value::Null;
	}
	Value value, result;
	Function::ResultListComposer resultListComposer(envBlock, context, result);
	for (int idx = 0; Next(sig, value); idx++) {
		ValueList valListArg(value, Value(static_cast<Number>(idx)));
		Context contextSub(valListArg);
		Value resultElem = pFuncBlock->Eval(envBlock, sig, contextSub);
		AScript_BlockSignalHandlerInLoop(sig, resultElem, Value::Null)
		resultListComposer.Store(resultElem);
	}
	if (sig.IsSignalled()) return Value::Null;
	return result;
}

//-----------------------------------------------------------------------------
// Iterator_Constant
//-----------------------------------------------------------------------------
Iterator_Constant::~Iterator_Constant()
{
}

bool Iterator_Constant::Next(Signal sig, Value &value)
{
	value = _value;
	return true;
}

//-----------------------------------------------------------------------------
// Iterator_Mapping
//-----------------------------------------------------------------------------
Iterator_Mapping::Iterator_Mapping(Environment &env, Signal sig,
									Function *pFunc, Context &context) :
	Iterator(false), _env(env), _pFunc(pFunc)
{
	pFunc->PrepareIteratorsForMap(sig, _iterOwner, context.GetArgs());
}

Iterator_Mapping::~Iterator_Mapping()
{
	Function::Delete(_pFunc);
}

bool Iterator_Mapping::Next(Signal sig, Value &value)
{
	ValueList valList;
	if (!_iterOwner.Next(sig, valList)) return false;
	Context context(valList);
	value = _pFunc->Eval(_env, sig, context);
	if (sig.IsSignalled()) return false;
	return true;
}

//-----------------------------------------------------------------------------
// Iterator_MemberMapping
//-----------------------------------------------------------------------------
Iterator_MemberMapping::~Iterator_MemberMapping()
{
	Expr::Delete(_pExpr);
	Iterator::Delete(_pIterator);
}

bool Iterator_MemberMapping::Next(Signal sig, Value &value)
{
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	ObjectBase *pObjEach = valueSelfEach.ExtractObject(sig);
	if (sig.IsSignalled()) return false;
	Environment &env = *pObjEach;
	value = _pExpr->Exec(env, sig);
	if (value.IsFunction()) {
		Object *pObj = new Object_WrappedMethod(env.GetClass_Function(),
							value.GetFunction()->IncRef(), valueSelfEach);
		value = Value(pObj, VTYPE_Function, true);
	}
	return true;
}

//-----------------------------------------------------------------------------
// Iterator_MethodMapping
//-----------------------------------------------------------------------------
Iterator_MethodMapping::~Iterator_MethodMapping()
{
	Expr::Delete(_pExprCaller);
	Iterator::Delete(_pIterator);
}

bool Iterator_MethodMapping::Next(Signal sig, Value &value)
{
	const Function *pFuncSuccRequester = NULL;
	Value valueSelfEach;
	if (!_pIterator->Next(sig, valueSelfEach)) return false;
	value = _pExprCaller->EvalEach(_env, sig, valueSelfEach, &pFuncSuccRequester);
	return true;
}

//-----------------------------------------------------------------------------
// Iterator_Fork
//-----------------------------------------------------------------------------
Iterator_Fork::Iterator_Fork(Environment &env, Signal sig,
									Function *pFunc, Context &context) :
	Iterator(false), _env(env), _pFunc(pFunc)
{
	pFunc->PrepareIteratorsForMap(sig, _iterOwner, context.GetArgs());
	_pValListToWrite = &_valListA;
	_pValListToRead = &_valListB;
	_pValueRead = _pValListToRead->begin();
}

Iterator_Fork::~Iterator_Fork()
{
	Function::Delete(_pFunc);
}

bool Iterator_Fork::Next(Signal sig, Value &value)
{
	if (_pValueRead == _pValListToRead->end()) {
		SwapList();
	} else {
		value = *_pValueRead++;
	}
	return true;
}

void Iterator_Fork::SwapList()
{
	_semaphore.Wait();
	ValueList *pValList = _pValListToWrite;
	_pValListToWrite = _pValListToRead;
	_pValListToRead = pValList;
	_pValListToWrite->clear();
	_pValueRead = _pValListToRead->begin();
	_semaphore.Release();
}

void Iterator_Fork::ForkProcess()
{
	IncRef();
	Start();
}

void Iterator_Fork::Run()
{
	Signal sig;
	ValueList valList;
	while (_iterOwner.Next(sig, valList)) {
		Context context(valList);
		Value value = _pFunc->Eval(_env, sig, context);
		if (sig.IsSignalled()) break;
	}
}

//-----------------------------------------------------------------------------
// Iterator_Delay
//-----------------------------------------------------------------------------
Iterator_Delay::~Iterator_Delay()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Delay::Next(Signal sig, Value &value)
{
	OAL::Sleep(_delay);
	return _pIterator->Next(sig, value);
}

//-----------------------------------------------------------------------------
// Iterator_Skip
//-----------------------------------------------------------------------------
Iterator_Skip::~Iterator_Skip()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Skip::Next(Signal sig, Value &value)
{
	bool flag = _pIterator->Next(sig, value);
	for (int i = 0; flag && i < _nSkip; i++) {
		Value valueTmp;
		flag = _pIterator->Next(sig, valueTmp);
	}
	return flag;
}

//-----------------------------------------------------------------------------
// Iterator_Align
//-----------------------------------------------------------------------------
Iterator_Align::~Iterator_Align()
{
	Iterator::Delete(_pIterator);
}

bool Iterator_Align::Next(Signal sig, Value &value)
{
	if (_cnt == 0) return false;
	if (!_pIterator->Next(sig, value)) value = _valueFill;
	if (_cnt > 0) _cnt--;
	return true;
}

//-----------------------------------------------------------------------------
// IteratorOwner
//-----------------------------------------------------------------------------
IteratorOwner::~IteratorOwner()
{
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator::Delete(*ppIterator);
	}
}

bool IteratorOwner::Next(Signal sig, ValueList &valList)
{
	valList.clear();
	foreach (IteratorOwner, ppIterator, *this) {
		Iterator *pIterator = *ppIterator;
		Value value;
		if (!pIterator->Next(sig, value)) return false;
		valList.push_back(value);
	}
	return true;
}

bool IteratorOwner::IsInfinite() const
{
	foreach_const (IteratorOwner, ppIterator, *this) {
		if (!(*ppIterator)->IsInfinite()) return false;
	}
	return true;
}

}
