#include "File.h"

#if defined(HAVE_DIRENT_H)
#include <dirent.h>		// struct stat
#endif

namespace AScript {

//-----------------------------------------------------------------------------
// DateTime
//-----------------------------------------------------------------------------
#if defined(HAVE_WINDOWS_H)
DateTime::DateTime(const SYSTEMTIME &st)
{
	SetSystemTime(st);
}

DateTime::DateTime(const FILETIME &ft, bool toLocalFlag)
{
	SYSTEMTIME stUTC;
	::FileTimeToSystemTime(&ft, &stUTC);
	if (toLocalFlag) {
		SYSTEMTIME stLocal;
		::SystemTimeToTzSpecificLocalTime(NULL, &stUTC, &stLocal);
		SetSystemTime(stLocal);
	} else {
		SetSystemTime(stUTC);
	}
}

void DateTime::SetSystemTime(const SYSTEMTIME &st)
{
	_year		= static_cast<short>(st.wYear);
	_month		= static_cast<char>(st.wMonth);
	_day		= static_cast<char>(st.wDay);
	_hour		= static_cast<char>(st.wHour);
	_minute		= static_cast<char>(st.wMinute);
	_second		= static_cast<char>(st.wSecond);
	_microsec	= static_cast<long>(st.wMilliseconds) * 1000;
}
#else
DateTime::DateTime(const struct tm &tm)
{
	SetTime(tm);
}

DateTime::DateTime(time_t t, bool toLocalFlag)
{
	struct tm tm;
	if (toLocalFlag) {
		localtime_r(&t, &tm);
	} else {
		gmtime_r(&t, &tm);
	}
	SetTime(tm);
}

void DateTime::SetTime(const struct tm &tm)
{
	_year		= static_cast<short>(tm.tm_year) + 1900;
	_month		= static_cast<char>(tm.tm_mon);
	_day		= static_cast<char>(tm.tm_mday);
	_hour		= static_cast<char>(tm.tm_hour);
	_minute		= static_cast<char>(tm.tm_min);
	_second		= static_cast<char>(tm.tm_sec);
	_microsec	= 0;
}
#endif

String DateTime::ToString(const char *format) const
{
	String rtn;
	for (const char *p = format; *p != '\0'; p++) {
		if (*p != '%') {
			rtn.push_back(*p);
			continue;
		}
		char buff[32];
		p++;
		char ch = *p;
		if (ch == 'a') {			// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'A') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'b') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'B') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'c') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'd') {
			::sprintf(buff, "%02d", _day);
			rtn += buff;
		} else if (ch == 'H') {
			::sprintf(buff, "%02d", _hour);
			rtn += buff;
		} else if (ch == 'I') {
			::sprintf(buff, "%02d",
						(_hour == 0)? 12 : (_hour > 12)? _hour - 12 : _hour);
			rtn += buff;
		} else if (ch == 'j') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'm') {
			::sprintf(buff, "%02d", _month);
			rtn += buff;
		} else if (ch == 'M') {
			::sprintf(buff, "%02d", _minute);
			rtn += buff;
		} else if (ch == 'p') {		// locale
			
		} else if (ch == 'S') {
			::sprintf(buff, "%02d", _second);
			rtn += buff;
		} else if (ch == 'U') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'w') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'W') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'x') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'X') {		// locale
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == 'y') {
			::sprintf(buff, "%02d", _year % 100);
			rtn += buff;
		} else if (ch == 'Y') {
			::sprintf(buff, "%04d", _year);
			rtn += buff;
		} else if (ch == 'Z') {
			::sprintf(buff, "****");
			rtn += buff;
		} else if (ch == '%') {
			rtn.push_back(ch);
		} else {
			rtn.push_back(ch);
		}
	}
	return rtn;
}

int DateTime::Compare(const DateTime &dt1, const DateTime &dt2)
{
	long result = 0;
	if ((result = dt1._year - dt2._year) != 0) {
	} else if ((result = dt1._month - dt2._month) != 0) {
	} else if ((result = dt1._day - dt2._day) != 0) {
	} else if ((result = dt1._hour - dt2._hour) != 0) {
	} else if ((result = dt1._minute - dt2._minute) != 0) {
	} else if ((result = dt1._second - dt2._second) != 0) {
	} else if ((result = dt1._microsec - dt2._microsec) != 0) {
	}
	return (result < 0)? -1 : (result > 0)? +1 : 0;
}

//-----------------------------------------------------------------------------
// FileStat
//-----------------------------------------------------------------------------
#if defined(HAVE_WINDOWS_H)
FileStat::FileStat(const char *pathName, const BY_HANDLE_FILE_INFORMATION &attrData) :
	_pathName(pathName), _attr(0), _bytes(attrData.nFileSizeLow),
	_atime(attrData.ftLastAccessTime, true),
	_mtime(attrData.ftLastWriteTime, true),
	_ctime(attrData.ftCreationTime, true), _uid(0), _gid(0)
{
	if (attrData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
		_attr |= ATTR_Dir;
	} else {
		_attr |= ATTR_Reg;
	}
	if (attrData.dwFileAttributes & FILE_ATTRIBUTE_READONLY) {
		_attr |= 0666;
	} else {
		_attr |= 0777;
	}
}

FileStat::FileStat(const char *pathName, const WIN32_FILE_ATTRIBUTE_DATA &attrData) :
	_pathName(pathName), _attr(0), _bytes(attrData.nFileSizeLow),
	_atime(attrData.ftLastAccessTime, true),
	_mtime(attrData.ftLastWriteTime, true),
	_ctime(attrData.ftCreationTime, true), _uid(0), _gid(0)
{
	if (attrData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
		_attr |= ATTR_Dir;
	} else {
		_attr |= ATTR_Reg;
	}
	if (attrData.dwFileAttributes & FILE_ATTRIBUTE_READONLY) {
		_attr |= 0666;
	} else {
		_attr |= 0777;
	}
}
#else
FileStat::FileStat(const char *pathName, const struct stat &stat) :
	_pathName(pathName), _attr(0), _bytes(stat.st_size),
	_atime(stat.st_atime, true),
	_mtime(stat.st_mtime, true),
	_ctime(stat.st_ctime, true), _uid(stat.st_uid), _gid(stat.st_gid)
{
	if (S_ISDIR(stat.st_mode)) _attr |= ATTR_Dir;
	if (S_ISCHR(stat.st_mode)) _attr |= ATTR_Chr;
	if (S_ISBLK(stat.st_mode)) _attr |= ATTR_Blk;
	if (S_ISREG(stat.st_mode)) _attr |= ATTR_Reg;
	if (S_ISFIFO(stat.st_mode)) _attr |= ATTR_Fifo;
	if (S_ISLNK(stat.st_mode)) _attr |= ATTR_Lnk;
	if (S_ISSOCK(stat.st_mode)) _attr |= ATTR_Sock;
	_attr |= (stat.st_mode & 0777);
}
#endif

//-----------------------------------------------------------------------------
// FileLister
//-----------------------------------------------------------------------------
FileLister::~FileLister()
{
}

#if defined(HAVE_WINDOWS_H)
//-----------------------------------------------------------------------------
// FileLister_Win32
//-----------------------------------------------------------------------------
class FileLister_Win32 : public FileLister {
private:
	HANDLE _hFind;
public:
	inline FileLister_Win32(const char *dirName) :
						FileLister(dirName), _hFind(INVALID_HANDLE_VALUE) {}
	virtual ~FileLister_Win32();
	virtual bool Next(FileEntry &entry);
};

FileLister *FileLister::CreateInstance(const char *dirName)
{
	return new FileLister_Win32(dirName);
}

FileLister_Win32::~FileLister_Win32()
{
	::FindClose(_hFind);
}

bool FileLister_Win32::Next(FileEntry &entry)
{
	WIN32_FIND_DATA findData;
	if (_hFind == INVALID_HANDLE_VALUE) {
		String pathName(_dirName);
		if (!pathName.empty()) pathName += File::Separator;
		pathName += "*.*";
		_hFind = ::FindFirstFile(pathName.c_str(), &findData);
		if (_hFind == INVALID_HANDLE_VALUE) return false;
	} else if (!::FindNextFile(_hFind, &findData)) {
		return false;
	}
	while (::strcmp(findData.cFileName, ".") == 0 ||
			::strcmp(findData.cFileName, "..") == 0) {
		if (!::FindNextFile(_hFind, &findData)) return false;
	}
	entry = FileEntry(findData.cFileName,
		(findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)? true : false);
	return true;
}

#else

//-----------------------------------------------------------------------------
// FileLister_Linux
//-----------------------------------------------------------------------------
class FileLister_Linux : public FileLister {
private:
	DIR *_pDir;
public:
	FileLister_Linux(const char *dirName);
	virtual ~FileLister_Linux();
	virtual bool Next(FileEntry &entry);
};

FileLister *FileLister::CreateInstance(const char *dirName)
{
	return new FileLister_Linux(dirName);
}

FileLister_Linux::FileLister_Linux(const char *dirName) :
									FileLister(dirName), _pDir(NULL)
{
}

FileLister_Linux::~FileLister_Linux()
{
	if (_pDir != NULL) {
		closedir(_pDir);
	}
}

bool FileLister_Linux::Next(FileEntry &entry)
{
	if (_pDir == NULL) {
		_pDir = opendir(_dirName.empty()? "." : _dirName.c_str());
		if (_pDir == NULL) return false;
	}
	struct dirent *pEnt = readdir(_pDir);
	if (pEnt == NULL) return false;
	entry = FileEntry(pEnt->d_name, (pEnt->d_type == DT_DIR));
	return true;
}

#endif

//-----------------------------------------------------------------------------
// File
//-----------------------------------------------------------------------------
const char File::SeparatorWin = '\\';
const char File::SeparatorUnix = '/';
String *File::_pWorkingDir = NULL;
String *File::_pBaseDir = NULL;

#if defined(HAVE_WINDOWS_H)
const char File::Separator = SeparatorWin;

File::File() : _hFile(INVALID_HANDLE_VALUE), _needCloseFlag(false),
							_accessMode(ACCESS_Read), _textFlag(false)
{
}

File::File(const File &file) : _hFile(file._hFile), _needCloseFlag(file._needCloseFlag),
	_fileName(file._fileName), _accessMode(file._accessMode), _textFlag(file._textFlag)
{
}

File &File::Open(Signal sig, const char *fileName, const char *mode)
{
	Close();
	do {
		const char *p = mode;
		if (*p == 'r') {
			_accessMode = ACCESS_Read;
		} else if (*p == 'w') {
			_accessMode = ACCESS_Write;
		} else if (*p == 'a') {
			_accessMode = ACCESS_Write;
		} else {
			sig.SetError(ERR_IOError, "invalid open mode");
			return *this;
		}
		p++;
		_textFlag = true;
		for ( ; *p != '\0'; p++) {
			char ch = *p;
			if (ch == '+') {
				_accessMode = ACCESS_ReadWrite;
			} else if (ch == 't') {
				_textFlag = true;
			} else if (ch == 'b') {
				_textFlag = false;
			} else {
				sig.SetError(ERR_IOError, "invalid open mode");
				return *this;
			}
		}
	} while (0);
	_fileName = MakeAbsPath(fileName);
	DWORD dwDesiredAccess =
			(_accessMode == ACCESS_Read)? GENERIC_READ :
			(_accessMode == ACCESS_Write)? GENERIC_WRITE :
			(_accessMode == ACCESS_ReadWrite)? GENERIC_READ | GENERIC_WRITE :
			GENERIC_READ;
	DWORD dwShareMode = FILE_SHARE_READ;
	DWORD dwCreationDisposition =
			(_accessMode == ACCESS_Read)? OPEN_EXISTING :
			(_accessMode == ACCESS_Write)? CREATE_ALWAYS :
			(_accessMode == ACCESS_ReadWrite)? OPEN_ALWAYS :
			OPEN_EXISTING;
	_hFile = ::CreateFile(_fileName.c_str(), dwDesiredAccess, dwShareMode,
					NULL, dwCreationDisposition, FILE_ATTRIBUTE_NORMAL, NULL);
	if (_hFile == INVALID_HANDLE_VALUE) {
		sig.SetError(ERR_IOError, "can't open file '%s'", ExtractBaseName(fileName).c_str());
		return *this;
	}
	_needCloseFlag = true;
	return *this;
}

File &File::OpenStdin()
{
	Close();
	_hFile = ::GetStdHandle(STD_INPUT_HANDLE);
	if (_hFile != INVALID_HANDLE_VALUE) {
		_fileName = "stdin", _accessMode = ACCESS_Read, _textFlag = true;
	}
	return *this;
}

File &File::OpenStdout()
{
	Close();
	_hFile = ::GetStdHandle(STD_OUTPUT_HANDLE);
	if (_hFile != INVALID_HANDLE_VALUE) {
		_fileName = "stdout", _accessMode = ACCESS_Write, _textFlag = true;
	}
	return *this;
}

File &File::OpenStderr()
{
	Close();
	_hFile = ::GetStdHandle(STD_ERROR_HANDLE);
	if (_hFile != INVALID_HANDLE_VALUE) {
		_fileName = "stderr", _accessMode = ACCESS_Write, _textFlag = true;
	}
	return *this;
}

void File::Close()
{
	if (_needCloseFlag && _hFile != INVALID_HANDLE_VALUE) {
		::CloseHandle(_hFile);
		_hFile = INVALID_HANDLE_VALUE;
		_needCloseFlag = false;
		_fileName = "<invalid>";
	}
}

int File::Seek(long offset, int origin)
{
	if (_hFile == INVALID_HANDLE_VALUE) return -1;
	DWORD dwMoveMethod =
		(origin == SEEK_SET)? FILE_BEGIN :
		(origin == SEEK_CUR)? FILE_CURRENT :
		(origin == SEEK_END)? FILE_END : FILE_BEGIN;
	return ::SetFilePointer(_hFile, offset, NULL, dwMoveMethod);
}

int File::GetChar()
{
	if (_hFile == INVALID_HANDLE_VALUE) return -1;
	DWORD dwBytesRead;
	unsigned char buff[1];
	::ReadFile(_hFile, buff, sizeof(buff), &dwBytesRead, NULL);
	if (_textFlag) {
		while (dwBytesRead > 0 && buff[0] == '\r') {
			::ReadFile(_hFile, buff, sizeof(buff), &dwBytesRead, NULL);
		}
	}
	return (dwBytesRead == 0)? -1 : buff[0];
}

void File::PutChar(int ch)
{
	if (_hFile == INVALID_HANDLE_VALUE) return;
	unsigned char buff[2];
	DWORD dwBytesToWrite, dwBytesWritten;
	if (_textFlag && ch == '\n') {
		dwBytesToWrite = 2;
		buff[0] = '\r';
		buff[1] = static_cast<unsigned char>(ch);
		::WriteFile(_hFile, buff, dwBytesToWrite, &dwBytesWritten, NULL);
	} else {
		dwBytesToWrite = 1;
		buff[0] = static_cast<unsigned char>(ch);
		::WriteFile(_hFile, buff, dwBytesToWrite, &dwBytesWritten, NULL);
	}
}

size_t File::Read(void *buff, size_t bytes)
{
	DWORD dwBytesRead;
	::ReadFile(_hFile, buff, bytes, &dwBytesRead, NULL);
	return static_cast<size_t>(dwBytesRead);
}

bool File::GetFileStatByHandle(Signal sig, FileStat &fileStat) const
{
	unsigned long attr = 0;
	BY_HANDLE_FILE_INFORMATION attrData;
	String pathName = MakeAbsPath(_fileName.c_str(), NULL);
	if (::GetFileInformationByHandle(_hFile, &attrData) == 0) {
		sig.SetError(ERR_IOError, "failed to get file status of %s", pathName.c_str());
		return false;
	}
	fileStat = FileStat(pathName.c_str(), attrData);
	return true;
}

bool File::GetFileStat(Signal sig, const char *fileName, FileStat &fileStat)
{
	unsigned long attr = 0;
	WIN32_FILE_ATTRIBUTE_DATA attrData;
	String pathName = MakeAbsPath(fileName, NULL);
	if (::GetFileAttributesEx(pathName.c_str(), GetFileExInfoStandard, &attrData) == 0) {
		sig.SetError(ERR_IOError, "failed to get file status of %s", pathName.c_str());
		return false;
	}
	fileStat = FileStat(pathName.c_str(), attrData);
	return true;
}

bool File::IsExist(const char *pathName)
{
	WIN32_FILE_ATTRIBUTE_DATA attrData;
	return ::GetFileAttributesEx(pathName, GetFileExInfoStandard, &attrData) != 0;
}

#else // !defined(HAVE_WINDOWS_H)
const char File::Separator = SeparatorUnix;

File::File() : _fp(NULL), _needCloseFlag(false), _accessMode(ACCESS_Read), _textFlag(false)
{
}

File::File(const File &file) : _fp(file._fp), _needCloseFlag(file._needCloseFlag),
	_fileName(file._fileName), _accessMode(file._accessMode), _textFlag(file._textFlag)
{
}

File &File::Open(Signal sig, const char *fileName, const char *mode)
{
	Close();
	do {
		const char *p = mode;
		if (*p == 'r') {
			_accessMode = ACCESS_Read;
		} else if (*p == 'w') {
			_accessMode = ACCESS_Write;
		} else if (*p == 'a') {
			_accessMode = ACCESS_Write;
		} else {
			sig.SetError(ERR_IOError, "invalid open mode");
			return *this;
		}
		p++;
		_textFlag = true;
		for ( ; *p != '\0'; p++) {
			char ch = *p;
			if (ch == '+') {
				_accessMode = ACCESS_ReadWrite;
			} else if (ch == 't') {
				_textFlag = true;
			} else if (ch == 'b') {
				_textFlag = false;
			} else {
				sig.SetError(ERR_IOError, "invalid open mode");
				return *this;
			}
		}
	} while (0);
	_fileName = MakeAbsPath(fileName);
	_fp = ::fopen(_fileName.c_str(), mode);
	if (_fp == NULL) {
		sig.SetError(ERR_IOError,
				"can't open file '%s'", ExtractBaseName(fileName).c_str());
		return *this;
	}
	_needCloseFlag = true;
	return *this;
}

File &File::OpenStdin()
{
	_fp = stdin;
	_fileName = "stdin", _accessMode = ACCESS_Read, _textFlag = true;
	return *this;
}

File &File::OpenStdout()
{
	_fp = stdout;
	_fileName = "stdout", _accessMode = ACCESS_Write, _textFlag = true;
	return *this;
}

File &File::OpenStderr()
{
	_fp = stderr;
	_fileName = "stderr", _accessMode = ACCESS_Write, _textFlag = true;
	return *this;
}

void File::Close()
{
	if (_needCloseFlag && _fp != NULL) {
		::fclose(_fp);
		_fp = NULL;
		_needCloseFlag = false;
		_fileName = "<invalid>";
	}
}

int File::Seek(long offset, int origin)
{
	if (_fp == NULL) return -1;
	return ::fseek(_fp, offset, origin);
}

int File::GetChar()
{
	if (_fp == NULL) return -1;
	int ch = ::fgetc(_fp);
	if (ch != '\r') return ch;
	return ::fgetc(_fp);
}

void File::PutChar(int ch)
{
	if (_fp == NULL) return;
	::fputc(ch, _fp);
}

size_t File::Read(void *buff, size_t bytes)
{
	return ::fread(buff, 1, bytes, _fp);
}

bool File::GetFileStatByHandle(Signal sig, FileStat &fileStat) const
{
	unsigned long attr = 0;
	struct stat stat;
	String pathName = MakeAbsPath(_fileName.c_str(), NULL);
	if (::fstat(fileno(_fp), &stat) != 0) {
		sig.SetError(ERR_IOError, "failed to get file status of %s", pathName.c_str());
		return false;
	}
	fileStat = FileStat(pathName.c_str(), stat);
	return true;
}

bool File::GetFileStat(Signal sig, const char *fileName, FileStat &fileStat)
{
	unsigned long attr = 0;
	struct stat stat;
	String pathName = MakeAbsPath(fileName, NULL);
	if (::stat(pathName.c_str(), &stat) != 0) {
		sig.SetError(ERR_IOError, "failed to get file status of %s", pathName.c_str());
		return false;
	}
	fileStat = FileStat(pathName.c_str(), stat);
	return true;
}

bool File::IsExist(const char *pathName)
{
	struct stat stat;
	return ::stat(pathName, &stat) == 0;
}

#endif

void File::PutString(const char *str)
{
	for ( ; *str != '\0'; str++) PutChar(*str);
}

String File::ExtractDirName(const char *pathName)
{
	const char *p = pathName + ::strlen(pathName);
	for ( ; p >= pathName; p--) {
		if (IsSeparator(*p)) {
			String str;
			return RegulateName(str, pathName, p - pathName);
		}
	}
	return String("");
}

String File::ExtractBaseName(const char *pathName)
{
	const char *p = pathName + ::strlen(pathName);
	for ( ; p >= pathName; p--) {
		if (IsSeparator(*p)) {
			String str;
			return RegulateName(str, p + 1);
		}
	}
	return String(pathName);
}

void File::Split(const char *pathName, StringList &pathNameSepList)
{
	String field;
	if (IsSeparator(*pathName)) field.push_back(Separator);
	for (const char *p = pathName; *p != '\0'; p++) {
		if (IsSeparator(*p)) {
			if (!field.empty()) {
				pathNameSepList.push_back(field);
				field.clear();
			}
		} else {
			field.push_back(*p);
		}
	}
	if (!field.empty()) pathNameSepList.push_back(field);
}

bool File::HasWildCard(const char *pathName)
{
	for (const char *p = pathName; *p != '\0'; p++) {
		if (*p == '*' || *p == '?' || *p == '[' || *p == ']') {
			return true;
		}
	}
	return false;
}

// dirName must be specified as an absolute path.
String File::MakeAbsPath(const char *fileName, const char *dirName)
{
	String str;
	if (dirName == NULL || dirName[0] == '\0' || dirName[0] == '.' &&
					(dirName[1] == '\0' ||
					 dirName[1] == Separator && dirName[2] == '\0')) {
		dirName = GetWorkingDir().c_str();
	}
	if (IsSeparator(fileName[0])) {
		for (const char *p = dirName;
						*p != '\0' && !IsSeparator(*p); p++) {
			str += *p;
		}
	} else if (fileName[0] != '\0' && fileName[1] == ':') {
		// absolute path in Windows style.
	} else {
		int lenDirName = ::strlen(dirName);
		if (lenDirName > 0 && IsSeparator(*(dirName + lenDirName - 1))) {
			lenDirName--;
		}
		for (;;) {
			if (*fileName == '.' &&
					(IsSeparator(*(fileName + 1)) || *(fileName + 1) == '\0')) {
				fileName += IsSeparator(*(fileName + 2))? 2 : 1;
			} else if (*fileName == '.' && *(fileName + 1) == '.' &&
					(IsSeparator(*(fileName + 2)) || *(fileName + 2) == '\0')) {
				int len = lenDirName;
				if (len > 0) len--;
				for ( ; len > 0 && !IsSeparator(*(dirName + len)); len--) ;
				if (len > 0) lenDirName = len;
				fileName += IsSeparator(*(fileName + 2))? 3 : 2;
			} else {
				break;
			}
		}
		for (const char *p = dirName; lenDirName > 0; p++, lenDirName--) {
			str += IsSeparator(*p)? Separator : *p;
		}
		str += Separator;
	}
	return RegulateName(str, fileName);
}

String &File::RegulateName(String &result, const char *name)
{
	for (const char *p = name; *p != '\0'; p++) {
		result += IsSeparator(*p)? Separator : *p;
	}
	return result;
}

String &File::RegulateName(String &result, const char *name, size_t cnt)
{
	for (const char *p = name; *p != '\0' && cnt > 0; p++, cnt--) {
		result += IsSeparator(*p)? Separator : *p;
	}
	return result;
}

const String &File::GetWorkingDir()
{
	if (_pWorkingDir == NULL) {
		char *pathName;
		int maxLen = 256;
		for (int i = 0; i < 16; i++, maxLen += 128) {
			pathName = ::getcwd(NULL, maxLen);
			if (pathName != NULL) break;
		}
		if (pathName == NULL) ::exit(1);
		size_t len = ::strlen(pathName);
		if (len > 0 && pathName[len - 1] == Separator) pathName[len - 1] = '\0';
		_pWorkingDir = new String(pathName);
		::free(pathName);
	}
	return *_pWorkingDir;
}

#if defined(HAVE_WINDOWS_H)
void File::InitBaseDir(const char *argv0)
{
	char pathName[512];
	::GetModuleFileName(NULL, pathName, NUMBEROF(pathName)); // Win32 API
	char *p = pathName + ::strlen(pathName);
	for ( ; p >= pathName; p--) {
		if (IsSeparator(*p)) {
			*p = '\0';
			break;
		}
	}
	_pBaseDir = new String(pathName);
}

#else
void File::InitBaseDir(const char *argv0)
{
	char *dirName = ::getcwd(NULL, PATH_MAX);
	_pBaseDir = new String(dirName);
	::free(dirName);
}
#endif

bool File::IsMatchName(const char *pattern, const char *fileName, bool ignoreCaseFlag)
{
	if (*pattern == '\0') {
		return *fileName == '\0';
	} else if (*pattern == '*') {
		return IsMatchName(pattern + 1, fileName, ignoreCaseFlag) ||
			*fileName != '\0' && IsMatchName(pattern, fileName + 1, ignoreCaseFlag);
	} else if (*pattern == '?') {
		return *fileName != '\0' && IsMatchName(pattern + 1, fileName + 1, ignoreCaseFlag);
	} else if (*pattern == '[') {
		bool negFlag = false;
		pattern++;
		if (*pattern == '!') {
			negFlag = true;
			pattern++;
			for ( ; *pattern != ']' && *pattern != '\0'; pattern++) {
				if (CompareChar(*fileName, *pattern, ignoreCaseFlag) == 0) return false;
			}
		} else {
			for ( ; *pattern != ']' && *pattern != '\0'; pattern++) {
				if (CompareChar(*fileName, *pattern, ignoreCaseFlag) != 0) return false;
			}
		}
		if (*pattern == ']') pattern++;
		return IsMatchName(pattern, fileName + 1, ignoreCaseFlag);
	} else {
		return CompareChar(*pattern, *fileName, ignoreCaseFlag) == 0 &&
						IsMatchName(pattern + 1, fileName + 1, ignoreCaseFlag);
	}
}

//-----------------------------------------------------------------------------
// File::Formatter
//-----------------------------------------------------------------------------
void File::Formatter::PutChar(char ch)
{
	_file.PutChar(ch);
}

}
