#ifndef __ENVIRONMENT_H__
#define __ENVIRONMENT_H__

#include "Value.h"

namespace AScript {

class Class;
class Class_Function;
class Class_String;
class Class_List;
class Class_Matrix;
class Class_Dict;
class Class_File;
class Class_FileStat;
class Class_DateTime;
class Class_Iterator;
class Class_Expr;
class Class_Environment;
class Class_Error;
class Class_Semaphore;
class Class_Struct;

class Module;

class ICallable;

//-----------------------------------------------------------------------------
// EnvType
//-----------------------------------------------------------------------------
enum EnvType {
	ENVTYPE_Invalid,
	ENVTYPE_Root,
	ENVTYPE_Local,
	ENVTYPE_Block,
	ENVTYPE_Module,
	ENVTYPE_Class,
	ENVTYPE_Instance,
	ENVTYPE_Method,
	ENVTYPE_Lister,
	ENVTYPE_Outer,	// this type is not set as type of each frame
};

const char *GetEnvTypeName(EnvType envType);

//-----------------------------------------------------------------------------
// ValueTypeMap
//-----------------------------------------------------------------------------
typedef std::map<const Symbol *, ValueType, Symbol::KeyCompare_UniqNumber> ValueTypeMap;

//-----------------------------------------------------------------------------
// ClassMap
//-----------------------------------------------------------------------------
typedef std::map<ValueType, Class *> ClassMap;

//-----------------------------------------------------------------------------
// IntegratedModuleMap
//-----------------------------------------------------------------------------
typedef std::map<int, Module *> IntegratedModuleMap;

//-----------------------------------------------------------------------------
// SeparatedModuleMap
//-----------------------------------------------------------------------------
typedef std::map<String, Module *> SeparatedModuleMap;

//-----------------------------------------------------------------------------
// Module Entry Type
//-----------------------------------------------------------------------------
typedef void (*ModuleEntryType)(Environment &env, Signal sig);
typedef void (*ModuleTerminateType)(Module *pModule);

//-----------------------------------------------------------------------------
// IntegratedModule
//-----------------------------------------------------------------------------
class IntegratedModule {
private:
	String _name;
	ModuleEntryType _moduleEntry;
	ModuleTerminateType _moduleTerminate;
public:
	inline IntegratedModule(const char *name,
			ModuleEntryType moduleEntry, ModuleTerminateType moduleTerminate) :
		_name(name), _moduleEntry(moduleEntry), _moduleTerminate(moduleTerminate) {}
	inline void ModuleEntry(Environment &env, Signal sig) {
		(*_moduleEntry)(env, sig);
	}
	inline void ModuleTerminate(Module *pModule) {
		(*_moduleTerminate)(pModule);
	}
	bool IsMatch(const SymbolList &symbolOfModule) const;
};

//-----------------------------------------------------------------------------
// IntegratedModuleOwner
//-----------------------------------------------------------------------------
class IntegratedModuleOwner : public std::vector<IntegratedModule *> {
public:
	~IntegratedModuleOwner();
};

//-----------------------------------------------------------------------------
// Environment
//-----------------------------------------------------------------------------
class Environment {
public:
	class Global {
	private:
		ValueTypeMap	_valTypeMap;
		ClassMap		_classMap;
		IntegratedModuleMap _integratedModuleMap;
		SeparatedModuleMap _separatedModuleMap;
		StringList		_workingDirList;
	public:
		SymbolPool		*_pSymbolPool;
		bool			_echoFlag;
		Class			*_pClass;
		Class_Function	*_pClass_Function;
		Class_String	*_pClass_String;
		Class_List		*_pClass_List;
		Class_Matrix	*_pClass_Matrix;
		Class_Dict		*_pClass_Dict;
		Class_File		*_pClass_File;
		Class_FileStat	*_pClass_FileStat;
		Class_DateTime	*_pClass_DateTime;
		Class_Iterator	*_pClass_Iterator;
		Class_Expr		*_pClass_Expr;
		Class_Environment *_pClass_Environment;
		Class_Error		*_pClass_Error;
		Class_Semaphore	*_pClass_Semaphore;
		Class_Struct	*_pClass_Struct;
		Function		*_pFunc_Neg;
		Function		*_pFunc_Invert;
		Function		*_pFunc_Not;
		Function		*_pFunc_Plus;
		Function		*_pFunc_Minus;
		Function		*_pFunc_Multiply;
		Function		*_pFunc_Divide;
		Function		*_pFunc_Modulo;
		Function		*_pFunc_format;
		Function		*_pFunc_Power;
		Function		*_pFunc_Equal;
		Function		*_pFunc_NotEqual;
		Function		*_pFunc_Greater;
		Function		*_pFunc_Less;
		Function		*_pFunc_GreaterEq;
		Function		*_pFunc_LessEq;
		Function		*_pFunc_Compare;
		Function		*_pFunc_ContainCheck;
		Function		*_pFunc_Or;
		Function		*_pFunc_And;
		Function		*_pFunc_Xor;
		Function		*_pFunc_OrOr;
		Function		*_pFunc_AndAnd;
		Function		*_pFunc_Sequence;
		Function		*_pFunc_SequenceInf;
		Function		*_pFunc_acos;
		Function		*_pFunc_asin;
		Function		*_pFunc_atan;
		Function		*_pFunc_atan2;
		Function		*_pFunc_ceil;
		Function		*_pFunc_cos;
		Function		*_pFunc_cosh;
		Function		*_pFunc_exp;
		Function		*_pFunc_abs;
		Function		*_pFunc_floor;
		Function		*_pFunc_log;
		Function		*_pFunc_log10;
		Function		*_pFunc_sin;
		Function		*_pFunc_sinh;
		Function		*_pFunc_sqrt;
		Function		*_pFunc_tan;
		Function		*_pFunc_tanh;
		Module			*_pModule_sys;
	public:
		Global();
		~Global();
		inline static void Delete(Global *pGlobal) {
			delete pGlobal;
		}
		void Prepare(Environment &env);
		ValueType LookupValueType(const Symbol *pSymbol) const;
		const Class *LookupClass(ValueType valType) const;
		Module *LookupIntegratedModule(int id) const;
		void RegisterIntegratedModule(int id, Module *pModule);
		Module *LookupSeparatedModule(const char *pathName) const;
		void RegisterSeparatedModule(const char *pathName, Module *pModule);
	};
	class Frame {
	private:
		int _cntRef;
		EnvType _envType;
		Global *_pGlobal;
		std::auto_ptr<ValueMap> _pValueMap;
	public:
		Frame(const Frame &frame);
		Frame(EnvType envType, Global *pGlobal);
		virtual ~Frame();
		inline Frame *IncRef() { _cntRef++; return this; }
		inline int DecRef() { _cntRef--; return _cntRef; }
		inline int GetRefCnt() const { return _cntRef; }
		inline static void Delete(Frame *pFrame) {
			if (pFrame != NULL && pFrame->DecRef() <= 0) delete pFrame;
		}
		inline Frame *Clone() const { return new Frame(*this); }
		inline EnvType GetType() const { return _envType; }
		inline const char *GetTypeName() const { return GetEnvTypeName(_envType); }
		inline bool IsType(EnvType envType) const { return _envType == envType; }
		inline Global *GetGlobal() { return _pGlobal; }
		inline const Global *GetGlobal() const { return _pGlobal; }
		inline const ValueMap &GetValueMap() const {
			return (_pValueMap.get() == NULL)? ValueMap::Null : *_pValueMap;
		}
		void AssignValue(const Symbol *pSymbol, const Value &value);
		Value *LookupValue(const Symbol *pSymbol);
		void DbgPrint() const;
	};
	class FrameList : public std::list<Frame *> {
	public:
		inline bool IsExist(Frame *pFrame) const {
			return std::find(begin(), end(), pFrame) != end();
		}
	};
	typedef std::map<const Symbol *, Frame *, Symbol::KeyCompare_UniqNumber> FrameCache;
private:
	FrameList _frameList;
	FrameCache *_pFrameCache;
	static IntegratedModuleOwner _integratedModuleOwner;
public:
	Environment(const Environment &env);
	Environment(Environment *pEnvOuter, EnvType envType);
	virtual ~Environment();
	inline FrameList &GetFrameList()			{ return _frameList;						}
	inline const FrameList &GetFrameList() const{ return _frameList;						}
	inline Frame &GetTopFrame()					{ return *_frameList.front();				}
	inline const Frame &GetTopFrame() const		{ return *_frameList.front();				}
	inline Frame &GetBottomFrame()				{ return *_frameList.back();				}
	inline const Frame &GetBottomFrame() const	{ return *_frameList.back();				}
	inline EnvType GetType() const				{ return GetTopFrame().GetType(); }
	inline const char *GetTypeName() const		{ return GetTopFrame().GetTypeName(); }
	inline bool IsType(EnvType envType) const	{ return GetTopFrame().IsType(envType); }
	inline Global *GetGlobal()					{ return GetTopFrame().GetGlobal();			}
	inline const Global *GetGlobal() const		{ return GetTopFrame().GetGlobal();			}
	inline Class *GetClass_Object()				{ return GetGlobal()->_pClass;				}
	inline Class_Function *GetClass_Function()	{ return GetGlobal()->_pClass_Function;		}
	inline Class_String *GetClass_String()		{ return GetGlobal()->_pClass_String;		}
	inline Class_List *GetClass_List()			{ return GetGlobal()->_pClass_List;			}
	inline Class_Matrix *GetClass_Matrix()		{ return GetGlobal()->_pClass_Matrix;		}
	inline Class_Dict *GetClass_Dict()			{ return GetGlobal()->_pClass_Dict;			}
	inline Class_File *GetClass_File()			{ return GetGlobal()->_pClass_File;			}
	inline Class_FileStat *GetClass_FileStat()	{ return GetGlobal()->_pClass_FileStat;		}
	inline Class_DateTime *GetClass_DateTime()	{ return GetGlobal()->_pClass_DateTime;		}
	inline Class_Iterator *GetClass_Iterator()	{ return GetGlobal()->_pClass_Iterator;		}
	inline Class_Expr *GetClass_Expr()			{ return GetGlobal()->_pClass_Expr;			}
	inline Class_Environment *GetClass_Environment() { return GetGlobal()->_pClass_Environment; }
	inline Class_Error *GetClass_Error()		{ return GetGlobal()->_pClass_Error;		}
	inline Class_Semaphore *GetClass_Semaphore(){ return GetGlobal()->_pClass_Semaphore;	}
	inline Class_Struct *GetClass_Struct()		{ return GetGlobal()->_pClass_Struct;		}
	inline const Function &GetFunc_Neg()		{ return *GetGlobal()->_pFunc_Neg;			}
	inline const Function &GetFunc_Invert()		{ return *GetGlobal()->_pFunc_Invert;		}
	inline const Function &GetFunc_Not()		{ return *GetGlobal()->_pFunc_Not;			}
	inline const Function &GetFunc_Plus()		{ return *GetGlobal()->_pFunc_Plus;			}
	inline const Function &GetFunc_Minus()		{ return *GetGlobal()->_pFunc_Minus;		}
	inline const Function &GetFunc_Multiply()	{ return *GetGlobal()->_pFunc_Multiply;		}
	inline const Function &GetFunc_Divide()		{ return *GetGlobal()->_pFunc_Divide;		}
	inline const Function &GetFunc_Modulo()		{ return *GetGlobal()->_pFunc_Modulo;		}
	inline const Function &GetFunc_format()		{ return *GetGlobal()->_pFunc_format;		}
	inline const Function &GetFunc_Power()		{ return *GetGlobal()->_pFunc_Power;		}
	inline const Function &GetFunc_Equal()		{ return *GetGlobal()->_pFunc_Equal;		}
	inline const Function &GetFunc_NotEqual()	{ return *GetGlobal()->_pFunc_NotEqual;		}
	inline const Function &GetFunc_Greater()	{ return *GetGlobal()->_pFunc_Greater;		}
	inline const Function &GetFunc_Less()		{ return *GetGlobal()->_pFunc_Less;			}
	inline const Function &GetFunc_GreaterEq()	{ return *GetGlobal()->_pFunc_GreaterEq;	}
	inline const Function &GetFunc_LessEq()		{ return *GetGlobal()->_pFunc_LessEq;		}
	inline const Function &GetFunc_Compare()	{ return *GetGlobal()->_pFunc_Compare;		}
	inline const Function &GetFunc_ContainCheck() { return *GetGlobal()->_pFunc_ContainCheck;	}
	inline const Function &GetFunc_Or()			{ return *GetGlobal()->_pFunc_Or;			}
	inline const Function &GetFunc_And()		{ return *GetGlobal()->_pFunc_And;			}
	inline const Function &GetFunc_Xor()		{ return *GetGlobal()->_pFunc_Xor;			}
	inline const Function &GetFunc_OrOr()		{ return *GetGlobal()->_pFunc_OrOr;			}
	inline const Function &GetFunc_AndAnd()		{ return *GetGlobal()->_pFunc_AndAnd;		}
	inline const Function &GetFunc_Sequence()	{ return *GetGlobal()->_pFunc_Sequence;		}
	inline const Function &GetFunc_SequenceInf(){ return *GetGlobal()->_pFunc_SequenceInf;	}
	inline const Function &GetFunc_acos()		{ return *GetGlobal()->_pFunc_acos;			}
	inline const Function &GetFunc_asin()		{ return *GetGlobal()->_pFunc_asin;			}
	inline const Function &GetFunc_atan()		{ return *GetGlobal()->_pFunc_atan;			}
	inline const Function &GetFunc_atan2()		{ return *GetGlobal()->_pFunc_atan2;		}
	inline const Function &GetFunc_ceil()		{ return *GetGlobal()->_pFunc_ceil;			}
	inline const Function &GetFunc_cos()		{ return *GetGlobal()->_pFunc_cos;			}
	inline const Function &GetFunc_cosh()		{ return *GetGlobal()->_pFunc_cosh;			}
	inline const Function &GetFunc_exp()		{ return *GetGlobal()->_pFunc_exp;			}
	inline const Function &GetFunc_abs()		{ return *GetGlobal()->_pFunc_abs;			}
	inline const Function &GetFunc_floor()		{ return *GetGlobal()->_pFunc_floor;		}
	inline const Function &GetFunc_log()		{ return *GetGlobal()->_pFunc_log;			}
	inline const Function &GetFunc_log10()		{ return *GetGlobal()->_pFunc_log10;		}
	inline const Function &GetFunc_sin()		{ return *GetGlobal()->_pFunc_sin;			}
	inline const Function &GetFunc_sinh()		{ return *GetGlobal()->_pFunc_sinh;			}
	inline const Function &GetFunc_sqrt()		{ return *GetGlobal()->_pFunc_sqrt;			}
	inline const Function &GetFunc_tan()		{ return *GetGlobal()->_pFunc_tan;			}
	inline const Function &GetFunc_tanh()		{ return *GetGlobal()->_pFunc_tanh;			}
	inline Module *GetModule_sys()				{ return GetGlobal()->_pModule_sys;			}
	inline void SetEchoFlag(bool echoFlag)		{ GetGlobal()->_echoFlag = echoFlag;		}
	inline bool GetEchoFlag() const				{ return GetGlobal()->_echoFlag;			}
	void AssignValue(const Symbol *pSymbol, const Value &value, bool escalateFlag);
	void AssignFunction(Function *pFunc);
	Value *LookupValue(const Symbol *pSymbol, bool escalateFlag);
	inline const Value *LookupValue(const Symbol *pSymbol, bool escalateFlag) const {
		return const_cast<const Value *>(
				const_cast<Environment *>(this)->LookupValue(pSymbol, escalateFlag));
	}
	Function *LookupFunction(const Symbol *pSymbol, bool escalateFlag);
	virtual Value EvalGetter(Signal sig, const Symbol *pSymbol, bool &evaluatedFlag);
	virtual Value EvalSetter(Signal sig, const Symbol *pSymbol,
										const Value &value, bool &evaluatedFlag);
	virtual ICallable *GetCallable(Signal sig, const Symbol *pSymbol);
	ValueType LookupValueType(const Symbol *pSymbol, const Class **ppClass);
	inline const Class *LookupClass(ValueType valType) {
		return GetGlobal()->LookupClass(valType);
	}
	void AssignModule(Module *pModule);
	bool ImportModule(Signal sig, const Expr *pExpr, const SymbolSet &symbolSet);
	static bool IsBinaryModule(const char *pathName);
	void SetupBuiltIn(Signal sig, int argc, const char *argv[]);
	virtual bool IsModule() const;
	virtual bool IsClass() const;
	virtual bool IsObject() const;
	inline void Error(const char *fileName, int lineNo, const char *str) {
		::fprintf(stderr, "fatal error at line.%d in %s: %s\n", lineNo, fileName, str);
	}
	// virtual function of Console
	void DbgPrint() const;
	const char *GetPrompt(bool indentFlag);
	void PutPrompt(bool indentFlag);
	void PutChar(char ch);
	void PutString(const char *str);
	void PutValue(Signal sig, const Value &value);
	static void IntegrateModule(const char *name,
			ModuleEntryType moduleEntry, ModuleTerminateType moduleTerminate);
private:
	Module *ImportIntegratedModule(Signal sig, const SymbolList &symbolOfModule);
	Module *ImportSeparatedModule(Signal sig, const SymbolList &symbolOfModule);
	bool SearchSeparatedModuleFile(Signal sig,
							String &pathName, const SymbolList &symbolOfModule);
	Module *ImportSeparatedModule_Script(Signal sig, const char *pathName);
	Module *ImportSeparatedModule_Binary(Signal sig, const char *pathName);
};

class EnvironmentRoot : public Environment {
public:
	inline EnvironmentRoot() : Environment(NULL, ENVTYPE_Root) {}
	virtual ~EnvironmentRoot();
};

}

#endif
