﻿//-------------------------------------------------------------------------------------------------
// File : asdxApp.h
// Desc : Desktop Application Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <Windows.h>
#include <asdxStepTimer.h>
#include <atomic>


//-------------------------------------------------------------------------------------------------
// Linker
//-------------------------------------------------------------------------------------------------
#ifdef ASDX_AUTO_LINK
#pragma comment( lib, "d3d12.lib" )
#pragma comment( lib, "dxgi.lib" )
#pragma comment( lib, "dxguid.lib" )
#pragma comment( lib, "d3dcompiler.lib" )
#pragma comment( lib, "winmm.lib" )
#endif//ASDX_AUTO_LINK


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// MouseEventArgs structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct MouseEventArgs
{
    int     CursorX;            //!< カーソルのX座標です.
    int     CursorY;            //!< カーソルのY座標です.
    int     WheelDelta;         //!< ホイールの移動量です.
    bool    IsLeftDown;         //!< 左ボタンが押されているかどうか.
    bool    IsRightDown;        //!< 右ボタンが押されているかどうか.
    bool    IsMiddleDown;       //!< 中ボタンが押されているかどうか.
    bool    IsSide1Down;        //!< サイドボタン1が押されているかどうか.
    bool    IsSide2Down;        //!< サイドボタン2が押されているかどうか.

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    MouseEventArgs()
    : CursorX       (0)
    , CursorY       (0)
    , WheelDelta    (0)
    , IsLeftDown    (false)
    , IsRightDown   (false)
    , IsMiddleDown  (false)
    , IsSide1Down   (false)
    , IsSide2Down   (false)
    { /* DO_NOTHING */ }
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// KeyEventArgs structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct KeyEventArgs
{
    uint32_t    KeyCode;        //!< キーコードです.
    bool        IsKeyDown;      //!< キーが押されているかどうか.
    bool        IsAltDown;      //!< ALTキーが押されているかどうか.

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    KeyEventArgs()
    : KeyCode   (0)
    , IsKeyDown (false)
    , IsAltDown (false)
    { /* DO_NOTHING */ }
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// ResizeEventArgs structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct ResizeEventArgs
{
    int     Width;          //!< ウィンドウの横幅.
    int     Height;         //!< ウィンドウの縦幅.
    float   AspectRatio;    //!< ウィンドウのアスペクト比.

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    ResizeEventArgs()
    : Width         (0)
    , Height        (0)
    , AspectRatio   (0.0f)
    { /* DO_NOTHING */ }
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// FrameEventArgs
///////////////////////////////////////////////////////////////////////////////////////////////////
struct FrameEventArgs
{
    double  UpTimeSec;      //!< アプリの起動時間です.
    double  ElapsedSec;     //!< 前のフレームからの経過時間です.
    float   FramePerSec;    //!< FPSです.
    bool    IsStopDraw;     //!< 描画停止中かどうか.

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    FrameEventArgs()
    : UpTimeSec  (0)
    , ElapsedSec (0)
    , FramePerSec(0.0f)
    , IsStopDraw (false)
    { /* DO_NOTHING */ }
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// DropEventArgs
///////////////////////////////////////////////////////////////////////////////////////////////////
struct DropEventArgs
{
    wchar_t**     Files;          //!< ファイル名リストです. 
    uint32_t      FileCount;      //!< ファイル数です.

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    DropEventArgs()
    : Files     (nullptr)
    , FileCount (0)
    { /* DO_NOTHING */ }
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// AppBase class
///////////////////////////////////////////////////////////////////////////////////////////////////
class AppBase
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param[in]      title       アプリケーションのタイトル名.
    //! @param[in]      width       ウィンドウの横幅.
    //! @param[in]      height      ウィンドウの縦幅.
    //! @param[in]      hIcon       アイコンハンドル.
    //! @param[in]      hMenu       メニューハンドル.
    //! @param[in]      hAccel      アクセレレータテーブル.
    //---------------------------------------------------------------------------------------------
    AppBase( 
        LPWSTR  title,
        int     width,
        int     height,
        HICON   hIcon,
        HMENU   hMenu,
        HACCEL  hAccel);

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~AppBase();

    //---------------------------------------------------------------------------------------------
    //! @brief      アプリケーションを実行します.
    //---------------------------------------------------------------------------------------------
    void Run();

    //---------------------------------------------------------------------------------------------
    //! @brief      フォーカスを持つかどうかチェックします.
    //---------------------------------------------------------------------------------------------
    bool HasFocus() const;

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================
    HINSTANCE                   m_hInst;                    //!< インスタンスハンドルです.
    HWND                        m_hWnd;                     //!< ウィンドウハンドルです.
    LPWSTR                      m_Title;                    //!< タイトル名です.
    HICON                       m_hIcon;                    //!< アイコンです.
    HMENU                       m_hMenu;                    //!< メニューです.
    HACCEL                      m_hAccel;                   //!< アクセレレータテーブルです.
    int                         m_Width;                    //!< ウィンドウの横幅です.
    int                         m_Height;                   //!< ウィンドウの縦幅です.
    float                       m_AspectRatio;              //!< ウィンドウのアスペクト比です.
    StepTimer                   m_StepTimer;                //!< タイマーです.

    //=============================================================================================
    // protected methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化時の処理です.
    //!
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    virtual bool OnInit();

    //---------------------------------------------------------------------------------------------
    //! @brief      終了時の処理です.
    //---------------------------------------------------------------------------------------------
    virtual void OnTerm();

    //---------------------------------------------------------------------------------------------
    //! @brief      フレーム遷移時の処理です.
    //!
    //! @param[in]      args            フレームイベント引数.
    //---------------------------------------------------------------------------------------------
    virtual void OnFrameMove( const FrameEventArgs& args );

    //---------------------------------------------------------------------------------------------
    //! @brief      フレーム描画時の処理です.
    //!
    //! @param[in]      args            フレームイベント引数.
    //---------------------------------------------------------------------------------------------
    virtual void OnFrameRender( const FrameEventArgs& args );

    //---------------------------------------------------------------------------------------------
    //! @brief      リサイズ時の処理です.
    //!
    //! @param[in]      args            リサイズイベント引数.
    //---------------------------------------------------------------------------------------------
    virtual void OnResize( const ResizeEventArgs& args );

    //---------------------------------------------------------------------------------------------
    //! @brief      キーの処理です.
    //!
    //! @param[in]      args            キーイベント引数.
    //---------------------------------------------------------------------------------------------
    virtual void OnKey( const KeyEventArgs& args );

    //---------------------------------------------------------------------------------------------
    //! @brief      マウスの処理です.
    //!
    //! @param[in]      args            マウスイベント引数.
    //---------------------------------------------------------------------------------------------
    virtual void OnMouse( const MouseEventArgs& args );

    //---------------------------------------------------------------------------------------------
    //! @brief      ドロップ時の処理です.
    //!
    //! @param[in]      args            ドロップイベント引数.
    //---------------------------------------------------------------------------------------------
    virtual void OnDrop( const DropEventArgs& args );

    //---------------------------------------------------------------------------------------------
    //! @brief      メッセージプロシージャの処理です.
    //!
    //! @param[in]      hWnd            ウィンドウハンドル.
    //! @param[in]      msg             メッセージ.
    //! @param[in]      wp              メッセージの追加情報.
    //! @param[in]      lp              メッセージの追加情報.
    //---------------------------------------------------------------------------------------------
    virtual void OnMsgProc( HWND hWnd, UINT msg, WPARAM wp, LPARAM lp );

    //---------------------------------------------------------------------------------------------
    //! @brief      描画停止フラグを設定します.
    //!
    //! @param[in]      isStopDraw          描画を停止する場合は true を指定.
    //---------------------------------------------------------------------------------------------
    void StopDraw( bool isStopDraw );

    //---------------------------------------------------------------------------------------------
    //! @brief      描画停止フラグを取得します.
    //!
    //! @retval true    描画停止中です.
    //! @retval false   描画有効です.
    //---------------------------------------------------------------------------------------------
    bool IsStopDraw() const;    
    
    //---------------------------------------------------------------------------------------------
    //! @brief      フレームカウントを取得します.
    //!
    //! @return     フレームカウントを返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t GetFrameCount() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      0.5秒ごとのFPSを取得します.
    //!
    //! @return     0.5秒ごとのFPSを返却します.
    //! @note       各フレームにおける瞬間の FPS を取得する場合は FrameEventArgs から取得してください.
    //---------------------------------------------------------------------------------------------
    float GetFramePerSec() const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    std::atomic<uint32_t>   m_FrameCount;      //!< フレームカウント.
    std::atomic<float>      m_FramePerSec;     //!< 0.5秒ごとのFPS.
    std::atomic<bool>       m_IsStopDraw;      //!< 描画停止フラグです.
    double                  m_LastUpdateSec;   //!< 最後の更新時間.

    //=============================================================================================
    // private methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      アプリケーションの初期化処理です.
    //!
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool InitApp();

    //---------------------------------------------------------------------------------------------
    //! @brief      アプリケーションの終了処理です.
    //---------------------------------------------------------------------------------------------
    void TermApp();

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウの初期化処理です.
    //!
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool InitWnd();

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウの終了処理です.
    //---------------------------------------------------------------------------------------------
    void TermWnd();

    //---------------------------------------------------------------------------------------------
    //! @brief      メインループ処理です.
    //---------------------------------------------------------------------------------------------
    void MainLoop();

    //---------------------------------------------------------------------------------------------
    //! @brief      キーイベントを処理します.
    //!
    //! @param[in]      param       キーイベント引数です.
    //! @note       このメソッドは内部処理で, OnKey() を呼び出します.
    //!             また，このメソッドはウィンドウプロシージャからのアクセス専用メソッドですので，
    //!             アプリケーション側で呼び出しを行わないでください.
    //---------------------------------------------------------------------------------------------
    void DoKeyEvent( const KeyEventArgs& param );

    //---------------------------------------------------------------------------------------------
    //! @brief      リサイズイベントを処理します.
    //!
    //! @param[in]      param       リサイズイベント引数です.
    //! @note       このメソッドは内部処理で, OnResize() を呼び出します.
    //!             また，このメソッドはウィンドウプロシージャからのアクセス専用メソッドですので，
    //!             アプリケーション側で呼び出しを行わないでください.
    //---------------------------------------------------------------------------------------------
    void DoResizeEvent( const ResizeEventArgs& param );

    //---------------------------------------------------------------------------------------------
    //! @brief      マウスイベントを処理します.
    //!
    //! @param[in]      param       マウスイベント引数です.
    //! @note       このメソッドは内部処理で, OnMouse() を呼び出します.
    //!             また，このメソッドはウィンドウプロシージャからのアクセス専用メソッドですので，
    //!             アプリケーション側で呼び出しを行わないでください.
    //---------------------------------------------------------------------------------------------
    void DoMouseEvent( const MouseEventArgs& param );

    //---------------------------------------------------------------------------------------------
    //! @brief      ドロップイベントを処理します.
    //!
    //! @param[in]      param       ドロップイベント引数です.
    //! @note       このメソッドは内部処理で, OnDrop() を呼び出します.
    //!             また，このメソッドはウィンドウプロシージャからのアクセス専用メソッドですので，
    //!             アプリケーション側で呼び出しを行わないでください.
    //---------------------------------------------------------------------------------------------
    void DoDropEvent( const DropEventArgs& param );

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウプロシージャです.
    //!
    //! @param[in]      hWnd        ウィンドウハンドル.
    //! @param[in]      msg         メッセージ.
    //! @param[in]      wp          メッセージの追加情報.
    //! @param[in]      lp          メッセージの追加情報.
    //---------------------------------------------------------------------------------------------
    static LRESULT CALLBACK MsgProc( HWND hWnd, UINT msg, WPARAM wp, LPARAM lp );
};

} // namespace asdx
