﻿//-------------------------------------------------------------------------------------------------
// File : a3dResBMP.h
// Desc : Bitmap Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>
#include <asdxILoadable.h>
#include <asdxIDisposable.h>


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// BMP_COMPRESSION_TYPE enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum BMP_COMPRESSION_TYPE
{
    BMP_COMPRESSION_RGB       = 0,      // 無圧縮.
    BMP_COMPRESSION_RLE8      = 1,      // RLE圧縮 8 bits/pixel.
    BMP_COMPRESSION_RLE4      = 2,      // RLE圧縮 4 bits/pixel.
    BMP_COMPRESSION_BITFIELDS = 3,      // ビットフィールド.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// BMP_FILE_HEADER structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct BMP_FILE_HEADER
{
    u16     Type;           // ファイルタイプ 'BM'
    u32     Size;           // ファイルサイズ.
    u16     Reserved1;      // 予約領域 (0固定).
    u16     Reserved2;      // 予約領域 (0固定).
    u32     OffBits;        // ファイル先頭から画像データまでのオフセット.
};
#pragma pack( pop )

///////////////////////////////////////////////////////////////////////////////////////////////////
// BMP_INFO_HEADER structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct BMP_INFO_HEADER
{
    u32     Size;           // ヘッダサイズ (40固定).
    s32     Width;          // 画像の横幅.
    s32     Height;         // 画像の縦幅.
    u16     Planes;         // プレーン数 (1固定).
    u16     BitCount;       // 1ピクセルあたりのビット数.
    u32     Compression;    // 圧縮形式.
    u32     SizeImage;      // 画像データ部のサイズ.
    s32     XPelsPerMeter;  // 横方向の解像度.
    s32     YPelsPerMeter;  // 縦方向の解像度.
    u32     ClrUsed;        // 格納されているパレット数.
    u32     ClrImportant;   // 重要なパレットのインデックス.
};
#pragma pack( pop )


///////////////////////////////////////////////////////////////////////////////////////////////////
// BMP_CORE_HEADER structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct BMP_CORE_HEADER
{
    u32     Size;
    u16     Width;
    u16     Height;
    u16     Planes;
    u16     BitCount;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// ResBMP class
///////////////////////////////////////////////////////////////////////////////////////////////////
class ResBMP : public ILoadable, public IDisposable
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    ///////////////////////////////////////////////////////////////////////////////////////////////
    // Format
    ///////////////////////////////////////////////////////////////////////////////////////////////
    enum Format
    {
        Format_RGB = 0,     // RGB形式  3コンポーネント.
        Format_RGBX,        // RGBX形式 4コンポーネント.
    };

    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    ResBMP();

    //---------------------------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param[in]      value       コピー元の値です.
    //---------------------------------------------------------------------------------------------
    ResBMP( const ResBMP& value );

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~ResBMP();

    //---------------------------------------------------------------------------------------------
    //! @brief      ファイルから読み込みします.
    //!
    //! @param[in]      filename        ファイル名.
    //! @retval true    読み込みに成功.
    //! @retval false   読み込みに失敗.
    //---------------------------------------------------------------------------------------------
    bool Load( const wchar_t* filename ) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリを解放します.
    //---------------------------------------------------------------------------------------------
    void Dispose() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      画像の横幅を取得します.
    //!
    //! @return     画像の横幅を返却します.
    //---------------------------------------------------------------------------------------------
    const u32 GetWidth() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      画像の縦幅を取得します.
    //!
    //! @return     画像の縦幅を返却します.
    //---------------------------------------------------------------------------------------------
    const u32 GetHeight() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      フォーマットを取得します.
    //!
    //! @return     フォーマットを返却します.
    //---------------------------------------------------------------------------------------------
    const u32 GetFormat() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      ピクセルデータを取得します.
    //!
    //! @return     ピクセルデータを返却します.
    //---------------------------------------------------------------------------------------------
    const u8* GetPixels() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param[in]      value       代入する値です.
    //! @return     代入結果を返却します.
    //---------------------------------------------------------------------------------------------
    ResBMP& operator = ( const ResBMP& value );

    //---------------------------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //! 
    //! @param[in]      value       比較する値です.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //---------------------------------------------------------------------------------------------
    bool operator == ( const ResBMP& value ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param[in]      value       比較する値です.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //---------------------------------------------------------------------------------------------
    bool operator != ( const ResBMP& value ) const;

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // protected methods.
    //=============================================================================================
    /* NOTHING */

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    u32     m_Width;        //!< 画像の横幅です.
    u32     m_Height;       //!< 画像の縦幅です.
    u32     m_Format;       //!< フォーマットです.
    u8*     m_pPixels;      //!< ピクセルデータです.
    u32     m_HashKey;      //!< ハッシュキーです.

    //=============================================================================================
    // private methods.
    //=============================================================================================
    /* NOTHING */
};


} // namespace asdx