﻿//-------------------------------------------------------------------------------------------------
// File : asdxLogger.h
// Desc : Logger Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxTypedef.h>


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// LogLevel enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum class LogLevel : u32
{
    Verbose = 0,          //!< VERBOSEレベル (白).
    Info,                 //!< INFOレベル    (緑).
    Debug,                //!< DEBUGレベル   (青).
    Warning,              //!< WRARNINGレベル(黄).
    Error,                //!< ERRORレベル   (赤).
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// ILogger interface
///////////////////////////////////////////////////////////////////////////////////////////////////
struct ILogger
{
    //---------------------------------------------------------------------------------------------
    //! @brief      ログを出力します.
    //!
    //! @param[in]      level       ログレベルです.
    //! @param[in]      format      フォーマットです.
    //---------------------------------------------------------------------------------------------
    virtual void LogA( const LogLevel level, const char8* format, ... ) = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      ログを出力します.
    //!
    //! @param[in]      level       ログレベルです.
    //! @param[in]      format      フォーマットです.
    //---------------------------------------------------------------------------------------------
    virtual void LogW( const LogLevel level, const char16* format, ... ) = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      フィルタを設定します.
    //!
    //! @param[in]      filter      設定するフィルタ.
    //---------------------------------------------------------------------------------------------
    virtual void SetFilter( const LogLevel filter ) = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      設定されているフィルタを取得します.
    //!
    //! @return     設定されているフィルタを取得します.
    //---------------------------------------------------------------------------------------------
    virtual LogLevel GetFilter() = 0;
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// SystemLogger class
///////////////////////////////////////////////////////////////////////////////////////////////////
class SystemLogger : public ILogger, private NonCopyable
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      唯一のインスタンスを取得します.
    //!
    //! @return     シングルトンインスタンスを返却します.
    //---------------------------------------------------------------------------------------------
    static SystemLogger& GetInstance();

    //---------------------------------------------------------------------------------------------
    //! @brief      ログを出力します.
    //!
    //! @param[in]      level       ログレベルです.
    //! @param[in]      format      フォーマットです.
    //---------------------------------------------------------------------------------------------
    void LogA( const LogLevel level, const char8* format, ... );

    //---------------------------------------------------------------------------------------------
    //! @brief      ログを出力します.
    //!
    //! @param[in]      level       ログレベルです.
    //! @param[in]      format      フォーマットです.
    //---------------------------------------------------------------------------------------------
    void LogW( const LogLevel level, const char16* format, ... );

    //---------------------------------------------------------------------------------------------
    //! @brief      フィルタを設定します.
    //!             フィルタを設定したもののみが，ログに出力されるようになります.
    //!
    //! @param[in]      filter      設定するフィルタ.
    //---------------------------------------------------------------------------------------------
    void SetFilter( const LogLevel filter );

    //---------------------------------------------------------------------------------------------
    //! @brief      設定されているフィルタを取得します.
    //!
    //! @return     設定されているフィルタを取得します.
    //---------------------------------------------------------------------------------------------
    LogLevel  GetFilter();

protected:
    //=============================================================================================
    // protected variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // protected methods.
    //=============================================================================================
    /* NOTHING */

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    static SystemLogger     s_Instance;     //!< シングルトンインスタンスです.
    LogLevel                m_Filter;       //!< フィルターです.

    //=============================================================================================
    // private methods.
    //=============================================================================================
    SystemLogger();
};


} // namespace asdx


//-------------------------------------------------------------------------------------------------
// Macros
//-------------------------------------------------------------------------------------------------
#ifndef DLOGA
  #if defined(DEBUG) || defined(_DEBUG)
    #define DLOGA( fmt, ... )      asdx::SystemLogger::GetInstance().LogA( asdx::LogLevel::Debug, "[File: %s, Line: %d] "fmt"\n", __FILE__, __LINE__, ##__VA_ARGS__ )
  #else
    #define DLOGA( fmt, ... )      ((void)0)
  #endif//defined(DEBUG) || defined(_DEBUG)
#endif//DLOGA

#ifndef DLOGW
  #if defined(DEBUG) || defined(_DEBUG)
    #define DLOGW( fmt, ... )      asdx::SystemLogger::GetInstance().LogW( asdx::LogLevel::Debug, ASDX_WIDE("[File: %s, Line: %d] ") ASDX_WIDE(fmt) ASDX_WIDE("\n"), ASDX_WIDE(__FILE__), __LINE__, ##__VA_ARGS__ )
  #else
    #define DLOGW( fmt, ... )      ((void)0)
  #endif//defined(DEBUG) || defined(_DEBUG)
#endif//DLOGW

#ifndef ILOGA
#define ILOGA( fmt, ... )      asdx::SystemLogger::GetInstance().LogA( asdx::LogLevel::Info, fmt "\n", ##__VA_ARGS__ )
#endif//ILOGA

#ifndef ILOGW
#define ILOGW( fmt, ... )      asdx::SystemLogger::GetInstance().LogW( asdx::LogLevel::Info, ASDX_WIDE(fmt) ASDX_WIDE("\n"), ##__VA_ARGS__ );
#endif//ILOGW

#ifndef ELOGA
#define ELOGA( fmt, ... )      asdx::SystemLogger::GetInstance().LogA( asdx::LogLevel::Error, "[File: %s, Line: %d] " fmt "\n", __FILE__, __LINE__, ##__VA_ARGS__ )
#endif//ELOGA

#ifndef ELOGW
#define ELOGW( fmt, ... )      asdx::SystemLogger::GetInstance().LogW( asdx::LogLevel::Error, ASDX_WIDE("[File: %s, Line: %d] ") ASDX_WIDE(fmt) ASDX_WIDE("\n"), ASDX_WIDE(__FILE__), __LINE__, ##__VA_ARGS__ )
#endif//ELOGW

#if defined(UNICODE) || defined(_UNICODE)
    #define DLOG        DLOGW
    #define ILOG        ILOGW
    #define ELOG        ELOGW
#else
    #define DLOG        DLOGA
    #define ILOG        ILOGA
    #define ELOG        ELOGA
#endif

