﻿//-------------------------------------------------------------------------------------------------
// File : asdxRandom.cpp
// Desc : Random Number Generater Moudle
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxMath.h>


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// Random class
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
Random::Random( s32 seed )
{
    SetSeed( seed );
}

//-------------------------------------------------------------------------------------------------
//      コピーコンストラクタです.
//-------------------------------------------------------------------------------------------------
Random::Random( const Random& random )
{
    m_X = random.m_X;
    m_Y = random.m_Y;
    m_Z = random.m_Z;
    m_W = random.m_W;
}

//-------------------------------------------------------------------------------------------------
//      デストラクタです.
//-------------------------------------------------------------------------------------------------
Random::~Random()
{
    /* DO_NOTHING */
}

//-------------------------------------------------------------------------------------------------
//      ランダム種を設定します.
//-------------------------------------------------------------------------------------------------
void Random::SetSeed( s32 seed )
{
    m_X = 123456789;
    m_Y = 362436069;
    m_Z = 521288629;
    m_W = ( seed <= 0 ) ? 88675123 : seed;
}

//-------------------------------------------------------------------------------------------------
//      乱数をu32型として取得します.
//-------------------------------------------------------------------------------------------------
u32 Random::GetAsU32()
{
    u32 t = m_X ^ ( m_X << 11 );
    m_X = m_Y;
    m_Y = m_Z;
    m_Z = m_W;
    m_W = ( m_W ^ ( m_W >> 19 ) ) ^ ( t ^ ( t >> 8 ) );
    return m_W;
}

//-------------------------------------------------------------------------------------------------
//      乱数をs32型として取得します.
//-------------------------------------------------------------------------------------------------
s32 Random::GetAsS32()
{
    s32 x = GetAsU32() & 0x7fffffff;
    return x;
}

//-------------------------------------------------------------------------------------------------
//      指定された値範囲までの乱数をs32型として取得します.
//-------------------------------------------------------------------------------------------------
s32 Random::GetAsS32( s32 a )
{
    // 符号ビット切り捨て.
    s32 x = GetAsU32() & 0x7fffffff;
    x %= a;
    return x;
}

//-------------------------------------------------------------------------------------------------
//      指定された値範囲で乱数をs32型として取得します.
//-------------------------------------------------------------------------------------------------
s32 Random::GetAsS32( s32 a, s32 b )
{
    s32 x = GetAsU32() & 0x7fffffff;
    x %= ( b - a );
    x += a;
    return x;
}

//-------------------------------------------------------------------------------------------------
//      乱数をf32型として取得します.
//-------------------------------------------------------------------------------------------------
f32 Random::GetAsF32()
{
    return static_cast<f32>( GetAsU32() ) / U32_MAX;
}

//-------------------------------------------------------------------------------------------------
//		指定された値範囲までの乱数をf32型として取得します.
//-------------------------------------------------------------------------------------------------
f32 Random::GetAsF32( f32 a )
{
    f32 x = GetAsF32();
    x *= a;
    return x;
}

//-------------------------------------------------------------------------------------------------
//      指定された値範囲の乱数をf32型として取得します.
//-------------------------------------------------------------------------------------------------
f32 Random::GetAsF32( f32 a, f32 b )
{
    f32 x = GetAsF32();
    x *= ( b - a );
    x += a;
    return x;
}

//-------------------------------------------------------------------------------------------------
//      乱数をf64型として取得します.
//-------------------------------------------------------------------------------------------------
f64 Random::GetAsF64()
{
    return static_cast<f64>( GetAsU32() ) / U32_MAX;
}

//-------------------------------------------------------------------------------------------------
//      指定された値範囲までの乱数をf64型として取得します.
//-------------------------------------------------------------------------------------------------
f64 Random::GetAsF64( f64 a )
{
    f64 x = GetAsF64();
    x *= a;
    return x;
}

//-------------------------------------------------------------------------------------------------
//      指定された値範囲で乱数をf64型として取得します.
//-------------------------------------------------------------------------------------------------
f64 Random::GetAsF64( f64 a, f64 b )
{
    f64 x = GetAsF64();
    x *= ( b - a );
    x *= a;
    return x;
}

//-------------------------------------------------------------------------------------------------
//      代入演算子です.
//-------------------------------------------------------------------------------------------------
Random& Random::operator = ( const Random& random )
{
    m_X = random.m_X;
    m_Y = random.m_Y;
    m_Z = random.m_Z;
    m_W = random.m_W;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      等価演算子です.
//-------------------------------------------------------------------------------------------------
bool Random::operator == ( const Random& random ) const
{
    return ( this == &random );
}

//-------------------------------------------------------------------------------------------------
//      非等価演算子です.
//-------------------------------------------------------------------------------------------------
bool Random::operator != ( const Random& random ) const
{
    return  ( this != &random );
}

} // namespace asdx
