﻿//-------------------------------------------------------------------------------------------------
// File : asdxMouse.cpp
// Desc : Mouse Module.
// Copyright(c) Project Ausra. All right reserved.
//-------------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxHid.h>
#include <cstring>
#include <cassert>
#include <Windows.h>


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// Mouse class
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
Mouse::Mouse()
: m_CursorX    ( -1 )
, m_CursorY    ( -1 )
, m_PrevCursorX( -1 )
, m_PrevCursorY( -1 )
, m_Index      ( 0 )
{ memset( m_Button, false, sizeof(bool) * NUM_MOUSE_BUTTON * 2 ); }

//-------------------------------------------------------------------------------------------------
//      デストラクタです.
//-------------------------------------------------------------------------------------------------
Mouse::~Mouse()
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      マウスステートを更新します.
//-------------------------------------------------------------------------------------------------
void Mouse::UpdateState()
{
    m_Index = 1 - m_Index;
    m_PrevCursorX = m_CursorX;
    m_PrevCursorY = m_CursorY;

    POINT pt;
    GetCursorPos( &pt );
    m_CursorX = s32( pt.x );
    m_CursorY = s32( pt.y );

    m_Button[ m_Index ][ MOUSE_BUTTON_L ]  = ( GetAsyncKeyState( VK_LBUTTON )  & 0x8000 ) ? true : false;
    m_Button[ m_Index ][ MOUSE_BUTTON_R ]  = ( GetAsyncKeyState( VK_RBUTTON )  & 0x8000 ) ? true : false;
    m_Button[ m_Index ][ MOUSE_BUTTON_M ]  = ( GetAsyncKeyState( VK_MBUTTON )  & 0x8000 ) ? true : false;
    m_Button[ m_Index ][ MOUSE_BUTTON_X1 ] = ( GetAsyncKeyState( VK_XBUTTON1 ) & 0x8000 ) ? true : false;
    m_Button[ m_Index ][ MOUSE_BUTTON_X2 ] = ( GetAsyncKeyState( VK_XBUTTON2 ) & 0x8000 ) ? true : false;
}

//-------------------------------------------------------------------------------------------------
//! @brief      マウスの状態をリセットします.
//-------------------------------------------------------------------------------------------------
void Mouse::ResetState()
{
    m_CursorX     = -1;
    m_CursorY     = -1;
    m_PrevCursorX = -1;
    m_PrevCursorY = -1;
    memset( m_Button, false, sizeof(bool) * NUM_MOUSE_BUTTON * 2 );
}

//-------------------------------------------------------------------------------------------------
//      マウスのX座標を取得します.
//-------------------------------------------------------------------------------------------------
s32 Mouse::GetCursorX() const
{ return m_CursorX; }

//-------------------------------------------------------------------------------------------------
//      マウスのY座標を取得します.
//-------------------------------------------------------------------------------------------------
s32 Mouse::GetCursorY() const
{ return m_CursorY; }

//-------------------------------------------------------------------------------------------------
//      前のマウスのX座標を取得します.
//-------------------------------------------------------------------------------------------------
s32 Mouse::GetPrevCursorX() const
{ return m_PrevCursorX; }

//-------------------------------------------------------------------------------------------------
//      前のマウスのY座標を取得します.
//-------------------------------------------------------------------------------------------------
s32 Mouse::GetPrevCursorY() const
{ return m_PrevCursorY; }

//-------------------------------------------------------------------------------------------------
//      現在のカーソルと前のカーソルのX座標の差分を取得します.
//-------------------------------------------------------------------------------------------------
s32 Mouse::GetCursorDiffX() const
{ return m_CursorX - m_PrevCursorX; }

//-------------------------------------------------------------------------------------------------
//      現在のカーソルと前のカーソルのY座標の差分を取得します.
//-------------------------------------------------------------------------------------------------
s32 Mouse::GetCursorDiffY() const
{ return m_CursorY - m_PrevCursorY; }

//-------------------------------------------------------------------------------------------------
//      キーが押されたかどうかチェックします.
//-------------------------------------------------------------------------------------------------
bool Mouse::IsPush( const u32 button ) const
{
    assert( button < NUM_MOUSE_BUTTON );
    return m_Button[ m_Index ][ button ];
}

//-------------------------------------------------------------------------------------------------
//      キーが押されっぱなしかどうかチェックします.
//-------------------------------------------------------------------------------------------------
bool Mouse::IsDown( const u32 button ) const
{
    assert( button < NUM_MOUSE_BUTTON );
    return m_Button[ m_Index ][ button ] & ( !m_Button[ 1 - m_Index ][ button ] );
}

//-------------------------------------------------------------------------------------------------
//      ドラッグ中かどうかチェックします.
//-------------------------------------------------------------------------------------------------
bool Mouse::IsDrag( const u32 button ) const
{
    assert( button < NUM_MOUSE_BUTTON );
    return m_Button[ m_Index ][ button ] & m_Button[ 1 - m_Index ][ button ];
}

} // namespace asdx
