﻿//-------------------------------------------------------------------------------------------------
// File : asdxMath.inl
// Desc : Math Module.
// Copyright(c) Project Asura All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// Functions
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      ラジアンに変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 ToRadian( f32 degree )
{ return degree * ( F_PI / 180.0f ); }

//-------------------------------------------------------------------------------------------------
//      ラジアンに変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f64 ToRadian( f64 degree )
{ return degree * ( D_PI / 180.0 ); }

//-------------------------------------------------------------------------------------------------
//      度に変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 ToDegree( f32 radian )
{ return radian * ( 180.0f / F_PI ); }

//-------------------------------------------------------------------------------------------------
//      度に変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f64 ToDegree( f64 radian )
{ return radian * ( 180.0 / D_PI ); }

//-------------------------------------------------------------------------------------------------
//      ゼロかどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsZero( f32 value )
{ return fabs( value ) <= F_EPSILON; }

//-------------------------------------------------------------------------------------------------
//      ゼロかどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsZero( f64 value )
{ return abs( value ) <= D_EPSILON; }

//-------------------------------------------------------------------------------------------------
//      値が等しいかどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsEqual( f32 value1, f32 value2 )
{ return fabs( value1 - value2 ) <= F_EPSILON; }

//-------------------------------------------------------------------------------------------------
//      値が等しいかどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsEqual( f64 value1, f64 value2 )
{ return abs( value1 - value2 ) <= D_EPSILON; }

//-------------------------------------------------------------------------------------------------
//      非数かどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsNan( f32 value )
{ return ( value != value ); }

//-------------------------------------------------------------------------------------------------
//      非数かどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsNan( f64 value )
{ return ( value != value ); }

//-------------------------------------------------------------------------------------------------
//      無限大かどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsInf( f32 value )
{
    // ビット列に変換して，指数部がすべて 1 かどうかチェック.
    u32 f = *reinterpret_cast< u32* >( &value );
    return ((f & 0x7f800000) == 0x7f800000) && (value == value);
}

//-------------------------------------------------------------------------------------------------
//      無限大かどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool IsInf( f64 value )
{
    // ビット列に変換して，指数部がすべて 1 かどうかチェック.
    u64 d = *reinterpret_cast<u64*>( &value );
    return ((d & 0x7ff0000000000000) == 0x7ff0000000000000) && (value == value);
}

//-------------------------------------------------------------------------------------------------
//      階乗計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u32 Fact( u32 number )
{
    u32 result = 1;
    for( u32 i=1; i<=number; ++i )
    { result *= i; }
    return result;
}

//-------------------------------------------------------------------------------------------------
//      2重階乗を計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u32 DblFact( u32 number )
{
    u32 result = 1;
    u32 start = ( ( number % 2 ) == 0 ) ? 2 : 1;
    for( u32 i=start; i<=number; i+=2 )
    { result *= i; }
    return result;
}

//-------------------------------------------------------------------------------------------------
//      順列を計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u32 Perm( u32 n, u32 r )
{
    assert( n >= r );
    return Fact( n ) / Fact( n - r );
}

//-------------------------------------------------------------------------------------------------
//      組み合わせを計算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
u32 Comb( u32 n, u32 r )
{
    assert( n >= r );
    return Fact( n ) / ( Fact( n - r ) * Fact( r ) );
}

//-------------------------------------------------------------------------------------------------
//      32bit 浮動小数から 16bit 浮動小数に変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
f16 F32ToF16( f32 value )
{
    f16 result;

    // ビット列を崩さないままu32型に変換.
    u32 bit = *reinterpret_cast<u32*>( &value );

    // f32表現の符号bitを取り出し.
    u32 sign   = ( bit & 0x80000000U) >> 16U;

    // 符号部を削ぎ落す.
    bit     = bit & 0x7FFFFFFFU;

    // f16として表現する際に値がデカ過ぎる場合は，無限大にクランプ.
    if ( bit > 0x47FFEFFFU)
    { result = 0x7FFFU; }
    else
    {
        // 正規化されたf16として表現するために小さすぎる値は正規化されていない値に変換.
        if ( bit < 0x38800000U)
        {
            u32 shift = 113U - ( bit >> 23U);
            bit    = (0x800000U | ( bit & 0x7FFFFFU)) >> shift;
        }
        else
        {
            // 正規化されたf16として表現するために指数部に再度バイアスをかける
            bit += 0xC8000000U;
        }

        // f16型表現にする.
        result = (( bit + 0x0FFFU + (( bit >> 13U) & 1U)) >> 13U) & 0x7FFFU; 
    }

    // 符号部を付け足して返却.
    return static_cast<f16>( result | sign );
}

//-------------------------------------------------------------------------------------------------
//      16bit 浮動小数から　32bit 浮動小数に変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
f32 F16ToF32( f16 value )
{
    u32 exponent;
    u32 result;

    // 仮数
    u32 mantissa = static_cast<u32>( value & 0x03FF );

    // 正規化済みの場合.
    if ( ( value & 0x7C00 ) != 0 )
    {
        // 指数部を計算.
        exponent = static_cast<u32>( ( value >> 10 ) & 0x1F );
    }
    // 正規化されていない場合.
    else if ( mantissa != 0 )
    {
        // 結果となるf32で値を正規化する.
        exponent = 1;

        do {
            exponent--;
            mantissa <<= 1;
        } while ( ( mantissa & 0x0400 ) == 0);

        mantissa &= 0x03FF;
    }
    // 値がゼロの場合.
    else
    {
        // 指数部を計算.
        exponent = (u32)-112;
    }

    result = ( ( value & 0x8000 ) << 16) | // 符号部.
             ( ( exponent + 112 ) << 23) | // 指数部.
             ( mantissa << 13 );           // 仮数部.

    return *reinterpret_cast<f32*>( &result );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Lerp( f32 a, f32 b, f32 amount )
{
    return a + amount * ( b - a );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f64 Lerp( f64 a, f64 b, f64 amount )
{
    return a + amount * ( b - a );
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Vector2 structure
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2::Vector2()
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタ.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2::Vector2( const f32* pf )
{
    assert( pf != nullptr );
    x = pf[ 0 ];
    y = pf[ 1 ];
}

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタ.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2::Vector2( f32 nx, f32 ny )
: x( nx )
, y( ny )
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      f32*へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2::operator f32 *()
{ return static_cast<f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      const f32*へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2::operator const f32 *() const
{ return static_cast<const f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      加算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2& Vector2::operator += ( const Vector2& v )
{
    x += v.x;
    y += v.y;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      減算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2& Vector2::operator -= ( const Vector2& v )
{
    x -= v.x;
    y -= v.y;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      乗算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2& Vector2::operator *= ( f32 f )
{
    x *= f;
    y *= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      除算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2& Vector2::operator /= ( f32 f )
{
    assert( !IsZero( f ) );
    x /= f;
    y /= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2& Vector2::operator = ( const Vector2& value )
{
    x = value.x;
    y = value.y;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      正符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::operator + () const
{ return (*this); }

//-------------------------------------------------------------------------------------------------
//      負符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::operator - () const
{ return Vector2( -x, -y ); }

//-------------------------------------------------------------------------------------------------
//      加算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::operator + ( const Vector2& v ) const
{ return Vector2( x + v.x, y + v.y ); }

//-------------------------------------------------------------------------------------------------
//      減算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::operator - ( const Vector2& v ) const
{ return Vector2( x - v.x, y - v.y ); }

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::operator * ( f32 f ) const
{ return Vector2( x * f, y * f ); }

//-------------------------------------------------------------------------------------------------
//      除算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::operator / ( f32 f ) const
{
    assert( !IsZero( f ) );
    return Vector2( x / f, y / f );
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 operator * ( f32 f, const Vector2& v )
{ return Vector2( f * v.x, f * v.y ); }

//-------------------------------------------------------------------------------------------------
//      等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Vector2::operator == ( const Vector2& v ) const
{ 
    return IsEqual( x, v.x )
        && IsEqual( y, v.y );
}

//-------------------------------------------------------------------------------------------------
//      非等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Vector2::operator != ( const Vector2& v ) const
{
    return !IsEqual( x, v.x )
        || !IsEqual( y, v.y );
}

//-------------------------------------------------------------------------------------------------
//      長さを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector2::Length() const
{ return sqrtf( x * x + y * y ); }

//-------------------------------------------------------------------------------------------------
//      長さの2乗を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector2::LengthSq() const
{ return ( x * x + y * y ); }

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2& Vector2::Normalize()
{
    auto mag = Length();
    assert( mag > 0.0f );
    x /= mag;
    y /= mag;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      ゼロ除算を考慮して正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2& Vector2::SafeNormalize( const Vector2& set )
{
    auto mag = Length();
    if ( mag > 0.0f )
    {
        x /= mag;
        y /= mag;
        return (*this);
    }

    x = set.x;
    y = set.y;
    return (*this);
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Vector2 Methods
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      各成分の絶対値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Abs( const Vector2& value )
{ 
    return Vector2(
        fabs( value.x ), 
        fabs( value.y ) 
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の絶対値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Abs( const Vector2 &value, Vector2 &result )
{ 
    result.x = fabs( value.x );
    result.y = fabs( value.y );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Clamp( const Vector2& value, const Vector2& a, const Vector2& b )
{
    return Vector2(
        asdx::Clamp( value.x, a.x, b.x ),
        asdx::Clamp( value.y, a.y, b.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Clamp( const Vector2& value, const Vector2& a, const Vector2& b, Vector2 &result )
{
    result.x = asdx::Clamp( value.x, a.x, b.x );
    result.y = asdx::Clamp( value.y, a.y, b.y );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を0～1に収めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Saturate( const Vector2& value )
{
    return Vector2(
        asdx::Saturate( value.x ),
        asdx::Saturate( value.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を0～1に収めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Saturate( const Vector2& value, Vector2& result )
{
    result.x = asdx::Saturate( value.x );
    result.y = asdx::Saturate( value.y );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector2::Distance( const Vector2& a, const Vector2& b )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    return sqrtf( X * X + Y * Y );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Distance( const Vector2 &a, const Vector2 &b, f32 &result )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    result = sqrtf( X * X + Y * Y );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離の2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector2::DistanceSq( const Vector2& a, const Vector2& b )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    return X * X + Y * Y;
}

//-------------------------------------------------------------------------------------------------
//      2点間距離の2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::DistanceSq( const Vector2 &a, const Vector2 &b, f32 &result )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    result = X * X + Y * Y;
}

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector2::Dot( const Vector2& a, const Vector2& b )
{ return ( a.x * b.x + a.y * b.y ); }

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Dot( const Vector2 &a, const Vector2 &b, f32 &result )
{ result = a.x * b.x + a.y * b.y; }

//-------------------------------------------------------------------------------------------------
//      正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Normalize( const Vector2& value )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    return Vector2(
        value.x / mag,
        value.y / mag 
    );
}

//-------------------------------------------------------------------------------------------------
//      正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Normalize( const Vector2 &value, Vector2 &result )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    result.x = value.x / mag;
    result.y = value.y / mag;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::SafeNormalize( const Vector2& value, const Vector2& set )
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        return Vector2(
            value.x / mag,
            value.y / mag 
        );
    }

    return set;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::SafeNormalize( const Vector2& value, const Vector2& set, Vector2& result )
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        result.x = value.x / mag;
        result.y = value.y / mag;
    }
    else
    {
        result.x = set.x;
        result.y = set.y;
    }
}

//-------------------------------------------------------------------------------------------------
//      交差角を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector2::ComputeCrossingAngle( const Vector2& a, const Vector2& b )
{
    auto d = a.Length() * b.Length();
    if ( d <= 0.0f )
    { return 0.0f; }

    auto c = Vector2::Dot( a, b ) / d;
    if ( c >= 1.0f ) 
    { return 0.0f; }

    if ( c <= -1.0f )
    { return F_PI; }

    return acosf( c );
}

//-------------------------------------------------------------------------------------------------
//      交差角を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::ComputeCrossingAngle( const Vector2 &a, const Vector2 &b, f32 &result )
{
    auto d = a.Length() * b.Length();
    if ( d <= 0.0f )
    {
        result = 0.0f;
        return;
    }

    auto c = Vector2::Dot( a, b ) / d;
    if ( c >= 1.0f ) 
    {
        result = 0.0f;
        return;
    }

    if ( c <= -1.0f )
    {
        result = F_PI;
        return;
    }

    result = acosf( c );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最小値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Min( const Vector2& a, const Vector2& b )
{ 
    return Vector2(
        asdx::Min( a.x, b.x ),
        asdx::Min( a.y, b.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最小値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Min( const Vector2 &a, const Vector2 &b, Vector2 &result )
{
    result.x = asdx::Min( a.x, b.x );
    result.y = asdx::Min( a.y, b.y );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最大値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Max( const Vector2& a, const Vector2& b )
{
    return Vector2(
        asdx::Max( a.x, b.x ),
        asdx::Max( a.y, b.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最大値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Max( const Vector2 &a, const Vector2 &b, Vector2 &result )
{
    result.x = asdx::Max( a.x, b.x );
    result.y = asdx::Max( a.y, b.y );
}

//-------------------------------------------------------------------------------------------------
//      反射ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Reflect( const Vector2& i, const Vector2& n )
{
    auto dot = n.x * i.x + n.y * i.y;
    return Vector2(
        i.x - ( 2.0f * n.x ) * dot,
        i.y - ( 2.0f * n.y ) * dot 
    );
}

//-------------------------------------------------------------------------------------------------
//      反射ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Reflect( const Vector2 &i, const Vector2 &n, Vector2 &result )
{
    auto dot = n.x * i.x + n.y * i.y;
    result.x = i.x - ( 2.0f * n.x ) * dot;
    result.y = i.y - ( 2.0f * n.y ) * dot;
}

//-------------------------------------------------------------------------------------------------
//      屈折ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Refract( const Vector2& i, const Vector2& n, f32 eta )
{
    auto cosi   = ( -i.x * n.x ) + ( -i.y * n.y );
    auto cost2  = 1.0f - eta * eta * ( 1.0f - cosi * cosi );
    auto sign   = Sign( cost2 );
    auto sqrtC2 = sqrtf( fabs( cost2 ) );
    auto coeff  = eta * cosi - sqrtC2;

    return Vector2(
        sign * ( eta * i.x + coeff * n.x ),
        sign * ( eta * i.y + coeff * n.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      屈折ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Refract( const Vector2 &i, const Vector2 &n, f32 eta, Vector2 &result )
{
    auto cosi   =  ( -i.x * n.x ) + ( -i.y * n.y );
    auto cost2  = 1.0f - eta * eta * ( 1.0f - cosi * cosi );
    auto sign   = Sign( cost2 );
    auto sqrtC2 = sqrtf( fabs( cost2 ) );
    auto coeff  = eta * cosi - sqrtC2;

    result.x = sign * ( eta * i.x + coeff * n.x );
    result.y = sign * ( eta * i.y + coeff * n.y );
}

//-------------------------------------------------------------------------------------------------
//      重心座標を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Barycentric
(
    const Vector2& a,
    const Vector2& b,
    const Vector2& c,
    f32            f,
    f32            g 
)
{
    return Vector2(
        a.x + f * ( b.x - a.x ) + g * ( c.x - a.x ),
        a.y + f * ( b.y - a.y ) + g * ( c.y - a.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      重心座標を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Barycentric
(
    const Vector2&  a,
    const Vector2&  b,
    const Vector2&  c,
    f32             f,
    f32             g,
    Vector2&        result
)
{
    result.x = a.x + f * ( b.x - a.x ) + g * ( c.x - a.x );
    result.y = a.y + f * ( b.y - a.y ) + g * ( c.y - a.y );
}

//-------------------------------------------------------------------------------------------------
//      エルミートスプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Hermite
(
    const Vector2&  a,
    const Vector2&  t1,
    const Vector2&  b,
    const Vector2&  t2,
    f32             amount
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    Vector2 result;
    if ( amount <= 0.0f )
    {
        result.x = a.x;
        result.y = a.y;
    }
    else if ( amount >= 1.0f )
    {
        result.x = b.x;
        result.y = b.y;
    }
    else
    {
        result.x = ( 2.0f * a.x - 2.0f * b.x + t2.x + t1.x ) * c3 + ( 3.0f * b.x - 3.0f * a.x - 2.0f * t1.x - t2.x ) * c3 + t1.x * amount + a.x;
        result.y = ( 2.0f * a.y - 2.0f * b.y + t2.y + t1.y ) * c3 + ( 3.0f * b.y - 3.0f * a.y - 2.0f * t1.y - t2.y ) * c3 + t1.y * amount + a.y;
    }
    return result;
}

//-------------------------------------------------------------------------------------------------
//      エルミートスプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Hermite
(
    const Vector2&  a,
    const Vector2&  t1,
    const Vector2&  b,
    const Vector2&  t2,
    f32             amount,
    Vector2&        result
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    if ( amount <= 0.0f )
    {
        result.x = a.x;
        result.y = a.y;
    }
    else if ( amount >= 1.0f )
    {
        result.x = b.x;
        result.y = b.y;
    }
    else
    {
        result.x = ( 2.0f * a.x - 2.0f * b.x + t2.x + t1.x ) * c3 + ( 3.0f * b.x - 3.0f * a.x - 2.0f * t1.x - t2.x ) * c3 + t1.x * amount + a.x;
        result.y = ( 2.0f * a.y - 2.0f * b.y + t2.y + t1.y ) * c3 + ( 3.0f * b.y - 3.0f * a.y - 2.0f * t1.y - t2.y ) * c3 + t1.y * amount + a.y;
    }
}

//-------------------------------------------------------------------------------------------------
//      Catmull-Rom スプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::CatmullRom
(
    const Vector2&  a,
    const Vector2&  b,
    const Vector2&  c,
    const Vector2&  d,
    f32             amount
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    return Vector2(
        ( 0.5f * ( 2.0f * b.x + ( c.x - a.x ) * amount + ( 2.0f * a.x - 5.0f * b.x + 4.0f * c.x - d.x ) * c2 + ( 3.0f * b.x - a.x - 3.0f * c.x + d.x ) * c3 ) ),
        ( 0.5f * ( 2.0f * b.y + ( c.y - a.y ) * amount + ( 2.0f * a.y - 5.0f * b.y + 4.0f * c.y - d.y ) * c2 + ( 3.0f * b.y - a.y - 3.0f * c.y + d.y ) * c3 ) )
    );
}

//-------------------------------------------------------------------------------------------------
//      Catmull-Rom スプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::CatmullRom
(
    const Vector2& a,
    const Vector2& b,
    const Vector2& c,
    const Vector2& d,
    f32            amount,
    Vector2&       result
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    result.x = ( 0.5f * ( 2.0f * b.x + ( c.x - a.x ) * amount + ( 2.0f * a.x - 5.0f * b.x + 4.0f * c.x - d.x ) * c2 + ( 3.0f * b.x - a.x - 3.0f * c.x + d.x ) * c3 ) );
    result.y = ( 0.5f * ( 2.0f * b.y + ( c.y - a.y ) * amount + ( 2.0f * a.y - 5.0f * b.y + 4.0f * c.y - d.y ) * c2 + ( 3.0f * b.y - a.y - 3.0f * c.y + d.y ) * c3 ) );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Lerp( const Vector2& a, const Vector2& b, f32 amount )
{
    return Vector2(
        a.x + amount * ( b.x - a.x ),
        a.y + amount * ( b.y - a.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Lerp( const Vector2 &a, const Vector2 &b, f32 amount, Vector2 &result )
{
    result.x = a.x + amount * ( b.x - a.x );
    result.y = a.y + amount * ( b.y - a.y );
}

//-------------------------------------------------------------------------------------------------
//      3次方程式を用いて，2つの値の間を補間します
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::SmoothStep( const Vector2& a, const Vector2& b, f32 amount )
{
    auto s = asdx::Clamp( amount, 0.0f, 1.0f );
    auto u = ( s * s ) + ( 3.0f - ( 2.0f * s ) );
    return Vector2(
        a.x + u * ( b.x - a.x ),
        a.y + u * ( b.y - a.y )
    );
}

//-------------------------------------------------------------------------------------------------
//      3次方程式を用いて，2つの値の間を補間します
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::SmoothStep( const Vector2 &a, const Vector2 &b, f32 t, Vector2 &result )
{
    auto s = asdx::Clamp( t, 0.0f, 1.0f );
    auto u = ( s * s ) + ( 3.0f - ( 2.0f * s ) );
    result.x = a.x + u * ( b.x - a.x );
    result.y = a.y + u * ( b.y - a.y );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::Transform( const Vector2& position, const Matrix& matrix )
{
    return Vector2(
        ((position.x * matrix._11) + (position.y * matrix._21)) + matrix._41,
        ((position.x * matrix._12) + (position.y * matrix._22)) + matrix._42 );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::Transform( const Vector2 &position, const Matrix &matrix, Vector2 &result )
{
    result.x = ((position.x * matrix._11) + (position.y * matrix._21)) + matrix._41;
    result.y = ((position.x * matrix._12) + (position.y * matrix._22)) + matrix._42;
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，法線ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::TransformNormal( const Vector2& normal, const Matrix& matrix )
{
    return Vector2(
        (normal.x * matrix._11) + (normal.y * matrix._21),
        (normal.x * matrix._12) + (normal.y * matrix._22) );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，法線ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::TransformNormal( const Vector2 &normal, const Matrix &matrix, Vector2 &result )
{
    result.x = (normal.x * matrix._11) + (normal.y * matrix._21);
    result.y = (normal.x * matrix._12) + (normal.y * matrix._22);
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector2 Vector2::TransformCoord( const Vector2& coords, const Matrix& matrix )
{
    auto X = ( ( ((coords.x * matrix._11) + (coords.y * matrix._21)) ) + matrix._41);
    auto Y = ( ( ((coords.x * matrix._12) + (coords.y * matrix._22)) ) + matrix._42);
    auto W = ( ( ((coords.x * matrix._14) + (coords.y * matrix._24)) ) + matrix._44);
    return Vector2(
        X / W,
        Y / W 
    );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector2::TransformCoord( const Vector2 &coords, const Matrix &matrix, Vector2 &result )
{
    auto X = ( ( ((coords.x * matrix._11) + (coords.y * matrix._21)) ) + matrix._41);
    auto Y = ( ( ((coords.x * matrix._12) + (coords.y * matrix._22)) ) + matrix._42);
    auto W = ( ( ((coords.x * matrix._14) + (coords.y * matrix._24)) ) + matrix._44);

    result.x = X / W;
    result.y = Y / W;
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Vector3 structure
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3::Vector3()
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3::Vector3( const f32* pf )
{
    assert( pf != nullptr );
    x = pf[ 0 ];
    y = pf[ 1 ];
    z = pf[ 2 ];
}

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3::Vector3( const Vector2& value, f32 nz )
: x( value.x )
, y( value.y )
, z( nz )
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3::Vector3( f32 nx, f32 ny, f32 nz )
: x( nx )
, y( ny )
, z( nz )
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3::operator f32 *()
{ return static_cast<f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      const f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3::operator const f32 *() const
{ return static_cast<const f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      加算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3& Vector3::operator += ( const Vector3& v )
{
    x += v.x;
    y += v.y;
    z += v.z;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      減算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3& Vector3::operator -= ( const Vector3& v )
{
    x -= v.x;
    y -= v.y;
    z -= v.z;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      乗算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3& Vector3::operator *= ( f32 f )
{
    x *= f;
    y *= f;
    z *= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      除算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3& Vector3::operator /= ( f32 f )
{
    assert( !IsZero( f ) );
    x /= f;
    y /= f;
    z /= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3& Vector3::operator = ( const Vector3& value )
{
    x = value.x;
    y = value.y;
    z = value.z;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      正符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::operator + () const
{ return (*this); }

//-------------------------------------------------------------------------------------------------
//      負符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::operator - () const
{ return Vector3( -x, -y, -z ); }

//-------------------------------------------------------------------------------------------------
//      加算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::operator + ( const Vector3& v ) const
{ return Vector3( x + v.x, y + v.y, z + v.z ); }

//-------------------------------------------------------------------------------------------------
//      減算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::operator - ( const Vector3& v ) const
{ return Vector3( x - v.x, y - v.y, z - v.z ); }

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::operator * ( f32 f ) const
{ return Vector3( x * f, y * f, z * f ); }

//-------------------------------------------------------------------------------------------------
//      除算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::operator / ( f32 f ) const
{
    assert( !IsZero( f ) );
    return Vector3( x / f, y / f, z / f );
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 operator * ( f32 f, const Vector3& v )
{ return Vector3( f * v.x, f * v.y, f * v.z ); }

//-------------------------------------------------------------------------------------------------
//      等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Vector3::operator == ( const Vector3& v ) const
{
    return IsEqual( x, v.x )
        && IsEqual( y, v.y )
        && IsEqual( z, v.z );
}

//-------------------------------------------------------------------------------------------------
//      非等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Vector3::operator != ( const Vector3& v ) const
{ 
    return !IsEqual( x, v.x )
        || !IsEqual( y, v.y )
        || !IsEqual( z, v.z );
}

//-------------------------------------------------------------------------------------------------
//      ベクトルの大きさを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector3::Length() const
{ return sqrtf( x * x + y * y + z * z); }

//-------------------------------------------------------------------------------------------------
//      ベクトルの大きさの2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector3::LengthSq() const
{ return ( x * x + y * y + z * z); }

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3& Vector3::Normalize()
{
    auto mag = Length();
    assert( mag > 0.0f );
    x /= mag;
    y /= mag;
    z /= mag;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3& Vector3::SafeNormalize( const Vector3& set )
{
    auto mag = Length();
    if ( mag > 0.0f )
    {
        x /= mag;
        y /= mag;
        z /= mag;
        return (*this);
    }

    x = set.x;
    y = set.y;
    z = set.z;
    return (*this);
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Vector3 methods
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      各成分の絶対値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Abs( const Vector3& v )
{ 
    return Vector3(
        fabs( v.x ),
        fabs( v.y ),
        fabs( v.z ) 
     );
}

//-------------------------------------------------------------------------------------------------
//      各成分の絶対値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Abs( const Vector3 &value, Vector3 &result )
{ 
    result.x = fabs( value.x );
    result.y = fabs( value.y );
    result.z = fabs( value.z );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Clamp( const Vector3& value, const Vector3& a, const Vector3& b )
{
    return Vector3( 
        asdx::Clamp( value.x, a.x, b.x ),
        asdx::Clamp( value.y, a.y, b.y ),
        asdx::Clamp( value.z, a.z, b.z )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Clamp( const Vector3 &value, const Vector3 &a, const Vector3 &b, Vector3 &result )
{
    result.x = asdx::Clamp( value.x, a.x, b.x );
    result.y = asdx::Clamp( value.y, a.y, b.y );
    result.z = asdx::Clamp( value.z, a.z, b.z );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を0～1に収めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Saturate( const Vector3& value )
{
    return Vector3(
        asdx::Saturate( value.x ),
        asdx::Saturate( value.y ),
        asdx::Saturate( value.z )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の値を0～1に収めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Saturate( const Vector3& value, Vector3& result )
{
    result.x = asdx::Saturate( value.x );
    result.y = asdx::Saturate( value.y );
    result.z = asdx::Saturate( value.z );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector3::Distance( const Vector3& a, const Vector3& b )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    return sqrtf( X * X + Y * Y + Z * Z );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Distance( const Vector3 &a, const Vector3 &b, f32 &result )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    result = sqrtf( X * X + Y * Y + Z * Z );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離の2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector3::DistanceSq( const Vector3& a, const Vector3& b )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    return X * X + Y * Y + Z * Z;
}

//-------------------------------------------------------------------------------------------------
//      2点間距離の2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::DistanceSq( const Vector3 &a, const Vector3 &b, f32 &result )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    result = X * X + Y * Y + Z * Z;
}

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector3::Dot( const Vector3& a, const Vector3& b )
{ return ( a.x * b.x + a.y * b.y + a.z * b.z ); }

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Dot( const Vector3 &a, const Vector3 &b, f32 &result )
{ result = a.x * b.x + a.y * b.y + a.z * b.z; }

//-------------------------------------------------------------------------------------------------
//      外積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Cross( const Vector3& a, const Vector3& b )
{
    return Vector3( 
        ( a.y * b.z ) - ( a.z * b.y ),
        ( a.z * b.x ) - ( a.x * b.z ),
        ( a.x * b.y ) - ( a.y * b.x )
    );
}

//-------------------------------------------------------------------------------------------------
//      外積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Cross( const Vector3 &a, const Vector3 &b, Vector3 &result )
{
    result.x = ( a.y * b.z ) - ( a.z * b.y );
    result.y = ( a.z * b.x ) - ( a.x * b.z );
    result.z = ( a.x * b.y ) - ( a.y * b.x );
}

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Normalize( const Vector3& value )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    return Vector3(
        value.x / mag,
        value.y / mag,
        value.z / mag 
    );
}

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Normalize( const Vector3& value, Vector3 &result )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    result.x = value.x / mag;
    result.y = value.y / mag;
    result.z = value.z / mag;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::SafeNormalize( const Vector3& value, const Vector3& set )
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        return Vector3(
            value.x / mag,
            value.y / mag,
            value.z / mag
        );
    }

    return set;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::SafeNormalize( const Vector3& value, const Vector3& set, Vector3& result )
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        result.x = value.x / mag;
        result.y = value.y / mag;
        result.z = value.z / mag;
    }
    else
    {
        result.x = set.x;
        result.y = set.y;
        result.z = set.z;
    }
}

//-------------------------------------------------------------------------------------------------
//      三角形の面法線を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::ComputeNormal( const Vector3& p1, const Vector3& p2, const Vector3& p3 )
{
    auto v1 = p2 - p1;
    auto v2 = p3 - p1;
    auto result = Vector3::Cross( v1, v2 );
    return result.Normalize();
}

//-------------------------------------------------------------------------------------------------
//      三角形の面法線を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::ComputeNormal( const Vector3 &p1, const Vector3 &p2, const Vector3 &p3, Vector3 &result )
{
    auto v1 = p2 - p1;
    auto v2 = p3 - p1;
    Vector3::Cross( v1, v2, result );
    result.Normalize();
}

//-------------------------------------------------------------------------------------------------
//      四角形の面法線を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::ComputeQuadNormal
(
    const Vector3& p1,
    const Vector3& p2,
    const Vector3& p3,
    const Vector3& p4
)
{
    Vector3 result;
    auto n1a = Vector3::ComputeNormal( p1, p2, p3 );
    auto n1b = Vector3::ComputeNormal( p1, p3, p4 );
    auto n2a = Vector3::ComputeNormal( p2, p3, p4 );
    auto n2b = Vector3::ComputeNormal( p2, p4, p1 );
    if ( Vector3::Dot( n1a, n1b ) > Vector3::Dot( n2a, n2b ) )
    {
        result = n1a + n1b;
        result.Normalize();
    }
    else
    {
        result = n2a + n2b;
        result.Normalize();
    }
    return result;
}

//-------------------------------------------------------------------------------------------------
//      四角形の面法線を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::ComputeQuadNormal
(
    const Vector3 &p1,
    const Vector3 &p2,
    const Vector3 &p3,
    const Vector3 &p4,
    Vector3 &result
)
{
    auto n1a = Vector3::ComputeNormal( p1, p2, p3 );
    auto n1b = Vector3::ComputeNormal( p1, p3, p4 );
    auto n2a = Vector3::ComputeNormal( p2, p3, p4 );
    auto n2b = Vector3::ComputeNormal( p2, p4, p1 );
    if ( Vector3::Dot( n1a, n1b ) > Vector3::Dot( n2a, n2b ) )
    {
        result = n1a + n1b;
        result.Normalize();
    }
    else
    {
        result = n2a + n2b;
        result.Normalize();
    }
}

//-------------------------------------------------------------------------------------------------
//      交差角を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector3::ComputeCrossingAngle( const Vector3& a, const Vector3& b )
{
    auto d = a.Length() * b.Length();
    if ( d <= 0.0f ) 
    { return 0.0f; }

    auto c = Vector3::Dot( a, b ) / d;
    if ( c >= 1.0f )
    { return 0.0f; }

    if ( c <= -1.0f )
    { return F_PI; }

    return acosf( c );
}

//-------------------------------------------------------------------------------------------------
//      交差角を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::ComputeCrossingAngle( const Vector3 &a, const Vector3 &b, f32 &result )
{
    auto d = a.Length() * b.Length();
    if ( d <= 0.0f )
    {
        result = 0.0f;
        return;
    }

    auto c = Vector3::Dot( a, b ) / d;
    if ( c >= 1.0f ) 
    {
        result = 0.0f;
        return;
    }

    if ( c <= -1.0f )
    {
        result = F_PI;
        return;
    }

    result = acosf( c );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最小値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Min( const Vector3& a, const Vector3& b )
{ 
    return Vector3( 
        asdx::Min( a.x, b.x ),
        asdx::Min( a.y, b.y ),
        asdx::Min( a.z, b.z )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最小値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Min( const Vector3 &a, const Vector3 &b, Vector3 &result )
{
    result.x = asdx::Min( a.x, b.x );
    result.y = asdx::Min( a.y, b.y );
    result.z = asdx::Min( a.z, b.z );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最大値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Max( const Vector3& a, const Vector3& b )
{
    return Vector3(
        asdx::Max( a.x, b.x ),
        asdx::Max( a.y, b.y ),
        asdx::Max( a.z, b.z )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最大値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Max( const Vector3 &a, const Vector3 &b, Vector3 &result )
{
    result.x = asdx::Max( a.x, b.x );
    result.y = asdx::Max( a.y, b.y );
    result.z = asdx::Max( a.z, b.z );
}

//-------------------------------------------------------------------------------------------------
//      反射ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Reflect( const Vector3& i, const Vector3& n )
{
    auto dot = n.x * i.x + n.y * i.y + n.z * i.z;
    return Vector3(
        i.x - ( 2.0f * n.x ) * dot,
        i.y - ( 2.0f * n.y ) * dot,
        i.z - ( 2.0f * n.z ) * dot
    );
}

//-------------------------------------------------------------------------------------------------
//      反射ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Reflect( const Vector3 &i, const Vector3 &n, Vector3 &result )
{
    auto dot = n.x * i.x + n.y * i.y + n.z * i.z;
    result.x = i.x - ( 2.0f * n.x ) * dot;
    result.y = i.y - ( 2.0f * n.y ) * dot;
    result.z = i.z - ( 2.0f * n.z ) * dot;
}

//-------------------------------------------------------------------------------------------------
//      屈折ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Refract( const Vector3& i, const Vector3& n, f32 eta )
{
    auto cosi   = ( -i.x * n.x ) + ( -i.y * n.y ) + ( -i.z * n.z );
    auto cost2  = 1.0f - eta * eta * ( 1.0f - cosi * cosi );
    auto sign   = Sign( cost2 );
    auto sqrtC2 = sqrtf( fabs( cost2 ) );
    auto coeff  = eta * cosi - sqrtC2;

    return Vector3(
        sign * ( eta * i.x + coeff * n.x ),
        sign * ( eta * i.y + coeff * n.y ),
        sign * ( eta * i.z + coeff * n.z )
    );
}

//-------------------------------------------------------------------------------------------------
//      屈折ベクトルを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Refract( const Vector3 &i, const Vector3 &n, f32 eta, Vector3 &result )
{
    auto cosi   =  ( -i.x * n.x ) + ( -i.y * n.y ) + ( -i.z * n.z );
    auto cost2  = 1.0f - eta * eta * ( 1.0f - cosi * cosi );
    auto sign   = Sign( cost2 );
    auto sqrtC2 = sqrtf( fabs( cost2 ) );
    auto coeff  = eta * cosi - sqrtC2;

    result.x = sign * ( eta * i.x + coeff * n.x );
    result.y = sign * ( eta * i.y + coeff * n.y );
    result.z = sign * ( eta * i.z + coeff * n.z );
}

//-------------------------------------------------------------------------------------------------
//      重心座標を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Barycentric
(
    const Vector3&  a,
    const Vector3&  b,
    const Vector3&  c,
    f32             f,
    f32             g
)
{
    return Vector3(
        a.x + f * ( b.x - a.x ) + g * ( c.x - a.x ),
        a.y + f * ( b.y - a.y ) + g * ( c.y - a.y ),
        a.z + f * ( b.z - a.z ) + g * ( c.z - a.z )
    );
}

//-------------------------------------------------------------------------------------------------
//      重心座標を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Barycentric
(
    const Vector3&  a,
    const Vector3&  b,
    const Vector3&  c,
    f32             f,
    f32             g,
    Vector3&        result
)
{
    result.x = a.x + f * ( b.x - a.x ) + g * ( c.x - a.x );
    result.y = a.y + f * ( b.y - a.y ) + g * ( c.y - a.y );
    result.z = a.z + f * ( b.z - a.z ) + g * ( c.z - a.z );
}

//-------------------------------------------------------------------------------------------------
//      エルミートスプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Hermite
(
    const Vector3&  a,
    const Vector3&  t1,
    const Vector3&  b,
    const Vector3&  t2,
    f32             amount
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    Vector3 result;
    if ( amount <= 0.0f )
    {
        result.x = a.x;
        result.y = a.y;
        result.z = a.z;
    }
    else if ( amount >= 1.0f )
    {
        result.x = b.x;
        result.y = b.y;
        result.z = b.z;
    }
    else
    {
        result.x = ( 2.0f * a.x - 2.0f * b.x + t2.x + t1.x ) * c3 + ( 3.0f * b.x - 3.0f * a.x - 2.0f * t1.x - t2.x ) * c3 + t1.x * amount + a.x;
        result.y = ( 2.0f * a.y - 2.0f * b.y + t2.y + t1.y ) * c3 + ( 3.0f * b.y - 3.0f * a.y - 2.0f * t1.y - t2.y ) * c3 + t1.y * amount + a.y;
        result.z = ( 2.0f * a.z - 2.0f * b.z + t2.z + t1.z ) * c3 + ( 3.0f * b.z - 3.0f * a.z - 2.0f * t1.z - t2.z ) * c3 + t1.y * amount + a.z;
    }
    return result;
}

//-------------------------------------------------------------------------------------------------
//      エルミートスプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Hermite
(
    const Vector3&  a,
    const Vector3&  t1,
    const Vector3&  b,
    const Vector3&  t2,
    f32             amount,
    Vector3&        result
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    if ( amount <= 0.0f )
    {
        result.x = a.x;
        result.y = a.y;
        result.z = a.z;
    }
    else if ( amount >= 1.0f )
    {
        result.x = b.x;
        result.y = b.y;
        result.z = b.z;
    }
    else
    {
        result.x = ( 2.0f * a.x - 2.0f * b.x + t2.x + t1.x ) * c3 + ( 3.0f * b.x - 3.0f * a.x - 2.0f * t1.x - t2.x ) * c3 + t1.x * amount + a.x;
        result.y = ( 2.0f * a.y - 2.0f * b.y + t2.y + t1.y ) * c3 + ( 3.0f * b.y - 3.0f * a.y - 2.0f * t1.y - t2.y ) * c3 + t1.y * amount + a.y;
        result.z = ( 2.0f * a.z - 2.0f * b.z + t2.z + t1.z ) * c3 + ( 3.0f * b.z - 3.0f * a.z - 2.0f * t1.z - t2.z ) * c3 + t1.z * amount + a.z;
    }
}

//-------------------------------------------------------------------------------------------------
//      Catmull-Rom スプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::CatmullRom
(
    const Vector3&  a,
    const Vector3&  b,
    const Vector3&  c,
    const Vector3&  d,
    f32             amount
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    return Vector3(
        ( 0.5f * ( 2.0f * b.x + ( c.x - a.x ) * amount + ( 2.0f * a.x - 5.0f * b.x + 4.0f * c.x - d.x ) * c2 + ( 3.0f * b.x - a.x - 3.0f * c.x + d.x ) * c3 ) ),
        ( 0.5f * ( 2.0f * b.y + ( c.y - a.y ) * amount + ( 2.0f * a.y - 5.0f * b.y + 4.0f * c.y - d.y ) * c2 + ( 3.0f * b.y - a.y - 3.0f * c.y + d.y ) * c3 ) ),
        ( 0.5f * ( 2.0f * b.z + ( c.z - a.z ) * amount + ( 2.0f * a.z - 5.0f * b.z + 4.0f * c.z - d.z ) * c2 + ( 3.0f * b.z - a.z - 3.0f * c.z + d.z ) * c3 ) )
    );
}

//-------------------------------------------------------------------------------------------------
//      Catmull-Rom スプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::CatmullRom
(
    const Vector3&  a,
    const Vector3&  b,
    const Vector3&  c,
    const Vector3&  d,
    f32             amount,
    Vector3&        result
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    result.x = ( 0.5f * ( 2.0f * b.x + ( c.x - a.x ) * amount + ( 2.0f * a.x - 5.0f * b.x + 4.0f * c.x - d.x ) * c2 + ( 3.0f * b.x - a.x - 3.0f * c.x + d.x ) * c3 ) );
    result.y = ( 0.5f * ( 2.0f * b.y + ( c.y - a.y ) * amount + ( 2.0f * a.y - 5.0f * b.y + 4.0f * c.y - d.y ) * c2 + ( 3.0f * b.y - a.y - 3.0f * c.y + d.y ) * c3 ) );
    result.z = ( 0.5f * ( 2.0f * b.z + ( c.z - a.z ) * amount + ( 2.0f * a.z - 5.0f * b.z + 4.0f * c.z - d.z ) * c2 + ( 3.0f * b.z - a.z - 3.0f * c.z + d.z ) * c3 ) );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Lerp( const Vector3& a, const Vector3& b, f32 amount )
{
    return Vector3(
        a.x + amount * ( b.x - a.x ),
        a.y + amount * ( b.y - a.y ),
        a.z + amount * ( b.z - a.z ) 
    );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Lerp( const Vector3 &a, const Vector3 &b, f32 amount, Vector3 &result )
{
    result.x = a.x + amount * ( b.x - a.x );
    result.y = a.y + amount * ( b.y - a.y );
    result.z = a.z + amount * ( b.z - a.z );
}

//-------------------------------------------------------------------------------------------------
//      3次方程式を用いて，２つの値を補間します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::SmoothStep( const Vector3& a, const Vector3& b, f32 amount )
{
    auto s = asdx::Clamp( amount, 0.0f, 1.0f );
    auto u = ( s * s ) + ( 3.0f - ( 2.0f * s ) );
    return Vector3(
        a.x + u * ( b.x - a.x ),
        a.y + u * ( b.y - a.y ),
        a.z + u * ( b.z - a.z )
    );
}

//-------------------------------------------------------------------------------------------------
//      3次方程式を用いて，２つの値を補間します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::SmoothStep( const Vector3 &a, const Vector3 &b, f32 amount, Vector3 &result )
{ 
    auto s = asdx::Clamp( amount, 0.0f, 1.0f );
    auto u = ( s * s ) + ( 3.0f - ( 2.0f * s ) );
    result.x = a.x + u * ( b.x - a.x );
    result.y = a.y + u * ( b.y - a.y );
    result.z = a.z + u * ( b.z - a.z );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Transform( const Vector3& position, const Matrix& matrix )
{
    return Vector3(
        ( ((position.x * matrix._11) + (position.y * matrix._21)) + (position.z * matrix._31)) + matrix._41,
        ( ((position.x * matrix._12) + (position.y * matrix._22)) + (position.z * matrix._32)) + matrix._42,
        ( ((position.x * matrix._13) + (position.y * matrix._23)) + (position.z * matrix._33)) + matrix._43 );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Transform( const Vector3 &position, const Matrix &matrix, Vector3 &result )
{
    result.x = ( ((position.x * matrix._11) + (position.y * matrix._21)) + (position.z * matrix._31)) + matrix._41;
    result.y = ( ((position.x * matrix._12) + (position.y * matrix._22)) + (position.z * matrix._32)) + matrix._42;
    result.z = ( ((position.x * matrix._13) + (position.y * matrix._23)) + (position.z * matrix._33)) + matrix._43;
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，法線ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::TransformNormal( const Vector3& normal, const Matrix& matrix )
{
    return Vector3(
        ((normal.x * matrix._11) + (normal.y * matrix._21)) + (normal.z * matrix._31),
        ((normal.x * matrix._12) + (normal.y * matrix._22)) + (normal.z * matrix._32),
        ((normal.x * matrix._13) + (normal.y * matrix._23)) + (normal.z * matrix._33) );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，法線ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::TransformNormal( const Vector3 &normal, const Matrix &matrix, Vector3 &result )
{
    result.x = ((normal.x * matrix._11) + (normal.y * matrix._21)) + (normal.z * matrix._31);
    result.y = ((normal.x * matrix._12) + (normal.y * matrix._22)) + (normal.z * matrix._32);
    result.z = ((normal.x * matrix._13) + (normal.y * matrix._23)) + (normal.z * matrix._33);
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::TransformCoord( const Vector3& coords, const Matrix& matrix )
{
    auto X = ( ( ((coords.x * matrix._11) + (coords.y * matrix._21)) + (coords.z * matrix._31) ) + matrix._41);
    auto Y = ( ( ((coords.x * matrix._12) + (coords.y * matrix._22)) + (coords.z * matrix._32) ) + matrix._42);
    auto Z = ( ( ((coords.x * matrix._13) + (coords.y * matrix._23)) + (coords.z * matrix._33) ) + matrix._43);
    auto W = ( ( ((coords.x * matrix._14) + (coords.y * matrix._24)) + (coords.z * matrix._34) ) + matrix._44);
    return Vector3(
        X / W,
        Y / W,
        Z / W 
    );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::TransformCoord( const Vector3 &coords, const Matrix &matrix, Vector3 &result )
{
    auto X = ( ( ((coords.x * matrix._11) + (coords.y * matrix._21)) + (coords.z * matrix._31) ) + matrix._41);
    auto Y = ( ( ((coords.x * matrix._12) + (coords.y * matrix._22)) + (coords.z * matrix._32) ) + matrix._42);
    auto Z = ( ( ((coords.x * matrix._13) + (coords.y * matrix._23)) + (coords.z * matrix._33) ) + matrix._43);
    auto W = ( ( ((coords.x * matrix._14) + (coords.y * matrix._24)) + (coords.z * matrix._34) ) + matrix._44);

    result.x = X / W;
    result.y = Y / W;
    result.z = Z / W;
}

//-------------------------------------------------------------------------------------------------
//      スカラー3重積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector3::ScalarTriple( const Vector3& a, const Vector3& b, const Vector3& c )
{
    auto crossX = ( b.y * c.z ) - ( b.z * c.y );
    auto crossY = ( b.z * c.x ) - ( b.x * c.z );
    auto crossZ = ( b.x * c.y ) - ( b.y * c.x );

    return ( a.x * crossX ) + ( a.y * crossY ) + ( a.z * crossZ );
}

//-------------------------------------------------------------------------------------------------
//      スカラー3重積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::ScalarTriple( const Vector3& a, const Vector3& b, const Vector3& c, f32& result )
{
    auto crossX = ( b.y * c.z ) - ( b.z * c.y );
    auto crossY = ( b.z * c.x ) - ( b.x * c.z );
    auto crossZ = ( b.x * c.y ) - ( b.y * c.x );

    result = ( a.x * crossX ) + ( a.y * crossY ) + ( a.z * crossZ );
}

//-------------------------------------------------------------------------------------------------
//      ベクトル3重積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::VectorTriple( const Vector3& a, const Vector3& b, const Vector3& c )
{
    auto crossX = ( b.y * c.z ) - ( b.z * c.y );
    auto crossY = ( b.z * c.x ) - ( b.x * c.z );
    auto crossZ = ( b.x * c.y ) - ( b.y * c.x );

    return Vector3(
        ( ( a.y * crossZ ) - ( a.z * crossY ) ),
        ( ( a.z * crossX ) - ( a.x * crossZ ) ),
        ( ( a.x * crossY ) - ( a.y * crossX ) )
    );
}

//-------------------------------------------------------------------------------------------------
//      ベクトル3重積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::VectorTriple( const Vector3& a, const Vector3& b, const Vector3& c, Vector3& result )
{
    auto crossX = ( b.y * c.z ) - ( b.z * c.y );
    auto crossY = ( b.z * c.x ) - ( b.x * c.z );
    auto crossZ = ( b.x * c.y ) - ( b.y * c.x );

    result.x = ( a.y * crossZ ) - ( a.z * crossY );
    result.y = ( a.z * crossX ) - ( a.x * crossZ );
    result.z = ( a.x * crossY ) - ( a.y * crossX );
}

//-------------------------------------------------------------------------------------------------
//      四元数でベクトルを回転させます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::Rotate( const Vector3& value, const Quaternion& rotation )
{
    auto a = Quaternion( value.x, value.y, value.z, 0.0f );
    auto q = Quaternion::Conjugate( rotation );
    auto r = Quaternion::Multiply( q, a );
    r = Quaternion::Multiply( r, rotation );
    return Vector3( r.x, r.y, r.z );
}

//-------------------------------------------------------------------------------------------------
//      四元数でベクトルを回転させます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::Rotate( const Vector3& value, const Quaternion& rotation, Vector3& result )
{
    auto a = Quaternion( value.x, value.y, value.z, 0.0f );
    auto q = Quaternion::Conjugate( rotation );
    auto r = Quaternion::Multiply( q, a );
    r = Quaternion::Multiply( r, rotation );
    result.x = r.x;
    result.y = r.y;
    result.z = r.z;
}

//-------------------------------------------------------------------------------------------------
//      四元数でベクトルを逆回転させます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector3 Vector3::InverseRotate( const Vector3& value, const Quaternion& rotation )
{
    auto a = Quaternion( value.x, value.y, value.z, 0.0f );
    auto r = Quaternion::Multiply( rotation, a );
    auto q = Quaternion::Conjugate( rotation );
    r = Quaternion::Multiply( r, q );
    return Vector3( r.x, r.y, r.z );
}

//-------------------------------------------------------------------------------------------------
//      四元数でベクトルを逆回転させます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector3::InverseRotate( const Vector3& value, const Quaternion& rotation, Vector3& result )
{
    auto a = Quaternion( value.x, value.y, value.z, 0.0f );
    auto r = Quaternion::Multiply( rotation, a );
    auto q = Quaternion::Conjugate( rotation );
    r = Quaternion::Multiply( r, q );
    result.x = r.x;
    result.y = r.y;
    result.z = r.z;
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Vector4 structure
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4::Vector4()
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4::Vector4( const f32* pf )
{
    assert( pf != nullptr );
    x = pf[ 0 ];
    y = pf[ 1 ];
    z = pf[ 2 ];
    w = pf[ 3 ];
}

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4::Vector4( const Vector2& value, f32 nz, f32 nw )
: x( value.x )
, y( value.y )
, z( nz )
, w( nw )
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4::Vector4( const Vector3& value, f32 nw )
: x( value.x )
, y( value.y )
, z( value.z )
, w( nw )
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4::Vector4( f32 nx, f32 ny, f32 nz, f32 nw )
: x( nx )
, y( ny )
, z( nz )
, w( nw )
{ /* DO_NTOHING */ }

//-------------------------------------------------------------------------------------------------
//      f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4::operator f32 *()
{ return static_cast<f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      const f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4::operator const f32 *() const
{ return static_cast<const f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      加算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4& Vector4::operator += ( const Vector4& v )
{
    x += v.x;
    y += v.y;
    z += v.z;
    w += v.w;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      減算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4& Vector4::operator -= ( const Vector4& v )
{
    x -= v.x;
    y -= v.y;
    z -= v.z;
    w -= v.w;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      乗算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4& Vector4::operator *= ( f32 f )
{
    x *= f;
    y *= f;
    z *= f;
    w *= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      除算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4& Vector4::operator /= ( f32 f )
{
    assert( !IsZero( f ) );
    x /= f;
    y /= f;
    z /= f;
    w /= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4& Vector4::operator = ( const Vector4& value )
{
    x = value.x;
    y = value.y;
    z = value.z;
    w = value.w;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      正符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::operator + () const
{ return (*this); }

//-------------------------------------------------------------------------------------------------
//      負符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::operator - () const
{ return Vector4( -x, -y, -z, -w ); }

//-------------------------------------------------------------------------------------------------
//      加算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::operator + ( const Vector4& v ) const
{ return Vector4( x + v.x, y + v.y, z + v.z, w + v.w ); }

//-------------------------------------------------------------------------------------------------
//      減算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::operator - ( const Vector4& v ) const
{ return Vector4( x - v.x, y - v.y, z - v.z, w - v.w ); }

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::operator * ( f32 f ) const
{ return Vector4( x * f, y * f, z * f, w * f ); }

//-------------------------------------------------------------------------------------------------
//      除算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::operator / ( f32 f ) const
{
    assert( !IsZero( f ) );
    return Vector4( x / f, y / f, z / f, w / f );
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 operator * ( f32 f, const Vector4& v )
{ return Vector4( f * v.x, f * v.y, f * v.z, f * v.w ); }

//-------------------------------------------------------------------------------------------------
//      等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Vector4::operator == ( const Vector4& v ) const
{
    return IsEqual( x, v.x )
        && IsEqual( y, v.y )
        && IsEqual( z, v.z )
        && IsEqual( w, v.z );
}

//-------------------------------------------------------------------------------------------------
//      非等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Vector4::operator != ( const Vector4& v ) const
{ 
    return !IsEqual( x, v.x )
        || !IsEqual( y, v.y )
        || !IsEqual( z, v.z )
        || !IsEqual( w, v.w );
}

//-------------------------------------------------------------------------------------------------
//      ベクトルの大きさを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector4::Length() const
{ return sqrtf( x * x + y * y + z * z + w * w ); }

//-------------------------------------------------------------------------------------------------
//      ベクトルの大きさの2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector4::LengthSq() const
{ return ( x * x + y * y + z * z + w * w ); }

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4& Vector4::Normalize()
{
    auto mag = Length();
    assert( mag > 0.0f );
    x /= mag;
    y /= mag;
    z /= mag;
    w /= mag;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4& Vector4::SafeNormalize( const Vector4& set )
{
    auto mag = Length();
    if ( mag > 0.0f )
    {
        x /= mag;
        y /= mag;
        z /= mag;
        w /= mag;
    }
    else
    {
        x = set.x;
        y = set.y;
        z = set.z;
        w = set.w;
    }

    return (*this);
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Vector4  Methods
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      各成分の絶対値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Abs( const Vector4& value )
{ 
    return Vector4( 
        fabs( value.x ),
        fabs( value.y ),
        fabs( value.z ),
        fabs( value.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の絶対値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Abs( const Vector4 &value, Vector4 &result )
{ 
    result.x = fabs( value.x );
    result.y = fabs( value.y );
    result.z = fabs( value.z );
    result.w = fabs( value.w );
}

//-------------------------------------------------------------------------------------------------
//      値を指定された範囲内に制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Clamp( const Vector4& value, const Vector4& a, const Vector4& b )
{
    return Vector4( 
        asdx::Clamp( value.x, a.x, b.x ),
        asdx::Clamp( value.y, a.y, b.y ),
        asdx::Clamp( value.z, a.z, b.z ),
        asdx::Clamp( value.w, a.w, b.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      値を指定された範囲内に制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Clamp( const Vector4 &value, const Vector4 &a, const Vector4 &b, Vector4 &result )
{
    result.x = asdx::Clamp( value.x, a.x, b.x );
    result.y = asdx::Clamp( value.y, a.y, b.y );
    result.z = asdx::Clamp( value.z, a.z, b.z );
    result.w = asdx::Clamp( value.w, a.w, b.w );
}

//-------------------------------------------------------------------------------------------------
//      指定された値を0～1の範囲に制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Saturate( const Vector4& value )
{
    return Vector4(
        asdx::Saturate( value.x ),
        asdx::Saturate( value.y ),
        asdx::Saturate( value.z ),
        asdx::Saturate( value.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      指定された体を0～1の範囲に制限します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Saturate( const Vector4& value, Vector4& result )
{
    result.x = asdx::Saturate( value.x );
    result.y = asdx::Saturate( value.y );
    result.z = asdx::Saturate( value.z );
    result.w = asdx::Saturate( value.w );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector4::Distance( const Vector4& a, const Vector4& b )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    auto W = b.w - a.w;
    return sqrtf( X * X + Y * Y + Z * Z + W * W );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Distance( const Vector4 &a, const Vector4 &b, f32 &result )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    auto W = b.w - a.w;
    result = sqrtf( X * X + Y * Y + Z * Z + W * W );
}

//-------------------------------------------------------------------------------------------------
//      2点間距離の2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector4::DistanceSq( const Vector4& a, const Vector4& b )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    auto W = b.w - a.w;
    return X * X + Y * Y + Z * Z + W * W;
}

//-------------------------------------------------------------------------------------------------
//      2点間距離の2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::DistanceSq( const Vector4 &a, const Vector4 &b, f32 &result )
{
    auto X = b.x - a.x;
    auto Y = b.y - a.y;
    auto Z = b.z - a.z;
    auto W = b.w - a.w;
    result = X * X + Y * Y + Z * Z + W * W;
}

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector4::Dot( const Vector4& a, const Vector4& b )
{ return ( a.x * b.x + a.y * b.y + a.z * b.z + a.w * b.w ); }

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Dot( const Vector4 &a, const Vector4 &b, f32 &result )
{ result = a.x * b.x + a.y * b.y + a.z * b.z + a.w * b.w; }

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Normalize( const Vector4& value )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    return Vector4(
        value.x / mag,
        value.y / mag,
        value.z / mag,
        value.w / mag
    );
}

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Normalize( const Vector4 &value, Vector4 &result )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    result.x = value.x / mag;
    result.y = value.y / mag;
    result.z = value.z / mag;
    result.w = value.w / mag;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::SafeNormalize( const Vector4& value, const Vector4& set )
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        return Vector4(
            value.x / mag,
            value.y / mag,
            value.z / mag,
            value.w / mag
        );
    }

    return set;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::SafeNormalize( const Vector4& value, const Vector4& set, Vector4& result)
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        result.x = value.x / mag;
        result.y = value.y / mag;
        result.z = value.z / mag;
        result.w = value.w / mag;
    }
    else
    {
        result.x = set.x;
        result.y = set.y;
        result.z = set.z;
        result.w = set.w;
    }
}

//-------------------------------------------------------------------------------------------------
//      交差角を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Vector4::ComputeCrossingAngle( const Vector4& a, const Vector4& b )
{
    auto d = a.Length() * b.Length();
    if ( d <= 0.0f )
    { return 0.0f; }

    auto c = Vector4::Dot( a, b ) / d;
    if ( c >= 1.0f )
    { return 0.0f; }

    if ( c <= -1.0f ) 
    { return F_PI; }

    return acosf( c );
}

//-------------------------------------------------------------------------------------------------
//      交差角を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::ComputeCrossingAngle( const Vector4 &a, const Vector4 &b, f32 &result )
{
    auto d = a.Length() * b.Length();
    if ( d <= 0.0f )
    {
        result = 0.0f;
        return;
    }

    auto c = Vector4::Dot( a, b ) / d;
    if ( c >= 1.0f ) 
    {
        result = 0.0f;
        return;
    }

    if ( c <= -1.0f )
    {
        result = F_PI;
        return;
    }

    result = acosf( c );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最小値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Min( const Vector4& a, const Vector4& b )
{ 
    return Vector4( 
        asdx::Min( a.x, b.x ),
        asdx::Min( a.y, b.y ),
        asdx::Min( a.z, b.z ),
        asdx::Min( a.w, b.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最小値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Min( const Vector4 &a, const Vector4 &b, Vector4 &result )
{
    result.x = asdx::Min( a.x, b.x );
    result.y = asdx::Min( a.y, b.y );
    result.z = asdx::Min( a.z, b.z );
    result.w = asdx::Min( a.w, b.w );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最大値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Max( const Vector4& a, const Vector4& b )
{
    return Vector4( 
        asdx::Max( a.x, b.x ),
        asdx::Max( a.y, b.y ),
        asdx::Max( a.z, b.z ),
        asdx::Max( a.w, b.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      各成分の最大値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Max( const Vector4 &a, const Vector4 &b, Vector4 &result )
{
    result.x = asdx::Max( a.x, b.x );
    result.y = asdx::Max( a.y, b.y );
    result.z = asdx::Max( a.z, b.z );
    result.w = asdx::Max( a.w, b.w );
}

//-------------------------------------------------------------------------------------------------
//      重心座標を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Barycentric
(
    const Vector4&  a,
    const Vector4&  b,
    const Vector4&  c,
    f32             f,
    f32             g
)
{
    return Vector4(
        a.x + f * ( b.x - a.x ) + g * ( c.x - a.x ),
        a.y + f * ( b.y - a.y ) + g * ( c.y - a.y ),
        a.z + f * ( b.z - a.z ) + g * ( c.z - a.z ),
        a.w + f * ( b.w - a.w ) + g * ( c.w - a.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      重心座標を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Barycentric
(
    const Vector4&  a,
    const Vector4&  b,
    const Vector4&  c,
    f32             f,
    f32             g,
    Vector4&        result
)
{
    result.x = a.x + f * ( b.x - a.x ) + g * ( c.x - a.x );
    result.y = a.y + f * ( b.y - a.y ) + g * ( c.y - a.y );
    result.z = a.z + f * ( b.z - a.z ) + g * ( c.z - a.z );
    result.w = a.w + f * ( b.w - a.w ) + g * ( c.w - a.w );
}

//-------------------------------------------------------------------------------------------------
//      エルミートスプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Hermite
(
    const Vector4&  a,
    const Vector4&  t1,
    const Vector4&  b,
    const Vector4&  t2,
    f32             amount
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    Vector4 result;
    if ( amount <= 0.0f )
    {
        result.x = a.x;
        result.y = a.y;
        result.z = a.z;
        result.w = a.w;
    }
    else if ( amount >= 1.0f )
    {
        result.x = b.x;
        result.y = b.y;
        result.z = b.z;
        result.w = b.w;
    }
    else
    {
        result.x = ( 2.0f * a.x - 2.0f * b.x + t2.x + t1.x ) * c3 + ( 3.0f * b.x - 3.0f * a.x - 2.0f * t1.x - t2.x ) * c3 + t1.x * amount + a.x;
        result.y = ( 2.0f * a.y - 2.0f * b.y + t2.y + t1.y ) * c3 + ( 3.0f * b.y - 3.0f * a.y - 2.0f * t1.y - t2.y ) * c3 + t1.y * amount + a.y;
        result.z = ( 2.0f * a.z - 2.0f * b.z + t2.z + t1.z ) * c3 + ( 3.0f * b.z - 3.0f * a.z - 2.0f * t1.z - t2.z ) * c3 + t1.y * amount + a.z;
        result.w = ( 2.0f * a.w - 2.0f * b.w + t2.w + t1.w ) * c3 + ( 3.0f * b.w - 3.0f * a.w - 2.0f * t1.w - t2.w ) * c3 + t1.w * amount + a.w;
    }
    return result;
}

//-------------------------------------------------------------------------------------------------
//      エルミートスプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Hermite
(
    const Vector4&  a,
    const Vector4&  t1,
    const Vector4&  b,
    const Vector4&  t2,
    f32             amount,
    Vector4&        result
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    if ( amount <= 0.0f )
    {
        result.x = a.x;
        result.y = a.y;
        result.z = a.z;
        result.w = a.w;
    }
    else if ( amount >= 1.0f )
    {
        result.x = b.x;
        result.y = b.y;
        result.z = b.z;
        result.w = b.w;
    }
    else
    {
        result.x = ( 2.0f * a.x - 2.0f * b.x + t2.x + t1.x ) * c3 + ( 3.0f * b.x - 3.0f * a.x - 2.0f * t1.x - t2.x ) * c3 + t1.x * amount + a.x;
        result.y = ( 2.0f * a.y - 2.0f * b.y + t2.y + t1.y ) * c3 + ( 3.0f * b.y - 3.0f * a.y - 2.0f * t1.y - t2.y ) * c3 + t1.y * amount + a.y;
        result.z = ( 2.0f * a.z - 2.0f * b.z + t2.z + t1.z ) * c3 + ( 3.0f * b.z - 3.0f * a.z - 2.0f * t1.z - t2.z ) * c3 + t1.z * amount + a.z;
        result.w = ( 2.0f * a.w - 2.0f * b.w + t2.w + t1.w ) * c3 + ( 3.0f * b.w - 3.0f * a.w - 2.0f * t1.w - t2.w ) * c3 + t1.w * amount + a.w;
    }
}

//-------------------------------------------------------------------------------------------------
//      Catmull-Rom スプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::CatmullRom
(
    const Vector4&  a,
    const Vector4&  b,
    const Vector4&  c,
    const Vector4&  d,
    f32             amount
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    return Vector4(
        ( 0.5f * ( 2.0f * b.x + ( c.x - a.x ) * amount + ( 2.0f * a.x - 5.0f * b.x + 4.0f * c.x - d.x ) * c2 + ( 3.0f * b.x - a.x - 3.0f * c.x + d.x ) * c3 ) ),
        ( 0.5f * ( 2.0f * b.y + ( c.y - a.y ) * amount + ( 2.0f * a.y - 5.0f * b.y + 4.0f * c.y - d.y ) * c2 + ( 3.0f * b.y - a.y - 3.0f * c.y + d.y ) * c3 ) ),
        ( 0.5f * ( 2.0f * b.z + ( c.z - a.z ) * amount + ( 2.0f * a.z - 5.0f * b.z + 4.0f * c.z - d.z ) * c2 + ( 3.0f * b.z - a.z - 3.0f * c.z + d.z ) * c3 ) ),
        ( 0.5f * ( 2.0f * b.w + ( c.w - a.w ) * amount + ( 2.0f * a.w - 5.0f * b.w + 4.0f * c.w - d.w ) * c2 + ( 3.0f * b.w - a.w - 3.0f * c.w + d.w ) * c3 ) )
    );
}

//-------------------------------------------------------------------------------------------------
//      Catmul-Rom スプライン補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::CatmullRom
(
    const Vector4&  a,
    const Vector4&  b,
    const Vector4&  c,
    const Vector4&  d,
    f32             amount,
    Vector4&        result
)
{
    auto c2 = amount * amount;
    auto c3 = c2 * amount;

    result.x = ( 0.5f * ( 2.0f * b.x + ( c.x - a.x ) * amount + ( 2.0f * a.x - 5.0f * b.x + 4.0f * c.x - d.x ) * c2 + ( 3.0f * b.x - a.x - 3.0f * c.x + d.x ) * c3 ) );
    result.y = ( 0.5f * ( 2.0f * b.y + ( c.y - a.y ) * amount + ( 2.0f * a.y - 5.0f * b.y + 4.0f * c.y - d.y ) * c2 + ( 3.0f * b.y - a.y - 3.0f * c.y + d.y ) * c3 ) );
    result.z = ( 0.5f * ( 2.0f * b.z + ( c.z - a.z ) * amount + ( 2.0f * a.z - 5.0f * b.z + 4.0f * c.z - d.z ) * c2 + ( 3.0f * b.z - a.z - 3.0f * c.z + d.z ) * c3 ) );
    result.w = ( 0.5f * ( 2.0f * b.w + ( c.w - a.w ) * amount + ( 2.0f * a.w - 5.0f * b.w + 4.0f * c.w - d.w ) * c2 + ( 3.0f * b.w - a.w - 3.0f * c.w + d.w ) * c3 ) );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Lerp( const Vector4& a, const Vector4& b, f32 amount )
{
    return Vector4(
        a.x + amount * ( b.x - a.x ),
        a.y + amount * ( b.y - a.y ),
        a.z + amount * ( b.z - a.z ),
        a.w + amount * ( b.w - a.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Lerp( const Vector4 &a, const Vector4 &b, f32 amount, Vector4 &result )
{
    result.x = a.x + amount * ( b.x - a.x );
    result.y = a.y + amount * ( b.y - a.y );
    result.z = a.z + amount * ( b.z - a.z );
    result.w = a.w + amount * ( b.w - a.w );
}

//-------------------------------------------------------------------------------------------------
//      3次方程式を用いて，2つの値の間を補間します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::SmoothStep( const Vector4& a, const Vector4& b, f32 amount )
{
    auto s = asdx::Clamp( amount, 0.0f, 1.0f );
    auto u = ( s * s ) + ( 3.0f - ( 2.0f * s ) );
    return Vector4(
        a.x + u * ( b.x - a.x ),
        a.y + u * ( b.y - a.y ),
        a.z + u * ( b.z - a.z ),
        a.w + u * ( b.w - a.w )
    );
}

//-------------------------------------------------------------------------------------------------
//      3次方程式を用いて，2つの値の間を補完します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::SmoothStep( const Vector4 &a, const Vector4 &b, f32 amount, Vector4 &result )
{
    auto s = asdx::Clamp( amount, 0.0f, 1.0f );
    auto u = ( s * s ) + ( 3.0f - ( 2.0f * s ) );
    result.x = a.x + u * ( b.x - a.x );
    result.y = a.y + u * ( b.y - a.y );
    result.z = a.z + u * ( b.z - a.z );
    result.w = a.w + u * ( b.w - a.w );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Vector4 Vector4::Transform( const Vector4& position, const Matrix& matrix )
{
    return Vector4(
        ( ( ((position.x * matrix._11) + (position.y * matrix._21)) + (position.z * matrix._31) ) + (position.w * matrix._41)),
        ( ( ((position.x * matrix._12) + (position.y * matrix._22)) + (position.z * matrix._32) ) + (position.w * matrix._42)),
        ( ( ((position.x * matrix._13) + (position.y * matrix._23)) + (position.z * matrix._33) ) + (position.w * matrix._43)),
        ( ( ((position.x * matrix._14) + (position.y * matrix._24)) + (position.z * matrix._34) ) + (position.w * matrix._44)) );
}

//-------------------------------------------------------------------------------------------------
//      指定された行列を用いて，ベクトルを変換します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Vector4::Transform( const Vector4 &position, const Matrix &matrix, Vector4 &result )
{
    result.x = ( ( ((position.x * matrix._11) + (position.y * matrix._21)) + (position.z * matrix._31) ) + (position.w * matrix._41));
    result.y = ( ( ((position.x * matrix._12) + (position.y * matrix._22)) + (position.z * matrix._32) ) + (position.w * matrix._42));
    result.z = ( ( ((position.x * matrix._13) + (position.y * matrix._23)) + (position.z * matrix._33) ) + (position.w * matrix._43));
    result.w = ( ( ((position.x * matrix._14) + (position.y * matrix._24)) + (position.z * matrix._34) ) + (position.w * matrix._44));
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Matrix structure (row-major)
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix::Matrix()
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix::Matrix( const f32* pf )
{
    assert( pf != nullptr );
    memcpy( &_11, pf, sizeof(Matrix) );
}

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix::Matrix
(
    f32 _f11, f32 _f12, f32 _f13, f32 _f14,
    f32 _f21, f32 _f22, f32 _f23, f32 _f24,
    f32 _f31, f32 _f32, f32 _f33, f32 _f34,
    f32 _f41, f32 _f42, f32 _f43, f32 _f44 
)
{
    _11 = _f11; _12 = _f12; _13 = _f13; _14 = _f14;
    _21 = _f21; _22 = _f22; _23 = _f23; _24 = _f24;
    _31 = _f31; _32 = _f32; _33 = _f33; _34 = _f34;
    _41 = _f41; _42 = _f42; _43 = _f43; _44 = _f44;
}

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix::Matrix( const Vector4& v1, const Vector4& v2, const Vector4& v3, const Vector4& v4 )
{
    _11 = v1.x; _12 = v1.y; _13 = v1.z; _14 = v1.z;
    _21 = v2.x; _22 = v2.y; _23 = v2.z; _24 = v2.z;
    _31 = v3.x; _32 = v3.y; _33 = v3.z; _34 = v3.z;
    _41 = v4.x; _42 = v4.y; _43 = v4.z; _44 = v4.z;
}

//-------------------------------------------------------------------------------------------------
//      インデクサです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
f32& Matrix::operator () ( u32 iRow, u32 iCol )
{ return m[iRow][iCol]; }

//-------------------------------------------------------------------------------------------------
//      インデクサです(const版).
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
const f32& Matrix::operator () ( u32 iRow, u32 iCol ) const
{ return m[iRow][iCol]; }

//-------------------------------------------------------------------------------------------------
//      f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix::operator f32* ()
{ return static_cast<f32*>( &_11 ); }

//-------------------------------------------------------------------------------------------------
//      const f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix::operator const f32* () const 
{ return static_cast<const f32*>( &_11 ); }

//-------------------------------------------------------------------------------------------------
//      乗算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix& Matrix::operator *= ( const Matrix &value )
{
    auto m11 = ( _11 * value._11 ) + ( _12 * value._21 ) + ( _13 * value._31 ) + ( _14 * value._41 );
    auto m12 = ( _11 * value._12 ) + ( _12 * value._22 ) + ( _13 * value._32 ) + ( _14 * value._42 );
    auto m13 = ( _11 * value._13 ) + ( _12 * value._23 ) + ( _13 * value._33 ) + ( _14 * value._43 );
    auto m14 = ( _11 * value._14 ) + ( _12 * value._24 ) + ( _13 * value._34 ) + ( _14 * value._44 );

    auto m21 = ( _21 * value._11 ) + ( _22 * value._21 ) + ( _23 * value._31 ) + ( _24 * value._41 );
    auto m22 = ( _21 * value._12 ) + ( _22 * value._22 ) + ( _23 * value._32 ) + ( _24 * value._42 );
    auto m23 = ( _21 * value._13 ) + ( _22 * value._23 ) + ( _23 * value._33 ) + ( _24 * value._43 );
    auto m24 = ( _21 * value._14 ) + ( _22 * value._24 ) + ( _23 * value._34 ) + ( _24 * value._44 );

    auto m31 = ( _31 * value._11 ) + ( _32 * value._21 ) + ( _33 * value._31 ) + ( _34 * value._41 );
    auto m32 = ( _31 * value._12 ) + ( _32 * value._22 ) + ( _33 * value._32 ) + ( _34 * value._42 );
    auto m33 = ( _31 * value._13 ) + ( _32 * value._23 ) + ( _33 * value._33 ) + ( _34 * value._43 );
    auto m34 = ( _31 * value._14 ) + ( _32 * value._24 ) + ( _33 * value._34 ) + ( _34 * value._44 );

    auto m41 = ( _41 * value._11 ) + ( _42 * value._21 ) + ( _43 * value._31 ) + ( _44 * value._41 );
    auto m42 = ( _41 * value._12 ) + ( _42 * value._22 ) + ( _43 * value._32 ) + ( _44 * value._42 );
    auto m43 = ( _41 * value._13 ) + ( _42 * value._23 ) + ( _43 * value._33 ) + ( _44 * value._43 );
    auto m44 = ( _41 * value._14 ) + ( _42 * value._24 ) + ( _43 * value._34 ) + ( _44 * value._44 );

    _11 = m11;  _12 = m12;  _13 = m13;  _14 = m14;
    _21 = m21;  _22 = m22;  _23 = m23;  _24 = m24;
    _31 = m31;  _32 = m32;  _33 = m33;  _34 = m34;
    _41 = m41;  _42 = m42;  _43 = m43;  _44 = m44;

    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      加算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix& Matrix::operator += ( const Matrix& mat )
{
    _11 += mat._11; _12 += mat._12; _13 += mat._13; _14 += mat._14;
    _21 += mat._21; _22 += mat._22; _23 += mat._23; _24 += mat._24;
    _31 += mat._31; _32 += mat._32; _33 += mat._33; _34 += mat._34;
    _41 += mat._41; _42 += mat._42; _43 += mat._43; _44 += mat._44;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      減算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix& Matrix::operator -= ( const Matrix& mat )
{
    _11 -= mat._11; _12 -= mat._12; _13 -= mat._13; _14 -= mat._14;
    _21 -= mat._21; _22 -= mat._22; _23 -= mat._23; _24 -= mat._24;
    _31 -= mat._31; _32 -= mat._32; _33 -= mat._33; _34 -= mat._34;
    _41 -= mat._41; _42 -= mat._42; _43 -= mat._43; _44 -= mat._44;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      乗算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix& Matrix::operator *= ( f32 f )
{
    _11 *= f; _12 *= f; _13 *= f; _14 *= f;
    _21 *= f; _22 *= f; _23 *= f; _24 *= f;
    _31 *= f; _32 *= f; _33 *= f; _34 *= f;
    _41 *= f; _42 *= f; _43 *= f; _44 *= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      除算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix& Matrix::operator /= ( f32 f )
{
    assert( !IsZero( f ) );
    _11 /= f; _12 /= f; _13 /= f; _14 /= f;
    _21 /= f; _22 /= f; _23 /= f; _24 /= f;
    _31 /= f; _32 /= f; _33 /= f; _34 /= f;
    _41 /= f; _42 /= f; _43 /= f; _44 /= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix& Matrix::operator = ( const Matrix& value )
{
    memcpy( &_11, &value._11, sizeof(Matrix) );
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      正符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::operator + () const
{ return (*this); }

//-------------------------------------------------------------------------------------------------
//      負符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::operator - () const
{
    return Matrix(
        -_11, -_12, -_13, -_14,
        -_21, -_22, -_23, -_24,
        -_31, -_32, -_33, -_34,
        -_41, -_42, -_43, -_44 );
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::operator * ( const Matrix& value ) const
{
    return Matrix(
        ( _11 * value._11 ) + ( _12 * value._21 ) + ( _13 * value._31 ) + ( _14 * value._41 ),
        ( _11 * value._12 ) + ( _12 * value._22 ) + ( _13 * value._32 ) + ( _14 * value._42 ),
        ( _11 * value._13 ) + ( _12 * value._23 ) + ( _13 * value._33 ) + ( _14 * value._43 ),
        ( _11 * value._14 ) + ( _12 * value._24 ) + ( _13 * value._34 ) + ( _14 * value._44 ),

        ( _21 * value._11 ) + ( _22 * value._21 ) + ( _23 * value._31 ) + ( _24 * value._41 ),
        ( _21 * value._12 ) + ( _22 * value._22 ) + ( _23 * value._32 ) + ( _24 * value._42 ),
        ( _21 * value._13 ) + ( _22 * value._23 ) + ( _23 * value._33 ) + ( _24 * value._43 ),
        ( _21 * value._14 ) + ( _22 * value._24 ) + ( _23 * value._34 ) + ( _24 * value._44 ),

        ( _31 * value._11 ) + ( _32 * value._21 ) + ( _33 * value._31 ) + ( _34 * value._41 ),
        ( _31 * value._12 ) + ( _32 * value._22 ) + ( _33 * value._32 ) + ( _34 * value._42 ),
        ( _31 * value._13 ) + ( _32 * value._23 ) + ( _33 * value._33 ) + ( _34 * value._43 ),
        ( _31 * value._14 ) + ( _32 * value._24 ) + ( _33 * value._34 ) + ( _34 * value._44 ),

        ( _41 * value._11 ) + ( _42 * value._21 ) + ( _43 * value._31 ) + ( _44 * value._41 ),
        ( _41 * value._12 ) + ( _42 * value._22 ) + ( _43 * value._32 ) + ( _44 * value._42 ),
        ( _41 * value._13 ) + ( _42 * value._23 ) + ( _43 * value._33 ) + ( _44 * value._43 ),
        ( _41 * value._14 ) + ( _42 * value._24 ) + ( _43 * value._34 ) + ( _44 * value._44 )
    );
}

//-------------------------------------------------------------------------------------------------
//      加算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::operator + ( const Matrix& mat ) const
{
    return Matrix(
        _11 + mat._11, _12 + mat._12, _13 + mat._13, _14 + mat._14,
        _21 + mat._21, _22 + mat._22, _23 + mat._23, _24 + mat._24,
        _31 + mat._31, _32 + mat._32, _33 + mat._33, _34 + mat._34,
        _41 + mat._41, _42 + mat._42, _43 + mat._43, _44 + mat._44 );
}

//-------------------------------------------------------------------------------------------------
//      減算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::operator - ( const Matrix& mat ) const
{
    return Matrix(
        _11 - mat._11, _12 - mat._12, _13 - mat._13, _14 - mat._14,
        _21 - mat._21, _22 - mat._22, _23 - mat._23, _24 - mat._24,
        _31 - mat._31, _32 - mat._32, _33 - mat._33, _34 - mat._34,
        _41 - mat._41, _42 - mat._42, _43 - mat._43, _44 - mat._44 );
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::operator * ( f32 f ) const
{
    return Matrix( 
        _11 * f, _12 * f, _13 * f, _14 * f,
        _21 * f, _22 * f, _23 * f, _24 * f,
        _31 * f, _32 * f, _33 * f, _34 * f,
        _41 * f, _42 * f, _43 * f, _44 * f );
}

//-------------------------------------------------------------------------------------------------
//      除算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::operator / ( f32 f ) const
{
    assert( !IsZero( f ) );
    return Matrix(
        _11 / f, _12 / f, _13 / f, _14 / f,
        _21 / f, _22 / f, _23 / f, _24 / f,
        _31 / f, _32 / f, _33 / f, _34 / f,
        _41 / f, _42 / f, _43 / f, _44 / f);
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix operator * ( f32 f, const Matrix& mat )
{
    return Matrix(
        f * mat._11, f * mat._12, f * mat._13, f * mat._14,
        f * mat._21, f * mat._22, f * mat._23, f * mat._24,
        f * mat._31, f * mat._32, f * mat._33, f * mat._34,
        f * mat._41, f * mat._42, f * mat._43, f * mat._44 );
}

//-------------------------------------------------------------------------------------------------
//      等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
bool Matrix::operator == ( const Matrix& mat ) const
{ return ( 0 == memcmp( this, &mat, sizeof( Matrix ) ) ); }

//-------------------------------------------------------------------------------------------------
//      非等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
bool Matrix::operator != ( const Matrix& mat ) const
{ return ( 0 != memcmp( this, &mat, sizeof( Matrix ) ) ); }

//-------------------------------------------------------------------------------------------------
//      行列式を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
f32 Matrix::Determinant() const
{
    return
        _11 * _22 * _33 * _44 + _11 * _23 * _34 * _42 +
        _11 * _24 * _32 * _43 + _12 * _21 * _34 * _43 +
        _12 * _23 * _31 * _44 + _12 * _24 * _33 * _41 +
        _13 * _21 * _32 * _44 + _13 * _22 * _34 * _41 +
        _13 * _24 * _31 * _42 + _14 * _21 * _33 * _42 +
        _14 * _22 * _31 * _43 + _14 * _23 * _32 * _41 -
        _11 * _22 * _34 * _43 - _11 * _23 * _32 * _44 -
        _11 * _24 * _33 * _42 - _12 * _21 * _33 * _44 -
        _12 * _23 * _34 * _41 - _12 * _24 * _31 * _43 -
        _13 * _21 * _34 * _42 - _13 * _22 * _31 * _44 -
        _13 * _24 * _32 * _41 - _14 * _21 * _32 * _43 -
        _14 * _22 * _33 * _41 - _14 * _23 * _31 * _42;
}

//-------------------------------------------------------------------------------------------------
//      単位行列化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix& Matrix::Identity()
{
    _11 = _22 = _33 = _44 = 1.0f;
    _12 = _13 = _14 =
    _21 = _23 = _24 =
    _31 = _32 = _34 =
    _41 = _42 = _43 = 0.0f;
    return (*this);
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Matrix Methods (row-major)
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      単位行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateIdentity()
{
    return Matrix(
        1.0f, 0.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 1.0f, 0.0f,
        0.0f, 0.0f, 0.0f, 1.0f );
}

//-------------------------------------------------------------------------------------------------
//      単位行列かどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Matrix::IsIdentity( const Matrix &value )
{
    return (
        IsEqual( value.m[0][0], 1.0f ) && IsZero( value.m[0][1] ) && IsZero( value.m[0][2] ) && IsZero( value.m[0][3] ) &&
        IsZero( value.m[1][0] ) && IsEqual( value.m[1][1], 1.0f ) && IsZero( value.m[1][2] ) && IsZero( value.m[1][3] ) &&
        IsZero( value.m[2][0] ) && IsZero( value.m[2][1] ) && IsEqual( value.m[2][2] , 1.0f ) && IsZero( value.m[2][3] ) &&
        IsZero( value.m[3][0] ) && IsZero( value.m[3][1] ) && IsZero( value.m[3][2] ) && IsEqual( value.m[3][3], 1.0f ) );
}

//-------------------------------------------------------------------------------------------------
//      行列を転置します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::Transpose( const Matrix& value )
{
    return Matrix(
        value._11, value._21, value._31, value._41,
        value._12, value._22, value._32, value._42,
        value._13, value._23, value._33, value._43,
        value._14, value._24, value._34, value._44 );
}

//-------------------------------------------------------------------------------------------------
//      行列を転置します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::Transpose( const Matrix &value, Matrix &result )
{
    result._11 = value._11;
    result._12 = value._21;
    result._13 = value._31;
    result._14 = value._41;

    result._21 = value._12;
    result._22 = value._22;
    result._23 = value._32;
    result._24 = value._42;

    result._31 = value._13;
    result._32 = value._23;
    result._33 = value._33;
    result._34 = value._34;

    result._41 = value._14;
    result._42 = value._24;
    result._43 = value._34;
    result._44 = value._44;
}

//-------------------------------------------------------------------------------------------------
//      行列同士を乗算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::Multiply( const Matrix& a, const Matrix& b )
{
    return Matrix(
        ( a._11 * b._11 ) + ( a._12 * b._21 ) + ( a._13 * b._31 ) + ( a._14 * b._41 ),
        ( a._11 * b._12 ) + ( a._12 * b._22 ) + ( a._13 * b._32 ) + ( a._14 * b._42 ),
        ( a._11 * b._13 ) + ( a._12 * b._23 ) + ( a._13 * b._33 ) + ( a._14 * b._43 ),
        ( a._11 * b._14 ) + ( a._12 * b._24 ) + ( a._13 * b._34 ) + ( a._14 * b._44 ),

        ( a._21 * b._11 ) + ( a._22 * b._21 ) + ( a._23 * b._31 ) + ( a._24 * b._41 ),
        ( a._21 * b._12 ) + ( a._22 * b._22 ) + ( a._23 * b._32 ) + ( a._24 * b._42 ),
        ( a._21 * b._13 ) + ( a._22 * b._23 ) + ( a._23 * b._33 ) + ( a._24 * b._43 ),
        ( a._21 * b._14 ) + ( a._22 * b._24 ) + ( a._23 * b._34 ) + ( a._24 * b._44 ),

        ( a._31 * b._11 ) + ( a._32 * b._21 ) + ( a._33 * b._31 ) + ( a._34 * b._41 ),
        ( a._31 * b._12 ) + ( a._32 * b._22 ) + ( a._33 * b._32 ) + ( a._34 * b._42 ),
        ( a._31 * b._13 ) + ( a._32 * b._23 ) + ( a._33 * b._33 ) + ( a._34 * b._43 ),
        ( a._31 * b._14 ) + ( a._32 * b._24 ) + ( a._33 * b._34 ) + ( a._34 * b._44 ),

        ( a._41 * b._11 ) + ( a._42 * b._21 ) + ( a._43 * b._31 ) + ( a._44 * b._41 ),
        ( a._41 * b._12 ) + ( a._42 * b._22 ) + ( a._43 * b._32 ) + ( a._44 * b._42 ),
        ( a._41 * b._13 ) + ( a._42 * b._23 ) + ( a._43 * b._33 ) + ( a._44 * b._43 ),
        ( a._41 * b._14 ) + ( a._42 * b._24 ) + ( a._43 * b._34 ) + ( a._44 * b._44 )
    );
}

//-------------------------------------------------------------------------------------------------
//      行列同士を乗算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::Multiply( const Matrix &a, const Matrix &b, Matrix &result )
{
    result._11 = ( a._11 * b._11 ) + ( a._12 * b._21 ) + ( a._13 * b._31 ) + ( a._14 * b._41 );
    result._12 = ( a._11 * b._12 ) + ( a._12 * b._22 ) + ( a._13 * b._32 ) + ( a._14 * b._42 );
    result._13 = ( a._11 * b._13 ) + ( a._12 * b._23 ) + ( a._13 * b._33 ) + ( a._14 * b._43 );
    result._14 = ( a._11 * b._14 ) + ( a._12 * b._24 ) + ( a._13 * b._34 ) + ( a._14 * b._44 );

    result._21 = ( a._21 * b._11 ) + ( a._22 * b._21 ) + ( a._23 * b._31 ) + ( a._24 * b._41 );
    result._22 = ( a._21 * b._12 ) + ( a._22 * b._22 ) + ( a._23 * b._32 ) + ( a._24 * b._42 );
    result._23 = ( a._21 * b._13 ) + ( a._22 * b._23 ) + ( a._23 * b._33 ) + ( a._24 * b._43 );
    result._24 = ( a._21 * b._14 ) + ( a._22 * b._24 ) + ( a._23 * b._34 ) + ( a._24 * b._44 );

    result._31 = ( a._31 * b._11 ) + ( a._32 * b._21 ) + ( a._33 * b._31 ) + ( a._34 * b._41 );
    result._32 = ( a._31 * b._12 ) + ( a._32 * b._22 ) + ( a._33 * b._32 ) + ( a._34 * b._42 );
    result._33 = ( a._31 * b._13 ) + ( a._32 * b._23 ) + ( a._33 * b._33 ) + ( a._34 * b._43 );
    result._34 = ( a._31 * b._14 ) + ( a._32 * b._24 ) + ( a._33 * b._34 ) + ( a._34 * b._44 );

    result._41 = ( a._41 * b._11 ) + ( a._42 * b._21 ) + ( a._43 * b._31 ) + ( a._44 * b._41 );
    result._42 = ( a._41 * b._12 ) + ( a._42 * b._22 ) + ( a._43 * b._32 ) + ( a._44 * b._42 );
    result._43 = ( a._41 * b._13 ) + ( a._42 * b._23 ) + ( a._43 * b._33 ) + ( a._44 * b._43 );
    result._44 = ( a._41 * b._14 ) + ( a._42 * b._24 ) + ( a._43 * b._34 ) + ( a._44 * b._44 );
}

//-------------------------------------------------------------------------------------------------
//      スカラー乗算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::Multiply( const Matrix& value, f32 scaleFactor )
{
    return Matrix(
        value._11 * scaleFactor, value._12 * scaleFactor, value._13 * scaleFactor, value._14 * scaleFactor,
        value._21 * scaleFactor, value._22 * scaleFactor, value._23 * scaleFactor, value._24 * scaleFactor,
        value._31 * scaleFactor, value._32 * scaleFactor, value._33 * scaleFactor, value._34 * scaleFactor,
        value._41 * scaleFactor, value._42 * scaleFactor, value._43 * scaleFactor, value._44 * scaleFactor );
}

//-------------------------------------------------------------------------------------------------
//      スカラー乗算します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::Multiply( const Matrix &value, f32 scaleFactor, Matrix &result )
{
    result._11 = value._11 * scaleFactor;
    result._12 = value._12 * scaleFactor;
    result._13 = value._13 * scaleFactor;
    result._14 = value._14 * scaleFactor;

    result._21 = value._21 * scaleFactor;
    result._22 = value._22 * scaleFactor;
    result._23 = value._23 * scaleFactor;
    result._24 = value._24 * scaleFactor;

    result._31 = value._31 * scaleFactor;
    result._32 = value._32 * scaleFactor;
    result._33 = value._33 * scaleFactor;
    result._34 = value._34 * scaleFactor;

    result._41 = value._41 * scaleFactor;
    result._42 = value._42 * scaleFactor;
    result._43 = value._43 * scaleFactor;
    result._44 = value._44 * scaleFactor;
}

//-------------------------------------------------------------------------------------------------
//      行列同士を乗算し，乗算結果を転置します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::MultiplyTranspose( const Matrix& a, const Matrix& b )
{
    return Matrix(
        ( a._11 * b._11 ) + ( a._12 * b._21 ) + ( a._13 * b._31 ) + ( a._14 * b._41 ),
        ( a._11 * b._12 ) + ( a._12 * b._22 ) + ( a._13 * b._32 ) + ( a._14 * b._42 ),
        ( a._11 * b._13 ) + ( a._12 * b._23 ) + ( a._13 * b._33 ) + ( a._14 * b._43 ),
        ( a._11 * b._14 ) + ( a._12 * b._24 ) + ( a._13 * b._34 ) + ( a._14 * b._44 ),

        ( a._21 * b._11 ) + ( a._22 * b._21 ) + ( a._23 * b._31 ) + ( a._24 * b._41 ),
        ( a._21 * b._12 ) + ( a._22 * b._22 ) + ( a._23 * b._32 ) + ( a._24 * b._42 ),
        ( a._21 * b._13 ) + ( a._22 * b._23 ) + ( a._23 * b._33 ) + ( a._24 * b._43 ),
        ( a._21 * b._14 ) + ( a._22 * b._24 ) + ( a._23 * b._34 ) + ( a._24 * b._44 ),

        ( a._31 * b._11 ) + ( a._32 * b._21 ) + ( a._33 * b._31 ) + ( a._34 * b._41 ),
        ( a._31 * b._12 ) + ( a._32 * b._22 ) + ( a._33 * b._32 ) + ( a._34 * b._42 ),
        ( a._31 * b._13 ) + ( a._32 * b._23 ) + ( a._33 * b._33 ) + ( a._34 * b._43 ),
        ( a._31 * b._14 ) + ( a._32 * b._24 ) + ( a._33 * b._34 ) + ( a._34 * b._44 ),

        ( a._41 * b._11 ) + ( a._42 * b._21 ) + ( a._43 * b._31 ) + ( a._44 * b._41 ),
        ( a._41 * b._12 ) + ( a._42 * b._22 ) + ( a._43 * b._32 ) + ( a._44 * b._42 ),
        ( a._41 * b._13 ) + ( a._42 * b._23 ) + ( a._43 * b._33 ) + ( a._44 * b._43 ),
        ( a._41 * b._14 ) + ( a._42 * b._24 ) + ( a._43 * b._34 ) + ( a._44 * b._44 )
    );
}

//-------------------------------------------------------------------------------------------------
//      行列同士を乗算し，乗算結果を転置します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::MultiplyTranspose( const Matrix &a, const Matrix &b, Matrix &result )
{
    result._11 = ( a._11 * b._11 ) + ( a._12 * b._21 ) + ( a._13 * b._31 ) + ( a._14 * b._41 );
    result._21 = ( a._11 * b._12 ) + ( a._12 * b._22 ) + ( a._13 * b._32 ) + ( a._14 * b._42 );
    result._31 = ( a._11 * b._13 ) + ( a._12 * b._23 ) + ( a._13 * b._33 ) + ( a._14 * b._43 );
    result._41 = ( a._11 * b._14 ) + ( a._12 * b._24 ) + ( a._13 * b._34 ) + ( a._14 * b._44 );

    result._12 = ( a._21 * b._11 ) + ( a._22 * b._21 ) + ( a._23 * b._31 ) + ( a._24 * b._41 );
    result._22 = ( a._21 * b._12 ) + ( a._22 * b._22 ) + ( a._23 * b._32 ) + ( a._24 * b._42 );
    result._32 = ( a._21 * b._13 ) + ( a._22 * b._23 ) + ( a._23 * b._33 ) + ( a._24 * b._43 );
    result._42 = ( a._21 * b._14 ) + ( a._22 * b._24 ) + ( a._23 * b._34 ) + ( a._24 * b._44 );

    result._13 = ( a._31 * b._11 ) + ( a._32 * b._21 ) + ( a._33 * b._31 ) + ( a._34 * b._41 );
    result._23 = ( a._31 * b._12 ) + ( a._32 * b._22 ) + ( a._33 * b._32 ) + ( a._34 * b._42 );
    result._33 = ( a._31 * b._13 ) + ( a._32 * b._23 ) + ( a._33 * b._33 ) + ( a._34 * b._43 );
    result._43 = ( a._31 * b._14 ) + ( a._32 * b._24 ) + ( a._33 * b._34 ) + ( a._34 * b._44 );

    result._14 = ( a._41 * b._11 ) + ( a._42 * b._21 ) + ( a._43 * b._31 ) + ( a._44 * b._41 );
    result._24 = ( a._41 * b._12 ) + ( a._42 * b._22 ) + ( a._43 * b._32 ) + ( a._44 * b._42 );
    result._34 = ( a._41 * b._13 ) + ( a._42 * b._23 ) + ( a._43 * b._33 ) + ( a._44 * b._43 );
    result._44 = ( a._41 * b._14 ) + ( a._42 * b._24 ) + ( a._43 * b._34 ) + ( a._44 * b._44 );
}

//-------------------------------------------------------------------------------------------------
//      逆行列を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::Invert( const Matrix& value )
{
    auto det = value.Determinant();
    assert( !IsZero( det ) );

    auto m11 = value._22*value._33*value._44 + value._23*value._34*value._42 + value._24*value._32*value._43 - value._22*value._34*value._43 - value._23*value._32*value._44 - value._24*value._33*value._42;
    auto m12 = value._12*value._34*value._43 + value._13*value._32*value._44 + value._14*value._33*value._42 - value._12*value._33*value._44 - value._13*value._34*value._42 - value._14*value._32*value._43;
    auto m13 = value._12*value._23*value._44 + value._13*value._24*value._42 + value._14*value._22*value._43 - value._12*value._24*value._43 - value._13*value._22*value._44 - value._14*value._23*value._42;
    auto m14 = value._12*value._24*value._33 + value._13*value._22*value._34 + value._14*value._23*value._32 - value._12*value._23*value._34 - value._13*value._24*value._32 - value._14*value._22*value._33;

    auto m21 = value._21*value._34*value._43 + value._23*value._31*value._44 + value._24*value._33*value._41 - value._21*value._33*value._44 - value._23*value._34*value._41 - value._24*value._31*value._43;
    auto m22 = value._11*value._33*value._44 + value._13*value._34*value._41 + value._14*value._31*value._43 - value._11*value._34*value._43 - value._13*value._31*value._44 - value._14*value._33*value._41;
    auto m23 = value._11*value._24*value._43 + value._13*value._21*value._44 + value._14*value._23*value._41 - value._11*value._23*value._44 - value._13*value._24*value._41 - value._14*value._21*value._43;
    auto m24 = value._11*value._23*value._34 + value._13*value._24*value._31 + value._14*value._21*value._33 - value._11*value._24*value._33 - value._13*value._21*value._34 - value._14*value._23*value._31;

    auto m31 = value._21*value._32*value._44 + value._22*value._34*value._41 + value._24*value._31*value._42 - value._21*value._34*value._42 - value._22*value._31*value._44 - value._24*value._32*value._41;
    auto m32 = value._11*value._34*value._42 + value._12*value._31*value._44 + value._14*value._32*value._41 - value._11*value._32*value._44 - value._12*value._34*value._41 - value._14*value._31*value._42;
    auto m33 = value._11*value._22*value._44 + value._12*value._24*value._41 + value._14*value._21*value._42 - value._11*value._24*value._42 - value._12*value._21*value._44 - value._14*value._22*value._41;
    auto m34 = value._11*value._24*value._32 + value._12*value._21*value._34 + value._14*value._22*value._31 - value._11*value._22*value._34 - value._12*value._24*value._31 - value._14*value._21*value._32;

    auto m41 = value._21*value._33*value._42 + value._22*value._31*value._43 + value._23*value._32*value._41 - value._21*value._32*value._43 - value._22*value._33*value._41 - value._23*value._31*value._42;
    auto m42 = value._11*value._32*value._43 + value._12*value._33*value._41 + value._13*value._31*value._42 - value._11*value._33*value._42 - value._12*value._31*value._43 - value._13*value._32*value._41;
    auto m43 = value._11*value._23*value._42 + value._12*value._21*value._43 + value._13*value._22*value._41 - value._11*value._22*value._43 - value._12*value._23*value._41 - value._13*value._21*value._42;
    auto m44 = value._11*value._22*value._33 + value._12*value._23*value._31 + value._13*value._21*value._32 - value._11*value._23*value._32 - value._12*value._21*value._33 - value._13*value._22*value._31;

    return Matrix(
        m11 / det, m12 / det, m13 / det, m14 / det,
        m21 / det, m22 / det, m23 / det, m24 / det,
        m31 / det, m32 / det, m33 / det, m34 / det,
        m41 / det, m42 / det, m43 / det, m44 / det );
}

//-------------------------------------------------------------------------------------------------
//      逆行列を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::Invert( const Matrix &value, Matrix &result )
{ 
    auto det = value.Determinant();
    assert( det != 0.0f );

    result._11 = value._22*value._33*value._44 + value._23*value._34*value._42 + value._24*value._32*value._43 - value._22*value._34*value._43 - value._23*value._32*value._44 - value._24*value._33*value._42;
    result._12 = value._12*value._34*value._43 + value._13*value._32*value._44 + value._14*value._33*value._42 - value._12*value._33*value._44 - value._13*value._34*value._42 - value._14*value._32*value._43;
    result._13 = value._12*value._23*value._44 + value._13*value._24*value._42 + value._14*value._22*value._43 - value._12*value._24*value._43 - value._13*value._22*value._44 - value._14*value._23*value._42;
    result._14 = value._12*value._24*value._33 + value._13*value._22*value._34 + value._14*value._23*value._32 - value._12*value._23*value._34 - value._13*value._24*value._32 - value._14*value._22*value._33;

    result._21 = value._21*value._34*value._43 + value._23*value._31*value._44 + value._24*value._33*value._41 - value._21*value._33*value._44 - value._23*value._34*value._41 - value._24*value._31*value._43;
    result._22 = value._11*value._33*value._44 + value._13*value._34*value._41 + value._14*value._31*value._43 - value._11*value._34*value._43 - value._13*value._31*value._44 - value._14*value._33*value._41;
    result._23 = value._11*value._24*value._43 + value._13*value._21*value._44 + value._14*value._23*value._41 - value._11*value._23*value._44 - value._13*value._24*value._41 - value._14*value._21*value._43;
    result._24 = value._11*value._23*value._34 + value._13*value._24*value._31 + value._14*value._21*value._33 - value._11*value._24*value._33 - value._13*value._21*value._34 - value._14*value._23*value._31;

    result._31 = value._21*value._32*value._44 + value._22*value._34*value._41 + value._24*value._31*value._42 - value._21*value._34*value._42 - value._22*value._31*value._44 - value._24*value._32*value._41;
    result._32 = value._11*value._34*value._42 + value._12*value._31*value._44 + value._14*value._32*value._41 - value._11*value._32*value._44 - value._12*value._34*value._41 - value._14*value._31*value._42;
    result._33 = value._11*value._22*value._44 + value._12*value._24*value._41 + value._14*value._21*value._42 - value._11*value._24*value._42 - value._12*value._21*value._44 - value._14*value._22*value._41;
    result._34 = value._11*value._24*value._32 + value._12*value._21*value._34 + value._14*value._22*value._31 - value._11*value._22*value._34 - value._12*value._24*value._31 - value._14*value._21*value._32;

    result._41 = value._21*value._33*value._42 + value._22*value._31*value._43 + value._23*value._32*value._41 - value._21*value._32*value._43 - value._22*value._33*value._41 - value._23*value._31*value._42;
    result._42 = value._11*value._32*value._43 + value._12*value._33*value._41 + value._13*value._31*value._42 - value._11*value._33*value._42 - value._12*value._31*value._43 - value._13*value._32*value._41;
    result._43 = value._11*value._23*value._42 + value._12*value._21*value._43 + value._13*value._22*value._41 - value._11*value._22*value._43 - value._12*value._23*value._41 - value._13*value._21*value._42;
    result._44 = value._11*value._22*value._33 + value._12*value._23*value._31 + value._13*value._21*value._32 - value._11*value._23*value._32 - value._12*value._21*value._33 - value._13*value._22*value._31;

    result._11 /= det;
    result._12 /= det;
    result._13 /= det;
    result._14 /= det;

    result._21 /= det;
    result._22 /= det;
    result._23 /= det;
    result._24 /= det;

    result._31 /= det;
    result._32 /= det;
    result._33 /= det;
    result._34 /= det;

    result._41 /= det;
    result._42 /= det;
    result._43 /= det;
    result._44 /= det;
}

//-------------------------------------------------------------------------------------------------
//      拡大・縮小行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateScale( f32 scale )
{
    return Matrix(
        scale, 0.0f, 0.0f, 0.0f,
        0.0f, scale, 0.0f, 0.0f,
        0.0f, 0.0f, scale, 0.0f,
        0.0f, 0.0f, 0.0f, 1.0f );
}

//-------------------------------------------------------------------------------------------------
//      拡大・縮小行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
void Matrix::CreateScale( f32 scale, Matrix &result )
{
    result._11 = scale;    result._12 = 0.0f;    result._13 = 0.0f;     result._14 = 0.0f;
    result._21 = 0.0f;     result._22 = scale;   result._23 = 0.0f;     result._24 = 0.0f;
    result._31 = 0.0f;     result._32 = 0.0f;    result._33 = scale;    result._34 = 0.0f;
    result._41 = 0.0f;     result._42 = 0.0f;    result._43 = 0.0f;     result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      拡大・縮小行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::CreateScale( f32 xScale, f32 yScale, f32 zScale )
{
    return Matrix(
        xScale, 0.0f,   0.0f,   0.0f,
        0.0f,   yScale, 0.0f,   0.0f,
        0.0f,   0.0f,   zScale, 0.0f,
        0.0f,   0.0f,   0.0f,   1.0f );
}

//-------------------------------------------------------------------------------------------------
//      拡大・縮小行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateScale( f32 xScale, f32 yScale, f32 zScale, Matrix &result )
{
    result._11 = xScale;    result._12 = 0.0f;      result._13 = 0.0f;      result._14 = 0.0f;
    result._21 = 0.0f;      result._22 = yScale;    result._23 = 0.0f;      result._24 = 0.0f;
    result._31 = 0.0f;      result._32 = 0.0f;      result._33 = zScale;    result._34 = 0.0f;
    result._41 = 0.0f;      result._42 = 0.0f;      result._43 = 0.0f;      result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      拡大・縮小行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateScale( const Vector3& scales )
{
    return Matrix(
        scales.x,   0.0f,       0.0f,       0.0f,
        0.0f,       scales.y,   0.0f,       0.0f,
        0.0f,       0.0f,       scales.z,   0.0f,
        0.0f,       0.0f,       0.0f,       1.0f ); 
}

//-------------------------------------------------------------------------------------------------
//      拡大・縮小行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateScale( const Vector3 &scales, Matrix &result )
{
    result._11 = scales.x;  result._12 = 0.0f;      result._13 = 0.0f;      result._14 = 0.0f;
    result._21 = 0.0f;      result._22 = scales.y;  result._23 = 0.0f;      result._24 = 0.0f;
    result._31 = 0.0f;      result._32 = 0.0f;      result._33 = scales.z;  result._34 = 0.0f;
    result._41 = 0.0f;      result._42 = 0.0f;      result._43 = 0.0f;      result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      平行移動行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::CreateTranslation( f32 xPos, f32 yPos, f32 zPos )
{
    return Matrix(
        1.0f, 0.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 1.0f, 0.0f,
        xPos, yPos, zPos, 1.0f );
}

//-------------------------------------------------------------------------------------------------
//      平行移動行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateTranslation( f32 xPos, f32 yPos, f32 zPos, Matrix &result )
{
    result._11 = 1.0f;  result._12 = 0.0f;  result._13 = 0.0f;  result._14 = 0.0f;
    result._21 = 0.0f;  result._22 = 1.0f;  result._23 = 0.0f;  result._24 = 0.0f;
    result._31 = 0.0f;  result._32 = 0.0f;  result._33 = 1.0f;  result._34 = 0.0f;
    result._41 = xPos;  result._42 = yPos;  result._43 = zPos;  result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      平行移動行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateTranslation( const Vector3& pos )
{
    return Matrix(
        1.0f, 0.0f, 0.0f, 0.0f,
        0.0f, 1.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 1.0f, 0.0f,
        pos.x, pos.y, pos.z, 1.0f );
}

//-------------------------------------------------------------------------------------------------
//      平行移動行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateTranslation( const Vector3 &pos, Matrix &result )
{
    result._11 = 1.0f;  result._12 = 0.0f;  result._13 = 0.0f;  result._14 = 0.0f;
    result._21 = 0.0f;  result._22 = 1.0f;  result._23 = 0.0f;  result._24 = 0.0f;
    result._31 = 0.0f;  result._32 = 0.0f;  result._33 = 1.0f;  result._34 = 0.0f;
    result._41 = pos.x; result._42 = pos.y; result._43 = pos.z; result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      X軸周りの回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::CreateRotationX( const f32 radian )
{
    auto cosRad = cosf(radian);
    auto sinRad = sinf(radian);
    return Matrix(
        1.0f,   0.0f,   0.0f,   0.0f,
        0.0f,   cosRad, sinRad, 0.0f,
        0.0f,  -sinRad, cosRad, 0.0f,
        0.0f,   0.0f,   0.0f,   1.0f );
}

//-------------------------------------------------------------------------------------------------
//      X軸周りの回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
void Matrix::CreateRotationX( const f32 radian, Matrix &result )
{
    auto cosRad = cosf( radian );
    auto sinRad = sinf( radian );

    result._11 = 1.0f;      result._12 = 0.0f;      result._13 = 0.0f;      result._14 = 0.0f;
    result._21 = 0.0f;      result._22 = cosRad;    result._23 = sinRad;    result._24 = 0.0f;
    result._31 = 0.0f;      result._32 = -sinRad;   result._33 = cosRad;    result._34 = 0.0f;
    result._41 = 0.0f;      result._42 = 0.0f;      result._43 = 0.0f;      result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      Y軸周りの回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::CreateRotationY( const f32 radian )
{
    auto cosRad = cosf( radian );
    auto sinRad = sinf( radian );

    return Matrix(
        cosRad, 0.0f,  -sinRad, 0.0f,
        0.0f,   1.0f,   0.0f,   0.0f,
        sinRad, 0.0f,   cosRad, 0.0f,
        0.0f,   0.0f,   0.0f,   1.0f );
}

//-------------------------------------------------------------------------------------------------
//      Y軸周りの回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateRotationY( const f32 radian, Matrix &result )
{
    auto cosRad = cosf( radian );
    auto sinRad = sinf( radian );

    result._11 = cosRad;    result._12 = 0.0f;  result._13 = -sinRad;   result._14 = 0.0f;
    result._21 = 0.0f;      result._22 = 1.0f;  result._23 = 0.0f;      result._24 = 0.0f;
    result._31 = sinRad;    result._32 = 0.0f;  result._33 = cosRad;    result._34 = 0.0f;
    result._41 = 0.0f;      result._42 = 0.0f;  result._43 = 0.0f;      result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      Z軸周りの回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateRotationZ( const f32 radian )
{
    auto cosRad = cosf( radian );
    auto sinRad = sinf( radian );

    return Matrix( 
        cosRad, sinRad, 0.0f, 0.0f,
       -sinRad, cosRad, 0.0f, 0.0f,
        0.0f,   0.0f,   1.0f, 0.0f,
        0.0f,   0.0f,   0.0f, 1.0f );
}

//-------------------------------------------------------------------------------------------------
//      Z軸周りの回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateRotationZ( const f32 radian, Matrix &result )
{
    auto cosRad = cosf( radian );
    auto sinRad = sinf( radian );

    result._11 = cosRad;    result._12 = sinRad;    result._13 = 0.0f;    result._14 = 0.0f;
    result._21 = -sinRad;   result._22 = cosRad;    result._23 = 0.0f;    result._24 = 0.0f;
    result._31 = 0.0f;      result._32 = 0.0f;      result._33 = 1.0f;    result._34 = 0.0f;
    result._41 = 0.0f;      result._42 = 0.0f;      result._43 = 0.0f;    result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      四元数から回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateFromQuaternion( const Quaternion& qua )
{
    auto xx = qua.x * qua.x;
    auto yy = qua.y * qua.y;
    auto zz = qua.z * qua.z;
    auto xy = qua.x * qua.y;
    auto yw = qua.y * qua.w;
    auto yz = qua.y * qua.z;
    auto xw = qua.x * qua.w;
    auto zx = qua.z * qua.x;
    auto zw = qua.z * qua.w;

    return Matrix(
        1.0f - (2.0f * (yy + zz)),
        2.0f * (xy + zw),
        2.0f * (zx - yw),
        0.0f,

        2.0f * (xy - zw),
        1.0f - (2.0f * (zz + xx)), 
        2.0f * (yz + xw),
        0.0f,

        2.0f * (zx + yw), 
        2.0f * (yz - xw),
        1.0f - (2.0f * (yy + xx)),
        0.0f,

        0.0f,
        0.0f,
        0.0f,
        1.0f );
}

//-------------------------------------------------------------------------------------------------
//      四元数から回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateFromQuaternion( const Quaternion &qua, Matrix &result )
{
    auto xx = qua.x * qua.x;
    auto yy = qua.y * qua.y;
    auto zz = qua.z * qua.z;
    auto xy = qua.x * qua.y;
    auto yw = qua.y * qua.w;
    auto yz = qua.y * qua.z;
    auto xw = qua.x * qua.w;
    auto zx = qua.z * qua.x;
    auto zw = qua.z * qua.w;

    result._11 = 1.0f - (2.0f * (yy + zz));
    result._12 = 2.0f * (xy + zw);
    result._13 = 2.0f * (zx - yw);
    result._14 = 0.0f;

    result._21 = 2.0f * (xy - zw);
    result._22 = 1.0f - (2.0f * (zz + xx)); 
    result._23 = 2.0f * (yz + xw);
    result._24 = 0.0f;

    result._31 = 2.0f * (zx + yw); 
    result._32 = 2.0f * (yz - xw);
    result._33 = 1.0f - (2.0f * (yy + xx));
    result._34 = 0.0f;

    result._41 = 0.0f;
    result._42 = 0.0f;
    result._43 = 0.0f;
    result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      指定された軸と角度から回転行列を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateFromAxisAngle( const Vector3& axis, f32 radian )
{
    auto sinRad = sinf(radian);
    auto cosRad = cosf(radian);
    auto a = 1.0f -cosRad;
    
    auto ab = axis.x * axis.y * a;
    auto bc = axis.y * axis.z * a;
    auto ca = axis.z * axis.x * a;
    auto tx = axis.x * axis.x;
    auto ty = axis.y * axis.y;
    auto tz = axis.z * axis.z;

    return Matrix(
        tx + cosRad * (1.0f - tx),
        ab + axis.z * sinRad,
        ca - axis.y * sinRad,
        0.0f,

        ab - axis.z * sinRad,
        ty + cosRad * (1.0f - ty),
        bc + axis.x * sinRad,
        0.0f,

        ca + axis.y * sinRad,
        bc - axis.x * sinRad,
        tz + cosRad * (1.0f - tz),
        0.0f,

        0.0f,
        0.0f,
        0.0f,
        1.0f );
}

//-------------------------------------------------------------------------------------------------
//      指定された軸と角度から回転行列を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateFromAxisAngle( const Vector3 &axis, f32 radian, Matrix &result )
{
    auto sinRad = sinf(radian);
    auto cosRad = cosf(radian);
    auto a = 1.0f -cosRad;
    
    auto ab = axis.x * axis.y * a;
    auto bc = axis.y * axis.z * a;
    auto ca = axis.z * axis.x * a;
    auto tx = axis.x * axis.x;
    auto ty = axis.y * axis.y;
    auto tz = axis.z * axis.z;

    result._11 = tx + cosRad * (1.0f - tx);
    result._12 = ab + axis.z * sinRad;
    result._13 = ca - axis.y * sinRad;
    result._14 = 0.0f;

    result._21 = ab - axis.z * sinRad;
    result._22 = ty + cosRad * (1.0f - ty);
    result._23 = bc + axis.x * sinRad;
    result._24 = 0.0f;

    result._31 = ca + axis.y * sinRad;
    result._32 = bc - axis.x * sinRad;
    result._33 = tz + cosRad * (1.0f - tz);
    result._34 = 0.0f;

    result._41 = 0.0f;
    result._42 = 0.0f;
    result._43 = 0.0f;
    result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      ヨー・ピッチ・ロール角から回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateRotationFromYawPitchRoll( f32 yaw, f32 pitch, f32 roll )
{
    auto value = Quaternion::CreateFromYawPitchRoll( yaw, pitch, roll );
    return Matrix::CreateFromQuaternion( value );
}

//-------------------------------------------------------------------------------------------------
//      ヨー・ピッチ・ロール角から回転行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateRotationFromYawPitchRoll( f32 yaw, f32 pitch, f32 roll, Matrix& result )
{
    auto value = Quaternion::CreateFromYawPitchRoll( yaw, pitch, roll );
    Matrix::CreateFromQuaternion( value, result );
}

//-------------------------------------------------------------------------------------------------
//      注視点を基にビュー行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateLookAt
(
    const Vector3& cameraPosition,
    const Vector3& cameraTarget,
    const Vector3& cameraUpVector
)
{
    auto zaxis = cameraPosition - cameraTarget;
    zaxis.Normalize();

    auto xaxis = Vector3::Cross(cameraUpVector, zaxis);
    xaxis.Normalize();

    auto yaxis = Vector3::Cross(zaxis, xaxis);
    yaxis.Normalize();

    return Matrix(
        xaxis.x, yaxis.x, zaxis.x, 0.0f,
        xaxis.y, yaxis.y, zaxis.y, 0.0f,
        xaxis.z, yaxis.z, zaxis.z, 0.0f,
        -Vector3::Dot(xaxis, cameraPosition),
        -Vector3::Dot(yaxis, cameraPosition),
        -Vector3::Dot(zaxis, cameraPosition),
        1.0f);
}

//-------------------------------------------------------------------------------------------------
//      注視点を基にビュー行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateLookAt
(
    const Vector3&  cameraPosition,
    const Vector3&  cameraTarget,
    const Vector3&  cameraUpVector,
    Matrix&         result
)
{
    auto zaxis = cameraPosition - cameraTarget;
    zaxis.Normalize();

    auto xaxis = Vector3::Cross(cameraUpVector, zaxis);
    xaxis.Normalize();

    auto yaxis = Vector3::Cross(zaxis, xaxis);
    yaxis.Normalize();

    result._11 = xaxis.x;
    result._12 = yaxis.x;
    result._13 = zaxis.x;
    result._14 = 0.0f;

    result._21 = xaxis.y;
    result._22 = yaxis.y;
    result._23 = zaxis.y;
    result._24 = 0.0f;

    result._31 = xaxis.z;
    result._32 = yaxis.z;
    result._33 = zaxis.z;
    result._34 = 0.0f;

    result._41 = -Vector3::Dot(xaxis, cameraPosition);
    result._42 = -Vector3::Dot(yaxis, cameraPosition);
    result._43 = -Vector3::Dot(zaxis, cameraPosition);
    result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      視線ベクトルを基にビュー行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateLookTo
(
    const Vector3& cameraPosition,
    const Vector3& cameraDir,
    const Vector3& cameraUpVector
)
{
    auto zaxis = -cameraDir;
    zaxis.Normalize();

    auto xaxis = Vector3::Cross(cameraUpVector, zaxis);
    xaxis.Normalize();

    auto yaxis = Vector3::Cross(zaxis, xaxis);
    yaxis.Normalize();

    return Matrix(
        xaxis.x, yaxis.x, zaxis.x, 0.0f,
        xaxis.y, yaxis.y, zaxis.y, 0.0f,
        xaxis.z, yaxis.z, zaxis.z, 0.0f,
        -Vector3::Dot(xaxis, cameraPosition),
        -Vector3::Dot(yaxis, cameraPosition),
        -Vector3::Dot(zaxis, cameraPosition),
        1.0f);
}

//-------------------------------------------------------------------------------------------------
//      視線ベクトルを基にビュー行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateLookTo
(
    const Vector3&  cameraPosition,
    const Vector3&  cameraDir,
    const Vector3&  cameraUpVector,
    Matrix&         result
)
{
    auto zaxis = -cameraDir;
    zaxis.Normalize();

    auto xaxis = Vector3::Cross(cameraUpVector, zaxis);
    xaxis.Normalize();

    auto yaxis = Vector3::Cross(zaxis, xaxis);
    yaxis.Normalize();

    result._11 = xaxis.x;
    result._12 = yaxis.x;
    result._13 = zaxis.x;
    result._14 = 0.0f;

    result._21 = xaxis.y;
    result._22 = yaxis.y;
    result._23 = zaxis.y;
    result._24 = 0.0f;

    result._31 = xaxis.z;
    result._32 = yaxis.z;
    result._33 = zaxis.z;
    result._34 = 0.0f;

    result._41 = -Vector3::Dot(xaxis, cameraPosition);
    result._42 = -Vector3::Dot(yaxis, cameraPosition);
    result._43 = -Vector3::Dot(zaxis, cameraPosition);
    result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      透視投影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Matrix Matrix::CreatePerspective
(
    f32 width,
    f32 height,
    f32 nearClip,
    f32 farClip
)
{
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( nearClip - farClip ));
    auto range = farClip / nearClip;

    return Matrix(
        2.0f * nearClip / width, 
        0.0f, 
        0.0f,
        0.0f,

        0.0f,
        2.0f * nearClip / height,
        0.0f,
        0.0f,

        0.0f,
        0.0f,
        range,
        -1.0f,

        0.0f,
        0.0f,
        range * nearClip,
        0.0f);
}

//-------------------------------------------------------------------------------------------------
//      透視投影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreatePerspective
(
    f32     width,
    f32     height,
    f32     nearClip,
    f32     farClip,
    Matrix& result
)
{
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( nearClip - farClip ));
    auto range = farClip / (nearClip - farClip);

    result._11 = 2.0f * nearClip / width;
    result._12 = 0.0f;
    result._13 = 0.0f;
    result._14 = 0.0f;

    result._21 = 0.0f;
    result._22 = 2.0f * nearClip / height;
    result._23 = 0.0f;
    result._24 = 0.0f;

    result._31 = 0.0f;
    result._32 = 0.0f;
    result._33 = range;
    result._34 = -1.0f;
    
    result._41 = 0.0f;
    result._42 = 0.0f;
    result._43 = range * nearClip;
    result._44 = 0.0f;
}

//-------------------------------------------------------------------------------------------------
//      視野角に基づいて透視投影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreatePerspectiveFieldOfView
(
    f32 fieldOfView,
    f32 aspectRatio,
    f32 nearClip,
    f32 farClip
)
{
    assert( !IsZero( aspectRatio ) );
    assert( !IsZero( nearClip - farClip ) );
    auto sinFov = std::sin( 0.5f * fieldOfView );
    auto cosFov = std::cos( 0.5f * fieldOfView );
    auto height = cosFov / sinFov;
    auto width  = height / aspectRatio;
    auto range  = farClip / ( nearClip - farClip );

    return Matrix(
        width,
        0.0f,
        0.0f,
        0.0f,

        0.0f,
        height,
        0.0f,
        0.0f,

        0.0f,
        0.0f,
        range,
        -1.0f,

        0.0f,
        0.0f,
        range * nearClip,
        0.0f );
}

//-------------------------------------------------------------------------------------------------
//      視野角に基づいて透視投影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreatePerspectiveFieldOfView
( 
    f32     fieldOfView,
    f32     aspectRatio,
    f32     nearClip,
    f32     farClip,
    Matrix& result
)
{
    assert( !IsZero( aspectRatio ) );
    assert( !IsZero( nearClip - farClip ) );
    auto sinFov = std::sin( 0.5f * fieldOfView );
    auto cosFov = std::cos( 0.5f * fieldOfView );
    auto height = cosFov / sinFov;
    auto width  = height / aspectRatio;
    auto range  = farClip / ( nearClip - farClip );

    result._11 = width;
    result._12 = 0.0f;
    result._13 = 0.0f;
    result._14 = 0.0f;

    result._21 = 0.0f;
    result._22 = height;
    result._23 = 0.0f;
    result._24 = 0.0f;

    result._31 = 0.0f;
    result._32 = 0.0f;
    result._33 = range;
    result._34 = -1.0f;

    result._41 = 0.0f;
    result._42 = 0.0f;
    result._43 = range * nearClip;
    result._44 = 0.0f;
}

//-------------------------------------------------------------------------------------------------
//      カスタマイズした透視投影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreatePerspectiveOffCenter
(
    f32 left,
    f32 right,
    f32 bottom,
    f32 top,
    f32 nearClip,
    f32 farClip
)
{
    auto width  = right - left;
    auto height = top - bottom;
    auto depth  = nearClip - farClip;
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( depth ) );

    return Matrix(
        2.0f * nearClip / width,
        0.0f,
        0.0f,
        0.0f,

        0.0f,
        2.0f * nearClip / height,
        0.0f,
        0.0f,

        (left + right) / width,
        (top + bottom) / height,
        farClip / depth,
        -1.0f,

        0.0f,
        0.0f,
        nearClip * farClip/ depth,
        0.0f );
}

//-------------------------------------------------------------------------------------------------
//      カスタマイズした透視投影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreatePerspectiveOffCenter
(
    f32     left,
    f32     right,
    f32     bottom,
    f32     top,
    f32     nearClip,
    f32     farClip,
    Matrix& result
)
{
    auto width  = right - left;
    auto height = top - bottom;
    auto depth  = nearClip - farClip;
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( depth ) );

    result._11 = 2.0f * nearClip / width;
    result._12 = 0.0f;
    result._13 = 0.0f;
    result._14 = 0.0f;

    result._21 = 0.0f;
    result._22 = 2.0f * nearClip / height;
    result._23 = 0.0f;
    result._24 = 0.0f;

    result._31 = (left + right) / width;
    result._32 = (top + bottom) / height;
    result._33 = farClip / depth;
    result._34 = -1.0f;

    result._41 = 0.0f;
    result._42 = 0.0f;
    result._43 = nearClip * farClip/ depth;
    result._44 = 0.0f;
}

//-------------------------------------------------------------------------------------------------
//      正射影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateOrthographic
(
    f32 width,
    f32 height,
    f32 nearClip,
    f32 farClip
)
{
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( nearClip - farClip ) );
    auto range = 1.0f / (nearClip - farClip);

    return Matrix(
        2.0f / width,
        0.0f,
        0.0f,
        0.0f,
    
        0.0f,
        2.0f / height,
        0.0f,
        0.0f,

        0.0f,
        0.0f,
        range,
        0.0f,

        0.0f,
        0.0f,
        range * nearClip,
        1.0f );
}

//-------------------------------------------------------------------------------------------------
//      正射影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateOrthographic
(
    f32     width,
    f32     height,
    f32     nearClip,
    f32     farClip,
    Matrix& result
)
{
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( nearClip - farClip ) );
    auto range = 1.0f / (nearClip - farClip);

    result._11 = 2.0f / width;
    result._12 = 0.0f;
    result._13 = 0.0f;
    result._14 = 0.0f;
    
    result._21 = 0.0f;
    result._22 = 2.0f / height;
    result._23 = 0.0f;
    result._24 = 0.0f;

    result._31 = 0.0f;
    result._32 = 0.0f;
    result._33 = range;
    result._34 = 0.0f;

    result._41 = 0.0f;
    result._42 = 0.0f;
    result._43 = range * nearClip;
    result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      カスタマイズした正射影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateOrthographicOffCenter
(
    f32 left,
    f32 right,
    f32 bottom,
    f32 top,
    f32 nearClip,
    f32 farClip
)
{
    auto width  = right - left;
    auto height = bottom - top;
    auto depth  = farClip - nearClip;
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( depth ) );

    return Matrix(
        2.0f / width,
        0.0f,
        0.0f,
        0.0f,

        0.0f,
        2.0f / height,
        0.0f,
        0.0f,

        0.0f,
        0.0f,
        1.0f / depth,
        0.0f,

        (left + right) / width,
        (top + bottom) / height,
        nearClip / depth,
        1.0f
    );
}

//-------------------------------------------------------------------------------------------------
//      カスタマイズした正射影行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateOrthographicOffCenter
(
    f32     left,
    f32     right,
    f32     bottom,
    f32     top,
    f32     nearClip,
    f32     farClip,
    Matrix& result
)
{
    auto width  = right - left;
    auto height = bottom - top;
    auto depth  = nearClip - farClip;
    assert( !IsZero( width ) );
    assert( !IsZero( height ) );
    assert( !IsZero( depth ) );

    result._11 = 2.0f / width;
    result._12 = 0.0f;
    result._13 = 0.0f;
    result._14 = 0.0f;

    result._21 = 0.0f;
    result._22 = 2.0f / height;
    result._23 = 0.0f;
    result._24 = 0.0f;

    result._31 = 0.0f;
    result._32 = 0.0f;
    result._33 = 1.0f / depth;
    result._34 = 0.0f;

    result._41 = (left + right) / width;
    result._42 = (top + bottom) / height;
    result._43 = nearClip / depth;
    result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      2つの行列を線形補間します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::Lerp( const Matrix& a, const Matrix& b, f32 amount )
{
    return Matrix(
        a._11 + amount * ( b._11 - a._11 ),
        a._12 + amount * ( b._12 - a._12 ),
        a._13 + amount * ( b._13 - a._13 ),
        a._14 + amount * ( b._14 - a._14 ),

        a._21 + amount * ( b._21 - a._21 ),
        a._22 + amount * ( b._22 - a._22 ),
        a._23 + amount * ( b._23 - a._23 ),
        a._24 + amount * ( b._24 - a._24 ),

        a._31 + amount * ( b._31 - a._31 ),
        a._32 + amount * ( b._32 - a._32 ),
        a._33 + amount * ( b._33 - a._33 ),
        a._34 + amount * ( b._34 - a._34 ),

        a._41 + amount * ( b._41 - a._41 ),
        a._42 + amount * ( b._42 - a._42 ),
        a._43 + amount * ( b._43 - a._43 ),
        a._44 + amount * ( b._44 - a._44 )
    );
}

//-------------------------------------------------------------------------------------------------
//      2つの行列を線形補間します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::Lerp( const Matrix& a, const Matrix& b, f32 amount, Matrix &result )
{
    result._11 = a._11 + amount * ( b._11 - a._11 );
    result._12 = a._12 + amount * ( b._12 - a._12 );
    result._13 = a._13 + amount * ( b._13 - a._13 );
    result._14 = a._14 + amount * ( b._14 - a._14 );

    result._21 = a._21 + amount * ( b._21 - a._21 );
    result._22 = a._22 + amount * ( b._22 - a._22 );
    result._23 = a._23 + amount * ( b._23 - a._23 );
    result._24 = a._24 + amount * ( b._24 - a._24 );

    result._31 = a._31 + amount * ( b._31 - a._31 );
    result._32 = a._32 + amount * ( b._32 - a._32 );
    result._33 = a._33 + amount * ( b._33 - a._33 );
    result._34 = a._34 + amount * ( b._34 - a._34 );

    result._41 = a._41 + amount * ( b._41 - a._41 );
    result._42 = a._42 + amount * ( b._42 - a._42 );
    result._43 = a._43 + amount * ( b._43 - a._43 );
    result._44 = a._44 + amount * ( b._44 - a._44 );
}

//-------------------------------------------------------------------------------------------------
//      ビルボード行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateBillboard( const Matrix& value )
{
    return Matrix(
        value._11, value._21, value._31, 0.0f,
        value._12, value._22, value._32, 0.0f,
        value._13, value._23, value._33, 0.0f,
        value._14, value._24, value._34, 1.0f);
}

//-------------------------------------------------------------------------------------------------
//      ビルボード行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateBillboard( const Matrix& value, Matrix& result )
{
    result._11 = value._11;     result._12 = value._21;    result._13 = value._31;      result._41 = 0.0f;
    result._21 = value._12;     result._22 = value._22;    result._23 = value._32;      result._24 = 0.0f;
    result._31 = value._13;     result._32 = value._23;    result._33 = value._33;      result._34 = 0.0f;
    result._41 = value._14;     result._42 = value._24;    result._43 = value._34;      result._44 = 1.0f;
}

//-------------------------------------------------------------------------------------------------
//      Y軸ビルボード行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Matrix Matrix::CreateBillboardAxisY( const Matrix& value )
{
    return Matrix(
        value._11,  0.0f,   value._31,  0.0f,
             0.0f,  1.0f,        0.0f,  0.0f,
        value._13,  0.0f,   value._33,  0.0f,
        value._14,  0.0f,   value._34,  1.0f);
}

//-------------------------------------------------------------------------------------------------
//      Y軸ビルボード行列を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Matrix::CreateBillboardAxisY( const Matrix& value, Matrix& result )
{
    result._11 = value._11; result._12 = 0.0f;      result._13 = value._31; result._14 = 0.0f;
    result._21 = 0.0f;      result._22 = 1.0f;      result._23 = 0.0f;      result._24 = 0.0f;
    result._31 = value._13; result._32 = 0.0f;      result._33 = value._33; result._34 = 0.0f;
    result._41 = value._14; result._42 = value._24; result._43 = value._34; result._44 = 1.0f;
}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Quaternion
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion::Quaternion()
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion::Quaternion( const f32* pf )
{
    assert( pf != nullptr );
    x = pf[ 0 ];
    y = pf[ 1 ];
    z = pf[ 2 ];
    w = pf[ 3 ];
}

//-------------------------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion::Quaternion( f32 nx, f32 ny, f32 nz, f32 nw )
: x( nx )
, y( ny )
, z( nz )
, w( nw )
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion::operator f32* ()
{ return static_cast<f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      const f32* 型へのキャストです.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion::operator const f32* () const
{ return static_cast<const f32*>( &x ); }

//-------------------------------------------------------------------------------------------------
//      加算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion& Quaternion::operator += ( const Quaternion& q )
{
    x += q.x;
    y += q.y;
    z += q.z;
    w += q.w;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      減算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion& Quaternion::operator -= ( const Quaternion& q )
{
    x -= q.x;
    y -= q.y;
    z -= q.z;
    w -= q.w;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      乗算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion& Quaternion::operator *= ( const Quaternion& q )
{
    auto X = ( q.x * w ) + ( x * q.w ) + ( q.y * z ) - ( q.z * y );
    auto Y = ( q.y * w ) + ( y * q.w ) + ( q.z * x ) - ( q.x * z );
    auto Z = ( q.z * w ) + ( z * q.w ) + ( q.x * y ) - ( q.y * x );
    auto W = ( q.w * w ) - ( q.x * x ) - ( q.y * y ) + ( q.z * z );
    x = X;
    y = Y;
    z = Z;
    w = W;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      乗算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion& Quaternion::operator *= ( f32 f )
{
    x *= f;
    y *= f;
    z *= f;
    w *= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      除算代入演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion& Quaternion::operator /= ( f32 f )
{
    assert( !IsZero( f ) );
    x /= f;
    y /= f;
    z /= f;
    w /= f;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      正符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion Quaternion::operator + () const
{ return (*this); }

//-------------------------------------------------------------------------------------------------
//      負符号演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion Quaternion::operator - () const
{ return Quaternion( -x, -y, -z, -w ); }

//-------------------------------------------------------------------------------------------------
//      加算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion Quaternion::operator + ( const Quaternion& q ) const
{ return Quaternion( x + q.x, y + q.y, z + q.z, w + q.z ); }

//-------------------------------------------------------------------------------------------------
//      減算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion Quaternion::operator - ( const Quaternion& q ) const
{ return Quaternion( x - q.x, y - q.y, z - q.z, w - q.z ); }

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion Quaternion::operator * ( const Quaternion& q ) const
{ 
    return Quaternion(
        ( q.x * w ) + ( x * q.w ) + ( q.y * z ) - ( q.z * y ),
        ( q.y * w ) + ( y * q.w ) + ( q.z * x ) - ( q.x * z ),
        ( q.z * w ) + ( z * q.w ) + ( q.x * y ) - ( q.y * x ),
        ( q.w * w ) - ( q.x * x ) - ( q.y * y ) + ( q.z * z )
   );
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion Quaternion::operator * ( f32 f ) const
{ return Quaternion( x * f, y * f, z * f, w *f ); }

//-------------------------------------------------------------------------------------------------
//      除算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion Quaternion::operator / ( f32 f ) const
{ 
    assert( !IsZero( f ) );
    return Quaternion( x / f, y / f, z / f, w / f ); 
}

//-------------------------------------------------------------------------------------------------
//      乗算演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion operator * ( f32 f, const Quaternion& q )
{ return Quaternion( f * q.x, f * q.y, f * q.z, f * q.w ); }

//-------------------------------------------------------------------------------------------------
//      等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
bool Quaternion::operator == ( const Quaternion& q ) const
{ 
    return IsEqual( x, q.x )
        && IsEqual( y, q.y )
        && IsEqual( z, q.z )
        && IsEqual( w, q.w );
}

//-------------------------------------------------------------------------------------------------
//      非等価比較演算子です.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Quaternion::operator != ( const Quaternion& q ) const
{ 
    return !IsEqual( x, q.x )
        || !IsEqual( y, q.y )
        || !IsEqual( z, q.z )
        || !IsEqual( w, q.w );
}

//-------------------------------------------------------------------------------------------------
//      四元数の大きさを求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Quaternion::Length() const
{ return sqrtf( x * x + y * y + z * z + w * w ); }

//-------------------------------------------------------------------------------------------------
//      四元数の大きさの2乗値を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Quaternion::LengthSq() const
{ return ( x * x + y * y + z * z + w * w ); }

//-------------------------------------------------------------------------------------------------
//      正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE 
Quaternion& Quaternion::Normalize()
{
    auto mag = Length();
    assert( mag > 0.0f );
    x /= mag;
    y /= mag;
    z /= mag;
    w /= mag;
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion& Quaternion::SafeNormalize( const Quaternion& set )
{
    auto mag = Length();
    if ( mag > 0.0f )
    {
        x /= mag;
        y /= mag;
        z /= mag;
        w /= mag;
    }
    else
    {
        x = set.x;
        y = set.y;
        z = set.z;
        w = set.w;
    }
    return (*this);
}

//-------------------------------------------------------------------------------------------------
//      単位四元数化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion& Quaternion::Identity()
{
    x = 0.0f;
    y = 0.0f;
    z = 0.0f;
    w = 1.0f;
    return (*this);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
// Quaternion Methods
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      単位四元数を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::CreateIdentity()
{ return Quaternion( 0.0f, 0.0f, 0.0f, 1.0f ); }

//-------------------------------------------------------------------------------------------------
//      単位四元数かどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Quaternion::IsIdentity( const Quaternion& value )
{
    return IsEqual( value.x, 0.0f )
        && IsEqual( value.y, 0.0f )
        && IsEqual( value.z, 0.0f )
        && IsEqual( value.w, 1.0f );
}

//-------------------------------------------------------------------------------------------------
//      正規化されているかどうかチェックします.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
bool Quaternion::IsNormalized( const Quaternion& value )
{
    return IsZero( 1.0f - value.Length() );
}

//-------------------------------------------------------------------------------------------------
//      乗算を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::Multiply( const Quaternion& a, const Quaternion& b )
{
    return Quaternion(
        ( b.x * a.w ) + ( a.x * b.w ) + ( b.y * a.z ) - ( b.z * a.y ),
        ( b.y * a.w ) + ( a.y * b.w ) + ( b.z * a.x ) - ( b.x * a.z ),
        ( b.z * a.w ) + ( a.z * b.w ) + ( b.x * a.y ) - ( b.y * a.x ),
        ( b.w * a.w ) - ( b.x * a.x ) - ( b.y * a.y ) + ( b.z * a.z )
   );
}

//-------------------------------------------------------------------------------------------------
//      四元数同士の乗算を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::Multiply( const Quaternion& a, const Quaternion& b, Quaternion& result )
{
    result.x = ( b.x * a.w ) + ( a.x * b.w ) + ( b.y * a.z ) - ( b.z * a.y );
    result.y = ( b.y * a.w ) + ( a.y * b.w ) + ( b.z * a.x ) - ( b.x * a.z );
    result.z = ( b.z * a.w ) + ( a.z * b.w ) + ( b.x * a.y ) - ( b.y * a.x );
    result.w = ( b.w * a.w ) - ( b.x * a.x ) - ( b.y * a.y ) + ( b.z * a.z );
}

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
f32 Quaternion::Dot( const Quaternion& a, const Quaternion& b )
{ return ( a.x * b.x + a.y * b.y + a.z * b.z + a.w * b.w ); }

//-------------------------------------------------------------------------------------------------
//      内積を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::Dot( const Quaternion &a, const Quaternion &b, f32 &result )
{ result = ( a.x * b.x + a.y * b.y + a.z * b.z + a.w * b.w ); }

//-------------------------------------------------------------------------------------------------
//      共役な四元数を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::Conjugate( const Quaternion& value )
{ return Quaternion( -value.x, -value.y, -value.z, value.w ); }

//-------------------------------------------------------------------------------------------------
//      共役な四元数を求めます.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::Conjugate( const Quaternion &value, Quaternion &result )
{
    result.x = -value.x;
    result.y = -value.y;
    result.z = -value.z;
    result.w = value.w;
}

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::Normalize( const Quaternion& value )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    return Quaternion(
        value.x / mag,
        value.y / mag,
        value.z / mag,
        value.w / mag
    );
}

//-------------------------------------------------------------------------------------------------
//      正規化を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::Normalize( const Quaternion& value, Quaternion &result )
{
    auto mag = value.Length();
    assert( mag > 0.0f );
    result.x /= mag;
    result.y /= mag;
    result.z /= mag;
    result.w /= mag;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion  Quaternion::SafeNormalize( const Quaternion& value, const Quaternion& set )
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        return Quaternion(
            value.x / mag,
            value.y / mag,
            value.z / mag,
            value.w / mag
        );
    }

    return set;
}

//-------------------------------------------------------------------------------------------------
//      零除算を考慮して正規化します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::SafeNormalize( const Quaternion& value, const Quaternion& set, Quaternion& result )
{
    auto mag = value.Length();
    if ( mag > 0.0f )
    {
        result.x = value.x / mag;
        result.y = value.y / mag;
        result.z = value.z / mag;
        result.w = value.w / mag;
    }
    else
    {
        result.x = set.x;
        result.y = set.y;
        result.z = set.z;
        result.w = set.w;
    }
}

//-------------------------------------------------------------------------------------------------
//      ヨー・ピッチ・ロール角から四元数を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::CreateFromYawPitchRoll( f32 yaw, f32 pitch, f32 roll )
{
    auto sr = sinf( roll  * 0.5f );
    auto cr = cosf( roll  * 0.5f );
    auto sp = sinf( pitch * 0.5f );
    auto cp = cosf( pitch * 0.5f );
    auto sy = sinf( yaw   * 0.5f );
    auto cy = cosf( yaw   * 0.5f );

    return Quaternion(
        -(sy * sp * cr) + (cy * cp * sr),
         (cy * sp * cr) + (sy * cp * sr),
        -(cy * sp * sr) + (sy * cp * cr),
         (cy * cp * cr) + (sy * sp * sr) );
}

//-------------------------------------------------------------------------------------------------
//      ヨー・ピッチ・ロール角から四元数を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::CreateFromYawPitchRoll( f32 yaw, f32 pitch, f32 roll, Quaternion& result )
{
    auto sr = sinf( roll  * 0.5f );
    auto cr = cosf( roll  * 0.5f );
    auto sp = sinf( pitch * 0.5f );
    auto cp = cosf( pitch * 0.5f );
    auto sy = sinf( yaw   * 0.5f );
    auto cy = cosf( yaw   * 0.5f );

    result.x = -(sy * sp * cr) + (cy * cp * sr);
    result.y =  (cy * sp * cr) + (sy * cp * sr);
    result.z = -(cy * sp * sr) + (sy * cp * cr);
    result.w =  (cy * cp * cr) + (sy * sp * sr);
}

//-------------------------------------------------------------------------------------------------
//      指定された軸と角度から四元数を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::CreateFromAxisAngle( const Vector3& axis, f32 radian )
{
    auto halfRad = radian * 0.5f;
    auto sinX = sinf( halfRad );
    return Quaternion(
        axis.x * sinX,
        axis.y * sinX,
        axis.z * sinX,
        cosf( halfRad )
   );
}

//-------------------------------------------------------------------------------------------------
//      指定された軸と角度から四元数を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::CreateFromAxisAngle( const Vector3& axis, f32 radian, Quaternion& result )
{
    auto halfRad = radian * 0.5f;
    auto sinX = sinf( halfRad );
    result.x = axis.x * sinX;
    result.y = axis.y * sinX;
    result.z = axis.z * sinX;
    result.w = cosf( halfRad );
}

//-------------------------------------------------------------------------------------------------
//      回転行列から四元数を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::CreateFromRotationMatrix( const Matrix& value )
{
    auto tr = value._11 + value._22 + value._33;
    if ( tr > 0.0f )
    {
        auto s = sqrtf( tr + 1.0f );
        auto w = s * 0.5f;
        s = 0.5f / s;
        return Quaternion(
            ( value._23 - value._32 ) * s,
            ( value._31 - value._13 ) * s,
            ( value._12 - value._21 ) * s,
            w 
        );
    }
    if ( ( value._11 >= value._22 ) && ( value._11 >= value._33 ) )
    {
        auto s = sqrtf( 1.0f + value._11 - value._22 - value._33 );
        auto x = s * 0.5f;
        s = 0.5f / s;
        return Quaternion(
            x,
            ( value._12 + value._21 ) * s,
            ( value._13 + value._31 ) * s,
            ( value._23 - value._32 ) * s
        );
    }
    if ( value._22 > value._33 )
    {
        auto s = sqrtf( 1.0f + value._22 - value._11 - value._33 );
        auto y = s * 0.5f;
        s = 0.5f / s;
        return Quaternion(
            ( value._21 + value._12 ) * s,
            y,
            ( value._32 + value._23 ) * s,
            ( value._31 - value._13 ) * s
        );
    }
    auto s = sqrtf( 1.0f + value._33 - value._11 - value._22 );
    auto z = s * 0.5f;
    s = 0.5f / s;
    return Quaternion(
        ( value._31 + value._13 ) * s,
        ( value._32 + value._23 ) * s,
        z,
        ( value._12 - value._21 ) * s
    );
}

//-------------------------------------------------------------------------------------------------
//      回転行列から四元数を生成します.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::CreateFromRotationMatrix( const Matrix& value, Quaternion& result )
{
    auto tr = value._11 + value._22 + value._33;
    if ( tr > 0.0f )
    {
        auto s = sqrtf( tr + 1.0f );

        result.w = s * 0.5f;
        s = 0.5f / s;

        result.x = ( value._23 - value._32 ) * s;
        result.y = ( value._31 - value._13 ) * s;
        result.z = ( value._12 - value._21 ) * s;
        return;
    }
    if ( ( value._11 >= value._22 ) && ( value._11 >= value._33 ) )
    {
        auto s = sqrtf( 1.0f + value._11 - value._22 - value._33 );
        result.x = 0.5f * s;

        s = 0.5f / s;
        result.y = ( value._12 + value._21 ) * s;
        result.z = ( value._13 + value._31 ) * s;
        result.w = ( value._23 - value._32 ) * s;
        return;
    }
    if ( value._22 > value._33 )
    {
        auto s = sqrtf( 1.0f + value._22 - value._11 - value._33 );
        result.y = 0.5f * s;
        s = 0.5f / s;

        result.x = ( value._21 + value._12 ) * s;
        result.z = ( value._32 + value._23 ) * s;
        result.w = ( value._31 - value._13 ) * s;
        return;
    }
    auto s = sqrtf( 1.0f + value._33 - value._11 - value._22 );
    result.z = 0.5f * s;
    s = 0.5f / s;
    result.x = ( value._31 + value._13 ) * s;
    result.y = ( value._32 + value._23 ) * s;
    result.w = ( value._12 - value._21 ) * s;
}

//-------------------------------------------------------------------------------------------------
//      球面線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::Slerp
(
    const Quaternion&   a,
    const Quaternion&   b,
    const f32           amount
)
{
    if ( amount <= 0.0f )
    { return a; }

    if ( amount >= 1.0f )
    { return b; }

    if ( a == b )
    { return b; }

    Quaternion temp;
    auto cosom = Quaternion::Dot( a, b );
    if ( cosom < 0.0f )
    { 
        temp = -b;
        cosom = -cosom;
    }
    else
    { temp = b; }

    auto scale0 = 0.0f;
    auto scale1 = 0.0f;
    if ( 1.0f - cosom > 1e-6f )
    {
        scale0 = 1.0f - cosom * cosom;
        auto sinom = 1.0f / sqrtf( scale0 );
        auto omega = atan2f( scale0 * sinom, cosom );
        scale0 = sinf( ( 1.0f - amount ) * omega ) * sinom;
        scale1 = sinf( amount * omega ) * sinom;
    }
    else
    {
        scale0 = 1.0f - amount;
        scale1 = amount;
    }

    return Quaternion(
        scale0 * a.x + scale1 * temp.x,
        scale0 * a.y + scale1 * temp.y,
        scale0 * a.z + scale1 * temp.z,
        scale0 * a.w + scale1 * temp.w);
}

//-------------------------------------------------------------------------------------------------
//      球面線形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::Slerp
(
    const Quaternion&   a,
    const Quaternion&   b,
    const f32           amount,
    Quaternion&         result
)
{
    if ( amount <= 0.0f )
    {
        result = a;
        return;
    }

    if ( amount > 1.0f )
    {
        result = b;
        return;
    }

    if ( a == b )
    {
        result = b;
        return;
    }

   Quaternion temp;
    auto cosom = Quaternion::Dot( a, b );
    if ( cosom < 0.0f )
    { 
        temp = -b;
        cosom = -cosom;
    }
    else
    { temp = b; }

    auto scale0 = 0.0f;
    auto scale1 = 0.0f;
    if ( 1.0f - cosom > 1e-6f )
    {
        scale0 = 1.0f - cosom * cosom;
        auto sinom = 1.0f / sqrtf( scale0 );
        auto omega = atan2f( scale0 * sinom, cosom );
        scale0 = sinf( ( 1.0f - amount ) * omega ) * sinom;
        scale1 = sinf( amount * omega ) * sinom;
    }
    else
    {
        scale0 = 1.0f - amount;
        scale1 = amount;
    }

    result.x = scale0 * a.x + scale1 * temp.x;
    result.y = scale0 * a.y + scale1 * temp.y;
    result.z = scale0 * a.z + scale1 * temp.z;
    result.w = scale0 * a.w + scale1 * temp.w;
}

//-------------------------------------------------------------------------------------------------
//      球面四角形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
Quaternion Quaternion::Squad
(
    const Quaternion&   q,
    const Quaternion&   a,
    const Quaternion&   b,
    const Quaternion&   c,
    f32                 amount
)
{
    auto d = Quaternion::Slerp( q, c, amount );
    auto e = Quaternion::Slerp( a, b, amount );
    return Quaternion::Slerp( d, e, 2.0f * amount * ( 1.0f - amount ) );
}

//-------------------------------------------------------------------------------------------------
//      球面四角形補間を行います.
//-------------------------------------------------------------------------------------------------
ASDX_INLINE
void Quaternion::Squad
(
    const Quaternion&   q,
    const Quaternion&   a,
    const Quaternion&   b,
    const Quaternion&   c,
    f32                 amount,
    Quaternion&         result
)
{
    Quaternion d, e;
    Quaternion::Slerp( q, c, amount, d );
    Quaternion::Slerp( a, b, amount, e );
    Quaternion::Slerp( d, e, 2.0f * amount * ( 1.0f - amount ), result );
}


////////////////////////////////////////////////////////////////////////////////////
// OrthonormalBasis structure
////////////////////////////////////////////////////////////////////////////////////

//----------------------------------------------------------------------------------
//      コンストラクタです.
//----------------------------------------------------------------------------------
ASDX_INLINE
OrthonormalBasis::OrthonormalBasis() = default;

//----------------------------------------------------------------------------------
//      引数付きコンストラクタです.
//----------------------------------------------------------------------------------
ASDX_INLINE
OrthonormalBasis::OrthonormalBasis
(
    const Vector3& nu,
    const Vector3& nv,
    const Vector3& nw 
)
: u( nu )
, v( nv )
, w( nw )
{ /* DO_NOTHING */ }

//----------------------------------------------------------------------------------
//      コピーコンストラクタです.
//----------------------------------------------------------------------------------
ASDX_INLINE
OrthonormalBasis::OrthonormalBasis( const OrthonormalBasis& value )
: u( value.u )
, v( value.v )
, w( value.w )
{ /* DO_NOTHING */ }

//----------------------------------------------------------------------------------
//      U方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromU( const Vector3& value )
{
    Vector3 n( 1.0f, 0.0f, 0.0f );
    Vector3 m( 0.0f, 1.0f, 0.0f );
    
    u = Vector3::Normalize( value );
    v = Vector3::Cross( u, n );
    if ( v.Length() < ONB_EPSILON )
    { v = Vector3::Cross( u, m ); }
    w = Vector3::Cross( v, u );
}

//----------------------------------------------------------------------------------
//      V方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromV( const Vector3& value )
{
    Vector3 n ( 1.0f, 0.0f, 0.0f );
    Vector3 m ( 0.0f, 1.0f, 0.0f );

    v = Vector3::Normalize( value );
    u = Vector3::Cross( v, n );
    if ( u.LengthSq() < ONB_EPSILON )
    { u = Vector3::Cross( v, m ); }
    w = Vector3::Cross( v, u );
}

//----------------------------------------------------------------------------------
//      W方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromW( const Vector3& value )
{
    Vector3 n ( 1.0f, 0.0f, 0.0f );
    Vector3 m ( 0.0f, 1.0f, 0.0f );

    w = Vector3::Normalize( value );
    u = Vector3::Cross( w, n );
    if ( u.Length() < ONB_EPSILON )
    { u = Vector3::Cross( w, m ); }
    u.Normalize();

    v = Vector3::Cross( u, w );
    v.Normalize();
}

//----------------------------------------------------------------------------------
//      U方向とV方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromUV( const Vector3& _u, const Vector3& _v )
{
    u = Vector3::Normalize( _u );
    w = Vector3::Normalize( Vector3::Cross( _u, _v ) );
    v = Vector3::Cross( w, v );
}

//----------------------------------------------------------------------------------
//      V方向とU方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromVU( const Vector3& _v, const Vector3& _u )
{
    v = Vector3::Normalize( _v );
    w = Vector3::Normalize( Vector3::Cross( _u, _v ) );
    u = Vector3::Cross( v, w );
}

//----------------------------------------------------------------------------------
//      U方向とW方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromUW( const Vector3& _u, const Vector3& _w )
{
    u = Vector3::Normalize( _u );
    v = Vector3::Normalize( Vector3::Cross( _w, _u ) );
    w = Vector3::Cross( u, v );
}

//----------------------------------------------------------------------------------
//      W方向とU方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromWU( const Vector3& _w, const Vector3& _u )
{
    w = Vector3::Normalize( _w );
    v = Vector3::Normalize( Vector3::Cross( _w, _u ) );
    u = Vector3::Cross( v, w );
}

//----------------------------------------------------------------------------------
//      V方向とW方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromVW( const Vector3& _v, const Vector3& _w )
{
    v = Vector3::Normalize( _v );
    u = Vector3::Normalize( Vector3::Cross( _v, _w ) );
    w = Vector3::Cross( u, v );
}

//----------------------------------------------------------------------------------
//      W方向とV方向から基底を構築します.
//----------------------------------------------------------------------------------
ASDX_INLINE
void OrthonormalBasis::InitFromWV( const Vector3& _w, const Vector3& _v )
{
    w = Vector3::Normalize( _w );
    u = Vector3::Normalize( Vector3::Cross( _v, _w ) );
    v = Vector3::Cross( w, u );
}

//----------------------------------------------------------------------------------
//      等価演算子です.
//----------------------------------------------------------------------------------
ASDX_INLINE
bool OrthonormalBasis::operator == ( const OrthonormalBasis& value ) const
{
    return ( u == value.u )
        && ( v == value.v )
        && ( w == value.w );
}

//----------------------------------------------------------------------------------
//      非等価演算子です.
//----------------------------------------------------------------------------------
ASDX_INLINE
bool OrthonormalBasis::operator != ( const OrthonormalBasis& value ) const
{
    return ( u != value.u )
        || ( v != value.v )
        || ( w != value.w );
}


} // namespace asdx

