﻿//-------------------------------------------------------------------------------------------------
// File : asdxDescHeap.h
// Desc : Descriptor Heap Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <asdxRef.h>
#include <d3d12.h>


namespace asdx {

///////////////////////////////////////////////////////////////////////////////////////////////////
// DescHandle class
///////////////////////////////////////////////////////////////////////////////////////////////////
class DescHandle
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    DescHandle();

    //---------------------------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //---------------------------------------------------------------------------------------------
    DescHandle( D3D12_CPU_DESCRIPTOR_HANDLE handleCpu );

    //---------------------------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //---------------------------------------------------------------------------------------------
    DescHandle
    (
        D3D12_CPU_DESCRIPTOR_HANDLE handleCpu,
        D3D12_GPU_DESCRIPTOR_HANDLE handleGpu
    );

    //---------------------------------------------------------------------------------------------
    //! @brief      CPUハンドルを持つかどうかチェックします.
    //---------------------------------------------------------------------------------------------
    bool HasHandleCpu() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      GPUハンドルを持つかどうかチェックします.
    //---------------------------------------------------------------------------------------------
    bool HasHandleGpu() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      CPUハンドルを取得します.
    //---------------------------------------------------------------------------------------------
    D3D12_CPU_DESCRIPTOR_HANDLE GetHandleCpu() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      GPUハンドルを取得します.
    //---------------------------------------------------------------------------------------------
    D3D12_GPU_DESCRIPTOR_HANDLE GetHandleGpu() const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    D3D12_CPU_DESCRIPTOR_HANDLE m_HandleCpu;        //!< CPUハンドルです.
    D3D12_GPU_DESCRIPTOR_HANDLE m_HandleGpu;        //!< GPUハンドルです.

    //=============================================================================================
    // private methods.
    //=============================================================================================
    /* NOTHING */
};


///////////////////////////////////////////////////////////////////////////////////////////////////
// DescHeap class
///////////////////////////////////////////////////////////////////////////////////////////////////
class DescHeap final
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    static const D3D12_CPU_DESCRIPTOR_HANDLE InvalidHandleCPU;      //!< 無効なCPUハンドルです.
    static const D3D12_GPU_DESCRIPTOR_HANDLE InvalidHandleGPU;      //!< 無効なGPUハンドルです.
    static const u32                         TypeCount = 4;         //!< ディスクリプタヒープタイプの数です.

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    DescHeap();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    ~DescHeap();

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化処理を行います.
    //!
    //! @param[in]      pDevice         デバイスです.
    //! @param[in]      pDesc           設定です.
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool Init( 
        ID3D12Device*               pDevice,
        D3D12_DESCRIPTOR_HEAP_DESC* pDesc);

    //---------------------------------------------------------------------------------------------
    //! @brief      終了処理を行います.
    //---------------------------------------------------------------------------------------------
    void Term();

    //---------------------------------------------------------------------------------------------
    //! @brief      ディスクリプタヒープを取得します.
    //!
    //! @return     ディスクリプタヒープを返却します.
    //---------------------------------------------------------------------------------------------
    ID3D12DescriptorHeap* GetPtr() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      アロー演算子です.
    //---------------------------------------------------------------------------------------------
    ID3D12DescriptorHeap* operator -> () const;

    //---------------------------------------------------------------------------------------------
    //! @brief      CPUハンドルを取得します.
    //!
    //! @param[in]      index       先頭からのインデックスです.
    //! @return     CPUハンドルを返却します.
    //---------------------------------------------------------------------------------------------
    D3D12_CPU_DESCRIPTOR_HANDLE GetHandleCPU( const u32 index ) const;

    //---------------------------------------------------------------------------------------------
    //! @brief      GPUハンドルを取得します.
    //!
    //! @param[in]      index       先頭からのインデックスです.
    //! @return     GPUハンドルを返却します.
    //---------------------------------------------------------------------------------------------
    D3D12_GPU_DESCRIPTOR_HANDLE GetHandleGPU( const u32 index ) const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    RefPtr<ID3D12DescriptorHeap> m_Heap;    //!< ディスクリプタヒープです.
    u32                          m_Size;    //!< インクリメントサイズです.
};


} // namespace asdx
