﻿// Vistaの効果を有効にするフラグ（不要の場合はコメントアウト）
#define USE_VISTA_EFFECTS

using System;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace AppliStation.Util
{
	/// <summary>
	/// Win32ネイティブメソッドを叩いてGUI操作するための関数群のクラス
	/// </summary>
	public sealed class NativeMethods
	{
		/// <summary>
		/// 呼び出し禁止
		/// </summary>
		private NativeMethods()
		{
		}

		/// <summary>
		/// WindowsVista向け、プログレスバーステータス(色)を設定する
		/// </summary>
		/// <param name="progBar">対象のプログレスバー</param>
		/// <param name="state">状態。(1:Normal,2:Error,3:Paused)</param>
		public static void ProgressBar_SetState(ProgressBar progBar, uint state)
		{
#if USE_VISTA_EFFECTS
			try {
				// status := (PBST_NORMAL | PBST_ERROR | PBST_PAUSED)
				// SendMessage(progressBar.Handle, PBM_SETSTATE, state, 0);
				SendMessage(progBar.Handle, 0x410, state, 0);
			} catch (Exception) {
			}
#endif
		}
		
		#region タスクバーおよびタイトルバーのフラッシュ
		
		/// <summary>
		/// タスクバーおよびタイトルバーボタンのフラッシュの設定フラグ
		/// </summary>
		public enum FlashFlag : uint {
			/// <summary>
			/// 点滅の停止
			/// </summary>
			Stop = 0,
			/// <summary>
			/// タイトルバーを点滅
			/// </summary>
			Caption = 1,
			/// <summary>
			/// タスクバーボタンを点滅
			/// </summary>
			Tray = 2,
			/// <summary>
			/// タイトルバーとタスクバーボタンを点滅
			/// </summary>
			All = 3,
			/// <summary>
			/// Stopが設定されるまで点滅する
			/// </summary>
			Timer = 4,
			/// <summary>
			/// フォアグラウンドの状態になるまで点滅
			/// </summary>
			TimerNoFG = 12,
		}
		
		[StructLayout(LayoutKind.Sequential)]
		struct FLASHWINFO
		{
		    public int cbSize;
		    public IntPtr hWnd;
		    public FlashFlag dwFlags;
		    public uint uCount;
		    public uint dwTimeout;
		}
		
		/// <summary>
		/// タスクバーおよびタイトルバーボタンを点滅させる
		/// </summary>
		/// <param name="form">対象フォーム</param>
		/// <param name="flag">点滅パラメータフラグ</param>
		/// <param name="count">点滅回数</param>
		/// <param name="timeout">点滅の間隔(ミリ秒)</param>
		/// <returns></returns>
		public static bool Form_FlashWindow(Form form, FlashFlag flag, uint count, uint timeout)
		{
			try {
				FLASHWINFO info = new FLASHWINFO();
				info.cbSize = Marshal.SizeOf(typeof(FLASHWINFO));
				info.hWnd = form.Handle;
				info.dwFlags = flag;
				info.uCount = count;
				info.dwTimeout = timeout;
				
				return FlashWindowEx(ref info) == 0;
			} catch (Exception) {
				return false;
			}
		}
		
		[DllImport("user32.dll")]
		static extern Int32 FlashWindowEx(ref FLASHWINFO pwfi);
		
		#endregion
		
		#region ListView関連
		
		/// <summary>
		/// リストビューにダブルバッファでの描画をするか否かを設定する
		/// </summary>
		/// <remarks>マウスでの選択に半透明ツールを採用するか否かもこの設定に依存</remarks>
		/// <param name="listView">対象のリストビュー</param>
		/// <param name="bEnable">ダブルバッファでの描画をするとき<code>true</code></param>
		public static void ListView_SetDoubleBuffer(ListView listView, bool bEnable)
		{
			try {
				// SendMessage(listView.Handle, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_DOUBLEBUFFER, bEnable? LVS_EX_DOUBLEBUFFER:0);
				SendMessage(listView.Handle, 0x1036, 0x00010000, ((bEnable)? 0x00010000u:0x0u));
			} catch (Exception) {
			}
		}
		
		/// <summary>
		/// 選択されたアイテムの部分の背景にグラデーションがかかった感じになる、
		/// Vista以降でのエクスプローラの見た目をListViewに反映させる。
		/// </summary>
		/// <remarks>Vista未満のバージョンでは何もしない。</remarks>
		/// <param name="listView">対象のListView</param>
		public static void ListView_EnableVistaExplorerTheme(ListView listView)
		{
#if USE_VISTA_EFFECTS
			// Vista未満はなにもしない
			OperatingSystem os = Environment.OSVersion;
			if (os.Platform != PlatformID.Win32NT || os.Version.Major < 6) return;
			
			try {
				SetWindowTheme(listView.Handle, "explorer", null);
			} catch (Exception) {
			}
#endif
		}
		
		/// <summary>
		/// ヘッダに"すべて選択"に似たチェックボックスを作るか否かを指定する
		/// </summary>
		/// <remarks>このオプションを設定するとVistaエクスプローラでの「チェックボックスを使用して項目を選択する」と同様の動作になる</remarks>
		/// <param name="listView">対象のListBox</param>
		/// <param name="bAutoCheckSelect">チェックボックスを使用して項目を選択するとき<code>true</code></param>
		public static void ListView_SetAutoCheckSelect(ListView listView, bool bAutoCheckSelect)
		{
#if USE_VISTA_EFFECTS
			try {
				// SendMessage(listView.Handle, LVM_SETEXTENDEDLISTVIEWSTYLE, LVS_EX_AUTOCHECKSELECT, bAutoCheckSelect?LVS_EX_AUTOCHECKSELECT:0);
				SendMessage(listView.Handle, 0x1036, 0x08000000, (bAutoCheckSelect)?0x08000000u:0x0u);
			} catch (Exception) {
			}
#endif
		}
		
		#region ColumnHeaderのソートの三角印用
		
		[StructLayout(LayoutKind.Sequential, CharSet=CharSet.Auto)]
		internal struct HD_ITEM
		{
			public uint    mask;
			public int     cxy;
			[MarshalAs(UnmanagedType.LPTStr)]public string   pszText;
			public IntPtr   hbm;
			public int     cchTextMax;
			public int     fmt;
			[MarshalAs(UnmanagedType.LPTStr)]public string  lParam;
			public int     iImage;        // index of bitmap in ImageList
			public int     iOrder;
		}
		
		internal static IntPtr ListView_GetHeader(ListView listview)
		{
			// SendMessage(hWnd, LVM_GETHEADER, 0, NULL);
			return SendMessage(listview.Handle, 0x101E, 0, 0);
		}
		
		/// <summary>
		/// WinXP以降、ソートの矢印を表示
		/// </summary>
		/// <param name="listView">対象のListView</param>
		/// <param name="column">表示する矢印のヘッダ</param>
		/// <param name="order">ソートの昇順・降順</param>
		public static void ColumnHeader_SetSortState(ListView listView, int column, SortOrder order)
		{
			try {
				// SendMessage(hWnd, LVM_GETHEADER, NULL, NULL);
				IntPtr hWnd = SendMessage(listView.Handle, 0x101F, 0, 0);
				
				HD_ITEM hdi = new HD_ITEM();
				hdi.mask = 0x0004; // HDI_FORMAT;
				for (int i = 0; i < listView.Columns.Count; i++) {
					// SendMessage(hWnd, HDM_GETITEMW, i, &hdi);
					SendMessage(hWnd, 0x120b, i, ref hdi);
					
					const int HDF_SORTUP = 0x400;
					const int HDF_SORTDOWN = 0x200;
					
					if (i != column || order == SortOrder.None) {
						hdi.fmt = hdi.fmt & ~(HDF_SORTUP | HDF_SORTDOWN);
					} else if (order == SortOrder.Ascending) { // 昇順
						hdi.fmt = hdi.fmt & ~HDF_SORTDOWN | HDF_SORTUP;
					} else if (order == SortOrder.Descending) { // 降順
						hdi.fmt = hdi.fmt & ~HDF_SORTUP | HDF_SORTDOWN;
					}
					
					// SendMessage(hWnd, HDM_SETITEMW, i, &hdi);
					SendMessage(hWnd, 0x120c, i, ref hdi);
				}
			} catch (Exception) {
			}
		}
		
		#endregion
		
		#endregion
		
		#region EnableWindow(コメントアウト)
//		/// <summary>
//		/// 指定されたコントロール(ウィンドウ)への、
//		/// キーボード入力およびマウス入力を有効化または無効化
//		/// </summary>
//		/// <param name="control">対象のコントロールのハンドラ</param>
//		/// <param name="bEnable">有効にするか無効にするかを指定</param>
//		/// <returns>直前にウィンドウが無効状態だった場合はtrueを返す</returns>
//		public static bool Control_EnableWindow(Control ctrl, bool bEnable)
//		{
//			try {
//				return EnableWindow(ctrl.Handle, bEnable);
//			} catch {
//				ctrl.Enabled = bEnable;
//				return true;
//			}
//		}
//		
//		/// <summary>
//		/// 指定されたコントロール(ウィンドウ)への、
//		/// キーボード入力およびマウス入力を有効化または無効化
//		/// </summary>
//		/// <param name="control">対象のコントロールのハンドラ</param>
//		/// <param name="bEnable">有効にするか無効にするかを指定</param>
//		/// <returns>直前にウィンドウが無効状態だった場合はtrueを返す</returns>
//		[DllImport("user32.dll")]
//		public static extern bool EnableWindow(IntPtr hWnd, bool bEnable);
		#endregion
		
		[DllImport("user32.dll", CharSet=CharSet.Auto)]
		internal static extern IntPtr SendMessage( IntPtr hWnd, UInt32 Msg, UInt32 wParam, UInt32 lParam);
		
		[DllImport("user32.dll", CharSet=CharSet.Auto)]
		internal static extern IntPtr SendMessage( IntPtr hWnd, UInt32 Msg, int wParam, ref HD_ITEM lParam);

		[DllImport("uxtheme.dll", ExactSpelling=true, CharSet=CharSet.Unicode)]
		internal static extern int SetWindowTheme(IntPtr hWnd, String pszSubAppName, String pszSubIdList);
	}
}
