/**/
#include <dlfcn.h>
#include <stdlib.h>
#include "context.h"

extern LISP sym_t;

#define MAX_CONTEXT 128

struct anthy_context;

struct anthy_conv_stat{
  int nr_segment;
};

struct anthy_segment_stat{
  int nr_candidate;
  int seg_len;
};

static struct anthy_api {
  int (*init)();
  struct anthy_context *(*create_context)();
  void (*release_context)(struct anthy_context *);
  void (*set_string)(struct anthy_context *, char *);
  void (*get_stat)(struct anthy_context *, struct anthy_conv_stat *);
  void (*get_segment_stat)(struct anthy_context *, int,
			   struct anthy_segment_stat *);
  int (*get_segment)(struct anthy_context *, int, int, char *, int);
  void (*resize_segment)(struct anthy_context *, int, int);
  void (*commit_segment)(struct anthy_context *, int, int);
} api;

struct context {
  struct anthy_context *ac;
} *context_slot;

static int
get_anthy_api()
{
  void *diclib, *lib;

  diclib = dlopen("libanthydic.so", RTLD_GLOBAL |RTLD_NOW);
  if (!diclib) {
    fprintf (stdout, "%s\n", dlerror());
    return -1;
  }
  lib = dlopen("libanthy.so", RTLD_NOW);
  if (!lib) {
    fprintf (stdout, "%s\n", dlerror());
    dlclose(diclib);
    return -1;
  }
  api.init = dlsym(lib, "anthy_init");
  api.create_context = dlsym(lib, "anthy_create_context");
  api.release_context = dlsym(lib, "anthy_release_context");
  api.set_string = dlsym(lib, "anthy_set_string");
  api.get_stat = dlsym(lib, "anthy_get_stat");
  api.get_segment_stat = dlsym(lib, "anthy_get_segment_stat");
  api.get_segment = dlsym(lib, "anthy_get_segment");
  api.resize_segment = dlsym(lib, "anthy_resize_segment");
  api.commit_segment = dlsym(lib, "anthy_commit_segment");
  if (api.init && api.create_context &&
      api.release_context && api.set_string &&
      api.get_stat && api.get_segment_stat &&
      api.resize_segment && api.commit_segment) {
    return 0;
  }
  return -1;
}

static struct anthy_context *
get_anthy_context(int id)
{
  if (id < 0 || id >= MAX_CONTEXT) {
    return NULL;
  }
  return context_slot[id].ac;
}

static LISP
init_anthy_lib()
{
  int i;
  if (get_anthy_api() == -1) {
    return NIL;
  }
  if (api.init() == -1) {
    return NIL;
  }
  context_slot = malloc(sizeof(struct context) *
			MAX_CONTEXT);
  for (i = 0; i < MAX_CONTEXT; i++) {
    context_slot[i].ac = NULL;
  }
  return sym_t;
}

static LISP
create_context()
{
  int i;
  for (i = 0; i < MAX_CONTEXT; i++) {
    if (!context_slot[i].ac) {
      context_slot[i].ac = api.create_context();
      return flocons(i);
    }
  }
  return NIL;
}


static LISP
release_context(LISP id_)
{
  int id = get_c_long(id_);
  if (context_slot[id].ac) {
    api.release_context(context_slot[id].ac);
    context_slot[id].ac = NULL;
  }
  return NIL;
}

static LISP
set_string(LISP id_, LISP str_)
{
  int id = get_c_long(id_);
  char *str;
  struct anthy_context *ac = get_anthy_context(id);
  if (!ac) {
    return NIL;
  }
  str = uim_get_c_string(str_);
  api.set_string(ac, str);
  free(str);
  return NIL;
}

static LISP
get_nr_segments(LISP id_)
{
  int id = get_c_long(id_);
  struct anthy_conv_stat acs;
  struct anthy_context *ac = get_anthy_context(id);
  if (!ac) {
    return NIL;
  }
  api.get_stat(ac, &acs);

  return flocons(acs.nr_segment);
}

static LISP
get_nr_candidates(LISP id_, LISP nth_)
{
  int id, nth;
  struct anthy_context *ac;
  struct anthy_conv_stat cs;
  id = get_c_long(id_);
  nth = get_c_long(nth_);
  ac = get_anthy_context(id);
  if (!ac) {
    return NIL;
  }
  api.get_stat(ac, &cs);
  if (nth < cs.nr_segment) {
    struct anthy_segment_stat ss;
    api.get_segment_stat(ac, nth, &ss);
    return flocons(ss.nr_candidate);
  }
  return NIL;
}

static LISP
get_nth_candidate(LISP id_, LISP seg_, LISP nth_)
{
  int id = get_c_long(id_);
  int seg = get_c_long(seg_);
  int nth  = get_c_long(nth_);
  int buflen;
  char *buf;
  struct anthy_context *ac = get_anthy_context(id);
  struct anthy_conv_stat cs;
  if (!ac) {
    return NIL;
  }
  buflen = api.get_segment(ac, seg, nth, NULL, 0);
  if (buflen == -1) {
    return NIL;
  }
  buf = alloca(buflen+1);
  api.get_segment(ac, seg, nth, buf, buflen+1);
  return strcons(buflen, buf);
}

static LISP
resize_segment(LISP id_, LISP seg_, LISP cnt_)
{
  int id = get_c_long(id_);
  int seg = get_c_long(seg_);
  int cnt = get_c_long(cnt_);
  struct anthy_context *ac = get_anthy_context(id);
  api.resize_segment(ac, seg, cnt);
  return NIL;
}

static LISP
commit_segment(LISP id_, LISP s_, LISP nth_)
{
  int id = get_c_long(id_);
  int s = get_c_long(s_);
  int nth = get_c_long(nth_);
  struct anthy_context *ac = get_anthy_context(id);
  api.commit_segment(ac, s, nth);
  return NIL;
}

void
uim_init_anthy()
{
  init_subr_0("anthy-lib-init", init_anthy_lib);

  init_subr_0("anthy-lib-alloc-context", create_context);
  init_subr_1("anthy-lib-free-context", release_context);

  init_subr_2("anthy-lib-set-string", set_string);
  init_subr_1("anthy-lib-get-nr-segments",get_nr_segments);
  init_subr_2("anthy-lib-get-nr-candidates", get_nr_candidates);
  init_subr_3("anthy-lib-get-nth-candidate", get_nth_candidate);
  init_subr_3("anthy-lib-resize-segment", resize_segment);
  init_subr_3("anthy-lib-commit-segment", commit_segment);
}
