// WinApp.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _WIN_APP_H_
#define _WIN_APP_H_

#include "Window.h"
#include <vector>
#include <list>
using std::list;


namespace Manah {
namespace Windows {


// CMessageArguments class definition
/////////////////////////////////////////////////////////////////////////////

class CMessageArguments {
	friend class CWinApp;

	// RXgN^
public:
	CMessageArguments() {}
private:
	CMessageArguments(const CMessageArguments& rhs);

	// Zq
public:
	template<class T>
	CMessageArguments& operator %(const T& rhs) {
		basic_stringstream<TCHAR> ss;
		ss << rhs;
		m_listArgs.push_back(ss.str());
		return *this;
	}
private:
	operator =(const CMessageArguments& rhs);

	// f[^o
private:
  list<tstring> m_listArgs;

};


// CWinApp class definition
/////////////////////////////////////////////////////////////////////////////

class CWinApp : public CObject {
	// RXgN^
public:
	CWinApp(const Controls::CWindow* pWindow = 0);
	virtual ~CWinApp();

	// \bh
public:
	static void		GetCommandLineArguments(const TCHAR* pszCmdLine, std::vector<tstring>& vecArgs);
	void			GetCommandLineArguments(std::vector<tstring>& vecArgs) const;	// these method just split
																					// arguments by space
	Controls::CWindow*	GetMainWindow() const;
	int				LoadString(UINT nID, TCHAR* lpszString, int cchMax) const;
	tstring			LoadString(UINT nID) const;
	tstring			LoadMessage(DWORD dwID, const CMessageArguments& args = CMessageArguments()) const;
	HCURSOR			LoadCursor(UINT nCursorID) const;
	HCURSOR			LoadCursor(const TCHAR* lpszCursorName) const;
	HCURSOR			LoadStandardCursor(const TCHAR* lpszCursorName) const;
	HICON			LoadIcon(UINT nIconID) const;
	HICON			LoadIcon(const TCHAR* lpszIconName) const;
	HICON			LoadStandardIcon(const TCHAR* lpszIconName) const;
	HBITMAP			LoadBitmap(UINT nID) const;
	HBITMAP			LoadBitmap(const TCHAR* lpszBitmapName) const;
	void			LoadAccelerators(UINT nID);
	void			LoadAccelerators(const TCHAR* lpszTableName);
	UINT			GetProfileInt(const TCHAR* lpszSectionName, const TCHAR* lpszKeyName, UINT nDefault);
	bool			WriteProfileInt(const TCHAR* lpszSectionName, const TCHAR* lpszKeyName, UINT nValue);
	const TCHAR*	GetProfileString(const TCHAR* lpszSectionName,
						const TCHAR* lpszKeyName, const TCHAR* lpszDefault = 0);
	bool			WriteProfileString(const TCHAR* lpszSectionName,
						const TCHAR* lpszKeyName, const TCHAR* lpszString);
	bool			GetProfileStruct(const TCHAR* lpszSectionName,
						const TCHAR* lpszKeyName, void* lpStruct, UINT uSize);
	bool			WriteProfileStruct(const TCHAR* lpszSectionName,
						const TCHAR* lpszKeyName, void* lpStruct, UINT uSize);

	virtual bool	InitApplication();
	virtual bool	InitInstance();
	virtual int		Run();
	virtual bool	PreTranslateMessage(MSG* pMsg);
protected:
	virtual LRESULT	DispatchEvent(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) = 0;
	virtual void	GetModelessDialogList(std::list<HWND>& listDlgs) const = 0;

	// f[^o
protected:
	Controls::CWindow*	m_pMainWindow;
	HINSTANCE			m_hInstance;				// application instance
//	const TCHAR*		m_pszCmdLine;				// command line
	TCHAR				m_szModulePath[MAX_PATH];	// executing application path
	TCHAR				m_szINIPath[MAX_PATH];		// INI file path
	TCHAR				m_szHelpPath[MAX_PATH];		// help file (*.hlp, *.chm) path
	HACCEL				m_hAcceleratorTable;
};

} /* namespace Windows */
} /* namespace Manah */


// CWinApp class implementation
/////////////////////////////////////////////////////////////////////////////

using Manah::Windows::CWinApp;
using Manah::Windows::CMessageArguments;

inline CWinApp::CWinApp(const Manah::Windows::Controls::CWindow* pWindow /* = 0 */)
		: m_pMainWindow(const_cast<Manah::Windows::Controls::CWindow*>(pWindow)), m_hAcceleratorTable(0) {
	TCHAR*	psz = 0;

	m_hInstance = ::GetModuleHandle(0);
	::GetModuleFileName(0, m_szModulePath, MAX_PATH);
	psz = _tcsrchr(m_szModulePath, _T('.'));
	_tcscpy(m_szINIPath, m_szModulePath);
	_tcscpy(m_szINIPath + (psz - m_szModulePath + 1), _T("ini"));
	_tcscpy(m_szHelpPath, m_szModulePath);
	_tcscpy(m_szHelpPath + (psz - m_szModulePath + 1), _T("chm"));
}

inline CWinApp::~CWinApp() {
}

inline void CWinApp::GetCommandLineArguments(const TCHAR* pszCmdLine, std::vector<tstring>& vecArgs) {
	assert(pszCmdLine != 0);
	TCHAR*			pszEach = new TCHAR[_tcslen(pszCmdLine) + 1];
	TCHAR			ch;
	size_t			i = 0;
	size_t			cchEach = 0;
	bool			bInQuote = false;

	vecArgs.clear();
	while(true) {
		ch = *(pszCmdLine + i);
		if(ch == 0) {	// terminator
			if(cchEach != 0) {
				*(pszEach + cchEach) = 0;
				vecArgs.push_back(pszEach);
			}
			break;
		}
		if(!bInQuote) {
			if(ch == _T(' ')) {
				if(cchEach != 0) {
					*(pszEach + cchEach) = 0;
					vecArgs.push_back(pszEach);
					cchEach = 0;
				}
			} else if(ch == _T('\"'))
				bInQuote = true;
			else {
				*(pszEach + cchEach) = ch;
				++cchEach;
			}
		} else {
			if(ch == _T('\"'))
				bInQuote = false;
			else {
				*(pszEach + cchEach) = ch;
				++cchEach;
			}
		}
		++i;
	}
	delete[] pszEach;
}

inline void CWinApp::GetCommandLineArguments(std::vector<tstring>& vecArgs) const {
	AssertValid();

	const TCHAR*	pszCmdLine = ::GetCommandLine();
	int				i = 0;
	TCHAR			ch;
	TCHAR*			psz;

	while(true) {
		ch = *(pszCmdLine + i);
		if(ch == 0) {
			vecArgs.clear();
			return;
		} else if(ch == _T('\"')) {
			psz = _tcschr(pszCmdLine + i + 1, _T('\"'));
			if(psz == 0) {
				vecArgs.clear();
				return;
			}
			i = psz - pszCmdLine + 1;
		} else if(ch == _T('\'')) {
			psz = _tcschr(pszCmdLine + i + 1, _T('\''));
			if(psz == 0) {
				vecArgs.clear();
				return;
			}
			i = psz - pszCmdLine + 1;
		} else if(ch == _T(' ')) {
			++i;
			break;
		} else
			++i;
	}
	GetCommandLineArguments(pszCmdLine + i, vecArgs);
}

inline Manah::Windows::Controls::CWindow* CWinApp::GetMainWindow() const {
	AssertValid();
	return m_pMainWindow;
}

inline UINT CWinApp::GetProfileInt(const TCHAR* lpszSectionName, const TCHAR* lpszKeyName, UINT nDefault) {
	AssertValid();
	return ::GetPrivateProfileInt(lpszSectionName, lpszKeyName, nDefault, m_szINIPath);
}

inline const TCHAR* CWinApp::GetProfileString(const TCHAR* lpszSectionName,
		const TCHAR* lpszKeyName, const TCHAR* lpszDefault /* = 0 */) {
	AssertValid();
	static TCHAR	szReturned[16384];

	::GetPrivateProfileString(lpszSectionName, lpszKeyName,
		(lpszDefault != 0) ? lpszDefault : _T(""), szReturned, 16384, m_szINIPath);
	return szReturned;
}

inline bool CWinApp::GetProfileStruct(
	   const TCHAR* lpszSectionName, const TCHAR* lpszKeyName, void* lpStruct, UINT uSize) {
	AssertValid();
	return toBoolean(::GetPrivateProfileStruct(lpszSectionName, lpszKeyName, lpStruct, uSize, m_szINIPath));
}

inline bool CWinApp::InitApplication() {
	return true;
}

inline bool CWinApp::InitInstance() {
	return true;
}

inline void CWinApp::LoadAccelerators(UINT nID) {
	AssertValid();
	m_hAcceleratorTable = ::LoadAccelerators(m_hInstance, MAKEINTRESOURCE(nID));
}

inline void CWinApp::LoadAccelerators(const TCHAR* lpszTableName) {
	AssertValid();
	m_hAcceleratorTable = ::LoadAccelerators(m_hInstance, lpszTableName);
}

inline HBITMAP CWinApp::LoadBitmap(UINT nID) const {
	AssertValid();
	return ::LoadBitmap(m_hInstance, MAKEINTRESOURCE(nID));
}

inline HBITMAP CWinApp::LoadBitmap(const TCHAR* lpszBitmapName) const {
	AssertValid();
	return ::LoadBitmap(m_hInstance, lpszBitmapName);
}

inline HCURSOR CWinApp::LoadCursor(UINT nCursorID) const {
	AssertValid();
	return ::LoadCursor(m_hInstance, MAKEINTRESOURCE(nCursorID));
}

inline HCURSOR CWinApp::LoadCursor(const TCHAR* lpszCursorName) const {
	AssertValid();
	return ::LoadCursor(m_hInstance, lpszCursorName);
}

inline tstring CWinApp::LoadMessage(DWORD dwID, const CMessageArguments& args /* = CMessageArguments() */) const {
	AssertValid();

	void*								psz = 0;
	TCHAR**								pp = new TCHAR*[args.m_listArgs.size()];
	std::list<tstring>::const_iterator	it;
	size_t							    i;

	for(it = args.m_listArgs.begin(), i = 0; it != args.m_listArgs.end(); ++it, ++i)
		pp[i] = const_cast<TCHAR*>(it->c_str());
	::FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_HMODULE
		| FORMAT_MESSAGE_ARGUMENT_ARRAY, m_hInstance, dwID, LANG_USER_DEFAULT,
		reinterpret_cast<TCHAR*>(&psz), 0, reinterpret_cast<va_list*>(pp));
	delete[] pp;
	if(psz != 0) {
		tstring str = static_cast<TCHAR*>(psz);
		::LocalFree(psz);
		return str;
	} else
		return _T("");
}

inline HCURSOR CWinApp::LoadStandardCursor(const TCHAR* lpszCursorName) const {
	AssertValid();
	return ::LoadCursor(0, lpszCursorName);
}

inline HICON CWinApp::LoadIcon(UINT nIconID) const {
	AssertValid();
	return ::LoadIcon(m_hInstance, MAKEINTRESOURCE(nIconID));
}

inline HICON CWinApp::LoadIcon(const TCHAR* lpszIconName) const {
	AssertValid();
	return ::LoadIcon(m_hInstance, lpszIconName);
}

inline HICON CWinApp::LoadStandardIcon(const TCHAR* lpszIconName) const {
	AssertValid();
	return ::LoadIcon(0, lpszIconName);
}

inline int CWinApp::LoadString(UINT nID, TCHAR* lpszString, int cchMax) const {
	AssertValid();
	return ::LoadString(m_hInstance, nID, lpszString, cchMax);
}

inline tstring CWinApp::LoadString(UINT nID) const {
	AssertValid();
	TCHAR	szString[1024];
	::LoadString(m_hInstance, nID, szString, 1024);
	return tstring(szString);
}

inline bool CWinApp::PreTranslateMessage(MSG* pMsg) {
	return false;
}

inline int CWinApp::Run() {
	MSG								msg;
	std::list<HWND>					listDlgs;
	std::list<HWND>::const_iterator	it;
	int								f;
	bool							bIsDlgMsg;

	while(true) {
		bIsDlgMsg = false;
		f = ::GetMessage(&msg, 0, 0, 0);
		assert(f != -1);
		if(f == 0)
			break;
		GetModelessDialogList(listDlgs);
		for(it = listDlgs.begin(); it != listDlgs.end(); ++it) {
			if(::IsWindow(*it) && ::IsDialogMessage(*it, &msg)) {
				bIsDlgMsg = true;
				break;
			}
		}
		if(!bIsDlgMsg) {
			if(m_hAcceleratorTable != 0) {
				if(!::TranslateAccelerator(msg.hwnd, m_hAcceleratorTable, &msg)
						&& !PreTranslateMessage(&msg)) {
					::TranslateMessage(&msg);
					::DispatchMessage(&msg);
				}
			} else if(!PreTranslateMessage(&msg)) {
				::TranslateMessage(&msg);
				::DispatchMessage(&msg);
			}
		}
	}

	return 0;
}

inline bool CWinApp::WriteProfileInt(const TCHAR* lpszSectionName, const TCHAR* lpszKeyName, UINT nValue) {
	AssertValid();

	TCHAR	szWrite[12];
	wsprintf(szWrite, _T("%d"), nValue);
	return toBoolean(::WritePrivateProfileString(lpszSectionName, lpszKeyName, szWrite, m_szINIPath));
}

inline bool CWinApp::WriteProfileString(const TCHAR* lpszSectionName, const TCHAR* lpszKeyName, const TCHAR* lpszString) {
	AssertValid();
	return toBoolean(::WritePrivateProfileString(lpszSectionName, lpszKeyName, lpszString, m_szINIPath));
}

inline bool CWinApp::WriteProfileStruct(
	   const TCHAR* lpszSectionName, const TCHAR* lpszKeyName, void* lpStruct, UINT uSize) {
	AssertValid();
	return toBoolean(::WritePrivateProfileStruct(lpszSectionName, lpszKeyName, lpStruct, uSize, m_szINIPath));
}

#endif /* _WIN_APP_H_ */

/* [EOF] */