// win_utils.h
// (c) 2004 exeal

#ifndef _WIN_UTILS_H_
#define _WIN_UTILS_H_

#include "Object.h"
#include "DC.h"


namespace Manah {
namespace Windows {

// this header contains following classes:
class CPoint;
class CSize;
class CRect;
class CFileFind;
class CClipboard;


// CPoint class definition
/////////////////////////////////////////////////////////////////////////////

class CPoint : public tagPOINT {
	// constructors
public:
	CPoint();
	CPoint(int xInit, int yInit);
	CPoint(const POINT& pt);
	CPoint(const SIZE& size);
	CPoint(DWORD dwPoint);

	// operators
public:
	bool	operator ==(const POINT& pt) const;
	bool	operator !=(const POINT& size) const;
	CPoint&	operator +=(const POINT& rhs);
	CPoint&	operator +=(const SIZE& rhs);
	CPoint&	operator -=(const POINT& rhs);
	CPoint&	operator -=(const SIZE& rhs);
	CPoint	operator +(const POINT& pt) const;
	CPoint	operator +(const SIZE& size) const;
	CRect	operator +(const CRect& rect) const;
	CPoint	operator -() const;
	CPoint	operator -(const POINT& pt) const;
	CPoint	operator -(const SIZE& size) const;
	CRect	operator -(const RECT& rect) const;

	// methods
public:
	void	Offset(int xOffset, int yOffset);
	void	Offset(const POINT& pt);
	void	Offset(const SIZE& size);
};


// CSize class definition
/////////////////////////////////////////////////////////////////////////////

class CSize : public tagSIZE {
	// constructors
public:
	CSize();
	CSize(int xInit, int yInit);
	CSize(const SIZE& size);
	CSize(const POINT& pt);
	CSize(DWORD dwSize);

	// operators
public:
	bool	operator ==(const SIZE& size) const;
	bool	operator !=(const SIZE& size) const;
	CSize&	operator +=(const SIZE& rhs);
	CSize&	operator -=(const SIZE& rhs);
	CSize	operator +(const SIZE& size) const;
	CPoint	operator +(const POINT& pt) const;
	CRect	operator +(const RECT& rect) const;
	CSize	operator -() const;
	CSize	operator -(const SIZE& size) const;
	CPoint	operator -(const POINT& pt) const;
	CRect	operator -(const RECT& rect) const;
};


// CRect class definition
/////////////////////////////////////////////////////////////////////////////

class CRect : public tagRECT {
	// constructors
public:
	CRect();
	CRect(int leftInit, int topInit, int rightInit, int bottomInit);
	CRect(const RECT& rect);
	CRect(const RECT* lpRect);
	CRect(const POINT& pt, const SIZE& size);
	CRect(const POINT& ptLeftTop, const POINT& ptRightBottom);

	// operators
public:
			operator const RECT*() const;
			operator RECT*() const;
	CRect&	operator =(const RECT& rhs);
	bool	operator ==(const RECT& rhs) const;
	bool	operator !=(const RECT& rhs) const;
	CRect&	operator +=(const POINT& rhs);
	CRect&	operator +=(const SIZE& rhs);
	CRect&	operator +=(const RECT& rhs);
	CRect&	operator -=(const POINT& rhs);
	CRect&	operator -=(const SIZE& rhs);
	CRect&	operator -=(const RECT& rhs);
	CRect&	operator &=(const RECT& rhs);
	CRect&	operator |=(const RECT& rhs);
	CRect	operator +(const POINT& pt) const;
	CRect	operator +(const SIZE& size) const;
	CRect	operator +(const RECT& rect) const;
	CRect	operator -(const POINT& pt) const;
	CRect	operator -(const SIZE& size) const;
	CRect	operator -(const RECT& rect) const;
	CRect	operator &(const RECT& rect) const;
	CRect	operator |(const RECT& rect) const;

	// methods
public:
	int		Width() const;
	int		Height() const;
	CSize	Size() const;
	CPoint	TopLeft() const;
	CPoint	BottomRight() const;
	CPoint	CenterPoint() const;
	bool	IsRectEmpty() const;
	bool	IsRectNull() const;
	bool	PtInRect(const POINT& pt) const;
	void	SetRect(int left, int top, int right, int bottom);
	void	SetRectEmpty();
	void	CopyRect(const RECT& rect);
	bool	EqualRect(const RECT& rect) const;
	void	InflateRect(int x, int y);
	void	InflateRect(const SIZE& size);
	void	InflateRect(const RECT& rect);
	void	InflateRect(int left, int top, int right, int bottom);
	void	DeflateRect(int x, int y);
	void	DeflateRect(const SIZE& size);
	void	DeflateRect(const RECT& rect);
	void	DeflateRect(int left, int top, int right, int bottom);
	void	NormalizeRect();
	void	OffsetRect(int x, int y);
	void	OffsetRect(const POINT& pt);
	void	OffsetRect(const SIZE& size);
	bool	SubtractRect(const RECT& rect1, const RECT& rect2);
	bool	IntersectRect(const RECT& rect1, const RECT& rect2);
	bool	UnionRect(const RECT& rect1, const RECT& rect2);
};


// CFileFind class definition
/////////////////////////////////////////////////////////////////////////////

class CFileFind {
	// constructors
public:
	CFileFind();
	~CFileFind();

	// methods
public:
	/* attributes */
	void			GetCreationTime(FILETIME& timeStamp) const;
	std::tstring	GetFileName() const;
	std::tstring	GetFilePath() const;
	ULONGLONG		GetFileSize() const;
	std::tstring	GetFileTitle() const;
	std::tstring	GetFileUrl() const;
	void			GetLastAccessTime(FILETIME& timeStamp) const;
	void			GetLastWriteTime(FILETIME& timeStamp) const;
	std::tstring	GetRoot() const;
	bool			IsArchived() const;
	bool			IsCompressed() const;
	bool			IsDirectory() const;
	bool			IsDots() const;
	bool			IsHidden() const;
	bool			IsNormal() const;
	bool			IsReadOnly() const;
	bool			IsSystem() const;
	bool			IsTemporary() const;
	bool			MatchesMask(DWORD dwMask) const;

	/* operations */
	void	Close();
	bool	Find(const TCHAR* lpszName = _T("*.*"));
	bool	FindNext();

	// data members
private:
	HANDLE			m_hFind;
	WIN32_FIND_DATA	m_wfd;
	bool			m_bFound;
};


// CClipboard class definition
/////////////////////////////////////////////////////////////////////////////

class CClipboard {
	// methods
public:
	/* open and read the clipboard text */
	static DWORD	GetClipboardTextSize(HWND hWnd);	// return SIZE (byte length) of clipboard text
	static bool		ReadClipboardText(					// read multi-bytes text from the clipboard
		HWND hWnd, char* pszText, std::size_t cch);		//  as format CF_TEXT
	static bool		ReadClipboardText(					// read wide-characters from the clipboard
		HWND hWnd, wchar_t* pszText, std::size_t cch);	//  as format CF_UNICODETEXT

	/* open and set the clipboard with CF_TEXT format text */
	static bool	SetClipboardText(		// set multi-bytes text to the clipboard
		HWND hWnd, const char* pszText, std::size_t cch = -1);
	static bool	SetClipboardText(		// set wide-characters to the clipboard
		HWND hWnd, const wchar_t* pszText, std::size_t cch = -1);

	/* basic clipboard operations (getter will return permanent object) */
	static void*	GetClipboardData(HWND hWnd,
						UINT nFormat, DWORD* pdwSize) throw(std::invalid_argument);
	static bool		SetClipboardData(HWND hWnd,
						UINT nFormat, const void* pData, DWORD dwSize) throw(std::invalid_argument);
	static void*	GetFirstAvailableClipboardData(HWND hWnd,
						UINT* pnFormat, DWORD* pdwSize) throw(std::invalid_argument);
};


// CPoint class implementation
/////////////////////////////////////////////////////////////////////////////

inline CPoint::CPoint() {
}

inline CPoint::CPoint(int xInit, int yInit) {
	x = xInit;
	y = yInit;
}

inline CPoint::CPoint(const POINT& pt) {
	x = pt.x;
	y = pt.y;
}

inline CPoint::CPoint(const SIZE& size) {
	x = size.cx;
	y = size.cy;
}

inline CPoint::CPoint(DWORD dwPoint) {
	x = LOWORD(dwPoint);
	y = HIWORD(dwPoint);
}

inline bool CPoint::operator !=(const POINT& pt) const {
	return (x != pt.x || y != pt.y);
}

inline CPoint CPoint::operator +(const POINT& pt) const {
	return CPoint(*this) += pt;
}

inline CPoint CPoint::operator +(const SIZE& size) const {
	return CPoint(*this) + size;
}

inline CRect CPoint::operator +(const CRect& rect) const {
	return CRect(rect.left + x, rect.top + y, rect.right + x, rect.bottom + y);
}

inline CPoint& CPoint::operator +=(const POINT& rhs) {
	x += rhs.x;
	y += rhs.y;
	return *this;
}

inline CPoint& CPoint::operator +=(const SIZE& rhs) {
	x += rhs.cx;
	y += rhs.cy;
	return *this;
}

inline CPoint CPoint::operator -() const {
	return CPoint(-x, -y);
}

inline CPoint CPoint::operator -(const POINT& pt) const {
	return CPoint(*this) -= pt;
}

inline CPoint CPoint::operator -(const SIZE& size) const {
	return CPoint(*this) -= size;
}

inline CRect CPoint::operator -(const RECT& rect) const {
	return CRect(rect.left - x, rect.top - y, rect.right - x, rect.bottom - y);
}

inline CPoint& CPoint::operator -=(const POINT& rhs) {
	x -= rhs.x;
	y -= rhs.y;
	return *this;
}

inline CPoint& CPoint::operator -=(const SIZE& rhs) {
	x -= rhs.cx;
	y -= rhs.cy;
	return *this;
}

inline bool CPoint::operator ==(const POINT& pt) const {
	return (x == pt.x && y == pt.y);
}

inline void CPoint::Offset(int xOffset, int yOffset) {
	x += xOffset;
	y += yOffset;
}

inline void CPoint::Offset(const POINT& pt) {
	*this += pt;
}

inline void CPoint::Offset(const SIZE& size) {
	*this += size;
}


// CSize class implementation
/////////////////////////////////////////////////////////////////////////////

inline CSize::CSize() {
}

inline CSize::CSize(int cxInit, int cyInit) {
	cx = cxInit;
	cy = cyInit;
}

inline CSize::CSize(const SIZE& size) {
	cx = size.cx;
	cy = size.cy;
}

inline CSize::CSize(const POINT& pt) {
	cx = pt.x;
	cy = pt.y;
}

inline CSize::CSize(DWORD dwSize) {
	cx = LOWORD(dwSize);
	cy = HIWORD(dwSize);
}

inline bool CSize::operator !=(const SIZE& size) const {
	return (cx != size.cx || cy != size.cy);
}

inline CSize CSize::operator +(const SIZE& size) const {
	return CSize(*this) += size;
}

inline CPoint CSize::operator +(const POINT& pt) const {
	return CSize(pt.x + cx, pt.y + cy);
}

inline CRect CSize::operator +(const RECT& rect) const {
	return CRect(rect.left + cx, rect.top + cy, rect.right + cx, rect.bottom + cy);
}

inline CSize& CSize::operator +=(const SIZE& rhs) {
	cx += rhs.cx;
	cy += rhs.cy;
	return *this;
}

inline CSize CSize::operator -() const {
	return CSize(-cx, -cy);
}

inline CSize CSize::operator -(const SIZE& size) const {
	return CSize(*this) -= size;
}

inline CPoint CSize::operator -(const POINT& pt) const {
	return CSize(pt.x - cx, pt.y - cy);
}

inline CRect CSize::operator -(const RECT& rect) const {
	return CRect(rect.left - cx, rect.top - cy, rect.right - cx, rect.bottom - cy);
}

inline CSize& CSize::operator -=(const SIZE& rhs) {
	cx -= rhs.cx;
	cy -= rhs.cy;
	return *this;
}

inline bool CSize::operator ==(const SIZE& size) const {
	return (cx == size.cx && cy == size.cy);
}


// CRect class implementation
/////////////////////////////////////////////////////////////////////////////

inline CRect::CRect() {
}

inline CRect::CRect(int leftInit, int topInit, int rightInit, int bottomInit) {
	left = leftInit;
	top = topInit;
	right = rightInit;
	bottom = bottomInit;
}

inline CRect::CRect(const RECT& rect) {
	left = rect.left;
	top = rect.top;
	right = rect.right;
	bottom = rect.bottom;
}

inline CRect::CRect(const RECT* lpRect) {
	left = lpRect->left;
	top = lpRect->top;
	right = lpRect->right;
	bottom = lpRect->bottom;
}

inline CRect::CRect(const POINT& pt, const SIZE& size) {
	left = pt.x;
	top = pt.y;
	right = pt.x + size.cx;
	bottom = pt.y + size.cy;
}

inline CRect::CRect(const POINT& ptLeftTop, const POINT& ptRightBottom) {
	left = ptLeftTop.x;
	top = ptLeftTop.y;
	right = ptRightBottom.x;
	bottom = ptRightBottom.y;
}

inline bool CRect::operator !=(const RECT& rhs) const {
	return !EqualRect(rhs);
}

inline CRect CRect::operator &(const RECT& rect) const {
	return CRect(*this) &= rect;
}

inline CRect& CRect::operator &=(const RECT& rhs) {
	IntersectRect(*this, rhs);
	return *this;
}

inline CRect CRect::operator +(const POINT& pt) const {
	return CRect(*this) += pt;
}

inline CRect CRect::operator +(const SIZE& size) const {
	return CRect(*this) += size;
}

inline CRect CRect::operator +(const RECT& rect) const {
	return CRect(*this) += rect;
}

inline CRect& CRect::operator +=(const POINT& rhs) {
	OffsetRect(rhs);
	return *this;
}

inline CRect& CRect::operator +=(const SIZE& rhs) {
	InflateRect(rhs);
	return *this;
}

inline CRect& CRect::operator +=(const RECT& rhs) {
	left += rhs.left;
	top += rhs.top;
	right += rhs.right;
	bottom += rhs.bottom;
	return *this;
}

inline CRect CRect::operator -(const POINT& pt) const {
	return CRect(*this) -= pt;
}

inline CRect CRect::operator -(const SIZE& size) const {
	return CRect(*this) -= size;
}

inline CRect CRect::operator -(const RECT& rect) const {
	return CRect(*this) -= rect;
}

inline CRect& CRect::operator -=(const POINT& rhs) {
	OffsetRect(-rhs.x, -rhs.y);
	return *this;
}

inline CRect& CRect::operator -=(const SIZE& rhs) {
	DeflateRect(rhs);
	return *this;
}

inline CRect& CRect::operator -=(const RECT& rhs) {
	left -= rhs.left;
	top -= rhs.top;
	right -= rhs.right;
	bottom -= rhs.bottom;
	return *this;
}

inline CRect& CRect::operator =(const RECT& rhs) {
	SetRect(rhs.left, rhs.top, rhs.right, rhs.bottom);
}

inline bool CRect::operator ==(const RECT& rhs) const {
	return EqualRect(rhs);
}

inline CRect::operator const RECT *() const {
	return this;
}

inline CRect::operator RECT *() const {
	return const_cast<CRect*>(this);
}

inline CRect CRect::operator |(const RECT& rect) const {
	return CRect(*this) |= rect;
}

inline CRect& CRect::operator |=(const RECT& rhs) {
	UnionRect(*this, rhs);
	return *this;
}

inline CPoint CRect::BottomRight() const {
	return CPoint(right, bottom);
}

inline CPoint CRect::CenterPoint() const {
	return CPoint((right - left) / 2, (bottom - top) / 2);
}

inline void CRect::CopyRect(const RECT& rect) {
	*this = rect;
}

inline void CRect::DeflateRect(int x, int y) {
	InflateRect(-x, -y);
}

inline void CRect::DeflateRect(const SIZE& size) {
	InflateRect(-size.cx, -size.cy);
}

inline void CRect::DeflateRect(const RECT& rect) {
	SetRect(left + rect.left, top + rect.top, right - rect.right, bottom - rect.bottom);
}

inline void CRect::DeflateRect(int left_, int top_, int right_, int bottom_) {
	SetRect(left + left_, top + top_, right - right_, bottom - bottom_);
}

inline bool CRect::EqualRect(const RECT& rect) const {
	return toBoolean(::EqualRect(*this, &rect));
}

inline int CRect::Height() const {
	return bottom - top;
}

inline void CRect::InflateRect(int x, int y) {
	::InflateRect(this, x, y);
}

inline void CRect::InflateRect(const SIZE& size) {
	::InflateRect(this, size.cx, size.cy);
}

inline void CRect::InflateRect(const RECT& rect) {
	SetRect(left - rect.left, top - rect.top, right + rect.right, bottom + rect.bottom);
}

inline void CRect::InflateRect(int left_, int top_, int right_, int bottom_) {
	SetRect(left - left_, top - top_, right + right_, bottom + bottom_);
}

inline bool CRect::IntersectRect(const RECT& rect1, const RECT& rect2) {
	return toBoolean(::IntersectRect(*this, &rect1, &rect2));
}

inline bool CRect::IsRectEmpty() const {
	return toBoolean(::IsRectEmpty(this));
}

inline bool CRect::IsRectNull() const {
	return toBoolean(left == 0 && top == 0 && right == 0 && bottom == 0);
}

inline void CRect::NormalizeRect() {
	if(top > bottom)
		std::swap(top, bottom);
	if(left > right)
		std::swap(left, right);
}

inline void CRect::OffsetRect(int x, int y) {
	::OffsetRect(*this, x, y);
}

inline void CRect::OffsetRect(const POINT& pt) {
	::OffsetRect(*this, pt.x, pt.y);
}

inline void CRect::OffsetRect(const SIZE& size) {
	::OffsetRect(*this, size.cx, size.cy);
}

inline bool CRect::PtInRect(const POINT& pt) const {
	return toBoolean(::PtInRect(*this, pt));
}

inline void CRect::SetRect(int left, int top, int right, int bottom) {
	::SetRect(*this, left, top, right, bottom);
}

inline void CRect::SetRectEmpty() {
	::SetRectEmpty(*this);
}

inline CSize CRect::Size() const {
	return CSize(right - left, bottom - top);
}

inline bool CRect::SubtractRect(const RECT& rect1, const RECT& rect2) {
	return toBoolean(::SubtractRect(this, &rect1, &rect2));
}

inline CPoint CRect::TopLeft() const {
	return CPoint(left, top);
}

inline bool CRect::UnionRect(const RECT& rect1, const RECT& rect2) {
	return toBoolean(::UnionRect(this, &rect1, &rect2));
}

inline int CRect::Width() const {
	return right - left;
}


// CFileFind class implementation
/////////////////////////////////////////////////////////////////////////////

inline CFileFind::CFileFind() : m_hFind(0), m_bFound(false) {
}

inline CFileFind::~CFileFind() {
	Close();
}

inline void CFileFind::Close() {
	if(m_hFind != 0) {
		::FindClose(m_hFind);
		m_hFind = 0;
		m_bFound = false;
	}
}

inline bool CFileFind::Find(const TCHAR* lpszName /* = _T("*.*") */) {
	Close();
	assert(lpszName != 0);
	assert(std::_tcslen(lpszName) < MAX_PATH);

	std::_tcscpy(m_wfd.cFileName, lpszName);
	m_hFind = ::FindFirstFile(lpszName, &m_wfd);
	if(m_hFind == INVALID_HANDLE_VALUE) {
		m_hFind = 0;
		return false;
	}
	m_bFound = true;
	return true;
}

inline bool CFileFind::FindNext() {
	if(m_hFind != 0 && m_bFound)
		m_bFound = toBoolean(::FindNextFile(m_hFind, &m_wfd));
	return m_bFound;
}

inline void CFileFind::GetCreationTime(FILETIME& timeStamp) const {
	assert(m_bFound);
	timeStamp = m_wfd.ftCreationTime;
}

inline std::tstring CFileFind::GetFileName() const {
	assert(m_bFound);
	return m_wfd.cFileName;
}

inline std::tstring CFileFind::GetFilePath() const {
	assert(m_bFound);
	TCHAR	szPath[MAX_PATH];
	return (_tfullpath(szPath, m_wfd.cFileName, MAX_PATH) != 0) ? szPath : _T("");
}

inline ULONGLONG CFileFind::GetFileSize() const {
	assert(m_bFound);

	ULARGE_INTEGER	size = {0};
	size.HighPart = m_wfd.nFileSizeHigh;
	size.LowPart = m_wfd.nFileSizeLow;
	return size.QuadPart;
}

inline std::tstring CFileFind::GetFileTitle() const {
	assert(m_bFound);

	const std::tstring	strName = GetFileName();
	if(!strName.empty()) {
		TCHAR	szTitle[MAX_PATH];
		_tsplitpath(strName.c_str(), 0, 0, szTitle, 0);
		return szTitle;
	}
	return _T("");
}

inline std::tstring CFileFind::GetFileUrl() const {
	assert(m_bFound);
	const std::tstring	strPath = GetFilePath();
	return !strPath.empty() ? (_T("file://") + strPath) : _T("");
}

inline void CFileFind::GetLastAccessTime(FILETIME& timeStamp) const {
	assert(m_bFound);
	timeStamp = m_wfd.ftLastAccessTime;
}

inline void CFileFind::GetLastWriteTime(FILETIME& timeStamp) const {
	assert(m_bFound);
	timeStamp = m_wfd.ftLastWriteTime;
}

inline std::tstring CFileFind::GetRoot() const {
	assert(m_bFound);
	TCHAR	szPath[MAX_PATH];
	return (_tfullpath(szPath, m_wfd.cFileName, MAX_PATH) != 0) ? szPath : _T("");
}

inline bool CFileFind::IsArchived() const {
	return MatchesMask(FILE_ATTRIBUTE_ARCHIVE);
}

inline bool CFileFind::IsCompressed() const {
	return MatchesMask(FILE_ATTRIBUTE_COMPRESSED);
}

inline bool CFileFind::IsDirectory() const {
	return MatchesMask(FILE_ATTRIBUTE_DIRECTORY);
}

inline bool CFileFind::IsDots() const {
	return IsDirectory() && (std::_tcscmp(m_wfd.cFileName, _T(".")) == 0 || std::_tcscmp(m_wfd.cFileName, _T("..")) == 0);
}

inline bool CFileFind::IsHidden() const {
	return MatchesMask(FILE_ATTRIBUTE_HIDDEN);
}

inline bool CFileFind::IsNormal() const {
	return MatchesMask(FILE_ATTRIBUTE_NORMAL);
}

inline bool CFileFind::IsReadOnly() const {
	return MatchesMask(FILE_ATTRIBUTE_READONLY);
}

inline bool CFileFind::IsSystem() const {
	return MatchesMask(FILE_ATTRIBUTE_SYSTEM);
}

inline bool CFileFind::IsTemporary() const {
	return MatchesMask(FILE_ATTRIBUTE_TEMPORARY);
}

inline bool CFileFind::MatchesMask(DWORD dwMask) const {
	assert(m_bFound);
	return toBoolean(m_wfd.dwFileAttributes & dwMask);
}


// CClipboard class implementation
/////////////////////////////////////////////////////////////////////////////

inline void* CClipboard::GetClipboardData(HWND hWnd, UINT nFormat, DWORD* pdwSize) throw(std::invalid_argument) { 
	void*	pData = 0;
	DWORD	dwSize;
	HGLOBAL	hGlobal = 0;

	if(pdwSize == 0)
		throw std::invalid_argument("Third argument can not be null!");
	if(!::IsClipboardFormatAvailable(nFormat) || !::OpenClipboard(hWnd))
		return 0;
	hGlobal = ::GetClipboardData(nFormat);
	if(hGlobal == 0)
		return ::CloseClipboard(), 0;
	dwSize = ::GlobalSize(hGlobal);
	pData = malloc(dwSize);
	if(pData == 0)
		return ::CloseClipboard, ::GlobalUnlock(hGlobal), 0;
	std::memcpy(pData, GlobalLock(hGlobal), dwSize);
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	*pdwSize = dwSize;

	return pData;
}

inline DWORD CClipboard::GetClipboardTextSize(HWND hWnd) {
	LONG	dwSize;
	HGLOBAL	hGlobal;

	if(!::IsClipboardFormatAvailable(CF_TEXT))
		return -1;

	::OpenClipboard(hWnd);
	hGlobal = ::GetClipboardData(CF_TEXT);
	if(hGlobal == 0){
		::CloseClipboard();
		return -1;
	}
	dwSize = ::GlobalSize(hGlobal);
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	return dwSize;
}

inline void* CClipboard::GetFirstAvailableClipboardData(
		HWND hWnd, UINT* pnFormat, DWORD* pdwSize) throw(std::invalid_argument) {
	UINT	nFormat;
	void*	pData = 0;

	if(pnFormat == 0 || pdwSize == 0)
		throw std::invalid_argument("Second and third argument can not be null!");
	if(!::OpenClipboard(hWnd))
		return 0;
	nFormat = ::EnumClipboardFormats(0);
	if(nFormat == 0)
		return 0;
	pData = CClipboard::GetClipboardData(hWnd, nFormat, pdwSize);
	*pnFormat = (pData != 0) ? nFormat : 0;
	return pData;
}

inline bool CClipboard::ReadClipboardText(HWND hWnd, char* pszText, std::size_t cch) {
	HGLOBAL	hGlobal;
	char*	pszString;

	if(!::IsClipboardFormatAvailable(CF_TEXT))
		return false;

	::OpenClipboard(hWnd);
	hGlobal = ::GetClipboardData(CF_TEXT);
	if(hGlobal == 0)
		return ::CloseClipboard(), false;
	pszString = static_cast<char*>(::GlobalLock(hGlobal));
	std::strncpy(pszText, pszString, cch);
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	return true;
}

inline bool CClipboard::ReadClipboardText(HWND hWnd, wchar_t* pszText, std::size_t cch) {
	HGLOBAL		hGlobal;
	char*		psz = 0;
	wchar_t*	pwsz = 0;

	if(::IsClipboardFormatAvailable(CF_UNICODETEXT)) {
		::OpenClipboard(hWnd);
		hGlobal = ::GetClipboardData(CF_UNICODETEXT);
		if(hGlobal == 0)
			return ::CloseClipboard(), false;
		pwsz = static_cast<wchar_t*>(::GlobalLock(hGlobal));
		std::wcsncpy(pszText, pwsz, cch);
	} else if(::IsClipboardFormatAvailable(CF_TEXT)) {
		::OpenClipboard(hWnd);
		hGlobal = ::GetClipboardData(CF_TEXT);
		if(hGlobal == 0)
			return ::CloseClipboard(), false;
		psz = static_cast<char*>(::GlobalLock(hGlobal));
		::MultiByteToWideChar(CP_ACP, 0, psz, cch * 2, pszText, cch);
	} else
		return false;
	::GlobalUnlock(hGlobal);
	::CloseClipboard();

	return true;
}

inline bool CClipboard::SetClipboardData(
		HWND hWnd, UINT nFormat, const void* pData, DWORD dwSize) throw(std::invalid_argument) {
	HGLOBAL	hGlobal = 0;
	void*	pTempData = 0;
	bool	bReturn;

	if(pData == 0)
		throw std::invalid_argument("Third argument can not be null!");
	if(!::OpenClipboard(hWnd))
		return 0;
	::EmptyClipboard();
	hGlobal = ::GlobalAlloc(GHND, dwSize);
	if(hGlobal == 0)
		return ::CloseClipboard(), false;
	pTempData = ::GlobalLock(hGlobal);
	std::memcpy(pTempData, pData, dwSize);
	::GlobalUnlock(hGlobal);
	bReturn = (::SetClipboardData(nFormat, hGlobal) != 0);
	::CloseClipboard();

	return bReturn;
}

inline bool CClipboard::SetClipboardText(HWND hWnd, const char* pszText, std::size_t cch /* = -1 */) {
	HGLOBAL	hGlobal;
	char*	pszString;

	if(cch == -1)
		cch = std::strlen(pszText);
	hGlobal = ::GlobalAlloc(GHND, cch + 1);
	if(hGlobal == 0)
		return false;
	pszString = static_cast<char*>(::GlobalLock(hGlobal));
	std::strncpy(pszString, pszText, cch);
	::GlobalUnlock(hGlobal);
	if(!::OpenClipboard(hWnd)){
		::GlobalFree(hGlobal);
		return false;
	}
	::EmptyClipboard();
	::SetClipboardData(CF_TEXT, hGlobal);
	::CloseClipboard();

	return true;
}

inline bool CClipboard::SetClipboardText(HWND hWnd, const wchar_t* pszText, std::size_t cch /* = -1 */) {
	HGLOBAL		hGlobal;
	wchar_t*	pwszString;

	if(cch == -1)
		cch = std::wcslen(pszText);
	hGlobal = ::GlobalAlloc(GHND, (cch + 1) * 2);
	if(hGlobal == 0)
		return false;
	pwszString = static_cast<wchar_t*>(::GlobalLock(hGlobal));
	std::wcsncpy(pwszString, pszText, cch);
	::GlobalUnlock(hGlobal);
	if(!::OpenClipboard(hWnd)){
		::GlobalFree(hGlobal);
		return false;
	}
	::EmptyClipboard();
	::SetClipboardData(CF_UNICODETEXT, hGlobal);
	::CloseClipboard();

	return true;
}

} /* namespace Windows */
} /* namespace Manah */

#endif /* _WIN_UTILS_H_ */

/* [EOF] */