// TabCtrl.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _TAB_CTRL_H_
#define _TAB_CTRL_H_
#include "Window.h"
#include "ImageList.h"
#include "ToolTipCtrl.h"


// CTabCtrl class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CTabCtrl : public CCommonControl<CTabCtrl> {
	DEFINE_CLASS_NAME(WC_TABCONTROL)

	// \bh
public:
	/*  */
	CImageList*	GetImageList() const;
	CImageList*	SetImageList(const CImageList* pImageList);
	int			GetItemCount() const;
	bool		GetItem(int iItem, TCITEM& tabCtrlItem) const;
	bool		SetItem(int iItem, const TCITEM& tabCtrlItem);
	bool		SetItemExtra(int nBytes);
	bool		GetItemRect(int iItem, LPRECT lpRect) const;
	int			GetCurSel() const;
	int			SetCurFocus(int iItem);
	int			SetCurSel(int iItem);
	SIZE		SetItemSize(const SIZE& size);
	void		SetPadding(const SIZE& size);
	int			GetRowCount() const;
	HWND		GetToolTips() const;
	void		SetToolTips(HWND hwndTip);
	int			GetCurFocus() const;
	int			SetMinTabWidth(int cx);
	DWORD		GetExtendedStyle() const;
	DWORD		SetExtendedStyle(DWORD dwNewStyle, DWORD dwExMask = 0);
	bool		GetItemState(int iItem, DWORD dwMask, DWORD& dwState) const;
	bool		SetItemState(int iItem, DWORD dwMask, DWORD dwState);

	/*  */
	void	AdjustRect(bool bLarger, RECT& rect);
	bool	DeleteItem(int iItem);
	bool	DeleteAllItems();
	void	DeselectAll(bool bExcludeFocus);
	bool	InsertItem(int iItem, const TCITEM& item);
	bool	InsertItem(int iItem, const TCHAR* lpszItem);
	bool	InsertItem(int iItem, const TCHAR* lpszItem, int iImage);
	bool	InsertItem(UINT nMask, int iItem, const TCHAR* lpszItem, int iImage, LPARAM lParam);
	bool	HighlightItem(int iItem, bool bHighlight = true);
	int		HitTest(TCHITTESTINFO& hitTestInfo) const;
	void	RemoveImage(int iImage);

protected:
	virtual void	DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct);
};


// CTabCtrl class implementation
/////////////////////////////////////////////////////////////////////////////

inline void CTabCtrl::AdjustRect(bool bLarger, RECT& rect) {
	AssertValidAsWindow();
	TabCtrl_AdjustRect(m_hWnd, bLarger, &rect);
}

inline bool CTabCtrl::DeleteAllItems() {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_DeleteAllItems(m_hWnd));
}

inline void CTabCtrl::DeselectAll(bool bExcludeFocus) {
	AssertValidAsWindow();
	TabCtrl_DeselectAll(m_hWnd, bExcludeFocus);
}

inline bool CTabCtrl::DeleteItem(int iItem) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_DeleteItem(m_hWnd, iItem));
}

inline void CTabCtrl::DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct) {
}

inline int CTabCtrl::GetCurFocus() const {
	AssertValidAsWindow();
	return TabCtrl_GetCurFocus(m_hWnd);
}

inline int CTabCtrl::GetCurSel() const {
	AssertValidAsWindow();
	return TabCtrl_GetCurSel(m_hWnd);
}

inline DWORD CTabCtrl::GetExtendedStyle() const {
	AssertValidAsWindow();
	return TabCtrl_GetExtendedStyle(m_hWnd);
}

inline CImageList* CTabCtrl::GetImageList() const {
	AssertValidAsWindow();
	return CImageList::FromHandle(TabCtrl_GetImageList(m_hWnd));
}

inline bool CTabCtrl::GetItem(int iItem, TCITEM& tabCtrlItem) const {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_GetItem(m_hWnd, iItem, &tabCtrlItem));
}

inline int CTabCtrl::GetItemCount() const {
	AssertValidAsWindow();
	return TabCtrl_GetItemCount(m_hWnd);
}

inline bool CTabCtrl::GetItemRect(int iItem, LPRECT lpRect) const {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_GetItemRect(m_hWnd, iItem, lpRect));
}

inline bool CTabCtrl::GetItemState(int iItem, DWORD dwMask, DWORD& dwState) const {
	AssertValidAsWindow();

	TCITEM	item;
	item.mask = TCIF_STATE;
	item.dwStateMask = dwMask;
	if(!GetItem(iItem, item))
		return false;
	dwState = item.dwState;
	return true;
}

inline int CTabCtrl::GetRowCount() const {
	AssertValidAsWindow();
	return TabCtrl_GetRowCount(m_hWnd);
}

inline HWND CTabCtrl::GetToolTips() const {
	AssertValidAsWindow();
	return TabCtrl_GetToolTips(m_hWnd);
}

inline bool CTabCtrl::HighlightItem(int iItem, bool bHighlight /* = true */) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_HighlightItem(m_hWnd, iItem, bHighlight));
}

inline int CTabCtrl::HitTest(TCHITTESTINFO& hitTestInfo) const {
	AssertValidAsWindow();
	return TabCtrl_HitTest(m_hWnd, &hitTestInfo);
}

inline bool CTabCtrl::InsertItem(int iItem, const TCITEM& item) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, &item));
}

inline bool CTabCtrl::InsertItem(int iItem, const TCHAR* lpszItem) {
	AssertValidAsWindow();
	TCITEM	item;

	item.mask = TCIF_TEXT;
	item.pszText = const_cast<TCHAR*>(lpszItem);
	item.cchTextMax = std::_tcslen(lpszItem);
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, &item));
}

inline bool CTabCtrl::InsertItem(int iItem, const TCHAR* lpszItem, int iImage) {
	AssertValidAsWindow();
	TCITEM	item;

	item.mask = TCIF_IMAGE | TCIF_TEXT;
	item.pszText = const_cast<TCHAR*>(lpszItem);
	item.cchTextMax = std::_tcslen(lpszItem);
	item.iImage = iImage;
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, &item));
}

inline bool CTabCtrl::InsertItem(UINT nMask, int iItem, const TCHAR* lpszItem, int iImage, LPARAM lParam) {
	AssertValidAsWindow();
	TCITEM	item;

	item.mask = nMask;
	item.pszText = const_cast<TCHAR*>(lpszItem);
	item.cchTextMax = std::_tcslen(lpszItem);
	item.iImage = iImage;
	item.lParam = lParam;
	return toBoolean(TabCtrl_InsertItem(m_hWnd, iItem, &item));
}

inline void CTabCtrl::RemoveImage(int iImage) {
	AssertValidAsWindow();
	TabCtrl_RemoveImage(m_hWnd, iImage);
}

inline int CTabCtrl::SetCurSel(int iItem) {
	AssertValidAsWindow();
	return TabCtrl_SetCurSel(m_hWnd, iItem);
}

inline int CTabCtrl::SetCurFocus(int iItem) {
	AssertValidAsWindow();
	return TabCtrl_SetCurFocus(m_hWnd, iItem);
}

inline DWORD CTabCtrl::SetExtendedStyle(DWORD dwNewStyle, DWORD dwExMask /* = 0 */) {
	AssertValidAsWindow();
	return SendMessage(TCM_SETEXTENDEDSTYLE, dwExMask, dwNewStyle);
}

inline CImageList* CTabCtrl::SetImageList(const CImageList* pImageList) {
	AssertValidAsWindow();
	return CImageList::FromHandle(TabCtrl_SetImageList(m_hWnd, pImageList->GetSafeHandle()));
}

inline bool CTabCtrl::SetItem(int iItem, const TCITEM& tabCtrlItem) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_SetItem(m_hWnd, iItem, &tabCtrlItem));
}

inline bool CTabCtrl::SetItemExtra(int nBytes) {
	AssertValidAsWindow();
	return toBoolean(TabCtrl_SetItemExtra(m_hWnd, nBytes));
}

inline SIZE CTabCtrl::SetItemSize(const SIZE& size) {
	AssertValidAsWindow();
	const DWORD	dw = TabCtrl_SetItemSize(m_hWnd, size.cx, size.cy);
	SIZE		s = {LOWORD(dw), HIWORD(dw)};
	return s;
}

inline bool CTabCtrl::SetItemState(int iItem, DWORD dwMask, DWORD dwState) {
	AssertValidAsWindow();
	TCITEM	item;
	item.mask = TCIF_STATE;
	item.dwState = dwState;
	item.dwStateMask = dwMask;
	return SetItem(iItem, item);
}

inline int CTabCtrl::SetMinTabWidth(int cx) {
	AssertValidAsWindow();
	return TabCtrl_SetMinTabWidth(m_hWnd, cx);
}

inline void CTabCtrl::SetPadding(const SIZE& size) {
	AssertValidAsWindow();
	TabCtrl_SetPadding(m_hWnd, size.cx, size.cy);
}

inline void CTabCtrl::SetToolTips(HWND hwndTip) {
	AssertValidAsWindow();
	TabCtrl_SetToolTips(m_hWnd, hwndTip);
}

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _TAB_CTRL */

/* [EOF] */