// LinkLable.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _LINK_LABEL_H_
#define _LINK_LABEL_H_
#include "ToolTipCtrl.h"
#include <stdexcept>


// CLinkLabel class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

// O͐FX@\ǃhł[߂

class CLinkLabel : public CCustomControl<CLinkLabel>, public CNoncopyable {
	DEFINE_WINDOW_CLASS() {
		lpszName = _T("ManahLinkLabel");
		nStyle = CS_BYTEALIGNCLIENT | CS_BYTEALIGNWINDOW | CS_DBLCLKS;
		cursor = MAKEINTRESOURCE(32649);	// IDC_HAND
	}

	// RXgN^
public:
	CLinkLabel(const TCHAR* pszCaption = 0, const TCHAR* pszTipText = 0);
	virtual ~CLinkLabel();

	// \bh
public:
	bool			Create(HWND hwndParent, HINSTANCE hInstance, int id = 0);
	const TCHAR*	GetTipText() const;
	void			SetTipText(const TCHAR* lpszText);
private:
	HFONT	_GetFont();

	// bZ[Wnh
protected:
	virtual OVERRIDE_DISPATCH_EVENT(CLinkLabel);
	virtual void	OnKillFocus(HWND hwndNew);								// WM_KILLFOCUS
	virtual void	OnLButtonDown(UINT nFlags, POINT pt);					// WM_LBUTTONDOWN
	virtual void	OnLButtonUp(UINT nFlags, POINT pt);						// WM_LBUTTONUP
	virtual void	OnPaint(GDI::CPaintDC& dc);								// WM_PAINT
	virtual bool	OnSetCursor(HWND hWnd, UINT nHitTest, UINT message);	// WM_SETCURSOR
	virtual void	OnSetFocus(HWND hwndOld);								// WM_SETFOCUS
	virtual void	OnSetText(const TCHAR* lpszText);						// WM_SETTEXT

	// f[^o
private:
	TCHAR*	m_pszTipText;
};


// CLinkLabel class implementation
/////////////////////////////////////////////////////////////////////////////

inline CLinkLabel::CLinkLabel(const TCHAR* pszCaption /* = 0 */, const TCHAR* pszTipText /* = 0 */) : m_pszTipText(0) {
	if(pszCaption != 0)
		SetWindowText(pszCaption);
	if(pszTipText != 0)
		SetTipText(pszTipText);
}

inline CLinkLabel::~CLinkLabel() {
	delete[] m_pszTipText;
}

inline bool CLinkLabel::Create(HWND hwndParent, HINSTANCE hInstance, int id /* = 0 */) {
	AssertValid();
	assert(hwndParent == 0 || toBoolean(::IsWindow(hwndParent)));

	if(!CCustomControl<CLinkLabel>::Create(hwndParent, DefaultWindowRect(), 0, WS_CHILD | WS_TABSTOP | WS_VISIBLE))
		return false;
	if(id != 0)
#ifdef _WIN64
		SetWindowLongPtr(GWLP_ID, id);
#else
		SetWindowLong(GWL_ID, id);
#endif /* _WIN64 */
	return true;
}

inline LRESULT CLinkLabel::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValid();

	if(PreTranslateMessage(message, wParam, lParam))
		return false;

	switch(message) {
	case WM_GETDLGCODE:
		return DLGC_BUTTON | DLGC_UNDEFPUSHBUTTON;
	case WM_KEYDOWN:
		if(!toBoolean(GetStyle() & WS_DISABLED) && wParam == VK_RETURN)
#ifdef _WIN64
			GetParent().SendMessage(WM_COMMAND, GetWindowLongPtr(GWLP_ID), reinterpret_cast<LPARAM>(m_hWnd));
#else
			GetParent().SendMessage(WM_COMMAND, GetWindowLong(GWL_ID), reinterpret_cast<LPARAM>(m_hWnd));
#endif /* _WIN64 */
		break;
	case WM_SETTEXT:
		OnSetText(reinterpret_cast<TCHAR*>(lParam));
		break;
	}

	return CCustomControl<CLinkLabel>::DispatchEvent(message, wParam, lParam);
}

inline HFONT CLinkLabel::_GetFont() {
	LOGFONT	lf;
	::GetObject(GetParent().GetFont(), sizeof(LOGFONT), &lf);
	lf.lfUnderline = true;
	return ::CreateFontIndirect(&lf);	// may return null...
}

inline const TCHAR* CLinkLabel::GetTipText() const {
	AssertValid();
	return m_pszTipText;
}

inline void CLinkLabel::SetTipText(const TCHAR* lpszText) {
	AssertValid();
	assert(lpszText != 0);

	delete[] m_pszTipText;
	m_pszTipText = new TCHAR[std::_tcslen(lpszText) + 1];
	std::_tcscpy(m_pszTipText, lpszText);
}

inline void CLinkLabel::OnKillFocus(HWND hwndNew) {
	GDI::CClientDC	dc = GetDC();
	RECT			rect;

	GetClientRect(rect);
	::OffsetRect(&rect, -rect.left, -rect.top);
	dc.DrawFocusRect(rect);
}

inline void CLinkLabel::OnLButtonDown(UINT nFlags, POINT pt) {
	SetFocus();
}

inline void CLinkLabel::OnLButtonUp(UINT nFlags, POINT pt) {
	if(toBoolean(GetStyle() & WS_DISABLED))
		return;
	::SendMessage(GetParent(), WM_COMMAND,
#ifdef _WIN64
		GetWindowLongPtr(GWLP_ID),
#else
		GetWindowLong(GWL_ID),
#endif /* _WIN64 */
		reinterpret_cast<LPARAM>(m_hWnd));
}

inline void CLinkLabel::OnPaint(GDI::CPaintDC& dc) {
#ifndef COLOR_HOTLIGHT
	const int	COLOR_HOTLIGHT = 26;
#endif /* !COLOR_HOTLIGHT */
	const std::size_t	cch = GetWindowTextLength();

	if(cch == 0)
		return;

	TCHAR* const	psz = new TCHAR[cch + 1];
	HFONT			hOldFont = dc.SelectObject(_GetFont());
	RECT			rect;

	GetWindowText(psz, cch + 1);
	dc.SetTextColor(::GetSysColor(toBoolean(GetStyle() & WS_DISABLED) ? COLOR_GRAYTEXT : COLOR_HOTLIGHT));
	dc.SetBkMode(TRANSPARENT);

	GetWindowRect(rect);
	::OffsetRect(&rect, -rect.left, -rect.top);
	::InflateRect(&rect, -1, -1);

	dc.DrawText(psz, cch, rect, DT_LEFT | DT_TOP | DT_SINGLELINE);
	delete[] psz;
	::DeleteObject(dc.SelectObject(hOldFont));
}

inline bool CLinkLabel::OnSetCursor(HWND hWnd, UINT nHitTest, UINT message) {
	if(toBoolean(GetStyle() & WS_DISABLED)) {
		::SetCursor(::LoadCursor(0, IDC_ARROW));
		return true;
	} else
		return false;
}

inline void CLinkLabel::OnSetFocus(HWND hwndOld) {
	RECT	rect;

	GetClientRect(rect);
	::OffsetRect(&rect, -rect.left, -rect.top);
	GetDC().DrawFocusRect(rect);
}

inline void CLinkLabel::OnSetText(const TCHAR* lpszText) {
	GDI::CClientDC	dc = GetDC();
	RECT			rect = {0, 0, 0, 0};

	HFONT	hOldFont = dc.SelectObject(_GetFont());
	dc.DrawText(lpszText, -1, rect, DT_CALCRECT);
	::DeleteObject(dc.SelectObject(hOldFont));
	rect.right += 2;
	rect.bottom += 2;
	SetWindowPos(0, rect, SWP_NOMOVE | SWP_NOZORDER);
}

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _LINK_LABEL_H_ */

/* [EOF] */