// File.h
// (c) 2003-2005 exeal

#ifndef _FILE_H_
#define _FILE_H_
#include "Object.h"


namespace Manah {
namespace Windows {
namespace IO {

typedef ushort	FileOpenFlag;
const FileOpenFlag
	modeRead		= 0x0000,	// ǂݎp
	modeWrite		= 0x0001,	// ݐp
	modeReadWrite	= 0x0002,	// ǂݏ\
	shareCompat		= 0x0000,	// gp
	shareExclusive	= 0x0010,	// vZX̓ǂݏ
	shareDenyWrite	= 0x0020,	// vZX݂̏
	shareDenyRead	= 0x0030,	// vZX̓ǂݎ
	shareDenyNone	= 0x0040,	// vZXƋL
	modeNoInherit	= 0x0080,	// qvZXŃt@CpȂ
	modeCreate		= 0x1000,	// (t@CɂĂ) t@C쐬
	modeNoTruncate	= 0x2000;	// modeCreate Ƃ̑gݍ킹Ŋ̃t@CjȂ
//	typeText		= 0x4000,
//	typeBinary		= 0x8000

class CFileException : std::runtime_error {
	// RXgN^
public:
	explicit CFileException(const std::string& what_arg) : std::runtime_error(what_arg) {}
};


template<bool bNoThrow>
class CFile : public CSelfAssertable, public CNoncopyable {
	// RXgN^
public:
	CFile();
	CFile(HANDLE hFile);
	CFile(const TCHAR* pszFileName, UINT nFlags) throw(CFileException);
	virtual ~CFile();

	// \bh
public:
	/* \z */
	virtual void	Abort();
	virtual CFile*	Duplicate() const throw(CFileException);
	virtual bool	Open(const TCHAR* pszFileName, FileOpenFlag flags) throw(CFileException);
	virtual bool	Close() throw(CFileException);

	/* Xg[ */
	virtual bool	Read(void* lpBuffer, DWORD dwCount, DWORD* pdwRead) throw(CFileException);
	virtual bool	Write(const void* lpBuffer, DWORD dwCount) throw(CFileException);
	virtual bool	Flush() throw(CFileException);

	/* bN */
	virtual bool	LockRange(DWORD dwPos, DWORD dwCount) throw(CFileException);
	virtual bool	UnlockRange(DWORD dwPos, DWORD dwCount) throw(CFileException);

	/*  */
	virtual DWORD	GetPosition() const throw(CFileException);
	virtual DWORD	GetFileSize(DWORD* pdwFileSizeHigh) const throw(CFileException);
	virtual bool	GetFileTime(LPFILETIME lpCreationTime,
						LPFILETIME lpLastAccessTime, LPFILETIME lpLastWriteTime) const throw(CFileException);
	virtual DWORD	GetCompressedFileSize(DWORD* lpFileSizeHigh) const throw(CFileException);
	virtual bool	IsOpened() const;

	// f[^o
private:
	HANDLE	m_hFile;		// file handle
protected:
	TCHAR*	m_pszFileName;	// full path
	bool	m_bAutoClose;	// whether close file when deleted object
};


// Global elements
/////////////////////////////////////////////////////////////////////////////

inline std::string GenerateErrorMessage() {
	std::string	strMessage;
	void*		lpszMessage = 0;
	DWORD		n = ::GetLastError();

	::FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER
		| FORMAT_MESSAGE_FROM_SYSTEM
		| FORMAT_MESSAGE_IGNORE_INSERTS,
		0, n, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		reinterpret_cast<char*>(&lpszMessage), 0, 0);
	strMessage = static_cast<char*>(lpszMessage);
	::LocalFree(lpszMessage);

	return strMessage;
}


// CFile class implementation
/////////////////////////////////////////////////////////////////////////////

template<bool bNoThrow>
inline CFile<bNoThrow>::CFile() : m_hFile(0), m_pszFileName(0), m_bAutoClose(false) {
}

template<bool bNoThrow>
inline CFile<bNoThrow>::CFile(HANDLE hFile) : m_hFile(hFile), m_pszFileName(0), m_bAutoClose(false) {
}

template<bool bNoThrow>
inline CFile<bNoThrow>::CFile(const TCHAR* pszFileName, UINT nFlags) : m_hFile(0), m_pszFileName(0) {
	assert(pszFileName != 0);
	if(bNoThrow)
		Open(pszFileName, nFlags);
	else {
		try {
			Open(pszFileName, nFlags);
		} catch(CFileException& /* e */) {
			throw;
		}
	}
}

template<bool bNoThrow>
inline CFile<bNoThrow>::~CFile() {
	if(m_hFile != 0 && m_bAutoClose)
		Close();
}

template<bool bNoThrow>
inline void CFile<bNoThrow>::Abort() {
	AssertValid();

	if(m_hFile != 0) {
		::CloseHandle(m_hFile);
		m_hFile = 0;
	}
	delete[] m_pszFileName;
	m_pszFileName = 0;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::Close() {
	AssertValid();

	if(m_hFile != 0) {
		if(!toBoolean(::CloseHandle(m_hFile))) {
			if(bNoThrow)
				return false;
			throw CFileException(GenerateErrorMessage());
		}
		m_hFile = 0;
	}
	m_bAutoClose = false;
	delete[] m_pszFileName;
	m_pszFileName = 0;
	return true;
}

template<bool bNoThrow>
inline CFile<bNoThrow>* CFile<bNoThrow>::Duplicate() const {
	AssertValid();

	if(m_hFile == 0)
		return 0;

	CFile*	pFile = new CFile();
	HANDLE	hFile;
	if(!toBoolean(::DuplicateHandle(::GetCurrentProcess(), m_hFile,
			::GetCurrentProcess(), &hFile, 0, false, DUPLICATE_SAME_ACCESS))) {
		delete pFile;
		if(bNoThrow)
			return false;
		throw CFileException(GenerateErrorMessage());
	}
	pFile->m_hFile = hFile;
	assert(pFile->m_hFile != 0);
	pFile->m_bAutoClose = m_bAutoClose;

	return pFile;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::LockRange(DWORD dwPos, DWORD dwCount) {
	AssertValid();
	assert(IsOpened());

	if(!toBoolean(::LockFile(m_hFile, dwPos, 0, dwCount, 0))) {
		if(bNoThrow)
			return false;
		throw CFileException(GenerateErrorMessage());
	}
	return true;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::Flush() {
	AssertValid();
	assert(IsOpened());

	if(!toBoolean(::FlushFileBuffers(m_hFile))) {
		if(bNoThrow)
			return false;
		throw CFileException(GenerateErrorMessage());
	}
	return true;
}

template<bool bNoThrow>
inline DWORD CFile<bNoThrow>::GetFileSize(DWORD* pdwFileSizeHigh) const {
	AssertValid();
	assert(IsOpened());

	const DWORD	dwSize = ::GetFileSize(m_hFile, pdwFileSizeHigh);
	if(dwSize == static_cast<DWORD>(-1) && ::GetLastError() != NO_ERROR) {
		if(bNoThrow)
			return false;
		throw CFileException(GenerateErrorMessage());
	}

	return dwSize;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::GetFileTime(LPFILETIME lpCreationTime,
		LPFILETIME lpLastAccessTime, LPFILETIME lpLastWriteTime) const throw(CFileException) {
	AssertValid();
	assert(IsOpened());

	const bool	bSucceeded = toBoolean(::GetFileTime(m_hFile, lpCreationTime, lpLastAccessTime, lpLastWriteTime));
	if(!bSucceeded && ::GetLastError() != NO_ERROR && !bNoThrow)
		throw CFileException(GenerateErrorMessage());
	return bSucceeded;
}

template<bool bNoThrow>
inline DWORD CFile<bNoThrow>::GetCompressedFileSize(DWORD* pdwFileSizeHigh) const {
	AssertValid();
	assert(IsOpened());

	const DWORD	dwSize = ::GetCompressedFileSize(m_pszFileName, pdwFileSizeHigh);
	if(dwSize == static_cast<DWORD>(-1) && ::GetLastError() != NO_ERROR && !bNoThrow)
		throw CFileException(GenerateErrorMessage());
	return dwSize;
}

template<bool bNoThrow>
inline DWORD CFile<bNoThrow>::GetPosition() const {
	AssertValid();
	assert(IsOpened());

	const DWORD dwPos = ::SetFilePointer(m_hFile, 0, 0, FILE_CURRENT);
	if(dwPos == static_cast<DWORD>(-1) && !bNoThrow)
		throw CFileException(GenerateErrorMessage());
	return dwPos;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::IsOpened() const {
	AssertValid();
	return m_hFile != 0;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::Open(const TCHAR* pszFileName, FileOpenFlag flags) {
	AssertValid();
	assert(pszFileName != 0);

	if(IsOpened() /*|| (nFlags & typeText) || (nFlags & typeBinary)*/) {
		if(bNoThrow)
			return false;
		else
			throw CFileException("File is already opened.");
	}

	HANDLE	hFile;
	DWORD	dwAccess, dwShareMode, dwCreateFlag;

	switch(flags & 0x03) {
	case modeRead:		dwAccess = GENERIC_READ;					break;
	case modeWrite:		dwAccess = GENERIC_WRITE;					break;
	case modeReadWrite:	dwAccess = GENERIC_READ | GENERIC_WRITE;	break;
	default:			assert(false);
	}

	switch(flags & 0x70) {
	case shareCompat:
	case shareExclusive:	dwShareMode = 0;									break;
	case shareDenyWrite:	dwShareMode = FILE_SHARE_READ;						break;
	case shareDenyRead:		dwShareMode = FILE_SHARE_WRITE;						break;
	case shareDenyNone:		dwShareMode = FILE_SHARE_READ | FILE_SHARE_WRITE;	break;
	}

	if(toBoolean(flags & modeCreate))
		dwCreateFlag = toBoolean(flags & modeNoTruncate) ? OPEN_ALWAYS : CREATE_ALWAYS;
	else
		dwCreateFlag = OPEN_EXISTING;

	hFile = ::CreateFile(pszFileName, dwAccess,
		dwShareMode, 0, dwCreateFlag, FILE_ATTRIBUTE_NORMAL, 0);
	if(hFile == INVALID_HANDLE_VALUE) {
		if(bNoThrow)
			return false;
		else
			throw CFileException(GenerateErrorMessage());
	}

	m_hFile = hFile;
	m_pszFileName = new TCHAR[std::_tcslen(pszFileName) + 1];
	std::_tcscpy(m_pszFileName, pszFileName);
	m_bAutoClose = true;

	return true;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::Read(void* lpBuffer, DWORD dwCount, DWORD* pdwRead) {
	AssertValid();
	assert(IsOpened());
	assert(lpBuffer != 0);

	if(dwCount == 0) {
		if(pdwRead != 0)
			*pdwRead = 0;
		return true;
	}

	DWORD	dwRead;
	if(!toBoolean(::ReadFile(m_hFile, lpBuffer, dwCount, &dwRead, 0))) {
		if(bNoThrow)
			return false;
		throw CFileException(GenerateErrorMessage());
	}
	if(pdwRead != 0)
		*pdwRead = dwRead;
	return true;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::UnlockRange(DWORD dwPos, DWORD dwCount) {
	AssertValid();
	assert(IsOpened());

	if(!toBoolean(::UnlockFile(m_hFile, dwPos, 0, dwCount, 0))) {
		if(bNoThrow)
			return false;
		throw CFileException(GenerateErrorMessage());
	}
	return true;
}

template<bool bNoThrow>
inline bool CFile<bNoThrow>::Write(const void* lpBuffer, DWORD dwCount) {
	AssertValid();
	assert(IsOpened());
	assert(lpBuffer != 0);

	if(dwCount == 0)
		return true;

	DWORD dwWritten;
	if(!toBoolean(::WriteFile(m_hFile, lpBuffer, dwCount, &dwWritten, 0))) {
		if(bNoThrow)
			return false;
		throw CFileException(GenerateErrorMessage());
	}
	return true;
}

} /* namespace IO */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _FILE_H_ */

/* [EOF] */