/*
* An experimental driver for AKI-H8-USB on Linux.
* Copyright (C) 2004 Masakazu Noguchi
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
* This driver is besed upon Greg Kroah-Hartman's USB skeleton driver v0.7.
* The author thanks to the developers of the skeleton.
* 
* Although the users of AKI-H8-USB can customise the device firmware, this 
* driver assumes the "Sample" firmware provided with the kit. Basically this 
* module simply reads from and writes to the bulk_out(0x02) and bulk_in(0x81)
* endpoints respectively.
*
*/



#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/poll.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/fcntl.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/smp_lock.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/usb.h>

#ifdef CONFIG_USB_DEBUG
	static int debug = 1;
#else
	static int debug;
#endif


#undef dbg
#define dbg(format, arg...) do { if (debug) printk(KERN_DEBUG __FILE__ ": " format "\n" , ## arg); } while (0)


#define DRIVER_VERSION "v0.1"
#define DRIVER_AUTHOR "Masakazu Noguchi"
#define DRIVER_DESC "The AKI-H8-USB Driver"

MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug, "Debug enabled or not");


#define USB_SKEL_VENDOR_ID	0xfffe
#define USB_SKEL_PRODUCT_ID	0x0010

/* register the IDs to the device table */
static struct usb_device_id vtable [] = {
	{ USB_DEVICE(USB_SKEL_VENDOR_ID, USB_SKEL_PRODUCT_ID) },
	{ }					/* Terminating entry */
};

MODULE_DEVICE_TABLE (usb, vtable);


/* define the bottom of device minor numbers */
#define MINOR_BASE	192

/* we can have up to this number of device plugged in at once */
#define MAX_DEVICES		16

struct akih8_prms {
	struct usb_device *	udev;			/* save off the usb device pointer */
	struct usb_interface *	interface;		/* the interface for this device */
	devfs_handle_t		devfs;			/* devfs device node */
	unsigned char		minor;			/* the starting minor number for this device */
	unsigned char		num_ports;		/* the number of ports this device has */
	char			num_interrupt_in;	/* number of interrupt in endpoints we have */
	char			num_bulk_in;		/* number of bulk in endpoints we have */
	char			num_bulk_out;		/* number of bulk out endpoints we have */

	unsigned char *		bulk_in_buffer;		/* the buffer to receive data */
	int			bulk_in_size;		/* the size of the receive buffer */
	__u8			bulk_in_endpointAddr;	/* the address of the bulk in endpoint */

	unsigned char *		bulk_out_buffer;	/* the buffer to send data */
	int			bulk_out_size;		/* the size of the send buffer */
	struct urb *		write_urb;		/* the urb used to send data */
	__u8			bulk_out_endpointAddr;	/* the address of the bulk out endpoint */

	struct tq_struct	tqueue;			/* task queue for line discipline waking up */
	int			open_count;		/* number of times this port has been opened */
	struct semaphore	sem;			/* locks this structure */
};


/* the global usb devfs handle */
extern devfs_handle_t usb_devfs_handle;


static ssize_t akih8_read	(struct file *file, char *buffer, size_t count, loff_t *ppos);
static ssize_t akih8_write	(struct file *file, const char *buffer, size_t count, loff_t *ppos);
static int akih8_ioctl		(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg);
static int akih8_open		(struct inode *inode, struct file *file);
static int akih8_release		(struct inode *inode, struct file *file);

static void * akih8_probe	(struct usb_device *dev, unsigned int ifnum, const struct usb_device_id *id);
static void akih8_disconnect	(struct usb_device *dev, void *ptr);

static void akih8_bulk_callback	(struct urb *urb);
static inline void akih8_delete (struct akih8_prms *dev);


/* array of pointers to our devices that are currently connected */
static struct akih8_prms		*minor_table[MAX_DEVICES];

/* lock to protect the minor_table structure */
static DECLARE_MUTEX (minor_table_mutex);

static struct file_operations akih8_fops = {
	owner:		THIS_MODULE,
	read:		akih8_read,
	write:		akih8_write,
	ioctl:		akih8_ioctl,
	open:		akih8_open,
	release:	akih8_release,
};


static struct usb_driver akih8_driver = {
	name:		"aki-h8-usb",
	probe:		akih8_probe,
	disconnect:	akih8_disconnect,
	fops:		&akih8_fops,
	minor:		MINOR_BASE,
	id_table:	vtable,
};


static inline void akih8_debug_data (const char *function, int size, const unsigned char *data)
{
	int i;

	if (!debug)
		return;

	printk (KERN_DEBUG __FILE__": %s - length = %d, data = ",
		function, size);
	for (i = 0; i < size; ++i) {
		printk ("%.2x ", data[i]);
	}
	printk ("\n");
}






static void akih8_bulk_callback (struct urb *urb)
{
	struct akih8_prms *dev = (struct akih8_prms *)urb->context;

	dbg(__FUNCTION__ " - minor %d", dev->minor);

	if ((urb->status != -ENOENT) &&
	    (urb->status != -ECONNRESET)) {
		dbg(__FUNCTION__ " - nonzero write bulk status received: %d",
		    urb->status);
		return;
	}

	return;
}




static int akih8_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct akih8_prms *dev;

	dev = (struct akih8_prms *)file->private_data;

	down (&dev->sem);

	if (dev->udev == NULL) {
		up (&dev->sem);
		return -ENODEV;
	}

	dbg(__FUNCTION__ " - minor %d, cmd 0x%.4x, arg %ld",
	    dev->minor, cmd, arg);


	up (&dev->sem);

	return -ENOTTY;
}





static ssize_t akih8_write (struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
	struct akih8_prms *dev;
	ssize_t bytes_written = 0;
	int retval = 0;

	dev = (struct akih8_prms *)file->private_data;

	dbg(__FUNCTION__ " - minor %d, count = %d", dev->minor, count);

	down (&dev->sem);

	if (dev->udev == NULL) {
		retval = -ENODEV;
		goto exit;
	}

	if (count == 0) {
		dbg(__FUNCTION__ " - write request of 0 bytes");
		goto exit;
	}

	/* see if we are already in the middle of a write */
	if (dev->write_urb->status == -EINPROGRESS) {
		dbg (__FUNCTION__ " - already writing");
		goto exit;
	}

	/* we can only write as much as 1 urb will hold */
	bytes_written = (count > dev->bulk_out_size) ?
				dev->bulk_out_size : count;

	if (copy_from_user(dev->write_urb->transfer_buffer, buffer, 
			   bytes_written)) {
		retval = -EFAULT;
		goto exit;
	}

	akih8_debug_data (__FUNCTION__, bytes_written,
			     dev->write_urb->transfer_buffer);

	FILL_BULK_URB(dev->write_urb, dev->udev,
		      usb_sndbulkpipe(dev->udev, dev->bulk_out_endpointAddr),
		      dev->write_urb->transfer_buffer, bytes_written,
		      akih8_bulk_callback, dev);

	retval = usb_submit_urb(dev->write_urb);
	if (retval) {
		err(__FUNCTION__ " - failed submitting write urb, error %d",
		    retval);
	} else {
		retval = bytes_written;
	}

exit:
	up (&dev->sem);

	return retval;
}






static ssize_t akih8_read (struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	struct akih8_prms *dev;
	int retval = 0;

	dev = (struct akih8_prms *)file->private_data;

	dbg(__FUNCTION__ " - minor %d, count = %d", dev->minor, count);

	down (&dev->sem);

	/* verify that the device wasn't unplugged */
	if (dev->udev == NULL) {
		up (&dev->sem);
		return -ENODEV;
	}

	retval = usb_bulk_msg (dev->udev,
			       usb_rcvbulkpipe (dev->udev,
						dev->bulk_in_endpointAddr),
			       dev->bulk_in_buffer, dev->bulk_in_size,
			       &count, HZ*10);

	if (!retval) {
		if (copy_to_user (buffer, dev->bulk_in_buffer, count))
			retval = -EFAULT;
		else
			retval = count;
	}

	up (&dev->sem);
	return retval;
}







static int akih8_release (struct inode *inode, struct file *file)
{
	struct akih8_prms *dev;
	int retval = 0;

	dev = (struct akih8_prms *)file->private_data;
	if (dev == NULL) {
		dbg (__FUNCTION__ " - object is NULL");
		return -ENODEV;
	}

	dbg(__FUNCTION__ " - minor %d", dev->minor);

	down (&minor_table_mutex);

	down (&dev->sem);

	if (dev->open_count <= 0) {
		dbg (__FUNCTION__ " - device not opened");
		retval = -ENODEV;
		goto exit_not_opened;
	}

	if (dev->udev == NULL) {
		up (&dev->sem);
		akih8_delete (dev);
		up (&minor_table_mutex);
		MOD_DEC_USE_COUNT;
		return 0;
	}

	--dev->open_count;
	if (dev->open_count <= 0) {
		/* shutdown any bulk writes that might be going on */
		usb_unlink_urb (dev->write_urb);
		dev->open_count = 0;
	}

	MOD_DEC_USE_COUNT;

exit_not_opened:
	up (&dev->sem);
	up (&minor_table_mutex);

	return retval;
}


static int akih8_open (struct inode *inode, struct file *file)
{
	struct akih8_prms *dev = NULL;
	int subminor;
	int retval = 0;

	dbg(__FUNCTION__);

	subminor = MINOR (inode->i_rdev) - MINOR_BASE;
	if ((subminor < 0) ||
	    (subminor >= MAX_DEVICES)) {
		return -ENODEV;
	}
	MOD_INC_USE_COUNT;

	down (&minor_table_mutex);
	dev = minor_table[subminor];
	if (dev == NULL) {
		up (&minor_table_mutex);
		MOD_DEC_USE_COUNT;
		return -ENODEV;
	}

	down (&dev->sem);

	up (&minor_table_mutex);

	++dev->open_count;

	/* save our object in the file's private structure */
	file->private_data = dev;

	up (&dev->sem);

	return retval;
}



static inline void akih8_delete (struct akih8_prms *dev)
{
	minor_table[dev->minor] = NULL;
	if (dev->bulk_in_buffer != NULL)
		kfree (dev->bulk_in_buffer);
	if (dev->bulk_out_buffer != NULL)
		kfree (dev->bulk_out_buffer);
	if (dev->write_urb != NULL)
		usb_free_urb (dev->write_urb);
	kfree (dev);
}



static void * akih8_probe(struct usb_device *udev, unsigned int ifnum, const struct usb_device_id *id)
{
	struct akih8_prms *dev = NULL;
	struct usb_interface *interface;
	struct usb_interface_descriptor *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	int minor;
	int buffer_size;
	int i;
	char name[10];

	if ((udev->descriptor.idVendor != USB_SKEL_VENDOR_ID) ||
	    (udev->descriptor.idProduct != USB_SKEL_PRODUCT_ID)) {
		printk("Vendor ID:%x\n", udev->descriptor.idVendor );
		printk("Product ID:%x\n", udev->descriptor.idProduct );
		printk("bLength:%x\n", udev->descriptor.bLength );
		printk("bDescriptorType:%x\n", udev->descriptor.bDescriptorType );
		printk("ID mismatch\n");
		return NULL;
	}
	/* select a "subminor" number (part of a minor number) */
	down (&minor_table_mutex);
	for (minor = 0; minor < MAX_DEVICES; ++minor) {
		if (minor_table[minor] == NULL)
			break;
	}
	if (minor >= MAX_DEVICES) {
		info ("Too many devices plugged in, can not handle this device.");
		goto exit;
	}



	dev = kmalloc (sizeof(struct akih8_prms), GFP_KERNEL);
	if (dev == NULL) {
		err ("Out of memory");
		goto exit;
	}
	memset (dev, 0x00, sizeof (*dev));
	minor_table[minor] = dev;

	interface = &udev->actconfig->interface[ifnum];

	init_MUTEX (&dev->sem);
	dev->udev = udev;
	dev->interface = interface;
	dev->minor = minor;

	printk("probe endpoints\n");
	iface_desc = &interface->altsetting[0];
	for (i = 0; i < iface_desc->bNumEndpoints; ++i) {
		endpoint = &iface_desc->endpoint[i];

		if ((endpoint->bEndpointAddress == 0x81) &&
		    ((endpoint->bmAttributes & 3) == 0x02)) {
			/* we found a bulk in endpoint */
			buffer_size = endpoint->wMaxPacketSize;
			dev->bulk_in_size = buffer_size;
			dev->bulk_in_endpointAddr = endpoint->bEndpointAddress;
			dev->bulk_in_buffer = kmalloc (buffer_size, GFP_KERNEL);
			printk("bulk in endpoint:%x\n", dev->bulk_in_endpointAddr);
			if (!dev->bulk_in_buffer) {
				err("Couldn't allocate bulk_in_buffer");
				goto error;
			}
		}

		if (((endpoint->bEndpointAddress == 0x02) ) &&
		    ((endpoint->bmAttributes & 3) == 0x02)) {
			/* we found a bulk out endpoint */
			dev->write_urb = usb_alloc_urb(0);
			if (!dev->write_urb) {
				err("No free urbs available");
				goto error;
			}
			buffer_size = endpoint->wMaxPacketSize;
			dev->bulk_out_size = buffer_size;
			dev->bulk_out_endpointAddr = endpoint->bEndpointAddress;
			dev->bulk_out_buffer = kmalloc (buffer_size, GFP_KERNEL);
			printk("bulk out endpoint:%x\n", dev->bulk_out_endpointAddr);
			if (!dev->bulk_out_buffer) {
				err("Couldn't allocate bulk_out_buffer");
				goto error;
			}
			FILL_BULK_URB(dev->write_urb, udev,
				      usb_sndbulkpipe(udev,
						      endpoint->bEndpointAddress),
				      dev->bulk_out_buffer, buffer_size,
				      akih8_bulk_callback, dev);
		}
	}


	/* initialize the devfs node for this device and register it */
	sprintf(name, "aki-h8:%d", dev->minor);

	dev->devfs = devfs_register (usb_devfs_handle, name,
				     DEVFS_FL_DEFAULT, USB_MAJOR,
				     MINOR_BASE + dev->minor,
				     S_IFCHR | S_IRUSR | S_IWUSR |
				     S_IRGRP | S_IWGRP | S_IROTH,
				     &akih8_fops, NULL);

	/* let the user know what node this device is now attached to */
	info ("The AKI-H8-USB device now attached to USB:%d ", dev->minor);
	goto exit;

error:
	akih8_delete (dev);
	dev = NULL;

exit:
	up (&minor_table_mutex);
	return dev;
}





static void akih8_disconnect(struct usb_device *udev, void *ptr)
{
	struct akih8_prms *dev;
	int minor;

	dev = (struct akih8_prms *)ptr;

	down (&minor_table_mutex);
	down (&dev->sem);
		
	minor = dev->minor;

	/* remove our devfs node */
	devfs_unregister(dev->devfs);

	/* if the device is not opened, then we clean up right now */
	if (!dev->open_count) {
		up (&dev->sem);
		akih8_delete (dev);
	} else {
		dev->udev = NULL;
		up (&dev->sem);
	}

	info("AKI-H8-USB #%d now disconnected", minor);
	up (&minor_table_mutex);
}



static int __init akih8_init(void)
{
	int result;

	/* register this driver with the USB subsystem */
	result = usb_register(&akih8_driver);
	if (result < 0) {
		err("usb_register failed for the "__FILE__" driver. Error number %d",
		    result);
		return -1;
	}

	info(DRIVER_DESC " " DRIVER_VERSION);
	return 0;
}


static void __exit akih8_exit(void)
{
	/* deregister this driver with the USB subsystem */
	usb_deregister(&akih8_driver);
}


module_init (akih8_init);
module_exit (akih8_exit);



MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");

