/* -*- c++ -*- */
#ifndef OSIXAKA2_GENERATOR_H__
#define OSIXAKA2_GENERATOR_H__

#include <osixaka2/platform.h>
#include <akaxiso2/akaxiso2.h>
#include <osixaka2/userdefs.h>
#include <osixaka2/processor.h>
#include <osixaka2/cpp_namer.h>
#include <fstream>

namespace osx {

  class output_file {
    std::string filename_;
    std::string outputdir_;
    std::string basedir_;
    static std::string escape_filename(const std::string &fn);
  public:
    output_file(const std::string &outputdir, const std::string &basedir); 
    void open(const std::string &filename);
    void close();
    void write_include(const std::string &include_name);
    void write_system_include(const std::string &include_name);
    void write_header();
    void write_footer();
    void newline();
    std::ofstream ostm_;
    const std::string &get_filename() const { return filename_; }
    void write_custom_includes(const includes &incs);
  };

  class generator_base {
  protected:
    registry &registry_;
    registry_unit *regunit_;
    cpp_namer namer_;
    aka::id_type target_ns_id_;
    std::ostream &ostm_;
    bool verbose_;
  public:
    generator_base(registry &reg, std::ostream &ostm, bool verbose) 
      : registry_(reg), regunit_(0), namer_(reg), target_ns_id_(aka::unregistered_token),
	ostm_(ostm), verbose_(verbose) { }

    std::string classname(const type_ref &type) const;
    std::string classname(const user_class &us) const;
    std::string classtype(const user_class &us) const;

    std::string leafname(const type_ref &type) const;
    std::string leafname(const user_class &us) const;

    std::string leaftype(const user_class &us) const;

    std::string membername(const aka::qname &name) const;
    std::string membername(const std::string &name) const;
    std::string get_prefix(aka::id_type nsid) const;
    void set_unit(const registry_unit &unit);
  };

  class generator : public generator_base {

    mutable output_file &elm_;
    mutable output_file &xiso_;
    mutable output_file &ximpl_;

    bool use_soft_array_;

    const_user_classes resolved_;
    const_user_classes generated_;

    bool is_class_resolved(const type_ref &type) const;
    void class_resolved(const user_class &us);
    bool is_class_generated(const user_class &us) const;
    void class_generated(const user_class &us);

  
    qname_set declared_leafs_;
    bool is_leaf_declared(const type_ref &type) const;
    void leaf_declared(const user_class &us);
  
    qname_set soft_resolving_;
    bool soft_resolve(const user_class &to_resolve, 
		      const_user_classes &to_generate, 
		      const_user_classes &forwarded);
    bool hard_resolve(const user_class &to_resolve, 
		      const_user_classes &to_generate, 
		      const_user_classes &processing);
  
    bool resolve_simpleType(const user_class &us, 
			    const_user_classes &simpleTypes,
			    bool generate_source);

    bool resolve_user_class(const user_class &to_resolve, 
			    const_user_classes &user_classes,
			    bool force_hard_resolve,
			    bool generate_source);

    void resolve_one_unit(const_user_classes &simpleTypes, 
			  const_user_classes &user_classes, 
			  bool generate);
  
    void generate_sequence(const user_class &def);
    void generate_choice(const user_class &def);
    void generate_all(const user_class &def);
    void generate_array(const user_class &def);
    void generate_simplecontent(const user_class &def);
    void generate_simpletype(const user_class &def);

    void write_ptrmember_forwarded_decl(const element_types &etypes) const; 

    void write_leaf_decl(const user_class &def, const std::string &leaftmp) const;
    void write_member_decls(const element_types &etypes) const;
    void write_attribute_decls(const attribute_defs &attrs) const;
    void write_member_leaves(const user_class &us) const;
    void write_item_leaves(const user_class &us) const;
    void write_attribute_leaves(const user_class &us) const;
    void write_occurrence_declaration(const user_class &us) const;

    void write_xmltype(const user_class &us) const;

    void begin_model_method(const user_class &us) const;
    void end_model_method() const;

    std::string get_ns_list(const xs::namespaceList &nslist) const;

    void begin_namespace();
    void end_namespace();

    void show_done(const user_class &def);
    void show_not_resolved(const user_class &def);

  public:
    generator(registry &reg,
	      output_file &elm, output_file &xiso, output_file &ximpl,
	      bool use_soft_array,
	      std::ostream &ostm, bool verbose) 
      : generator_base(reg, ostm, verbose), elm_(elm), xiso_(xiso), ximpl_(ximpl), 
	use_soft_array_(use_soft_array) {}

    void prepare();
    void resolve(const registry_unit &unit, bool generate);

    void write_include(const std::string &basename) const;
    void write_custom_includes(const registry_unit &unit) const;
    void open(const std::string &basename);
    void close();
  };

  class func_generator : public generator_base {
    mutable output_file &header_;
    mutable output_file &impl_;
    std::string instantiate_xiso_name_;
    void begin_instantiate_xiso() const;
    void end_instantiate_xiso() const;
    void generate_instantiate_xiso();
    void generate_serialize_methods();
  public:
    func_generator(registry &reg, output_file &header, output_file &impl, 
		   const std::string &instantiate_xiso_name, 
		   std::ostream &ostm, bool verbose) 
      : generator_base(reg, ostm, verbose), 
	header_(header), impl_(impl), instantiate_xiso_name_(instantiate_xiso_name) {}
    void write_includes();
    void generate();
    void open(const std::string &basename);
    void close();
  };
}

#endif
