#include "akaxisonizer.h"
#include <iostream>
#include <ctype.h>
#include <akaxiso2/util/string_funcs.h>

using namespace osx;


bool akaxisonizer::array_required(const user_class *def) const {
  
  switch (def->id_) {
  case sequence_id:
  case all_id:
  case simpletype_id:
  case simplecontent_id:
    return (def->get_occurrence().minOccurs_ != 1) || (def->get_occurrence().maxOccurs_ != 1);
  case choice_id:
  case array_id:
    break;
  }
  return false;
}


bool akaxisonizer::array_required(const element_type &etype) const {
  assert(!etype.type_.empty());
  const imported_type *imported = etype.type_.imported_;
  if (imported != 0)
    return (etype.get_occurrence().minOccurs_ != 1) || 
      (etype.get_occurrence().maxOccurs_ != 1);
  
  const user_class *def = etype.type_.us_;
  
  switch (def->id_) {
  case sequence_id:
  case all_id:
  case simpletype_id:
  case simplecontent_id:
    return (etype.get_occurrence().minOccurs_ != 1) || (etype.get_occurrence().maxOccurs_ != 1);
  case choice_id:
  case array_id:
    break;
  }
  return false;
}

bool akaxisonizer::is_array(const element_type &etype) const {
  return etype.get_occurrence().is_array();
}

bool akaxisonizer::occ_required(const aka::occurrence &occ) const {
  return occ.is_array();
}

void akaxisonizer::dereference() {
  dereference_elements(regunit_.elements_);
  dereference_user_classes(regunit_.complexTypes_);
  dereference_user_classes(regunit_.simpleTypes_);
  dereference_user_classes(regunit_.groups_);
  dereference_user_classes(regunit_.local_complexTypes_);
  dereference_user_classes(regunit_.local_simpleTypes_);
  dereference_user_classes(regunit_.local_classes_);
}

void akaxisonizer::dereference_user_classes(user_classes &uss) {
  for (user_classes::iterator it = uss.begin(); it != uss.end(); ++it) {
    user_class *us = it->second;
    for (element_types::iterator eit = us->etypes_.begin();
         eit != us->etypes_.end(); ++eit) {
      if (eit->type_.is_reference())
        eit->type_ = registry_.resolve_type_ref(eit->type_);
    }
    for (attribute_types::iterator ait = us->adefs_.attributes_.begin();
         ait != us->adefs_.attributes_.end(); ++ait) {
      if (ait->type_.is_reference())
        ait->type_ = registry_.resolve_type_ref(ait->type_);
    }
  }
}

void akaxisonizer::dereference_elements(element_defs &edefs) {

  element_defs::iterator it;
  for (it = regunit_.elements_.begin();
       it != regunit_.elements_.end(); ++it) {
    element_def &edef = it->second;
    aka::qname name = it->first;

    if (edef.type_.is_imported())
      continue;
    user_class *localCt = regunit_.local_complexTypes_.get(name);
    user_class *localSt = regunit_.local_simpleTypes_.get(name);
    if (localCt != 0) {
      regunit_.local_complexTypes_.erase(name);
      regunit_.complexTypes_.insert(name, localCt);
    }
    else if (localSt != 0) {
      regunit_.local_simpleTypes_.erase(name);
      regunit_.simpleTypes_.insert(name, localSt);
    }
  }

  for (it = edefs.begin(); it != edefs.end(); ++it) {
    if (it->second.type_.is_reference())
      it->second.type_ = registry_.resolve_type_ref(it->second.type_);
  }
}

void akaxisonizer::add_missings() {
  create_arrays(regunit_.complexTypes_, regunit_.complexTypes_);
  create_arrays(regunit_.groups_, regunit_.groups_);
  create_arrays(regunit_.simpleTypes_, regunit_.simpleTypes_);
  add_missings_for_classes(regunit_.complexTypes_);
  add_missings_for_classes(regunit_.groups_);
  add_missings_for_classes(regunit_.local_complexTypes_);
  add_missings_for_classes(regunit_.local_classes_);
}

void akaxisonizer::add_missings_for_classes(user_classes &uss) {
  for (user_classes::iterator it = uss.begin(); it != uss.end(); ++it) {
    add_missings_for_class(it->second);
  }
}  

// Add array definitions, and copy def->occ_ to element_type::occ_.
void akaxisonizer::add_missings_for_class(user_class *def) {
  
  if (def->id_ == array_id)
    return;
  
  // Confirm occurrence, and add array decls if required.
  for (element_types::iterator it = def->etypes_.begin();
       it != def->etypes_.end(); ++it) {
    
    // array for anyType will be inserted in replace_any().
    if (registry_.is_xs_any(it->type_.get_name())) 
      continue;
    
    // imported simpleType do not require array insertions.
    if (it->type_.is_imported())
      continue;
    
    const user_class *childdef = it->type_.us_;
    if (childdef == 0)
      continue;
    
    // sequence must not have holder.
    assert(!(array_required(*it) && 
             array_required(childdef) && 
             (childdef->id_ == sequence_id))
           );
    
    if (childdef->id_ == array_id) {
      if (!array_required(*it) && occ_required(childdef->get_occurrence())) {
        it->set_occurrence(childdef->get_occurrence());
      }
      continue;
    }
    if (childdef->id_ == choice_id) {
      // Add missings for choice.
      // move choice@minOccurs/maxOccurs to element_type.
      
      // both element and xs:choice are not single element.
      if (occ_required(it->get_occurrence()) && occ_required(childdef->get_occurrence())) {
        it->type_ = get_array(type_ref(childdef));
      }
      else if (!occ_required(it->get_occurrence()) && occ_required(childdef->get_occurrence())){
        it->set_occurrence(childdef->get_occurrence());
      }
      else if (occ_required(it->get_occurrence()) && occ_required(childdef->get_occurrence())) {
        // Nothing to do.
      }
    }
    else if (array_required(*it) && !array_required(childdef)) {
      it->type_ = get_array(type_ref(childdef));
      assert(!it->type_.empty());
    }
    else if (!array_required(*it) && array_required(childdef)) {
      it->type_ = get_array(type_ref(childdef));
      it->set_occurrence(childdef->get_occurrence());
      assert(!it->type_.empty());
    }
    else if (array_required(*it) && array_required(childdef) 
             && (childdef->id_ == all_id)) {
      assert(!"not implemented.");
    }
    else if (array_required(*it) && array_required(childdef) 
             && (childdef->id_ == sequence_id)) {
      assert(!"not implemented.");
    }
  }
}


void akaxisonizer::optimize() {
  if (verbose_)
    std::cout << "Optimization" << std::flush;

  bool optimizing;

  do {
    if (verbose_)
      std::cout << "." << std::flush;
    optimizing = optimize_classes(regunit_.complexTypes_);
    optimizing |= optimize_classes(regunit_.groups_);
    optimizing |= optimize_classes(regunit_.local_complexTypes_);
    optimizing |= optimize_classes(regunit_.local_classes_);
  } while (optimizing);

  if (verbose_)
    std::cout << "done." << std::endl;
}

bool akaxisonizer::optimize_classes(user_classes &uss) {
  bool optimizing = false;
  for (user_classes::iterator it = uss.begin(); it != uss.end(); ++it)
    optimizing |= simplify_class(it->second);    
  return optimizing;
}


bool akaxisonizer::simplify_class(user_class *def) {

  if ((def->id_ == sequence_id)
      && (def->id_ == choice_id)
      && (def->id_ == all_id)) {
    if (def->etypes_.empty()) // Nothing to optimize.
      return false;
  }

  if (verbose_)
    ostm_ << "simplify " << def->get_name().qualified() << "..." << std::flush;

  bool optimized = false;
  if (def->etypes_.size() == 1)
    optimized |= simplify_one_element_type(def);
  optimized |= simplify_children(def);

  if (verbose_) {
    if (optimized)
      ostm_ << "simplified." << std::endl;
    else
      ostm_ << "no change." << std::endl;
  }  

  return optimized;
}

bool akaxisonizer::simplify_children(user_class *def) {
  
  bool optimized = false;
  element_types::iterator it = def->etypes_.begin();

  while (it != def->etypes_.end()) {

    bool ignore = !regunit_.type_exists(it->type_) /* out of the present namespace. */
      || registry_.is_xs_any(it->type_.get_name()) /* xs:any is ignored. */
      || it->type_.is_imported() /* imported type, therefore could not be optimized. */
      || !regunit_.erasable(it->type_); /* not erasable (cf. topLevel types), */
                                        /* optimization should not be applied. */
    if (ignore) {
      ++it;
      continue;
    }

    const user_class *child = it->type_.us_;
    assert(child != 0);
    
    /** 
     * simplify array to ptrmember.
     * If occurrence of element is (0, 1), and child is array,
     * convert array to ptrmember.
     */
    if (((def->id_ == sequence_id) || (def->id_ == all_id)) && is_array(*it)) {
      if ((child->id_ == array_id)
          && (it->get_occurrence().minOccurs_ == 0) 
          && (it->get_occurrence().maxOccurs_ == 1)) {
        it->type_ = child->get_value_type();
        optimized = true;
	++it;
	continue;
      }
    }

    /**
     * sequence/array/sequence(one-element).
     * If child is array, and child array item is sequence,
     * and sequence has just one child element without attributes,
     * Then replace type to array of child element.
     * @test : XMLSchema.xsd and test6.xsd.
     */
    if ((def->id_ == sequence_id)
        && it->is_group_ // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
        && is_array(*it)
        && (child->id_ == array_id)) {
      const user_class *gcdef = child->get_value_type().us_;
      if ((gcdef != 0) && (gcdef->id_ == sequence_id)) {
        if ((gcdef->etypes_.size() == 1)
            && !gcdef->adefs_.has_attributes()
            && !is_array(gcdef->etypes_.front())) {
          const element_type &gc_etype = gcdef->etypes_.front();
          it->type_ = get_array(gc_etype.type_);
          if (it->name_ == aka::qname("*"))
            it->name_ = gc_etype.name_;
          it->is_group_ = gc_etype.is_group_;
          it->namespace_list_ = gc_etype.namespace_list_;
          optimized = true;
	  ++it;
	  continue;
        }
      }
    }
    
    /**
     * optimization for choice (minOccurs=1, maxOccurs=1) referred by array.
     * if def is sequence, and child is array,
     * and grand-child is choice, then array of choice is simplified...????
     * see osx.xsd for types declarations.
     */
    if ((def->id_ == sequence_id) && 
        (child->id_ == array_id)) {
      const type_ref &vn = child->get_value_type();
      const user_class *gcdef = vn.us_;
      if ((gcdef != 0) && 
          (gcdef->id_ == choice_id)) {
        /* add_missings is completed, therefore we will ignore gcdef->occ_. */
        assert(!gcdef->adefs_.has_attributes());
        it->type_ = vn;
        optimized = true;
	++it;
	continue;
      }
    }

    // one item choice optimization.
    if ((def->id_ == sequence_id)
        && it->is_group_ // !!!!!!!!!!!!!!!!!!!!
        && !child->adefs_.has_attributes()
        && (child->id_ == choice_id)
        && (child->etypes_.size() == 1)) {
      const element_type &etype = child->etypes_.front();
      
      if (!array_required(*it) || !array_required(etype)) {
        optimized = true;       
        if (is_array(*it) && !is_array(etype))
          it->type_ = get_array(etype.type_);
        else if (!is_array(*it) && is_array(etype)) {
          it->type_ = etype.type_;
          it->set_occurrence(etype.get_occurrence());
        }
        else if (!is_array(*it) && !is_array(etype)) {
          it->type_ = etype.type_;
        }
        else {
          optimized = false;
        }

        if (optimized) {
          it->name_ = etype.name_;
          it->is_group_ = etype.is_group_;
	  ++it;
	  continue;
        }
      }
    }


    // If def is sequence, and member is not array, 
    /// and the child is a sequence, and has just one element without attributes. 
    if ((def->id_ == sequence_id)
        && !is_array(*it)
        && (child->id_ == sequence_id)
        && (child->etypes_.size() == 1)
        && (!child->adefs_.has_attributes())) { 
      const element_type &gc_etype = child->etypes_.front();
      const user_class *gcdef = gc_etype.type_.us_;
      if ((gcdef != 0) && (gcdef->id_ == array_id)) {
        if (gc_etype.is_group_) {
          it->type_ = gc_etype.type_;
          it->set_occurrence(gc_etype.get_occurrence());
          optimized = true;
	  ++it;
	  continue;
        }
      }
    }
    
    // If element is group, and child is a sequence, and does not have any attributes,
    // move child elements to parent.
    if ((def->id_ == sequence_id)
        && it->is_group_
        && !is_array(*it)
        && (child->id_ == sequence_id)
        && !child->adefs_.has_attributes()) {
      def->etypes_.insert(it, child->etypes_.begin(), child->etypes_.end());
      element_types::iterator saved = it;
      ++it;
      def->etypes_.erase(saved);
      optimized = true;
      continue;
    }

    /* optimization is not applied, then go to the next. */
    ++it;
  }
  return optimized;
}  

bool akaxisonizer::simplify_one_element_type(user_class *def) {

  /** One child element optimization. */

  element_type &etype = def->etypes_.front();

  // Don't optimize toplevel types.
  if (!regunit_.erasable(etype.type_))
    return false;

  // If child element has occurrence, return false, becuase could not pass occurrence to 
  // parent element_type.
  if (occ_required(etype.get_occurrence()))
    return false;

  // restrict optimization application to it's nemspace-scope.
  if (!regunit_.type_exists(etype.type_))
    return false;

  const user_class *child = etype.type_.us_;
  assert(child != 0);

  if (!def->adefs_.has_attributes() && etype.is_group_) {

    // parent does not have attributes.
    // Child is one particle.
    // Therefore expand child elements to parent to erase anonymous types.

    switch (child->id_) {
    case sequence_id:
    case all_id:
    case choice_id:
      def->etypes_ = child->etypes_;
      def->adefs_ = child->adefs_;
      def->id_ = child->id_;
      return true;
    case array_id:
      def->set_value_type(child->get_value_type());
      def->etypes_.clear();
      def->id_ = child->id_;
      return true;
    case simpletype_id:
    case simplecontent_id:
      break; // Nothing to optimize.
    }
  }
  
  // child does not have attributes.
  // Move elements to parent. 
  bool child_expandable = (child->id_ != choice_id) && (child->id_ != array_id);
  if (!child->adefs_.has_attributes() && child_expandable && etype.is_group_) { 
    def->etypes_ = child->etypes_;
    def->id_ = child->id_;
    return true;;
  }
  return false;
}


bool akaxisonizer::is_ptrmember(const element_type &etype) const {
  if (etype.type_.is_imported()) // imported simpleType
    return (etype.get_occurrence().minOccurs_ == 0) && (etype.get_occurrence().maxOccurs_ == 1);
  
  const user_class *def = etype.type_.us_;

  switch (def->id_) {
  case sequence_id:
  case all_id:
  case simpletype_id:
  case simplecontent_id:
    return (etype.get_occurrence().minOccurs_ == 0) && (etype.get_occurrence().maxOccurs_ == 1);
  case array_id:
  case choice_id:
    return false;
//     if (!etype.occ_.in_range(0))      // Should be in optimization.!!!!!!
//       return true;
  }
  return false;
}



bool akaxisonizer::replace_any(const type_ref &type, element_type &etype) {

  if (type.get_name() == aka::qname("xs:anyType")) {
    if ((etype.get_occurrence().minOccurs_ == 0) && (etype.get_occurrence().maxOccurs_ == 1)) {
      etype.type_ = registry_.get_aka_any();
      etype.is_ptrmember_ = true;
    }
    else if (etype.get_occurrence().is_array()) {
      etype.type_ = registry_.get_aka_any_array();
      etype.occ_required_ = true;
    }
    else {
      etype.type_ = registry_.get_aka_any();
    }
    etype.namespace_list_.clear();
    etype.namespace_list_.push_back("##any");
    return true;
  }
  else if (type.get_name() == aka::qname("xs:anySimpleType")) {
    if ((etype.get_occurrence().minOccurs_ == 0) && (etype.get_occurrence().maxOccurs_ == 1)) {
      etype.type_ = registry_.get_std_string();
      etype.is_ptrmember_ = true;
    }
    else if (etype.get_occurrence().is_array()) {
      etype.type_ = registry_.get_std_string_array();
      etype.occ_required_ = true;
    }
    else {
      etype.type_ = registry_.get_std_string();
      etype.is_ptrmember_ = true;
    }
    return true;
  }
  return false;
}

bool akaxisonizer::replace_any(element_type &etype) {

  if (replace_any(etype.type_, etype))
    return true;

  const user_class *def = etype.type_.us_;
  if ((def != 0) && (def->id_ == array_id)) {
    if (replace_any(def->get_value_type(), etype))
      return true;
  }
  return false;
}

bool akaxisonizer::add_member_number(aka::qname &name, const std::string &prefix, int index) {

  std::string::size_type pos = name.local().find_first_of('*');
  if (pos == std::string::npos)
    return false;

  std::string local = name.local();

  local.resize(local.size() - 1);

  std::ostringstream ostm;
  ostm << index;
  local += prefix + ostm.rdbuf()->str();
  name = aka::qname(name.get_namespace_id(), local);
  return true;
}

void akaxisonizer::fix_member_name(user_class *def) {
  int seqindex = 0;
  int choindex = 0;
  int allindex = 0;
  int array_index = 0;

  for (element_types::iterator it = def->etypes_.begin();
       it != def->etypes_.end(); ++it) {

    if (it->type_.is_imported() || registry_.is_xs_any(it->type_.get_name())) {
      assert(it->name_.local().find_first_of('*') == std::string::npos);
      continue;
    }

    const user_class *child = it->type_.us_;
    assert(child != 0);
    if (child->id_ == sequence_id) {
      if (add_member_number(it->name_, "s", seqindex))
        ++seqindex;
    }
    else if (child->id_ == choice_id) {
      if (add_member_number(it->name_, "c", choindex))
        ++choindex;
    }
    else if (child->id_ == all_id) {
      if (add_member_number(it->name_, "a", allindex))
        ++allindex;
    }
    else if (child->id_ == array_id) {
      if (add_member_number(it->name_, "array", array_index))
        ++array_index;
    }
    assert(it->name_.local().find_first_of('*') == std::string::npos);
  }
}

void akaxisonizer::rename_duplicate_member(user_class *def) {

  if (def->id_ != sequence_id) {
    for (element_types::iterator it = def->etypes_.begin();
         it != def->etypes_.end(); ++it) {
      it->member_name_ = it->name_;
    }
  }    
  else {
    std::map<aka::qname, long, aka::qname_less> members;
    std::map<aka::qname, long, aka::qname_less> member_count;
    element_types::iterator it;
    for (it = def->etypes_.begin(); it != def->etypes_.end(); ++it) {
      members[it->name_] = 0;
      member_count[it->name_] = 0;
    }
    for (it = def->etypes_.begin(); it != def->etypes_.end(); ++it)
      ++ members[it->name_];
    
    for (it = def->etypes_.begin(); it != def->etypes_.end(); ++it) {
      if (members[it->name_] > 1) {
        it->member_name_ = it->name_;
        const std::string &memname = it->member_name_.local();
        if (isdigit(memname[memname.size() - 1])) {
          aka::qname member_name = it->member_name_;
          it->member_name_ = aka::qname(member_name.get_namespace_id(),
                                        member_name.local() + "_*");
        }
        else {
          aka::qname member_name = it->member_name_;
          it->member_name_ = aka::qname(member_name.get_namespace_id(),
                                        member_name.local() + '*');
        }
        add_member_number(it->member_name_, "", member_count[it->name_]);
        ++member_count[it->name_];
      }
      else {
        it->member_name_ = it->name_;
      }
    }
  }
}

bool akaxisonizer::replace_builtin(element_type &etype) {

  if (!etype.type_.is_imported())
      return false;

  if (is_ptrmember(etype))
    etype.is_ptrmember_ = true;
  else if (occ_required(etype.get_occurrence())) {
    const imported_type *array_type = registry_.get_imported_array_type(etype.type_.imported_);
    if (array_type == 0) {
      raise_name_error("type", etype.type_.get_name(), "corresponding array not found.", __FILE__, __LINE__);
    }
    etype.type_ = type_ref(array_type);
    etype.occ_required_ = true;
  }
  return true;
}


void akaxisonizer::set_array_properties(const user_class *def, element_type &etype) {

  switch (def->id_) {
  case sequence_id:
  case all_id:
    if (is_ptrmember(etype))
      etype.is_ptrmember_ = true;
    else if (occ_required(etype.get_occurrence()))
      etype.occ_required_ = true;
    break;
  case choice_id:
    if (occ_required(etype.get_occurrence()))
      etype.occ_required_ = true;
    break;
  case array_id:
    break; // Don't have to set array properties for array.
  case simpletype_id:
  case simplecontent_id:
    // does not have children.  No array available.
    break;
  }
}


void akaxisonizer::akaxisonize_particles(user_classes &uss) {
  
  for (user_classes::iterator cit = uss.begin(); cit != uss.end(); ++cit) {
    user_class *def = cit->second;

    fix_member_name(def);
    rename_duplicate_member(def);

    for (element_types::iterator eit = def->etypes_.begin();
         eit != def->etypes_.end(); ++eit) {

      // Must call replace_any() before repalce_builtin().
      if (replace_any(*eit)) 
        continue;
      else if (replace_builtin(*eit))
        continue;
      else 
        set_array_properties(def, *eit);
    }      

    for (attribute_types::iterator ait = def->adefs_.attributes_.begin();
         ait != def->adefs_.attributes_.end(); ++ait) {
      if (ait->type_.get_name() == aka::qname("xs:anySimpleType"))
        ait->type_ = registry_.get_std_string();
    }
  }
}

void akaxisonizer::akaxisonize() {

  // for simpleType.
  user_classes::iterator cit;
  for (cit = regunit_.simpleTypes_.begin(); cit != regunit_.simpleTypes_.end(); ++cit) {
    user_class *def = cit->second;
    //assert(def.id_ == simpletype_id);
    if (def->get_value_type().get_name() == aka::qname("xs:anySimpleType"))
      def->set_value_type(registry_.get_std_string());
  }
  for (cit = regunit_.local_simpleTypes_.begin(); 
       cit != regunit_.local_simpleTypes_.end(); ++cit) {
    user_class *def = cit->second;
    //assert(def.id_ == simpletype_id);
    if (def->get_value_type().get_name() == aka::qname("xs:anySimpleType"))
      def->set_value_type(registry_.get_std_string());
  }

  akaxisonize_particles(regunit_.complexTypes_);
  akaxisonize_particles(regunit_.groups_);
  akaxisonize_particles(regunit_.local_complexTypes_);
  akaxisonize_particles(regunit_.local_classes_);

  for (element_defs::iterator eit = regunit_.elements_.begin();
       eit != regunit_.elements_.end(); ++eit) {
    if (eit->second.type_.get_name() == aka::qname("xs:anyType"))
      eit->second.type_ = registry_.get_aka_wildcard();
    else if (eit->second.type_.get_name() == aka::qname("xs:anySimpleType"))
      eit->second.type_ = registry_.get_std_string();
  }
}

std::string akaxisonizer::get_type_string(const std::string &tag, 
                                          const user_class *us) {

  std::ostringstream ostm;
  ostm << "<" << tag 
       << " name=" << aka::quote(us->get_name());
  if (us->get_name() != us->get_classname())
    ostm << " osx:classname=" << aka::quote(us->get_classname());
  ostm << "/>";
  return ostm.str();
}

void akaxisonizer::check_names() {
  typedef std::vector<std::string> type_strings;
  typedef std::map<aka::qname, std::vector<std::string>, aka::qname_less> namemap;

  namemap names;

  user_classes::const_iterator it;
  for (it = regunit_.complexTypes_.begin(); 
       it != regunit_.complexTypes_.end(); ++it) {
    std::string type_string = get_type_string("xs:complexType", it->second);
    names[it->second->get_classname()].push_back(type_string);
  }
  for (it = regunit_.simpleTypes_.begin(); 
       it != regunit_.simpleTypes_.end(); ++it) {
    std::string type_string = get_type_string("xs:simpleType", it->second);
    names[it->second->get_classname()].push_back(type_string);
  }
  for (it = regunit_.groups_.begin(); 
       it != regunit_.groups_.end(); ++it) {
    std::string type_string = get_type_string("xs:group", it->second);
    names[it->second->get_classname()].push_back(type_string);
  }
  for (it = regunit_.local_complexTypes_.begin(); 
       it != regunit_.local_complexTypes_.end(); ++it) {
    std::string type_string = get_type_string("xs:complexType", it->second);
    names[it->second->get_classname()].push_back(type_string);
  }
  for (it = regunit_.local_simpleTypes_.begin(); 
       it != regunit_.local_simpleTypes_.end(); ++it) {
    std::string type_string = get_type_string("xs:simpleType", it->second);
    names[it->second->get_classname()].push_back(type_string);
  }
  for (it = regunit_.local_classes_.begin(); 
       it != regunit_.local_classes_.end(); ++it) {
    std::string type_string = get_type_string("(local class)", it->second);
    names[it->second->get_classname()].push_back(type_string);
  }

  for (namemap::const_iterator mit = names.begin();
       mit != names.end(); ++mit) {
    if (mit->second.size() > 1) {
      std::ostringstream ostm;
      ostm << regunit_.filename_ << ": class name confliction, " 
           << aka::quote(mit->first) << "." << std::endl;
      const type_strings &typestrs = mit->second;
      for (type_strings::const_iterator it = typestrs.begin();
           it != typestrs.end(); ++it) {
        ostm << " " << *it << std::endl;
      }
      throw aka::error(ostm.str(), __FILE__, __LINE__);
    }
  }
}


type_ref akaxisonizer::get_array(const type_ref &value_type) {

  const aka::qname &name = value_type.get_name();
  if (registry_.is_xs_any(name))
    return value_type;

  if (value_type.is_imported()) {
    const imported_type *imported = 
      registry_.get_imported_array_type(value_type.imported_);
    assert(imported != 0);
    return type_ref(imported);
  }

  const user_class *us = registry_.find_array_type(value_type.us_);
  if (us != 0) {
    assert(us->id_ == array_id);
    return type_ref(us);
  }

  
  if (regunit_.local_simpleTypes_.get(name) != 0)
    return registry_.create_array(value_type, regunit_.local_classes_);
  if (regunit_.local_complexTypes_.get(name) != 0)
    return registry_.create_array(value_type, regunit_.local_classes_);
  if (regunit_.local_classes_.get(name) != 0)
    return registry_.create_array(value_type, regunit_.local_classes_);
  
  assert(!"Must not reach here.");
  return type_ref();
}

void akaxisonizer::create_arrays(user_classes &uss, user_classes &arrays) {
  const_user_classes cuss;
  for (user_classes::iterator it = uss.begin(); it != uss.end(); ++it)
    cuss.push_back(it->second);
  for (const_user_classes::iterator qit = cuss.begin(); qit != cuss.end(); ++qit)
    registry_.create_array(type_ref(*qit), arrays);
}
