/* -*- c++ -*- */
#ifndef AKAXISO2_UNICODE_UTF8_TRAITS_H__
#define AKAXISO2_UNICODE_UTF8_TRAITS_H__

/**
 * @file akaxiso2/unicode/utf8_traits.h
 * @brief UTF-8 encoding traits.
 */

#include <akaxiso2/configuration.h>
#include <akaxiso2/exception.h>
#include <akaxiso2/unicode/ustring.h>
#include <akaxiso2/unicode/string_buffer.h>
#include <assert.h>

namespace aka2 {

  extern const int utf8_char_length_table[];

  struct utf8_traits {
    /**
     * compare UTF-8 string 
     * @param lhs UTF-8 string to be compared.
     * @param rhs UTF-8 string to be compared.
     * @return 0 if lhs == rhs, positive value if lhs > rhs, negative value if lhs < rhs.
     */
    static int compare(const char *lhs, const char *rhs) {
      return strcmp(lhs, rhs);
    }
    /**
     * compare UTF-8 string
     * @param lhs UTF-8 string to be compared.
     * @param rhs UTF-8 string to be compared.
     * @param size maximum length of strings to be comapred.
     * @return 0 if lhs == rhs, positive value if lhs > rhs, negative value if lhs < rhs.
     */
    static int compare(const char *lhs, const char *rhs, size_t size) {
      return strncmp(lhs, rhs, size);
    }
    
    /**
     * @brief find first match for UTF-8 string.
     * @param str target UTF-8 string.
     * @param length find from length'th char.
     * @param ch UTF-8 character to find.
     * @return the byte offset of the first match.
     */
    static int find_first(const char *str, size_t length, const char ch);
    
    /**
     * @brief get UTF-8 character length.
     * @param str UTF-8 string buffer
     * @return size of UTF-8 character(1-6).
     */
    inline static int char_length(const char *str) {
      int length = utf8_char_length_table[*reinterpret_cast<const unsigned char*>(str)];
      if (length == 0)
	throw aka2::error("Failed to get UTF-8 char length.", __FILE__, __LINE__);
      return length;
    }

    /**
     * @brief get string byte length.
     * @return length of buffer.
     */
    static size_t length(const char *buffer) { return strlen(buffer); }

    /**
     * @brief name of encoding.
     * @return char sequence of "UTF-8".
     */
    static const char *encoding();
    
  };

}

#endif
