/* -*- c++ -*- */
#ifndef AKAXISO2_UNICODE_STRING_BUFFER_H__
#define AKAXISO2_UNICODE_STRING_BUFFER_H__

/**
 * @file akaxiso2/unicode/string_buffer.h
 * @brief string buffer class template.
 */

#include <string>
#include <akaxiso2/unicode/ustring.h>

namespace aka2 {

  template<class T, class S>
  class T_string_buffer {
    /** hidden copy-ctor to forbid copying. */
    T_string_buffer(const T_string_buffer&);
  public:
    T_string_buffer(size_t initial_size = 1024) 
      : reserve_size_(initial_size), current_length_(0) { 
      buffer_ = new T[initial_size];
    }

    ~T_string_buffer() {
      delete [] buffer_;
      buffer_ = 0;
    }

    void insert_front(const T *Ts, size_t length) {
      size_t new_size = current_length_ + length;
      if (new_size > reserve_size_)
	resize(new_size);
      memmove(buffer_ + length, buffer_, current_length_ * sizeof(T));
      memcpy(buffer_, Ts, length * sizeof(T));
      current_length_ = new_size;
    }

    void append(const T *Ts, size_t length) {
      size_t new_size = current_length_ + length;
      if (new_size > reserve_size_)
	resize(new_size);
      memcpy(buffer_ + current_length_, Ts, length * sizeof(T));
      current_length_ = new_size;
    }

    void append(const T &t) {
      size_t new_size = current_length_ + 1;
      if (new_size > reserve_size_)
	resize(new_size);
      buffer_[current_length_] = t;
      ++current_length_;
    }

    void terminate() {
      size_t new_size = current_length_ + 1;
      if (new_size > reserve_size_)
	resize(new_size);
      buffer_[current_length_] = 0;
    }


    T* get_ptr() const {
      return buffer_;
    }

    T* get_end_ptr() {
      return buffer_ + current_length_;
    }

    T* get_end_ptr(size_t required_size) {
      size_t new_size = current_length_ + required_size;
      if (new_size > reserve_size_) {
	resize(new_size);
      }
      return buffer_ + current_length_;
    }

    T* get_buffer_end() const {
      return buffer_ + reserve_size_;
    }

    void commit_current_end(T *endptr) {
      current_length_ = endptr - buffer_;
    }

    void commit_additional_length(size_t length) {
      current_length_ += length;
    }

    void assign_to_string(S &str) {
      str.assign(buffer_, current_length_);
    }

    void erase_front(const size_t length) {
      current_length_ -= length;
      memmove(buffer_, buffer_ + length, current_length_ * sizeof(T));
    }

    void erase_by(const T *pos) {
      erase_front(pos - buffer_);
    }

    const S &get_string() {
      assign_to_string(string_);
      return string_;
    }

    void reserve(size_t new_reserve_size) {
      if (new_reserve_size > reserve_size_)
	resize(new_reserve_size);
    }

    void resize(size_t new_size) {
      reserve_size_ = new_size + (new_size >> 1);
      T *new_buffer = new T[reserve_size_];
      memcpy(new_buffer, buffer_, current_length_ * sizeof(T));
      delete [] buffer_;
      buffer_ = new_buffer;
      string_.reserve(reserve_size_); 
   }

    void clear() {
      current_length_ = 0;
    }

    bool empty() const {
      return current_length_ == 0;
    }

    size_t get_content_length() const { return current_length_; }
    size_t get_remain_length() const { return reserve_size_ - current_length_;  }

    size_t reserve_size_;
    size_t current_length_;
    T *buffer_;
    S string_;
  };


  typedef T_string_buffer<char, std::string> string_buffer;
  typedef T_string_buffer<uchar_t, ustring> ustring_buffer;
}

#endif
