/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_PIVOT_TRANSCODER_H__
#define AKAXISO2_FRAMEWORK_PIVOT_TRANSCODER_H__

/**
 * @file akaxiso2/transcoders/pivot_transcoder.h
 * @brief transcoder interface and base classes. 
 */

#include <string>
#include <akaxiso2/transcoders/transcoder.h>
#include <akaxiso2/unicode/string_buffer.h>
#include <akaxiso2/unicode/ustring.h>
#include <akaxiso2/unicode/utf8_traits.h>
#include <akaxiso2/unicode/ucs2_traits.h>
#include <akaxiso2/unicode/uniconv.h>

namespace aka2 {

  struct pivot_tag { };
  extern pivot_tag pivot;

}


#if !defined(AKAXISO2_INTERNAL_UCS2)


namespace aka2 {
  typedef std::string pstring;
  typedef char pchar_t;
  typedef struct utf8_traits pchar_traits;
  typedef string_buffer pstring_buffer;


  inline std::string ucs2_to_pivot(const ustring &str) {
    return uniconv::ucs2_to_utf8(str);
  }

  inline ustring pivot_to_ucs2(const std::string &str) {
    return uniconv::utf8_to_ucs2(str);
  }


}

#else


namespace aka2 {
  typedef ustring pstring;
  typedef uchar_t pchar_t;
  typedef struct ucs2_traits pchar_traits;
  typedef ustring_buffer pstring_buffer;


  inline const pstring &ucs2_to_pivot(const ustring &str) { return str; }
  inline const ustring &pivot_to_ucs2(const pstring &str) { return str; }

}

#endif


namespace aka2 {

  /**
   * @brief base class of local-code-page (lcp) transcoder
   */
  class pivot_transcoder {
    transcoder *from_pivot_;
    transcoder *to_pivot_;
    const struct char_mark *from_bom_;
    const struct char_mark *to_bom_;
    bool from_bom_to_check_;
    bool to_bom_to_check_;
    void reset();

    void to_pivot(const char *lcp, size_t length);
    void to_lcp(const pchar_t *pivot, size_t length);
    pivot_transcoder();
  public:
    ~pivot_transcoder();

    void to_pivot(pstring &pivot, const std::string &encstr);
    /**
     * @brief transcode string to pivot.
     * @param encstr converted string.
     * @param pivot string to be converted.
     */
    void to_lcp(std::string &encstr, const pstring &pivot);
    /**
     * @brief transcode string to pivot.
     * @param encstr string to be converted.
     * @return converted UTF-8 string.
     */
    pstring to_pivot(const std::string &encstr);
    /**
     * @brief transcode string to pivot.
     * @param pivot UTF-8 string to be converted.
     * @return converted string.
     */
    std::string to_lcp(const pstring &pivot);

    /**
     * @brief pstring_buffer version of to_pivot().
     * @param pivot_buff string_buffer to receive converted pchar_t sequence.
     * @param str string to be converted.
     */
    void to_pivot(pstring_buffer &pivot_buff, const std::string &str);

    /**
     * @brief string_buffer version of to_lcp().
     * @param lcp_buff string_buffer to receive converted lcp character sequence.
     * @param pivot pivot-encoded string to be converted.
     */
    void to_lcp(string_buffer &lcp_buff, const pstring &pivot);
    
    static pivot_transcoder *create(transcoder_factory trf); 

    static void initialize_static_transcoder(transcoder_factory trfact);
    static void uninitialize_static_transcoder();

  };

  pstring to_pivot(const std::string &str);
  std::string to_lcp(const pstring &str);
  void to_pivot(pstring_buffer &pivot_buff, const std::string &str);
  void to_lcp(string_buffer &lcp_buff, const pstring &pivot);
}


#endif
