#include "pivot_transcoder.h"
#include "../exception.h"
#include "encoding_name.h"
#include "../unicode/ucs2_traits.h"
#include "../unicode/utf8_traits.h"
#include "../util/mutex.h"


using namespace aka2;

pivot_tag aka2::pivot;



pstring pivot_transcoder::to_pivot(const std::string &encstr) {
  pstring result;
  to_pivot(result, encstr);
  return result;
}


std::string pivot_transcoder::to_lcp(const pstring &pivot) {
  std::string result;
  to_lcp(result, pivot);
  return result;
}


pivot_transcoder::pivot_transcoder()  {  }


pivot_transcoder::~pivot_transcoder() { 
  if (from_pivot_ || to_pivot_) {
    delete from_pivot_;
    delete to_pivot_;
    from_pivot_ = 0;
    to_pivot_ = 0;
  }
}


void pivot_transcoder::to_pivot(const char *lcp, size_t length) {
  to_pivot_->clear_transcoded_buffer();
  bool res = to_pivot_->transcode(lcp, length);

  if (!res) {
    reset();
    throw error("string_transcoder: incomplete input.", __FILE__, __LINE__);
  }
  
  if (to_bom_to_check_ && to_pivot_->clear_bom(to_bom_))
    to_bom_to_check_ = false;

  size_t pivot_length = to_pivot_->outbuf_.get_content_length();
  if (pivot_length % sizeof(pchar_t)) {
    throw error("wrong conversion, maybe internal error.", __FILE__, __LINE__);
  }
}

void pivot_transcoder::to_lcp(const pchar_t *pivot, size_t length) {
  from_pivot_->clear_transcoded_buffer();
  bool res = from_pivot_->transcode(reinterpret_cast<const char *>(pivot),
				    length * sizeof(pchar_t));

  if (!res) {
    reset();
    throw error("string_transcoder: incomplete input.", __FILE__, __LINE__);
  }
  
  if (from_bom_to_check_ && from_pivot_->clear_bom(from_bom_))
    from_bom_to_check_ = false;

}

void pivot_transcoder::to_pivot(pstring &pivot, const std::string &encstr) {
  if (encstr.empty()) {
    pivot.erase(pivot.size());
    return;
  }
  
  to_pivot(encstr.data(), encstr.size());

  pivot.assign(reinterpret_cast<const pchar_t*>(to_pivot_->outbuf_.get_ptr()),
	       to_pivot_->outbuf_.get_content_length() / sizeof(pchar_t));
  to_pivot_->inbuf_.clear();
  to_pivot_->outbuf_.clear();

}


void pivot_transcoder::to_lcp(std::string &encstr, const pstring &pivot) {
  if (pivot.empty()) {
    encstr.assign("");
    return;
  }

  to_lcp(pivot.data(), pivot.size());

  encstr.assign(from_pivot_->outbuf_.get_ptr(),
		from_pivot_->outbuf_.get_content_length());
  from_pivot_->inbuf_.clear();
  from_pivot_->outbuf_.clear();
}


void pivot_transcoder::reset() {
  from_pivot_->reset();
  from_pivot_->inbuf_.clear();
  from_pivot_->outbuf_.clear();

  to_pivot_->reset();
  to_pivot_->inbuf_.clear();
  to_pivot_->outbuf_.clear();

  /* transcoder supports char_mark, therefore it should be removed. */
  from_bom_to_check_ = (from_bom_ != 0);
  to_bom_to_check_ = (to_bom_ != 0);
}


void pivot_transcoder::to_pivot(pstring_buffer &pivot_buff, const std::string &encstr) {

  pivot_buff.clear();

  if (encstr.empty()) {
    pivot_buff.terminate();
    return;
  }

  to_pivot(encstr.data(), encstr.size());

  pivot_buff.append(reinterpret_cast<const pchar_t*>(to_pivot_->outbuf_.get_ptr()),
		    to_pivot_->outbuf_.get_content_length() / sizeof(pchar_t));
  pivot_buff.terminate();

  to_pivot_->inbuf_.clear();
  to_pivot_->outbuf_.clear();

}


void pivot_transcoder::to_lcp(string_buffer &lcp_buff, const pstring &pivot) {

  lcp_buff.clear();

  if (pivot.empty()) {
    lcp_buff.terminate();
    return;
  }

  to_lcp(pivot.data(), pivot.size());

  size_t length = from_pivot_->outbuf_.get_content_length();
  lcp_buff.append(from_pivot_->outbuf_.get_ptr(), length);
  lcp_buff.terminate();

  from_pivot_->inbuf_.clear();
  from_pivot_->outbuf_.clear();
}




pivot_transcoder *pivot_transcoder::create(transcoder_factory trf) {
  pivot_transcoder *tr = new pivot_transcoder();
  std::string pivot = pchar_traits::encoding();
  std::string lcp = get_default_encoding();

  tr->from_pivot_ = create_transcoder(lcp, pivot, trf);
  tr->from_pivot_->set_encoding_types(lcp, pivot);
  tr->from_bom_ = get_unicode_bom(lcp);

  tr->to_pivot_ = create_transcoder(pivot, lcp, trf);
  tr->to_pivot_->set_encoding_types(pivot, lcp);
  tr->to_bom_ = get_unicode_bom(pivot);

  tr->reset();
  return tr;
}



namespace {
  pivot_transcoder *static_ = 0;
  mutex mutex_;
}

void pivot_transcoder::initialize_static_transcoder(transcoder_factory trfact) {
  static_ = pivot_transcoder::create(trfact);
}

void pivot_transcoder::uninitialize_static_transcoder() {
  assert(static_ != 0);
  delete static_;
  static_ = 0;
}





pstring aka2::to_pivot(const std::string &str) {
  mutex_.lock();
  pstring result = static_->to_pivot(str);
  mutex_.unlock();
  return result;
}

std::string aka2::to_lcp(const pstring &str) {
  mutex_.lock();
  std::string result = static_->to_lcp(str);
  mutex_.unlock();
  return result;
}

void aka2::to_pivot(pstring_buffer &pivot_buff, const std::string &str) {
  mutex_.lock();
  static_->to_pivot(pivot_buff, str);
  mutex_.unlock();
}

void aka2::to_lcp(string_buffer &lcp_buff, const pstring &pivot) {
  mutex_.lock();
  static_->to_lcp(lcp_buff, pivot);
  mutex_.unlock();
}
