#include "icu_transcoder.h"

#ifdef AKAXISO2_USE_ICU

#include <unicode/ucnv.h>
#include "../exception.h"
#include "encoding_name.h"
#include <assert.h>

using namespace aka2;


UConverter* icu_transcoder::create_converter(const std::string &converterName) {
  UErrorCode err = U_ZERO_ERROR;
  UConverter *ucnv = ucnv_open(converterName.c_str(), &err);
  if (err != U_ZERO_ERROR) {
    const char *errname = u_errorName(err);
    throw aka2::error(errname, __FILE__, __LINE__);
  }
  return ucnv;
}


icu_transcoder::~icu_transcoder() {
  ucnv_close(ucnv_to_);
  ucnv_close(ucnv_from_);
}

bool icu_transcoder::from_inbuf() {

  const char *inbuf_current = inbuf_.get_ptr();
  const char *inbufend = inbuf_.get_end_ptr();
  UErrorCode err_from = U_ZERO_ERROR;
    
  while (inbuf_current < inbufend) {
      
    err_from = U_ZERO_ERROR;
    UChar *pivot = ubuffer_.get_end_ptr(inbufend - inbuf_current);
    UChar *pivotend = ubuffer_.get_buffer_end();
      
    err_from = U_ZERO_ERROR;
    ucnv_toUnicode(ucnv_from_, 
		   &pivot, pivotend, 
		   &inbuf_current, inbufend, 
		   0, FALSE, &err_from);
    ubuffer_.commit_current_end(pivot);
      
    switch (err_from) {
    case U_ZERO_ERROR:
      inbuf_.clear();
      return true;
    case U_TRUNCATED_CHAR_FOUND:
      inbuf_.erase_by(inbuf_current);
      return true;
    case U_INVALID_CHAR_FOUND:
    case U_ILLEGAL_CHAR_FOUND:
      // incoming string could not be transcoded to Unicode,
      return false;
    default:
      assert(!"Must not reach here.");
      break;
    }
  }
  // input length is 0.
  return true;
}

bool icu_transcoder::to_outbuf(const aka2::uchar_t *pivot_current) {

  const UChar *pivotend = ubuffer_.get_end_ptr();

  while (pivot_current < pivotend) {
    char *outbuf = outbuf_.get_end_ptr(pivotend - pivot_current);
    char *outbufend = outbuf_.get_buffer_end();
    
    UErrorCode err_to = U_ZERO_ERROR;
    ucnv_fromUnicode(ucnv_to_, 
		     &outbuf, outbufend, 
		     &pivot_current, pivotend, 
		     0, FALSE, &err_to);
    outbuf_.commit_current_end(outbuf);

    switch (err_to) {
    case U_BUFFER_OVERFLOW_ERROR: {
      continue;
    }
    case U_ZERO_ERROR: { // normal end.
      ubuffer_.clear();
      return true;
    }
    case U_INVALID_CHAR_FOUND:
    case U_ILLEGAL_CHAR_FOUND: {
      return false;
    }
    case U_TRUNCATED_CHAR_FOUND:
      throw error("UTF-16 sarrogate not supportd.", __FILE__, __LINE__);
    default:
      assert(!"Must not reach here.");
      break;
    }
  }
  return true;
}



void icu_transcoder::reset() {
  ubuffer_.clear();
  if (ucnv_to_)
    ucnv_reset(ucnv_to_);
  if (ucnv_from_)
    ucnv_reset(ucnv_from_);
}


transcoder *icu_transcoder::create(const std::string &tocode, const std::string &fromcode) {

  UConverter *ucnv_to = 0;
  UConverter *ucnv_from = 0;

  try {
    if (is_ucs2(tocode)) {
      ucnv_to = 0;
    }
    else if (is_utf8(tocode)) {
      ucnv_to = 0;
    }
    else {
      ucnv_to = create_converter(tocode);
    }


    if (is_ucs2(fromcode)) {
      ucnv_from = 0;
    }
    else if (is_utf8(fromcode)) {
      ucnv_from = 0;
    }
    else {
      ucnv_from = create_converter(fromcode);
    }
  }
  catch ( ... ) {
    if (ucnv_to)
      ucnv_close(ucnv_to);
    if (ucnv_from)
      ucnv_close(ucnv_from);
    throw;
  }

  return new icu_transcoder(ucnv_to, ucnv_from);
}



#endif
