#include "builtin_transcoders.h"
#include "../unicode/uniconv.h"
#include "encoding_name.h"
#include "../util/string_funcs.h"

using namespace aka2;

bool null_transcoder::transcode(const char *to_transcode, size_t length) {
  char *ptr = outbuf_.get_end_ptr(length); 
  memcpy(ptr, to_transcode, length);
  outbuf_.commit_additional_length(length);
  return true;
}

void null_transcoder::reset() {
}

bool utf8_to_ucs2_transcoder::transcode(const char *to_transcode, size_t length) {
  // must succeed in any case.
  uniconv::utf8_to_ucs2(outbuf_, to_transcode, length);
  return true;
}

void utf8_to_ucs2_transcoder::reset() {
}


bool ucs2_to_utf8_transcoder::transcode(const char *to_convert, size_t length) {
  inbuf_.append(to_convert, length);
  const aka2::uchar_t *uinbuf = reinterpret_cast<const aka2::uchar_t*>(inbuf_.get_ptr());
  size_t ulength = inbuf_.get_content_length() / 2;

  uniconv::ucs2_to_utf8(outbuf_, uinbuf, ulength);
  inbuf_.erase_by(reinterpret_cast<const char *>(uinbuf + ulength));

  return inbuf_.empty(); // If incomplete input, return false.
}

void ucs2_to_utf8_transcoder::reset() {
}


transcoder *builtin_transcoder::create(const std::string &tocode, const std::string &fromcode) {
  transcoder *tr = create_no_throw(tocode, fromcode);
  if (tr != 0)
    return tr;
  throw error("encoding " + quote(tocode) + " or " + quote(fromcode) + " not supported.", 
              __FILE__, __LINE__);
  return 0; // to suppress warning.
}

transcoder *builtin_transcoder::create_no_throw(const std::string &tocode, const std::string &fromcode) {
  std::string from_key = create_encoding_key(fromcode);
  std::string to_key = create_encoding_key(tocode);

  // null transcode.
  if (to_key == from_key)
    return new null_transcoder;

  /** use AKAXISO2-builtin transcoders */
  if (is_utf8(to_key) && is_ucs2(from_key))
    return new ucs2_to_utf8_transcoder;

  if (is_ucs2(to_key) && is_utf8(from_key))
    return new utf8_to_ucs2_transcoder;

  return 0;
}
