/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_MEMBERDEF_H__
#define AKAXISO2_FRAMEWORK_MEMBERDEF_H__

/**
 * @file akaxiso2/framework/memberdef.h
 * @brief member definitions in leaf classes of @ref aka_sequence and @ref aka_all
 */

#include <akaxiso2/framework/fixed.h>
#include <akaxiso2/framework/ptrmember.h>
#include <akaxiso2/framework/any.h>
#include <akaxiso2/framework/accessor.h>
#include <akaxiso2/framework/model_check.h>

namespace aka2 {

  /**
   * @brief template to define member for @ref aka_sequence and @ref aka_all .
   *
   * This class provides classes/functions to define child elements
   * for @ref aka_sequence and @ref aka_all.
   * @param L leaf class.
   * @param T value class.
   * @see aka2::sequence<>, aka2::all<>
   */
  template<class L, class T>
  struct memberdef : xmltype_statics<L> {

    static member_type *define_member(const std::string tagname, 
				      member_getter *mgetter,
				      element_op &op,
				      default_op *defop,
				      const occurrence &occ,
				      bool emptiable) {
      aka2::member_type mtype(mgetter, op, emptiable);
      mtype.set_name(qname(tagname)); 
      if (defop != 0)
	mtype.set_default_op(defop);
      if (!check_occurrence(op.get_schematype(), occ))
	throw tagged_error("element", tagname, "has wrong occurrence.", __FILE__, __LINE__);
      mtype.set_occurrence(occ);
      return L::register_membertype(mtype);
    }
    
    /**
     * @brief Helper class to define a serializable member in aka2::sequence<>, aka2::all<>. 
     *
     * Helper class to define serializable members in aka2::sequence<>, aka2::all<>.
     * @see aka2::memberdef::member
     */
    struct _member {
      /**
       * @brief Constructor for member definition
       * @param tagname tag name of member.
       * @param m pointer to a member of a value class.
       */
      template<class P, class V> 
      _member(const std::string &tagname, V P::* m) {
      	new_member(tagname, m, xiso::leaf<V>(), 1, 1, false);
      }
      /**
       * @brief Constructor for member array definition
       * @param tagname tag name of member.
       * @param m pointer to a member of a value class of T.
       * @param minOccurs minOccurs.
       * @param maxOccurs maxOccurs.
       */
      template<class P, class V> 
      _member(const std::string &tagname, V P::* m, int minOccurs, int maxOccurs) {
      	new_member(tagname, m, xiso::leaf<V>(), minOccurs, maxOccurs, false);
      }

      /**
       * @brief Constructor for member definition
       * @param tagname tag name of member.
       * @param m pointer to a member of a value class of T.
       * @param vl leaf class for V
       */
      template<class P, class V, class VL> 
      _member(const std::string &tagname, V P::* m, const VL& vl) {
	new_member(tagname, m, VL(), 1, 1, false);
      }
      /**
       * @brief Constructor for member array definition
       * @param tagname tag name of member.
       * @param m pointer to a member of a value class of T.
       * @param vl leaf class for VL.
       * @param minOccurs minOccurs.
       * @param maxOccurs maxOccurs.
       * @param emptiable allow empty array.
       */
      template<class P, class V, class VL> 
      _member(const std::string &tagname, V P::* m, const VL& vl, int minOccurs, int maxOccurs,
	      bool emptiable = false) {
	new_member(tagname, m, VL(), minOccurs, maxOccurs, emptiable);
      }

      /**
       * @brief set default value of member.
       * @param defval default value.
       */
      void set_default(const std::string &defval) {
	if (mtype_->get_schematype() != simpletype_id)
	  throw tagged_error("element", mtype_->get_name().qualified(), 
			     "default value should be simpleType.",
			     __FILE__, __LINE__);
      	mtype_->setup_default_value(defval, system_entity_complements());
      };

    private:
      template<class P, class V, class VL> 
      void new_member(const std::string &tagname, V P::* m, const VL&, 
		      int minOccurs, int maxOccurs,
		      bool emptiable) {
      	VL::initialize();
	member_getter *mgetter = 
	  create_ptr_getter(reinterpret_cast<T*>(0), m);	  
	mtype_ = define_member(tagname, 
			       mgetter, 
			       VL::dispatcher_, 
			       VL::create_default_op(),
			       aka2::occurrence(minOccurs, maxOccurs),
			       emptiable);
      }
      member_type *mtype_;
    };

    /**
     * @brief Typedef name of _member.
     *
     * Use member instead of _member.\n
     * _member is typedef'ed for VC6 workaround.
     */
    typedef _member member;

    /**
     * @brief Utility class to define a pointer-type member.
     *
     * Member must be aka2::deep_ptr<> template class.
     * Occurrence is [0, 1].
     * @see arrays_pointers
     */
    struct _ptrmember {
      /**
       * @brief Constructor to define a pointer-type member
       * @param tagname tag name
       * @param m pointer to member of a value class
       */
      template<class P, class V> 
      _ptrmember(const std::string &tagname, V P::* m) {
	new_ptr_member(tagname, m, xiso::leaf<TYPENAME V::value_type>(), false);
      }

      /**
       * @brief Constructor to define a pointer-type member
       * @param tagname tag name
       * @param m pointer to member of a value class
       * @param vl leaf class for V
       */
      template<class P, class V, class VL>
      _ptrmember(const std::string &tagname, V P::* m, const VL& vl) {
      	new_ptr_member(tagname, m, VL(), false);
      }
    private:
      template<class P, class V, class VL> 
      void new_ptr_member(const std::string &tagname, V P::* m, const VL&, 
			  bool emptiable) {
	VL::initialize();
	member_getter *mgetter = 
	  create_ptr_getter(reinterpret_cast<T*>(0), m);	  
	define_member(tagname, mgetter, 
		      aka2::ptrmember_op_stub<V, VL>::dispatcher_,
		      VL::create_default_op(),
		      occurrence(0, 1), emptiable);
      }
    };

    /**
     * @brief Typedef name of _ptrmember.
     *
     * _ptrmember is typedef'ed for VC6 workaround.
     */
    typedef _ptrmember ptrmember;

    /**
     * @brief Helper class to define serializable fixed-value member.
     *
     * Define fixed-value member with fixed_member struct. 
     * Value class does not have corresponding value because fixed value is not variable.
     * @param V Value type of a fixed value. 
     */
    template<class V>
    struct fixed_member { // Default value for complexType or mixed content will not be supported.
      fixed_member(const std::string &tagname, const std::string &fixed_value) {
      	new_member(tagname, xiso::leaf<V>(), fixed_value);
      }
      template<class VL>
      fixed_member(const std::string &tagname, const std::string &fixed_value, const VL &) {
      	new_member(tagname, VL(), fixed_value);
      }
    private:
      template<class VL>
      void new_member(const std::string &tagname, const VL&, const std::string &fixed_value) {
      	VL::initialize();
	if (VL::dispatcher_.get_schematype() != simpletype_id)
	  throw tagged_error("fixed", L::get_xmltype().qualified(), 
			     " fixed value should be simpleType.",
			     __FILE__, __LINE__);
	member_type *mtype = define_member(tagname, new null_getter(),
					   aka2::fixed<VL>::dispatcher_,
					   VL::create_default_op(),
					   aka2::occurrence(), false);
	mtype->setup_default_value(fixed_value, system_entity_complements());
      }
    };
    
    /**
     * @brief define an array member of fixed-values.
     *
     * Fixed value members are implemented as container of a fixed value.
     * Container element should be simpleType. 
     * @see aka2::memberdef::fixed_array
     */
    struct _fixed_array { 
      /**
       * @brief define a member array of a fixed-value.
       * @param tagname tagname
       * @param fixed_value fixed value
       * @param m pointer to array member of value class.
       * @param minOccurs minOccurs
       * @param maxOccurs maxOccurs
       */
      template<class P, class V>
      _fixed_array(const std::string &tagname, const std::string &fixed_value, V P::*m, 
		   int minOccurs, int maxOccurs) {
      	new_member(tagname, m, xiso::leaf<V>(),
		   minOccurs, maxOccurs, fixed_value);
      }

      /**
       * @brief define member array of a fixed value.
       * @param tagname tag name
       * @param fixed_value fixed value
       * @param m pointer to array member of value class.
       * @param vl leaf class of V
       * @param minOccurs minOccurs
       * @param maxOccurs maxOccurs
       * @param emptiable allow empty array.
       */
      template<class P, class V, class VL>
      _fixed_array(const std::string &tagname, const std::string &fixed_value, V P::*m, 
		   const VL& vl,
		   int minOccurs, int maxOccurs,
		   bool emptiable = false) {
      	new_member(tagname, m, VL(), minOccurs, maxOccurs, fixed_value, emptiable);
      }
    private:
      template<class P, class V, class VL>
      void new_member(const std::string &tagname, V P::* m, const VL&, 
		      int minOccurs, int maxOccurs,
		      const std::string &fixed_value,
		      bool emptiable) {
      	VL::initialize();
	typedef TYPENAME VL::item_leaf_type item_leaf_type;

	if (VL::dispatcher_.get_schematype() != array_id)
	  throw tagged_error("element", tagname, "should be array.", __FILE__, __LINE__);
	if (VL::dispatcher_.get_item_op().get_schematype() != fixed_id)
	  throw tagged_error("element", tagname, "should be fixed.", __FILE__, __LINE__);
	const fixed_op &fop = static_cast<const fixed_op&>(VL::dispatcher_.get_item_op());
	if (fop.get_value_op().get_schematype() != simpletype_id)
	  throw tagged_error("element", tagname, "fixed value should not a simpleType.",
			     __FILE__, __LINE__);
	
	member_getter *mgetter = create_ptr_getter(static_cast<const T*>(0), m);
	member_type *mtype = define_member(tagname, mgetter,
					   VL::dispatcher_,
					   item_leaf_type::create_default_op(),
					   aka2::occurrence(minOccurs, maxOccurs),
					   emptiable);
	mtype->setup_default_value(fixed_value, system_entity_complements());
      }
    };

    /**
     * @brief Typedef name of _fixed_array
     *
     * Use fixed_array instead of _fixed_array.
     * fixed_array is typedef'ed for VC6 workaround.
     */
    typedef _fixed_array fixed_array;

    /**
     * @brief define a pointer-type member of aka2::any.
     *
     * Pointer type of aka2::any should be declared as aka2::deep_ptr<aka2::any>.
     * Occurence is [0, 1].
     * @param tagname tag name
     * @param m pointer to aka2::deep_ptr<aka2::any> member
     * @param ns_list namespace list. 
     * available values are "##any", "##other:target_namespace_uri", 
     * "##local" or namespace uri list.
     */
    template<class P>
    void any_ptrmember(const std::string &tagname, aka2::deep_ptr<aka2::any> P::* m,
		       const std::string &ns_list = "##any") {
      typedef aka2::ptrmember_op_stub<aka2::deep_ptr<aka2::any>, aka2::any_op>
	op_type;
      member_getter *mgetter = 
	create_ptr_getter(reinterpret_cast<T*>(0), m);	  
      aka2::member_type 
	mtype(mgetter, op_type::dispatcher_, false);
      mtype.set_name(qname(tagname));
      mtype.set_occurrence(aka2::occurrence(0, 1));
      mtype.set_ns_list(ns_list);
      L::register_membertype(mtype);
    }

    /**
     * @brief define an aka2::any (xs:anyType) member.
     * @param tagname tag name
     * @param m pointer to an aka2::any member.
     * @param ns_list namespace list. 
     * available values are "##any", "##other:target_namespace_uri", 
     * "##local" or namespace uri list.
     */
    template<class P>
    void any(const std::string &tagname, aka2::any P::* m, 
	     const std::string &ns_list = "##any") {
      member_getter *mgetter = 
	create_ptr_getter(reinterpret_cast<T*>(0), m);	  
      aka2::member_type mtype(mgetter, any_op::dispatcher_, false);
      mtype.set_name(aka2::qname(tagname));
      mtype.set_ns_list(ns_list);
      L::register_membertype(mtype);
    }

    /**
     * @brief define an aka2::any_array member.
     * @param tagname tag name
     * @param m pointer to an aka2::any_array member.
     * @param minOccurs minOccurs
     * @param maxOccurs maxOccurs
     * @param ns_list namespace list. 
     * available values are "##any", "##other:target_namespace_uri", 
     * "##local" or namespace uri list.
     */
    template<class P>
    void any(const std::string &tagname, aka2::any_array P::* m, 
	     int minOccurs, int maxOccurs, const std::string &ns_list = "##any") {
      any(tagname, m, minOccurs, maxOccurs, false, ns_list);
    }

    /**
     * @brief define an aka2::any_array member.
     * @param tagname tag name
     * @param m pointer to an aka2::any_array member.
     * @param minOccurs minOccurs
     * @param maxOccurs maxOccurs
     * @param ns_list namespace list. 
     * available values are "##any", "##other:target_namespace_uri", 
     * "##local" or namespace uri list.
     * @param emptiable true if array could be empty even minOccurs is not zero.
     */
    template<class P>
    void any(const std::string &tagname, aka2::any_array P::* m, 
	     int minOccurs, int maxOccurs, bool emptiable,
	     const std::string &ns_list) {
      member_getter *mgetter = 
	create_ptr_getter(reinterpret_cast<T*>(0), m);	  
      aka2::member_type mtype(mgetter, any_array_op::dispatcher_, emptiable);
      mtype.set_name(aka2::qname(tagname));
      mtype.set_occurrence(aka2::occurrence(minOccurs, maxOccurs));
      mtype.set_ns_list(ns_list);
      L::register_membertype(mtype);
    }

    /**
     * @brief define a child element by using accessor.
     *
     * Accessor is defined with getter(G)/setter(S) classes.\n
     * Getter class (G) should have G::operator() to get pointer to the value.
     *
     * @code
     * template <class P, class V>
     * struct G {
     *   typedef V value_type;
     *   const V* operator()(const P& p) const;
     * };
     * @endcode
     *
     * Setter class (S) should have S::operator() to set value.
     *
     * @code
     * template<class P, class V>
     * struct S {
     *   typedef V value_type;
     *   void operator()(P &p, const V &value) const;
     * };
     * @endcode
     *
     * class P is the type of value class, and V if for value type of an accessor.
     * @see aka2::memberdef::accessor
     */
    struct _accessor {
      /**
       * @brief define a member by using accessor.
       *
       * @param tagname tagname.
       * @param g getter class.
       * @param s setter class.
       * @param vl leaf class of value.
       */
      template<class G, class S, class VL>
      _accessor(const std::string &tagname, const G &g, const S &s, const VL &vl) {
        new_member(tagname, g, s, VL(), 1, 1, false);
      }

      /**
       * @brief define an array member by using accessor.
       *
       * @param tagname tagname.
       * @param g getter class.
       * @param s setter class.
       * @param vl leaf class of value.
       * @param minOccurs minOccurs
       * @param maxOccurs maxOccurs
       * @param emptiable true if array could be empty even minOccurs is not zero.
       */
      template<class G, class S, class VL>
      _accessor(const std::string &tagname, const G &g, const S &s, const VL &vl,
		int minOccurs, int maxOccurs,
		bool emptiable = false) {
        new_member(tagname, g, s, VL(), minOccurs, maxOccurs, false);
      }
    private:
      template<class G, class S, class VL> 
      void new_member(const std::string &tagname, 
		      const G &g, const S &s, const VL&,
		      int minOccurs, int maxOccurs, bool emptiable) {
	VL::initialize();

	member_getter *mgetter = 
	  create_accessor_getter(reinterpret_cast<T*>(0), 
				 VL::dispatcher_,
				 g, s);	  
	define_member(tagname, 
		      mgetter,
		      VL::dispatcher_,
		      VL::create_default_op(),
		      aka2::occurrence(minOccurs, maxOccurs),
		      emptiable);
      }
    };

    /**
     * @brief Typedef name of _accessor.
     *
     * Use accessor intead of _accessor.\n
     * _accessor is typedef'ed for VC6 workaround.
     */
    typedef _accessor accessor;
  };
}

#endif
