/* -*- c++ -*- */
#ifndef AKAXISO2_FRAMEWORK_ENTITY_COMPLEMENTS_H__
#define AKAXISO2_FRAMEWORK_ENTITY_COMPLEMENTS_H__

/**
 * @file akaxiso2/framework/entity_complements.h
 * @brief preconditions for read/write simpleType values.
 */

#include <akaxiso2/framework/types.h>
#include <akaxiso2/util/shared_ptr.h>
#include <akaxiso2/transcoders/pivot_transcoder.h>
#include <map>
#include <set>

namespace aka2 {
  class prefix_map;

  void reset_for_new_document();

  /**
   * @brief complement operator
   * Base class for operator of entity_complement.
   */
  struct complement_operator {
    virtual ~complement_operator() { }
    /**
     * @brief reset complement before serialization/deserialization. 
     * @param e pointer of complement.
     */
    virtual void reset_for_new_document(void *e) const = 0;
  };

  /**
   * @brief template for 
   */
  template <class T>
  struct T_complement_operator : public complement_operator {
    virtual void reset_for_new_document(void *e) const {
      T* t = static_cast<T*>(e);
      t->clear();
    }
  };

  /**
   * @brief base class for user-defined complement.
   * entity_complement is the base class to give additional information 
   * for deserialization of @ref aka_simpletype.
   */
  struct entity_complement {
    /**
     * @brief constructor
     * @param complement instance of complement.
     * @param op instance of complement_operator.
     */
    entity_complement(void *complement, complement_operator *op)
      : complement_(complement), op_(op) { }
    void *complement_;
    shared_ptr<complement_operator> op_;
    
    /**
     * @brief reset complement before serialization/deserialization. 
     */
    void reset_for_new_document() const {
      op_->reset_for_new_document(complement_);
    }
  };


  typedef std::set<std::string> id_constraint;

  /**
   * @brief entity_complements to read/write simpletype values.
   *
   * entity_complements class are created by deserializer.\n
   * Created instances are pased to aka2::simpletype<> methods to give entity_complements 
   * to serialize/deserialize simpletype values during deserialization.\n
   * \n
   * This class will be exteded when we need other entity_complements for simpletypes.\n
   * Currently namespace prefix/URI asociation map is obtained.\n
   * @see @ref aka_simpletype
   */
  class entity_complements {
    typedef std::map<std::string, entity_complement> complements;
    complements complements_;

    shared_ptr<prefix_map> prefixes_;
    id_constraint id_constraint_;

    transcoder_factory trfact_;
    pivot_transcoder *transcoder_;

    void clear_complements();
    entity_complements(const entity_complements &); // hidden copy-ctor.

  protected:
    explicit entity_complements(transcoder_factory trfact);
  public:
    /**
     * @brief const version to get prefix and namespace URI asociation map.
     * @return get reference to current prefix_map.
     */
    prefix_map &get_prefixes() const { return *prefixes_; }

    std::string to_lcp(const pstring &pivot) {
      return transcoder_->to_lcp(pivot);
    }
    pstring to_pivot(const std::string &lcp) {
      return transcoder_->to_pivot(lcp);
    }

    /**
     * @brief add complement.
     * @param name complement name.
     * @param t instance of complement.
     */
    template<class T>
    void add_complement(const std::string &name, T &t) {
      add_complement(name, &t, new T_complement_operator<T>());
    }

    /**
     * @brief add complement.
     * @param name complement name.
     * @param t instance of complement.
     * @param op instance of complement_operator-derived class.
     */
    template<class T>
    void add_complement(const std::string &name, T &t, complement_operator *op) {
      add_complement_ptr(name, &t, op);
    }

    void add_complement_ptr(const std::string &name, void *p, complement_operator *prop_op);
    void *get_complement_ptr(const std::string &name) const;
    const void *const_get_complement_ptr(const std::string &name) const;

    /**
     * @brief check string is unique.
     * Check uniqueness of text value. 
     * @param value text value to check uniqueness.
     * @return true if xs:ID value is unique.
     */
    bool check_id(const std::string &value);

    /**
     * @brief clear all entity_complement.
     */
    void clear_all();

    /**
     * @brief reset entity_complements before serialization/deserialization.
     */
    void reset_for_new_document();

    static entity_complements *create(transcoder_factory trfact);
    static entity_complements *clone(entity_complements &ecomps,
				     bool copy_prefixes, 
				     transcoder_factory trfact);
    ~entity_complements();
  };


  /**
   * @brief get complement from entity_complements.
   * @param ecomps entity_complements to get complement from.
   * @param name name of complement.
   * @return reference to complement.
   */
  template<class T>
  inline T& get_complement(entity_complements &ecomps, const std::string &name) {
    return *static_cast<T*>(ecomps.get_complement_ptr(name));
  }
  
  /**
   * @brief const version of get_complement().
   * @param ecomps entity_complements to get complement from.
   * @param name name of complement.
   * @return reference to complement.
   */
  template<class T>
  inline const T& get_complement(const entity_complements &ecomps, 
				 const std::string &name) {
    return *static_cast<const T*>(ecomps.const_get_complement_ptr(name));
  }
}

#endif
