#include "transcoding_inputsource.h"

#ifdef AKAXISO2_USE_XERCESC

#include <iostream>
#include <fstream>

#include "xerces_handler.h"
#include "../deserializer/encoding_sensor.h"
#include "../../util/platform.h"
#include "../../util/scoped_ptr.h"
#include "../../transcoders/transcoder.h"

#include <xercesc/util/BinInputStream.hpp>
#include <xercesc/util/XMLString.hpp>

using namespace aka2;
using namespace xercesc;

namespace {

  class chars_to_XMLCh {
  public:
    chars_to_XMLCh(const char* chars) {
      xstr_ = xercesc::XMLString::transcode(chars);
    }
    ~chars_to_XMLCh() {
      xercesc::XMLString::release(&xstr_);
    }
    const XMLCh *xstr() const { return xstr_; }
  private:
    XMLCh *xstr_;
  };


  class TranscodingBinInputStream : public xercesc::BinInputStream {
  public:
    TranscodingBinInputStream(stateful_transcoder *tc) 
      : transcoder_(tc), pos_(0) { }
    
    virtual unsigned int curPos () const;
    virtual size_t readRawBytes(char *toFill, size_t maxToRead) = 0; 
    virtual unsigned int readBytes(XMLByte *const toFill, const unsigned int maxToRead);
  private:
    stateful_transcoder *transcoder_;
    string_buffer read_buffer_;
    unsigned int pos_;
  };

  unsigned int TranscodingBinInputStream::curPos () const {
    return pos_;
  }

  unsigned int TranscodingBinInputStream::readBytes(XMLByte *const toFill, 
						    const unsigned int maxToRead) {
    
    string_buffer &strbuf = transcoder_->get_buffer();
    if (strbuf.get_content_length() < maxToRead) {
      read_buffer_.clear();
      char *raw_read = read_buffer_.get_end_ptr(maxToRead);
      size_t read_size = readRawBytes(raw_read, maxToRead);
      transcoder_->transcode(raw_read, read_size);
    }
    
    unsigned int rest_length = strbuf.get_content_length();
    unsigned int to_read_now = 
      rest_length < maxToRead ? rest_length : maxToRead;
    memcpy(toFill, strbuf.get_ptr(), to_read_now);
    strbuf.erase_by(strbuf.get_ptr() + to_read_now);
    
    pos_ += to_read_now;
    return to_read_now;
  }
}



TranscodingInputSource::TranscodingInputSource(const char *systemId,
					       const char *publicId,
					       transcoder_factory trfact) 
  : transcoder_(0), trfact_(trfact) {
  setSystemId(chars_to_XMLCh(systemId).xstr());
  setPublicId(chars_to_XMLCh(publicId).xstr());
}


TranscodingInputSource::~TranscodingInputSource() {
  for (transcoder_cache::iterator it = trcache_.begin();
       it != trcache_.end(); ++it) {
    delete it->second;
  }
  trcache_.clear();
}


bool TranscodingInputSource::setup_internal(const char *buffer, size_t length) {
  std::string encoding = sense_encoding(buffer, length);
  if (encoding.empty())
    return false;

  std::string key = create_encoding_key(encoding);
  if (trcache_.find(key) != trcache_.end()) {
    transcoder_ = trcache_[key];
  }
  else {
    transcoder_ = stateful_transcoder::create("UTF-16", encoding.c_str(), 
					      false, trfact_);
    trcache_[key] = transcoder_;
  }
  setEncoding(chars_to_XMLCh("UTF-16").xstr());
  return true;
}


namespace {

  class BufferBinInputStream : public TranscodingBinInputStream {
  public:
    BufferBinInputStream(const char *buffer, size_t length,
			 stateful_transcoder *tc) 
      : TranscodingBinInputStream(tc), current_(buffer), end_(buffer + length) { }
    virtual size_t readRawBytes(char *toFill, size_t maxToRead);
  private:
    const char *current_;
    const char *end_;
  };

  unsigned int BufferBinInputStream::readRawBytes(char *toFill, 
						  unsigned int maxToRead) {
    size_t content_length = end_ - current_;
    size_t size_to_read = content_length < maxToRead ? content_length : maxToRead;

    memcpy(toFill, current_, content_length);
    current_ += size_to_read;
    return size_to_read;
  }

}

BufferInputSource::BufferInputSource(const char *systemId, 
				     const char *publicId,
				     transcoder_factory trfact,
				     const char *buffer, size_t length) 
  : TranscodingInputSource(systemId, publicId, trfact), 
    buffer_(buffer), length_(length) {
}

void BufferInputSource::setup() {
  setup_internal(buffer_, length_);
  assert(transcoder_ != 0);
}


xercesc::BinInputStream *BufferInputSource::makeStream() const {
  return new BufferBinInputStream(buffer_, length_, transcoder_);
}

namespace {

  class IStreamBinInputStream : public TranscodingBinInputStream {
  public:
    IStreamBinInputStream(std::istream &istm, stateful_transcoder *tc) 
      : TranscodingBinInputStream(tc), istm_(istm) { }
    virtual size_t readRawBytes(char *toFill, size_t maxToRead);
  private:
    std::istream &istm_;
  };

  size_t IStreamBinInputStream::readRawBytes(char *toFill, 
					     size_t maxToRead) {
    return readsome(istm_, toFill, maxToRead);
  }
  
}



IStreamInputSource::IStreamInputSource(const char *systemId,
				       const char *publicId,
				       transcoder_factory trfact,
				       std::istream &istm) 
  : TranscodingInputSource(systemId, publicId, trfact), 
    istm_(istm) { 
}

void IStreamInputSource::setup() {
  string_buffer buffer;
  while (true) {

    char *tofill = buffer.get_end_ptr(1024);
    size_t read_size = readsome(istm_, tofill, 
				buffer.get_remain_length());
    if (read_size == 0)
      break;
    buffer.commit_additional_length(read_size);
    if (setup_internal(buffer.get_ptr(), buffer.get_content_length()))
      break;
  }
  assert(transcoder_ != 0);
  transcoder_->transcode(buffer.get_ptr(), buffer.get_content_length());
}

xercesc::BinInputStream *IStreamInputSource::makeStream() const {
  return new IStreamBinInputStream(istm_, transcoder_);
}



#endif
