/* -*- c++ -*- */
#ifndef AKAXISO2_XML_SERIALIZER_XML_SERIALIZER_H__
#define AKAXISO2_XML_SERIALIZER_XML_SERIALIZER_H__

/**
 * @file akaxiso2/XML/serializer/xml_serializer.h 
 * @brief Base class of XML serializer 
 */

#include <akaxiso2/XML/serializer/serializer.h>
#include <akaxiso2/framework/operators.h>

namespace aka2 {

  /** 
   * @brief serialization function.
   *
   * Internally aka2::xml_serializer is used.
   * @param doc document to be serialized.
   * @param ostm std::ostream to output serialized XML document.
   */
  void serialize(const document &doc, std::ostream &ostm);


  struct wildcard;
  typedef std::vector<wildcard> wc_array;

  /** 
   * @brief Base class of XML serializer.
   *
   * This class implements XML serialization logic except for encoding-translation.
   * Derived classes are responsible to initialize serializer::formatter_ member.
   */


  /**
   * @brief XML serializer
   *
   * XML serializer by using default transcoder.
   */
  class xml_serializer : public serializer {
  public:
    explicit xml_serializer(const std::string &encoding = std::string(),
			    transcoder_factory create = 0);
    virtual ~xml_serializer();

    /** 
     * Property to use XML-namespace 1.1.
     * @param value true to use XML-namespace 1.1(default). false to dsiable.
     */
    void enable_ns11(bool value) { ns11_enabled_ = value; }

    /**
     * @brief Serialize document instance.
     * @param doc document instance
     * @param ostm std::ostream
     */
    void serialize(const document &doc, std::ostream &ostm);
    
    /**
     * @brief Serialize given root instance to std::ostream.
     * @param root root instance.
     * @param name tagname of root element.
     * @param ostm std::ostream to output serialized XML document.
     */
    template <class R>
    void serialize(const R &root, const std::string &name, std::ostream &ostm) {
      do_serialize(&root, qname(name), ostm);
    }

  private:
    bool ns11_enabled_;

    void do_serialize(const void *e, 
		      const qname &name, std::ostream &ostm);
    void serialize_internal(const void *e, const element_props &props, 
			    const qname &name);

    void write_element(const void *e, const element_props &props, const element_op &op);
    void write_element_entity(const void *e, const element_props &props, const element_op &op);

    void write_sequence(const qname &tagname,
                        const void *e, const sequence_op &sop);
    void write_sequence_entity(const void *e, const sequence_op &sop);

    void write_choice(const qname &tagname, 
                      const void *e, const choice_op &sop);
    void write_choice_entity(const void *e, const choice_op &sop);

    void write_all(const qname &tagname,
                   const void *e, const all_op &sop);
    void write_all_entity(const void *e, const all_op &sop);

    void write_simplecontent(const qname &tagname, 
                             const void *e, const simplecontent_op &sop);
    void write_array(const void *e, const element_props &props, const array_op &aop);
    void write_group_array(const void *e, const element_props &props, const array_op &aop);
    void write_ptrmember(const void *e, const element_props &props);
    void write_ptrmember_entity(const void *e, const element_props &props);

    void write_simpletype(const qname &tagname,
                          const void *e, const simpletype_op &sop);

    void write_fixed_simpletype(const element_props &props);


    void write_nill_element(const qname &tagname,
                            const void *e, const element_op &op);
    void write_beginning_tag(const qname& tagname,
                             const void *e, const element_op &op);
    void write_ending_tag(const qname &tagname);

    void write_attributes(const void *e, const element_op &op);
    void write_namespace_attributes();
    void write_ns(id_type nsid);
    void write_ns11_declaration(const qname &tagname, const element_op &op, const void *e);

    void write_any(const element_props &props, const any &an);
    void write_any_array(const element_props &props, const any_array &ar);
    void write_wildcard(const element_props &props, const wildcard &wc);
    void write_wildcard_array(const element_props &props, const wc_array &ar);
    void write_anyType(const qname &tagname, const wildcard &an);
    void write_anyType_array(const qname &tagname, const wc_array &ar);
    void write_ns_decl_if_new(id_type nsid);


    static bool is_sequence_empty(const void *e, const member_types &mtypes);
    static bool is_all_empty(const void *e, const member_types &mmap);
    static bool is_member_empty(const void *e, const member_type &mtype);
    static bool is_any_empty(const void *e, const member_type &mtype);
    static bool is_fixed_empty(const void *e, const member_type &mtype); 

  };

} // namespace aka2

#endif
