/* -*- c++ -*- */
#ifndef AKAXISO2_XML_SERIALIZER_FORMATTER_H__
#define AKAXISO2_XML_SERIALIZER_FORMATTER_H__

/**
 * @file akaxiso2/XML/serializer/formatter.h 
 * @brief Base class of formatter
 */

#include <akaxiso2/util/iosfwd.h>
#include <akaxiso2/unicode/ustring.h>
#include <akaxiso2/transcoders/stateful_transcoder.h>
#include <akaxiso2/transcoders/pivot_transcoder.h>

namespace aka2 {

  /** 
   * @brief Base class of formatter. 
   *
   * This class implements text-formatter functionality for XML serializer.\n
   * Any derived formatter classes should implement some pure-virtual members 
   * to achive encoding translation. 
   */
  class formatter {
  public:
    formatter(transcoder_factory create);
    virtual ~formatter();

    /**
     * @brief Set std::ostream to output text.
     * @param ostm std::ostream to output text.
     * @param encoding encoding name.
     *
     */
    void prepare(std::ostream &ostm, const std::string &encoding);
    /**
     * @brief Finish serialization.
     */ 
    void finish();

    void write_text_entity(const pstring &entity) {
      write_entity(entity.data(), entity.size(), 1);
    }
    void write_attribute_entity(const pstring &entity) {
      write_entity(entity.data(), entity.size(), 0);
    }

    /**
     * @brief Translate UCS-2 string to output-encoding specified by set_encoding 
     * and output converted string to ostream passed by prepare() method.
     * @param chars UTF-8 string to write.
     * @param length byte length of a given UTF-8 string.
     */
    void write_pchar(const pchar_t ch);
    void write_uni(const pchar_t *chars, size_t length);
    void write_uni(const pchar_t *chars);
    void write_uni(const pstring &str) { write_uni(str.data(), str.size()); }
  private:
    /** 
     * @brief Translating local-code-page to UCS2.
     * @param source source string in LCP to be converted to UCS2.
     * @return converted UCS2 string.
     */
    void write_entity(const pchar_t *ch, size_t length, const int escape_index);
    void set_binary_mode_for_stdio();
    void revert_stream_mode();

    transcoder_factory create_;
    std::ostream *ostm_;
    int saved_stream_mode_;
    int fd_;

    stateful_transcoder *to_out_;
    pstring_buffer escaped_entity_;
  };

} // namespace aka2


#endif
