#include "expat_parser.h"

#ifdef AKAXISO2_USE_EXPAT

#include "../../util/platform.h"
#include "../../util/sstream.h"
#include "../../util/string_funcs.h"
#include "../../util/scoped_ptr.h"
#include "../deserializer/encoding_sensor.h"

#include <fstream>

using namespace aka2;

expat_parser::expat_parser(transcoder_factory trfact) 
  : dochandler_(0), dtdhandler_(0), trfact_(trfact), buffer_size_(1024) { }

void expat_parser::process_attributes(const XML_Char **atts, sax_attributes &attrs) {
  for (const XML_Char **ptr = atts; *ptr != 0; ptr += 2) {
    attrs.push_back(sax_attribute(ptr[0], ptr[1]));
  }
}


void expat_parser::startElement(void *userData,
				const XML_Char *name,
				const XML_Char **atts) {
  sax_attributes saxattrs;
  expat_parser *handler = static_cast<expat_parser*>(userData);
  handler->process_attributes(atts, saxattrs);
  handler->dochandler_->startElement(name, saxattrs);
}


void expat_parser::endElement(void *userData,
			      const XML_Char *name) {

  expat_parser *handler = static_cast<expat_parser*>(userData);
  handler->dochandler_->endElement(name);
}

void expat_parser::characters(void *userData,
			       const XML_Char *data,
			       int len) {
  expat_parser *handler = static_cast<expat_parser*>(userData);
  handler->dochandler_->characters(data, len);
}

int expat_parser::unknownEncodingHandler(void *encodingHandlerData,
					 const XML_Char *name,
					 XML_Encoding *info) {
  static_cast<expat_parser*>(encodingHandlerData)->encoding_ = to_lcp(name);
  return XML_STATUS_OK;
}


void expat_parser::startDoctypeDeclHandler(void *userData,
					   const XML_Char *doctypeName,
					   const XML_Char *sysid,
					   const XML_Char *pubid,
					   int has_internal_subset) {
  expat_parser *parser = static_cast<expat_parser*>(userData);
  parser->dtdhandler_->startDoctypeDeclHandler(doctypeName,
					       sysid, pubid,
					       has_internal_subset);
}

void expat_parser::endDoctypeDeclHandler(void *userData) {
  expat_parser *parser = static_cast<expat_parser*>(userData);
  parser->dtdhandler_->endDoctypeDeclHandler();
}

int expat_parser::externalEntityRefHandler(XML_Parser p,
					   const XML_Char *context,
					   const XML_Char *base,
					   const XML_Char *systemId,
					   const XML_Char *publicId) {

  expat_parser *ep = static_cast<expat_parser*>(XML_GetUserData(p));
  string_buffer buffer(ep->buffer_size_);

  XML_Status res = XML_STATUS_ERROR;

  std::string fullpath;
  if (base != 0)
    fullpath = to_lcp(base);
  fullpath.append(to_lcp(systemId));

  std::ifstream istm(fullpath.c_str());

  if (!istm.is_open()) {
    std::string message = "Failed to open external DTD(" + fullpath + ").";
    throw error(message, __FILE__, __LINE__);
  }

  XML_Parser parser = 
    ep->create_external_entity_parser(p, 
				      to_lcp(context).c_str(), 
				      to_lcp(base).c_str(), 
				      pchar_traits::encoding(),
				      ep->dtdhandler_);
  
  scoped_ptr<stateful_transcoder> 
    tc(stateful_transcoder::create("UTF-8", ep->encoding_, false, ep->trfact_));
  
  try {
    do {
      buffer.clear();
      size_t read_length = readsome(istm, buffer.get_end_ptr(), 
				    buffer.get_remain_length());
      buffer.commit_additional_length(read_length);
      tc->clear_transcoded_buffer();
      tc->transcode(buffer.get_ptr(), buffer.get_content_length());
      
      res = ep->parse(parser, 
		      tc->get_buffer().get_ptr(), tc->get_buffer().get_content_length(), 
		      buffer.empty(), fullpath);
      if (res != XML_STATUS_OK)
	break;
    } while (!buffer.empty());
    
    if (res == XML_STATUS_OK) {
      ep->free_parser(parser);
      return res;
    }
  }
  catch ( ... ) {
    ep->free_parser(parser);
    throw;
  }
  
  positioned_error e = create_positioned_error(parser, fullpath, __FILE__, __LINE__);
  ep->free_parser(parser);
  throw e;

  return XML_STATUS_ERROR;
}


void expat_parser::elementDeclHandler(void *userData, const XML_Char *name, XML_Content *model) {
  expat_parser *ep = static_cast<expat_parser*>(userData);
  ep->dtdhandler_->elementDeclHandler(name, model);
  XML_FreeContentModel(ep->parser_stack_.top(), model);
}

void expat_parser::attlistDeclHandler(void *userData,
				      const XML_Char *elname,
				      const XML_Char *attname,
				      const XML_Char *att_type,
				      const XML_Char *dflt,
				      int isrequired) {
  expat_parser *ep = static_cast<expat_parser*>(userData);
  ep->dtdhandler_->attlistDeclHandler(elname, attname, 
				      att_type, dflt, isrequired);
}


void expat_parser::set_base(const std::string &base) {
  base_ = base;
}

XML_Status expat_parser::parse(XML_Parser p, 
			       const char *doc, int size, int is_final,
			       const std::string &source_name) {
  XML_Status res;
  try {
    res = XML_Parse(p, doc, size, is_final);
  }
  catch (positioned_error &e) {
    set_position(e, p, source_name);
    throw e;
  }
  catch ( const aka2::error &e ) {
    positioned_error pe(e);
    set_position(pe, p, source_name);
    throw;
  }
  catch ( ... ) {
    throw;
  }
  return res;
}


void expat_parser::parse_buffer(const char *buffer, int len,
				const std::string &source_name) {
  
  encoding_ = sense_encoding(buffer, len);
  if (encoding_.empty())
    encoding_ = "UTF-8";

  scoped_ptr<stateful_transcoder> 
    tc(stateful_transcoder::create("UTF-8", encoding_, false, trfact_));

  XML_Parser parser = create_parser("UTF-8", dochandler_, dtdhandler_);
  XML_Status res;
  
  tc->transcode(buffer, len);

  try {
    res = parse(parser, 
		tc->get_buffer().get_ptr(),
		tc->get_buffer().get_content_length(),
		1, source_name);
  }
  catch ( ... ) {
    free_parser(parser);
    throw;
  }

  if (res == XML_STATUS_OK) {
    free_parser(parser);
    return;
  }
  
  positioned_error e = create_positioned_error(parser, source_name, __FILE__, __LINE__);
  free_parser(parser);
  throw e;
}

void expat_parser::parse_istream(std::istream &istm, 
				 const std::string &source_name) {
  
  string_buffer buffer(buffer_size_);
  
  while (encoding_.empty()) {
    size_t read_length = readsome(istm, buffer.get_end_ptr(buffer_size_), 
				  buffer.get_remain_length());
    buffer.commit_additional_length(read_length);
    if (read_length == 0)
      break;
    encoding_ = sense_encoding(buffer.get_ptr(), buffer.get_content_length());
  }
  
  // Try translation with given encoding.
  scoped_ptr<stateful_transcoder> 
    tc(stateful_transcoder::create("UTF-8", encoding_, false, trfact_));
  
  XML_Parser parser = create_parser("UTF-8", dochandler_, dtdhandler_);
  XML_Status res;

  try {
    while (true) {
      tc->clear_transcoded_buffer();
      tc->transcode(buffer.get_ptr(), buffer.get_content_length());
      res = parse(parser, tc->get_buffer().get_ptr(), 
		  tc->get_buffer().get_content_length(),
		  tc->get_buffer().get_content_length() == 0, 
		  source_name);
      if (res != XML_STATUS_OK)
	break;
      if (buffer.empty())
	break;
      buffer.clear();
      size_t read_size = readsome(istm, buffer.get_ptr(), buffer.get_remain_length());
      buffer.commit_additional_length(read_size);
    };
    
    if (res == XML_STATUS_OK) {
      free_parser(parser);
      return;
    }
  }
  catch ( ... ) {
    free_parser(parser);
    throw;
  }

  positioned_error e = create_positioned_error(parser, source_name, __FILE__, __LINE__);
  free_parser(parser);
  throw e;

  assert(!"Must not reach here.");
}


void expat_parser::set_position(positioned_error &e,
				XML_Parser parser,
				const std::string &source_name) {
  int linenum = XML_GetCurrentLineNumber(parser);
  int columnnum = XML_GetCurrentColumnNumber(parser);
  e.set_position(source_name, linenum, columnnum);
}

positioned_error expat_parser::create_positioned_error(XML_Parser parser,
						       const std::string &source_name,
						       const char *file,
						       const unsigned long line) {
  std::ostringstream ostm;

  XML_Error errcode = XML_GetErrorCode(parser);
  const XML_LChar *msg = XML_ErrorString(errcode);
#if defined(XML_UNICODE_WCHAR_T)
  positioned_error e(to_lcp(ucs2_to_pivot(msg)), file, line);
#else
  positioned_error e(msg, file, line);
#endif
  set_position(e, parser, source_name); 

  return e;
}

XML_Parser expat_parser::create_parser(const char *encoding, 
				       expat_document_handler *dochandler,
				       expat_dtd_handler *dtdhandler) {

  pstring_buffer pivot_str;
  to_pivot(pivot_str, encoding);

  XML_Parser parser = XML_ParserCreate(pivot_str.get_ptr());
  XML_SetUserData(parser, this);
  XML_SetUnknownEncodingHandler(parser, &expat_parser::unknownEncodingHandler, this);
  
  if (!base_.empty()) {
    to_pivot(pivot_str, base_);
    XML_SetBase(parser, pivot_str.get_ptr());
  }

  if (dochandler_ != 0) {
    XML_SetStartElementHandler(parser, &expat_parser::startElement);
    XML_SetEndElementHandler(parser, &expat_parser::endElement);
    XML_SetCharacterDataHandler(parser, &expat_parser::characters);
  }
  if (dtdhandler_ != 0) {
    XML_SetParamEntityParsing(parser,
			      XML_PARAM_ENTITY_PARSING_UNLESS_STANDALONE);
    XML_SetDoctypeDeclHandler(parser, 
			      &expat_parser::startDoctypeDeclHandler,
			      &expat_parser::endDoctypeDeclHandler);
    XML_SetExternalEntityRefHandler(parser, 
				    &expat_parser::externalEntityRefHandler);
    XML_SetElementDeclHandler(parser, &expat_parser::elementDeclHandler);
    XML_SetAttlistDeclHandler(parser, &expat_parser::attlistDeclHandler);
  }

  parser_stack_.push(parser);

  return parser;
}

XML_Parser expat_parser::create_external_entity_parser(XML_Parser p,
						       const char *context,
						       const char *base,
						       const char *encoding,
						       expat_dtd_handler *dtdh) {

  pstring_buffer context_buf, encoding_buf;
  to_pivot(context_buf, context);
  to_pivot(encoding_buf, encoding);
  XML_Parser parser = XML_ExternalEntityParserCreate(p, 
						     context_buf.get_ptr(),
						     encoding_buf.get_ptr());

  parser_stack_.push(parser);

  XML_SetUserData(parser, this);
  XML_SetUnknownEncodingHandler(parser, &expat_parser::unknownEncodingHandler, this);
  if (base != 0) {
    pstring_buffer base_buf;
    to_pivot(base_buf, base);
    XML_SetBase(parser, base_buf.get_ptr());
  }

  if (dtdhandler_ != 0) {
    XML_SetParamEntityParsing(parser,
			      XML_PARAM_ENTITY_PARSING_UNLESS_STANDALONE);
    XML_SetDoctypeDeclHandler(parser, 
			      &expat_parser::startDoctypeDeclHandler,
			      &expat_parser::endDoctypeDeclHandler);
    XML_SetExternalEntityRefHandler(parser, 
				    &expat_parser::externalEntityRefHandler);
    XML_SetElementDeclHandler(parser, &expat_parser::elementDeclHandler);
    XML_SetAttlistDeclHandler(parser, &expat_parser::attlistDeclHandler);
  }
  return parser;

}


void expat_parser::free_parser(XML_Parser parser) {
  assert(!parser_stack_.empty());
  assert(parser_stack_.top() == parser);
  parser_stack_.pop();
  XML_ParserFree(parser);
}

#endif
