#include "sax2_adaptor.h"
#include "../../framework/document.h"
#include "../../framework/namespace_statics.h"
#include "document_handler.h"
#include "../../util/string_funcs.h"
#include "../../unicode/utf8_traits.h"
#include "../../framework/literal.h"

using namespace aka2;

#ifdef _MSC_VER
#include <malloc.h>
#endif


void sax2_adaptor::create_qname(const pstring &tagname) {
  if (pchar_traits::find_first(tagname.data(), tagname.size(), ':') == -1) {
    tagname_ = qname(pivot, ns_.get_current_decls().default_ns_id_, tagname);
  }
  else {
    tagname_ = qname(pivot, tagname, *pfs_);
    if (tagname_.get_namespace_id() == ns_.get_current_decls().default_ns_id_) {
      throw positioned_error("Prefixed tagname " + tag(to_lcp(tagname)) 
			     + " found though namespace defaulting is applied.",
			     __FILE__, __LINE__);
    }
  }
}



void ns_stack::inc_depth() {
  ns_decls decls;
  if (!stack_.empty())
    decls.default_ns_id_ = stack_.top().default_ns_id_;
  stack_.push(decls);
}

void ns_stack::dec_depth() {
  stack_.pop();
}

const ns_decls &ns_stack::get_current_decls() const {
  return stack_.top();
}

void ns_stack::set_default_ns_id(id_type id) {
  stack_.top().default_ns_id_ = id; 
}


bool ns_stack::process_namespace_attribute(const sax_attribute &attr, const nsdecl **decl) {

  if (pchar_traits::compare(literal::xmlns(), attr.name_.data(), 5) != 0)
    return false;

  pstring attrname = attr.name_;
  pstring uri = attr.value_;

  if ((uri.at(0) == '"') && (uri.at(uri.size() - 1) == '"')){
    uri.erase(0, 1);                // remove ""
    uri.erase(uri.length() - 1, 1);
  }

  nsdecl newentry;
  newentry.uri_ = uri;
  if (attrname.size() == 5) { // length of "xmlns"
    newentry.prefix_.erase(newentry.prefix_.size());
  }
  else if (pchar_traits::compare(attrname.data(), literal::xmlns_colon(), 6) == 0) {
    newentry.prefix_ = attrname.substr(6);
    if (newentry.prefix_.empty()) {
      std::string message = "xmlns:=" + quote(to_lcp(uri)) 
	+ ":namespace prefix is not specified.";
      throw error(message, __FILE__, __LINE__);
    }
  }

  stack_.top().prefixes_.push_back(newentry);
  *decl = &stack_.top().prefixes_.back();
  return true;

}


sax2_adaptor::sax2_adaptor() : handler_(0) { }


void sax2_adaptor::process_attributes(const sax_attributes &saxattrs, attribute_values &xattrs) {

  std::vector<size_t> non_ns_attrs;
  size_t i;

  size_t attrsize = saxattrs.size();
  for (i = 0; i < attrsize; ++i) {
    const nsdecl *decl = 0; 
    const sax_attribute &attr = saxattrs[i];
    if (ns_.process_namespace_attribute(attr, &decl)) {
      if (decl != 0)
	handler_->start_prefix_mapping(decl->prefix_, decl->uri_);
      if (decl->prefix_.empty()) // prefix is "", set default namespace id.
	ns_.set_default_ns_id(get_namespace_id(pivot, decl->uri_));
    }
    else {
      non_ns_attrs.push_back(i);
    }
  }
  
  attrsize = non_ns_attrs.size();
  for (i = 0; i < attrsize; ++i) {
    const sax_attribute &attr = saxattrs[non_ns_attrs[i]];
    qname name(pivot, attr.name_, *pfs_);
    std::pair<attribute_values::iterator, bool> res = 
      xattrs.insert(attribute_values::value_type(name, attr.value_));
    if (!res.second){
      throw tagged_error("attribute", name.qualified(), "appeared more than twice.",
			 __FILE__, __LINE__);
    }
  }
}


void sax2_adaptor::startElement(const pstring &tagname, sax_attributes &saxattrs) {

  attribute_values xattrs;
  ns_.inc_depth();

  process_attributes(saxattrs, xattrs);
  create_qname(tagname);
  handler_->startElement(tagname_, xattrs);
}


void sax2_adaptor::characters(const pchar_t *chars, const unsigned int length) {
  handler_->characters(chars, length);
}


void sax2_adaptor::endElement(const pstring &tagname) {

  create_qname(tagname);
  handler_->endElement(tagname_);

  const ns_decls &nsdecls = ns_.get_current_decls();
  const ns_decls::prefixes &prfs = nsdecls.prefixes_;
  size_t prfs_size = prfs.size();
  for (size_t index = 0; index < prfs_size; ++index)
    handler_->end_prefix_mapping(prfs[index].prefix_);
  ns_.dec_depth();
}

void sax2_adaptor::set_handler(document_handler &handler) {
  handler_ = &handler;
  pfs_ = &handler.get_entity_complements().get_prefixes();
}

